/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: Resources.cc,v 1.8 2003/11/12 06:04:04 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <sys/types.h>
#include <dirent.h>

/*
 * this is workaround, by some reason dirent.h defines DIR as a typedef
 * for struct __dirstream which is not defined anywhere.  07/12/02  vk
 */
#ifndef __dirstream
struct __dirstream {};
#endif

#include <sys/stat.h>
#ifndef __MINGW32__
#include <pwd.h>
#endif
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>

#include <iostream>
#include <fstream>
#include <cstdlib>

#include <fwbuilder/libfwbuilder-config.h>

#include "fwbuilder/Resources.hh"
#include "fwbuilder/Constants.hh"
#include "fwbuilder/FWObject.hh"
#include "fwbuilder/FWObjectDatabase.hh"
#include "fwbuilder/RuleElement.hh"
#include "fwbuilder/FWOptions.hh"
#include "fwbuilder/Firewall.hh"
#include "fwbuilder/Host.hh"
#include "fwbuilder/Tools.hh"

using namespace libfwbuilder;
using namespace std;

const string Resources::PLATFORM_RES_DIR_NAME = "platform";
const string Resources::OS_RES_DIR_NAME = "os";

Resources*  Resources::global_res = NULL;
map<string,Resources*>  Resources::platform_res;
map<string,Resources*>  Resources::os_res;


Resources::Resources() throw(FWException)
{
    doc=NULL;
}

Resources::Resources(const string &_resF) throw(FWException)
{
    doc=NULL;
    resfile=_resF;

    if (global_res==NULL) 
    {
        global_res=this;
        loadRes(_resF);
        loadSystemResources();
    } else
        loadRes(_resF);
}

string Resources::getXmlNodeContent(xmlNodePtr node)
{
    string res;
    char* cptr= (char*)( xmlNodeGetContent(node) );
    if (cptr!=NULL) 
    {
        res=cptr;
        free(cptr);
    }
    return res;
}

string Resources::getXmlNodeProp(xmlNodePtr node,string prop)
{
    string res;
    char* cptr=(char*)( xmlGetProp(node,TOXMLCAST(prop.c_str())));
    if (cptr!=NULL) 
    {
        res=cptr;
        free(cptr);
    }
    return res;
}

void Resources::loadRes(const std::string &rfile ) throw(FWException)
{

//    doc = xmlParseFile(rfile.c_str()); 
    doc = XMLTools::loadAndParseFile(rfile);

    if(!doc)   throw FWException("Error parsing "+rfile);

    root=xmlDocGetRootElement(doc);
   
    if(!root || !root->name || strcmp(FROMXMLCAST(root->name), "FWBuilderResources")!=0) 
    {
        xmlFreeDoc(doc);
        throw FWException("Invalid resources file "+rfile);
    }
}

void Resources::loadSystemResources() throw(FWException)
{
/*
 * Find and open resources for individual firewall platforms and OS.
 * If corresponding directory does not exist or is empty, then no
 * target platform support package is installed. So be it, not my
 * problem
 */
//    string resDir=Constants::getTemplateDirectory();

    string::size_type n=resfile.rfind("/");
    string resDir = resfile.substr(0,n);

    string platform_resources_dir= resDir + "/" +PLATFORM_RES_DIR_NAME;

    DIR *d=opendir(platform_resources_dir.c_str());
    if (d!=NULL)
    {
        struct dirent *de;
        while ( (de=readdir(d))!=NULL ) 
        {
            if (strcmp(de->d_name,".")==SAME || strcmp(de->d_name,"..")==SAME)
                continue;

            string pfile=de->d_name;
            string rfile=platform_resources_dir+"/"+pfile;
            string platform=pfile.substr(0, pfile.find(".xml"));

            Resources *tr=new Resources(rfile);
            platform_res[platform]=tr;
        }
    }
    closedir(d);

    string os_resources_dir= resDir + "/" +OS_RES_DIR_NAME;

    d=opendir(os_resources_dir.c_str());
    if (d!=NULL)
    {
        struct dirent *de;
        while ( (de=readdir(d))!=NULL ) 
        {
            if (strcmp(de->d_name,".")==SAME || strcmp(de->d_name,"..")==SAME)
                continue;

            string pfile=de->d_name;
            string rfile=os_resources_dir+"/"+de->d_name;
            string os=pfile.substr(0, pfile.find(".xml"));

            Resources *tr=new Resources(rfile);
            os_res[os]=tr;
        }
    }
    closedir(d);

#if 0
    cerr << "Loaded resources for the following modules :\n";

    map<string,string> p=getPlatforms();
    map<string,string>::iterator i1;
    for (i1=p.begin(); i1!=p.end(); ++i1)
    {
        cerr << (*i1).first << "  " << (*i1).second << endl;
    }

    p=getOS();
    for (i1=p.begin(); i1!=p.end(); ++i1)
    {
        cerr << (*i1).first << "  " << (*i1).second << endl;
    }
#endif
}

xmlNodePtr Resources::getXmlNode(const string& path)
{
    return XMLTools::getXmlNodeByPath(root,path);
}


string  Resources::getIconPath(const char* icon)
{
    string icn;

    icn= getResourceStr("/FWBuilderResources/Paths/Icndir");
    icn += "/";
    icn += getResourceStr(string("/FWBuilderResources/UI/Icons/")+icon);

    return icn;
}

string  Resources::getIconPath(const string& icon)
{
    return getIconPath(icon.c_str());
}

string  Resources::getResourceStr(const string& resource_path)
{
    xmlNodePtr node=XMLTools::getXmlNodeByPath(root,resource_path.c_str());
    if (node)  return getXmlNodeContent(node);
    return "";
}

int     Resources::getResourceInt(const string& resource_path)
{
    return atoi(getResourceStr(resource_path).c_str());
}

bool    Resources::getResourceBool(const string& resource_path)
{
    string res=getResourceStr(resource_path);
    return (res=="true" || res=="True");
}

string  Resources::getObjResourceStr(const FWObject *obj,
					  const string& resource_name)
{
    string objid   = obj->getId();
    string objtype = obj->getTypeName();
    string res;
    string resource="/FWBuilderResources/ObjectResources/"+objid+"/"+resource_name;
    res= getResourceStr(resource);
    if (res.empty()) 
    {
        string resource="/FWBuilderResources/TypeResources/"+objtype+"/"+resource_name;
        res= getResourceStr(resource);
        if (res.empty()) 
        {
            resource="/FWBuilderResources/TypeResources/DEFAULT/"+resource_name;
            res= getResourceStr(resource);
        }
    }

    if (res.empty())
        cerr << "Failed to locate resource for object " << obj->getName() 
             << " (" << obj->getId() << "), resource="
             << resource_name << endl;

    return res;
}

bool  Resources::getObjResourceBool(const FWObject *obj,
				    const string&  resource_name)
{
    string res=getObjResourceStr(obj,resource_name);
    return (res=="true" || res=="True");
}

/*
 *  call this using appropriate Resources object, like this:
 *  
 *  platform_res["iptables"]->getVersion()
 */
string  Resources::getVersion()
{
    xmlNodePtr pn=getXmlNode("/FWBuilderResources/");
    return getXmlNodeProp(pn,"version");
}

/*
 *  call this using appropriate Resources object, like this:
 *  
 *  platform_res["iptables"]->getCompiler()
 */
string  Resources::getCompiler()
{
    return getResourceStr("/FWBuilderResources/Target/compiler");
}

vector<string>    Resources::getListOfPlatforms()
{
    vector<string> vs;
    for (map<string,Resources*>::iterator i1=platform_res.begin();
         i1!=platform_res.end(); ++i1)
        vs.push_back( (*i1).first );
    return vs;
}



map<string,string>    Resources::getPlatforms()
{
    map<string,string> vs;
    for (map<string,Resources*>::iterator i1=platform_res.begin();
         i1!=platform_res.end(); ++i1)
    {
        string desc=(*i1).second->getResourceStr("/FWBuilderResources/Target/description");
        vs[ (*i1).first ]=desc;
    }
    return vs;
}


map<string,string>    Resources::getOS()
{
    map<string,string> vs;
    for (map<string,Resources*>::iterator i1=os_res.begin(); i1!=os_res.end(); ++i1)
    {
        string desc=(*i1).second->getResourceStr("/FWBuilderResources/Target/description");
        vs[ (*i1).first ]=desc;
    }
    return vs;
}


string Resources::getRuleElementResourceStr(const string &rel,
                                            const string &resource_name)

{
    xmlNodePtr  c,d;

    xmlNodePtr  dptr=Resources::global_res->getXmlNode("FWBuilderResources/RuleElements");

    assert (dptr!=NULL);

    for(c=dptr->xmlChildrenNode; c; c=c->next) 
    {
	if ( xmlIsBlankNode(c) ) continue;
        if (rel==getXmlNodeProp(c,"RuleElement")) 
        {
            d=XMLTools::getXmlChildNode(c,resource_name.c_str());
            if (d) 
            {
                return getXmlNodeContent(d);
            }
	}
    }
    return string("");
}

bool Resources::isSystem(const FWObject *o)
{
    return global_res->getObjResourceBool(o, "system");
}

string  Resources::getIconFileName(const FWObject *o)
{ 
    string res;
    
    res=global_res->getResourceStr("/FWBuilderResources/Paths/Icndir");
    res += "/";
    res += global_res->getObjResourceStr(o, "icon");

    return res;
}

string  Resources::getNegIconFileName(const FWObject *o)
{ 
    string res;
    
    res=global_res->getResourceStr("/FWBuilderResources/Paths/Icndir");
    res += "/";
    res += global_res->getObjResourceStr(o, "icon-neg");
    
    return res;
}

string  Resources::getRefIconFileName(const FWObject *o)
{ 
    string res;
    
    res=global_res->getResourceStr("/FWBuilderResources/Paths/Icndir");
    res += "/";
    res += global_res->getObjResourceStr(o, "icon-ref");
    
    return res;
}

string  Resources::getTreeIconFileName(const FWObject *o)
{ 
    string res;
    
    res=global_res->getResourceStr("/FWBuilderResources/Paths/Icndir");
    res += "/";
    res += global_res->getObjResourceStr(o, "icon-tree");
    
    return res;
}

void    Resources::setDefaultOption(FWObject *o,const string &xml_node)
{
    xmlNodePtr pn = XMLTools::getXmlNodeByPath(root,xml_node.c_str());
    if (pn==NULL) return;

    string optname=FROMXMLCAST(pn->name);
    string optval =getXmlNodeContent(pn);
    o->setStr(optname , optval);
}

void    Resources::setDefaultOptionsAll(FWObject *o,const string &xml_node)
{
    xmlNodePtr pn = XMLTools::getXmlNodeByPath(root , xml_node.c_str() );
    if (pn==NULL) return;

    xmlNodePtr opt;

    for(opt=pn->xmlChildrenNode; opt; opt=opt->next) 
    {
        if ( xmlIsBlankNode(opt) ) continue;
        setDefaultOption(o,xml_node+"/"+FROMXMLCAST(opt->name));
    }
}


void    Resources::setDefaultTargetOptions(const string &target,Firewall *fw)  throw (FWException)
{
    FWOptions *opt=fw->getOptionsObject();
    Resources *r=NULL;

    if (platform_res.count(target)!=0)      r=platform_res[target];
    if (r==NULL && os_res.count(target)!=0) r=os_res[target];
    if (r==NULL)
        throw FWException("Support module for target '"+target+"' is not available");

    r->setDefaultOptionsAll(opt,"/FWBuilderResources/Target/options");
}

void    Resources::setDefaultOptions(Host *h)
{
    FWOptions *opt=h->getOptionsObject();

    global_res->setDefaultOptionsAll(opt,
      "/FWBuilderResources/TypeResources/"+h->getTypeName()+"/options");
}

void    Resources::setDefaultProperties(FWObject *obj)
{
    global_res->setDefaultOptionsAll(obj,
      "/FWBuilderResources/TypeResources/"+obj->getTypeName()+"/properties");
}

string Resources::getTargetCapabilityStr(const string &target,
                                         const string &cap_name)  throw (FWException)
{
    Resources *r=NULL;

    if (platform_res.count(target)!=0)      r=platform_res[target];
    if (r==NULL && os_res.count(target)!=0) r=os_res[target];
    if (r==NULL)
        throw FWException("Support module for target '"+target+"' is not available");

    return r->getResourceStr("/FWBuilderResources/Target/capabilities/"+cap_name);
}

bool Resources::getTargetCapabilityBool(const string &target,
                                        const string &cap_name)  throw (FWException)
{
    string s=getTargetCapabilityStr(target,cap_name);
    return (s=="true" || s=="True");
}

string Resources::getTargetOptionStr(const string &target,
                                     const string &opt_name)  throw (FWException)
{
    Resources *r=NULL;

    if (platform_res.count(target)!=0)      r=platform_res[target];
    if (r==NULL && os_res.count(target)!=0) r=os_res[target];
    if (r==NULL)
        throw FWException("Support module for target '"+target+"' is not available");

    return r->getResourceStr("/FWBuilderResources/Target/options/"+opt_name);
}

bool  Resources::getTargetOptionBool(const string &target,
                                     const string &opt_name)  throw (FWException)
{
    string s=getTargetOptionStr(target,opt_name);
    return (s=="true" || s=="True");
}


