/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: FWObject.cc,v 1.32 2003/03/02 21:47:29 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <fwbuilder/Tools.hh>
#include <fwbuilder/XMLTools.hh>
#include <fwbuilder/FWObject.hh>
#include <fwbuilder/FWObjectDatabase.hh>
#include <fwbuilder/FWObjectReference.hh>

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>

#include <iostream>
#include <algorithm>
#include <functional>
#include <stack>

#include <assert.h>

using namespace std;
using namespace libfwbuilder;

const char *FWObject::TYPENAME={"UNDEF"};
string FWObject::NOT_FOUND="";


void FWObject::fromXML(xmlNodePtr root) throw(FWException)
{
    assert(root!=NULL);

    const char *n;

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("name")));
    if(n)
    {
        setName(n);
        FREEXMLBUFF(n);
    }

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("id")));
    if(n)
    {
        setId(n);
        FREEXMLBUFF(n);
    }

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("comment")));
    if(n)
    {
        setComment(XMLTools::unquote_linefeeds(n));
        FREEXMLBUFF(n);
    }

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("library")));
    if(n)
    {
        setLibrary(n);
        FREEXMLBUFF(n);
    }

    ref_counter=0;

    for(xmlNodePtr cur=root->xmlChildrenNode; cur; cur=cur->next) 
    {
        if(cur && !xmlIsBlankNode(cur))  
        {
            FWObject *o=FWObjectDatabase::db->createFromXML(cur);
            if(o!=NULL) 
            {
		add(o);
                try
                {
                    o->fromXML(cur);
                } catch(FWException &ex)
                {
                    map<string, string> &properties = ex.getProperties();
                    if(properties.find("failed_element")==properties.end())
                        properties["failed_element"]=o->getTypeName();
                    throw;
                }
	    }
        }
    }
    
    setDirty(false);
}

xmlNodePtr FWObject::toXML(xmlNodePtr xml_parent_node) throw(FWException)
{
    return toXML(xml_parent_node, true);
}

xmlNodePtr FWObject::toXML(xmlNodePtr parent, bool process_children) throw(FWException)
{
    xmlNodePtr me = xmlNewChild(parent, NULL, xml_name.empty()?STRTOXMLCAST(getTypeName()):STRTOXMLCAST(xml_name), NULL);

    for(map<string, string>::const_iterator i=data.begin(); i!=data.end(); ++i) 
    {
        const string &name  = (*i).first;
        const string &value = (*i).second;

        xmlAttrPtr pr = xmlNewProp(me, 
                                   STRTOXMLCAST(name) , 
                                   STRTOXMLCAST((name=="comment"?XMLTools::quote_linefeeds(value):value)));
        
        if(name=="id")
        {
            xmlAddID(NULL, parent->doc, STRTOXMLCAST(value), pr);
        } else if(name=="ref") 
        {
            xmlAddRef(NULL, parent->doc, STRTOXMLCAST(value), pr);
        }
    }

    if(process_children)
        for(list<FWObject*>::const_iterator j=begin(); j!=end(); ++j) 
            (*j)->toXML(me);
    
    return me;
}


FWObject::FWObject()
{
    setName("New object");
    
    ref_counter = 0;
    parent      = NULL;

    // When object created we assign it unique Id
    setId(FWObjectDatabase::db->generateUniqueId());

    setDirty(false);
}

FWObject::FWObject(const FWObject &c)
{
    *this=c;
}


/**
 * we do not need to do anything with tree root in FWObject, but
 * derived classes might want to use it to create children objects and
 * add references etc.
 */
FWObject::FWObject(const FWObject *root)
{
    setName("New object");
    
    ref_counter = 0    ;
    parent      = NULL ;
    
    // When object created we assign it unique Id
    setId(FWObjectDatabase::db->generateUniqueId());

    setDirty(false);
}

FWObject::~FWObject() 
{
    clearChildren();
    data.clear();
}

FWObject* FWObject::getParent() const
{
    return parent;
}

void FWObject::setParent(FWObject *p)
{
    parent=p;
}

void FWObject::setXMLName(const string &n)
{
    xml_name = n;
}

FWObject* FWObject::_find(const string& name) const
{
    const_iterator i=find_if(begin(),end(), FWObjectNameEQPredicate(name));
    return i==end()?NULL:(*i);
}

FWObject& FWObject::operator=(const FWObject &x)  throw(FWException)
{
    return duplicate(&x, FALSE);
}

FWObject& FWObject::duplicate(const FWObject *x, bool preserve_id) throw(FWException)
{
    shallowDuplicate(x,preserve_id);

    destroyChildren();

    for(list<FWObject*>::const_iterator m=x->begin(); m!=x->end(); ++m) 
    {
        FWObject *o=*m;
        FWObject *o1=FWObjectDatabase::db->create(o->getTypeName());
        if(!o1)
            throw FWException(string("Error creating object with type: ")+o->getTypeName());
        o1->duplicate(o, preserve_id);
        add(o1);
    }

    setDirty(true);
    return *this;
}

FWObject& FWObject::shallowDuplicate(const FWObject *x, bool preserve_id) throw(FWException)
{
    string id = getId();

    data = x->data;

    if(!preserve_id)
    {
        ref_counter = 0           ;
        xml_name    = x->xml_name ;
    } else if(id!="")   // some objects do not have ID per DTD (e.g. Src, Dst, etc.)
	setId(id);

    setDirty(true);
    return *this;
}

const string &FWObject::getName() const 
{ 
    return getStr("name"); 
}

void FWObject::setName(const string &n)   
{
    setStr("name",n);
    setDirty(true);
}

const string &FWObject::getLibrary() const
{
    return getStr("library"); 
}

void FWObject::setLibrary(const string& c)
{
    setStr("library",c);
    setDirty(true);
}

const string &FWObject::getComment() const
{ 
    return getStr("comment"); 
}

void FWObject::setComment(const string &c)
{
    setStr("comment",c);
    setDirty(true);
}

const string &FWObject::getId() const
{ 
    return getStr("id");
}

void FWObject::setId(const string &c)
{
    setStr("id",c);
    setDirty(true);
}

bool FWObject::exists(const string &name) const 
{
    return data.count(name)!=0; 
}

const string &FWObject::getStr(const string &name) const
{
    if(exists(name)) 
    {
        map<string,string>::const_iterator i=data.find(name);
        return (*i).second;
    } else
    {
        return NOT_FOUND;
    }
}

void FWObject::remStr(const string &name)
{
    if(exists(name)) 
    {
	map<string, string>::iterator m=data.find(name);
	if(m != data.end()) 
        {
	    data.erase(m);
	    setDirty(true);
	}
    }
}

void FWObject::setStr(const string &name, const string &val)
{
    data[name]=val;
    setDirty(true);
}

int FWObject::getInt(const string &name) const
{
    string s=getStr(name);
    if(s!="") 
        return( atol(s.c_str()) );
    else   
        return(-1);
}

void FWObject::setInt(const string &name, int val)
{
    gchar str[128];
    sprintf(str,"%d",val);
    setStr(name, str);
    setDirty(true);
}

bool FWObject::getBool(const string &name) const
{
    if(exists(name)) 
    {
	return(getStr(name)=="1" || 
               strcasecmp(getStr(name).c_str() , "true")==0);
    } else
        return false;
}

void FWObject::setBool(const string &name, bool val)
{
    setStr(name, (val)?"True":"False");
    setDirty(true);
}

void FWObject::setBool(const string &name, const string &val)
{
    if(!name.empty())
	setBool(name,
		(val=="1" || strcasecmp(val.c_str(),"true")==0)); 
}

void FWObject::Show()
{
    setBool("read",true);
}

void FWObject::Hide()
{
    setBool("read",false);
}



void FWObject::dump(bool recursive,bool brief,int offset)
{
    dump(cerr,recursive,brief,offset);
}

void   FWObject::dump(std::ostream &f,bool recursive,bool brief,int offset)
{
    FWObject *o;
    string    n;

    if (brief) {
	f << string(offset,' ');
	f << " Obj=" << this;
	f << " ID="  << getId();
	f << " Name=" << getName();
	f << " Type=" << getTypeName();
	f << " Library=" << getLibrary();

	if (FWReference::cast(this)!=0) {
	    f << " Ref=" << FWReference::cast(this)->getPointer();
	}

	f << endl;

	if (recursive) {
	    list<FWObject*>::iterator m;
	    for (m=begin(); m!=end(); ++m) {
		if (  (o=(*m))!=NULL)  o->dump(f,recursive,brief,offset+2);
	    }
	}
    } else {

	f << string(offset,' ') << string(16,'-') << endl;
	f << string(offset,' ') << "Obj:    " << this << endl;
	f << string(offset,' ') << "ID:     " << getId() << endl;
	f << string(offset,' ') << "Name:   " << getName() << endl;
	f << string(offset,' ') << "Ref.ctr:" << ref_counter << endl;
	f << string(offset,' ') << "Type:   " << getTypeName() << endl;
	f << string(offset,' ') << "Library:" << getLibrary() << endl;
//    f << string(offset,' ') << "Path:   " << getPath() << endl;
	n=(getParent()!=NULL)?getParent()->getName():"";
	f << string(offset,' ') << "Parent: " << getParent()
	  << "  name=" << n << endl;
	f << string(offset,' ') << "Root:   " << getRoot() << endl;

	map<string, string>::iterator d;
	for (d=data.begin(); d!=data.end(); ++d) {
	    if((*d).first=="name") 
                continue;
	    f << string(offset,' ');
	    f << (*d).first << ": " << (*d).second << endl;
	}
	if (recursive) {
	    list<FWObject*>::iterator m;
	    for (m=begin(); m!=end(); ++m) {
		if (  (o=(*m))!=NULL)  o->dump(f,recursive,brief,offset+2);
	    }
	}
    }

}

void FWObject::_adopt(FWObject *obj)
{
    obj->ref();
    obj->setParent(this);
}

void FWObject::addAt(const string& where_id, FWObject *obj)
{
    FWObject *p=getById( where_id , true );
    assert (p!=NULL);
    p->add(obj);
}

void FWObject::add(FWObject *obj,bool validate)
{
    if(!validate || validateChild(obj)) 
    {
	push_back(obj);
	_adopt(obj);
	setDirty(true);
    }
}

FWReference* FWObject::createRef()
{
    FWObjectReference *ref=new FWObjectReference();
    ref->setPointer(this);
    return ref;
}


void FWObject::addRef(FWObject *obj)
{
    if(validateChild(obj)) 
    {
	FWReference *oref = obj->createRef();
	obj->ref();

	push_back(oref);
	_adopt(oref);
	setDirty(true);
    }
}

void FWObject::insert_before(FWObject *o1, FWObject *obj)
{
    if(!obj) 
        return;
    
    list<FWObject*>::iterator m=find(begin(),end(),o1);
    if(m!=end())
    {
        insert(m, obj);
        _adopt(obj);
        setDirty(true);
    }
}

void FWObject::insert_after(FWObject *o1, FWObject *obj)
{
    if(!obj) 
        return;

    list<FWObject*>::iterator m=find(begin(),end(),o1);
    if(m!=end())
    {
        insert(++m,obj);
        _adopt(obj);
        setDirty(true);
    }
}

void FWObject::swapObjects(FWObject *o1, FWObject *o2)
{
    for(list<FWObject*>::iterator m=begin(); m!=end(); ++m) 
    {
        if(*m==o1) 
        {
            *m=o2;
        } else if(*m==o2)
        {
            *m=o1;
        }
    }
    setDirty(true);
}

void FWObject::remove(FWObject *obj, bool delete_if_last)
{
    FWObject::iterator fi=std::find(begin(), end(), obj);
    if(fi!=end())
    {
        erase(fi);
        setDirty(true);
        obj->unref();
        if(delete_if_last && !obj->ref_counter)
            delete obj;
    }
}

class RemoveChild 
{
    FWObject *r;

    public:

    RemoveChild(FWObject *_o) { r=_o; }
    
    void operator()(FWObject *obj) 
    {
	std::for_each(obj->begin(), obj->end(), RemoveChild(r));
        obj->remove(r, false);
    }
};

void FWObject::removeAllInstances(FWObject *rm)
{
    removeAllReferences(rm);
    (RemoveChild(rm))(this);
    
    if(!rm->ref_counter) 
        delete rm;
}

void FWObject::findAllReferences(const FWObject *obj, std::set<FWReference*> &res)
{
    string obj_id=obj->getId();
    for(list<FWObject*>::iterator m=begin(); m!=end(); ++m) 
    {
        FWObject *o=*m;
        FWReference *oref=FWReference::cast(o);
        if(oref)
        {
	    if(oref->getPointerId()==obj_id) 
                res.insert(oref);
	} else
        {
            o->findAllReferences(obj, res);
        }
    }
}

set<FWReference*> FWObject::findAllReferences(const FWObject *obj)
{
    set<FWReference*> res;
    findAllReferences(obj, res);
    return res;
}

void FWObject::removeRef(FWObject *obj)
{
    string obj_id=obj->getId();
    for(list<FWObject*>::iterator m=begin(); m!=end(); ++m) 
    {
        FWObject *o=*m;
        if(FWReference::cast(o)!=NULL)
        {
	    FWReference *oref=FWReference::cast(o);
	    if(oref && oref->getPointerId()==obj_id)
            {
                // do not delete object even if this reference was the last one (?)
		obj->unref();  
                // remove will delete o because reference counter goes to zero
		FWObject::remove(o); 
		return;
	    }
	}
    }
}

class RemoveReference {
    FWObject *r;

    public:
    RemoveReference(FWObject *_o) { r=_o; }
    void operator()(FWObject *obj) {
	std::for_each(obj->begin(), obj->end(), RemoveReference(r) );
	obj->removeRef(r);
    }
};

void FWObject::removeAllReferences(FWObject *rm)
{
    (RemoveReference(rm))(this);
//    std::for_each(begin(), end(), RemoveReference(obj) );
}

bool FWObject::validateChild(FWObject *obj)
{ 
    return true;

    /*
     *  Check if object "this" is a descendant of object "obj" to avoid loops
     *
     *  check disabled for now since we need to be able to add firewall to its
     *  own policy
     */
    FWObject *p;
    p=this;
    do {
	if (p==obj) return false;
    } while ((p=p->getParent())!=NULL);
    return true;
}

void FWObject::destroyChildren()
{
    for(list<FWObject*>::iterator m=begin(); m!=end(); ++m) 
    {
        FWObject *o=*m;
        o->destroyChildren();
        delete o;
    }
    clear();
}

/*
 *  even if I run this method with flag recursive=true, it does not
 *  guarantee that there will be no objects lost in the memory. If
 *  some children of this are referenced from somewhere else, they
 *  won't be deleted because their reference counter is >1. This is
 *  bad because it leads to a situation when object is not a part of
 *  the tree anymore, but reference to it does exist.
 *
 *  If this method is called with recursive=false, then it deletes
 *  only immediate children of this, leaving their children hanging in
 *  the memory. TODO: research whether we ever need to call it with
 *  recursive=false
 *
 *  In other words, this method leaves tree in inconsistent state. At
 *  this time I am just using it carefully, only when I copy objects
 *  between main tree and scratch pad, and when I create copies of
 *  objects. In both cases children of this will be immediately
 *  restored after call to clearChildren.
 *
 *                                              05/08/02 vk
 */
void FWObject::clearChildren(bool recursive)
{
    for(list<FWObject*>::iterator m=begin(); m!=end(); ++m) 
    {
        FWObject *o=*m;
        if (recursive) o->clearChildren(recursive);
        o->unref();
        if(o->ref_counter==0) 
            delete o;
    }
    clear();
    setDirty(true);
}

int FWObject::getChildrenCount()
{
    return(size());
}

FWObject* FWObject::getRoot() const
{
    const FWObject *p=this;

    while(p->getParent()!=NULL) p=p->getParent();
    return (FWObject*)p;
}

/*
 * returns true if this is either direct child of obj, or a child
 * of one of its children and so on.
 */
bool FWObject::isChildOf(FWObject *obj)
{
    if (this==obj) return false;
    FWObject *p=this;
    while (p!=NULL && p!=obj) p=p->getParent();
    return (p==obj);
}

FWObject* FWObject::getById  (const string &id, bool recursive)
{
    if(id==getId())  return this;
    
    list<FWObject*>::iterator j;
    for(j=begin(); j!=end(); ++j)     
    {
        FWObject *o=*j;
        string oid=o->getId();
        if(id==oid) return o;

        if(recursive && (o=o->getById(id, true))!=NULL ) return o;
    }
    return NULL; // not found
}


FWObject* FWObject::getFirstByType(const string &type_name) const
{
    const_iterator i=find_if(begin(),end(), FWObjectTypeNameEQPredicate(type_name));
    return i==end()?NULL:(*i);
}

list<FWObject*> FWObject::getByType(const string &type_name) const
{
    list<FWObject*> res;
    for(const_iterator i=begin(); i!=end(); ++i) 
    {
	i=find_if( i, end(), FWObjectTypeNameEQPredicate(type_name));
	if (i==end()) break;
	res.push_back(*i);
    } 
    return res; 
}

FWObjectTypedChildIterator FWObject::findByType(const std::string &type_name) const
{
    return FWObjectTypedChildIterator(this, type_name);
}

void  FWObject::setDirty(bool f, bool recursive)
{
    dirty=f;
    if(recursive) 
	for(list<FWObject*>::iterator m=begin(); m!=end(); ++m) 
            (*m)->setDirty(f, TRUE);
}

bool FWObject::isDirty(bool recursive)
{
    if(dirty)
        return TRUE;

    if(recursive) 
    {
	for(list<FWObject*>::iterator m=begin(); m!=end(); ++m)
            if((*m)->isDirty(recursive))
                return TRUE;
    }
    return FALSE;
}

bool FWObject::GUISortOrder(const FWObject *a, const FWObject *b)
{
    return a->getName() < b->getName();
}

class sort_order_func_adaptor 
{
    private:
    
    FWObject *o;
    
    public:
    
    explicit sort_order_func_adaptor(FWObject *p) { o=p; }
    
    bool operator()(const FWObject *a, const FWObject *b) 
    { 
        return o->GUISortOrder(a,b);
    }
};

void FWObject::sortChildren()
{
    this->sort(sort_order_func_adaptor(this));
    setDirty(true);
}

FWObjectTypedChildIterator::FWObjectTypedChildIterator(const FWObjectTypedChildIterator &o):
    type_name(o.type_name),real_iterator(o.real_iterator), _begin(o._begin), _end(o._end)
{
}

FWObjectTypedChildIterator::FWObjectTypedChildIterator(const FWObject *o, const std::string &_type_name)
{
    type_name     = _type_name ;
    _end          = o->end()   ;
    real_iterator = o->begin() ;
    // position to first element
    while(real_iterator!=_end && (*real_iterator)->getTypeName()!=type_name)
        real_iterator++;
    _begin = real_iterator;
}

bool FWObjectTypedChildIterator::operator==(const FWObject::const_iterator& __x) const 
{ 
    return real_iterator == __x; 
}

bool FWObjectTypedChildIterator::operator!=(const FWObject::const_iterator& __x) const 
{ 
    return real_iterator != __x; 
}
    
FWObject *FWObjectTypedChildIterator::operator*() const 
{ 
    return *real_iterator;
}

FWObjectTypedChildIterator& FWObjectTypedChildIterator::operator++() 
{
    if(real_iterator==_end)
        return *this;
    do
    {
        real_iterator++;
    } while(real_iterator!=_end && (*real_iterator)->getTypeName()!=type_name);
        return *this;
}

FWObjectTypedChildIterator& FWObjectTypedChildIterator::operator--() 
{ 
    if(real_iterator==_begin)
        return *this;
    do
    {
        real_iterator--;
    } while(real_iterator!=_begin && (real_iterator==_end || (*real_iterator)->getTypeName()!=type_name));
    return *this;
}

