/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "flux.h"
#include <stdlib.h>
#include <string.h>

/*

  Node list is circular:

  .-> . <-> . <-> . <-> . <-> . <-> . <-.
  |                                     |
  |               node_out    node_in   |
  |               |           |         |
  `-> . <-> . <-> o <-> O <-> o <-> . <-'
        | |
        | next
        prev

  New nodes are always inserted right in front of the "in" node,
  while node removal occurs only right behind the "out" node.

  The number of nodes is always tailored to the FIFO's data flux
  level. Less traffic, nodes are freed up - more traffic, nodes
  are allocated.
  
  ---

  Within nodes:

      byte_out                       byte_in
      |                              |
  [...ooooooooooooooo] [ooooooooooooo.....] 
  node_out             node_in

*/


struct fifobuf_node *_fifobuf_node_add(FIFOBUF *fib)
{
  struct fifobuf_node *node_new;

  /* Allocate node structures. */
  
  node_new = malloc(sizeof(struct fifobuf_node));
  if (!node_new) return(0);
  node_new->data = malloc(fib->bufsize);
  if (!node_new->data) { free(node_new); return(0); }

  if (!fib->node_out)  /* Is this the first node allocated to buffer? */
  {
    node_new->next = node_new;
    node_new->prev = node_new;
    fib->node_out = fib->node_in = node_new;
  }
  else
  {
    node_new->next = fib->node_in->next;  /* [IN]     [NEW]  -> [O] */
    node_new->prev = fib->node_in;        /* [IN] <-  [NEW]     [O] */
    fib->node_in->next->prev = node_new;  /* [IN]     [NEW] <-  [O] */
    fib->node_in->next = node_new;        /* [IN]  -> [NEW]     [O] */
  }

  fib->buffers_cur++;
  return(node_new);
}

/* NOTE: Does not check node for data content! Happily deletes
   used nodes. */

void _fifobuf_node_del(FIFOBUF *fib)
{
  struct fifobuf_node *node_del;

  if (!fib->buffers_cur) return;

  /* Unlink */

  node_del = fib->node_out->prev;
  node_del->prev->next = fib->node_out;  /* [O] -> (OLD)    [OUT] */
  fib->node_out->prev = node_del->prev;  /* [O]    (OLD) <- [OUT] */

  /* Free */
  
  free(node_del->data);
  free(node_del);

  /* Logistics update */
  
  fib->buffers_cur--;
  
  return;
}


FIFOBUF *fifobuf_new(unsigned short buffers_min,
                     unsigned short buffers_max,
                     unsigned long bufsize)
{
  FIFOBUF *fib;
  unsigned short i;

  /* Invalid parameters? */
  if (buffers_min < 2) buffers_min = 2;
  if (buffers_max < buffers_min) buffers_max = buffers_min;
  if (bufsize < 16) bufsize = 16;

  /* Initialize head */
  fib = malloc(sizeof(struct fifobuf_head));
  if (!fib) return(0);
  fib->buffers_min = buffers_min;
  fib->buffers_max = buffers_max;
  fib->buffers_cur = 0;
  fib->bufsize = bufsize;
  fib->enqueued = 0;
  fib->node_out = fib->node_in = 0;
  fib->byte_out = fib->byte_in = 0;
  
  /* Create minimum number of data nodes */
  for (i = 0; i < buffers_min; i++) _fifobuf_node_add(fib);

  return(fib);
}

void fifobuf_free(FIFOBUF *fib)
{
  /* Delete data nodes */
  while (fib->buffers_cur) _fifobuf_node_del(fib);

  free(fib);
  return;
}

long fifobuf_enqueue(FIFOBUF *fib, void *data, unsigned long len)
{
  int i, j;

  /* Max buffers cannot hold data? */
  if (len > fifobuf_free_max(fib)) return(-1);

  /* Add buffer nodes? */
  if (len > fifobuf_free_cur(fib))
  {
    j = _fifobuf_fitnodes(fib, fib->enqueued + len) - fib->buffers_cur;
    for (i = 0; i < j; i++) _fifobuf_node_add(fib);
  }

  /* Enqueue data bytes */

  for (i = len; i; )
  {
    if (fib->bufsize - fib->byte_in)
    {
      /* Last buffer can hold more */

      unsigned long copied_len;
      
      if (fib->bufsize - fib->byte_in < i)
        copied_len = fib->bufsize - fib->byte_in;
      else
        copied_len = i;

      memcpy(fib->node_in->data + fib->byte_in, data + len - i,
             copied_len);
      
      i -= copied_len;
      fib->byte_in += copied_len;
      fib->enqueued += copied_len;
    }
    else
    {
      /* Goto next buffer */

      fib->node_in = fib->node_in->next;
      fib->byte_in = 0;
    }
  }

  return(fib->enqueued);
}

unsigned long fifobuf_dequeue(FIFOBUF *fib, void *data, unsigned long len)
{
  unsigned long i;

  if (len > fib->enqueued) len = fib->enqueued;
    
  for (i = len; i; )
  {
    if (fib->bufsize - fib->byte_out)
    {
      unsigned long copied_len;
        
      if (fib->bufsize - fib->byte_out < i)
        copied_len = fib->bufsize - fib->byte_out;
      else
        copied_len = i;
        
      memcpy(data + len - i, fib->node_out->data + fib->byte_out,
              copied_len);
        
      i -= copied_len;
      fib->byte_out += copied_len;
      fib->enqueued -= copied_len;
    }
    else
    {
      fib->node_out = fib->node_out->next;
      fib->byte_out = 0;
    }
  }

  return(len);
}

void *fifobuf_dequeue_dyn(FIFOBUF *fib, unsigned long len)
{
  void *data;
  
  if (len > fib->enqueued) return(0);
  
  data = malloc(len + 1);
  if (!data) return(0);
  *((unsigned char *) data + len) = 0;

  if (fifobuf_dequeue(fib, data, len) > 0) return(data);
  return(0);
}

unsigned long fifobuf_peek(FIFOBUF *fib, void *data, unsigned long len)
{
  unsigned long i, byte_out;
  struct fifobuf_node *node_out;

  if (len > fib->enqueued) len = fib->enqueued;

  for (i = len, byte_out = fib->byte_out, node_out = fib->node_out; i; )
  {
    if (fib->bufsize - byte_out)
    {
      unsigned long copied_len;
        
      if (fib->bufsize - byte_out < i)
        copied_len = fib->bufsize - byte_out;
      else
        copied_len = i;
        
      memcpy(data + len - i, node_out->data + byte_out,
              copied_len);
        
      i -= copied_len;
      byte_out += copied_len;
    }
    else
    {
      node_out = node_out->next;
      byte_out = 0;
    }
  }

  return(len);
}

unsigned long fifobuf_drop(FIFOBUF *fib, unsigned long len)
{
  if (len > fib->enqueued) len = fib->enqueued;
  fib->enqueued -= len;
  
  while (len > fib->bufsize - fib->byte_out)
  {
    fib->node_out = fib->node_out->next;
    len -= fib->bufsize - fib->byte_out;
    fib->byte_out = 0;
  }

  fib->byte_out += len;

  return(fib->enqueued);
}

unsigned long fifobuf_do(FIFOBUF *fib, unsigned long len,
                         int (*func)(void *data,
                                     unsigned long len,
                                     void *extra),
                         void *extra)
{
  struct fifobuf_node *node_out;
  unsigned long i, byte_out;
  int r;
  
  if (len < fib->enqueued) len = fib->enqueued;

  for (i = len, byte_out = fib->byte_out, node_out = fib->node_out; i; )
  {
    if (fib->bufsize - byte_out)
    {
      unsigned long copied_len;
        
      if (fib->bufsize - byte_out < i)
        copied_len = fib->bufsize - byte_out;
      else
        copied_len = i;

      r = func(node_out->data + byte_out, copied_len, extra);
      if (r < 0) break;
      
      i -= copied_len;
      byte_out += copied_len;
    }
    else
    {
      node_out = node_out->next;
      byte_out = 0;
    }
  }

  return(i);
}
