/*
     This file is part of libextractor.
     (C) 2002, 2003 Vidyut Samanta and Christian Grothoff

     libextractor is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     libextractor is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with libextractor; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
 **/

#include "extractor_util.h"
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifndef MINGW
#include <sys/mman.h>
#include <netinet/in.h>
#endif


static struct EXTRACTOR_Keywords * addKeyword(EXTRACTOR_KeywordType type,
					      char * keyword,
					      struct EXTRACTOR_Keywords * next) {
  EXTRACTOR_KeywordList * result;

  if (keyword == NULL)
    return next;
  result = malloc(sizeof(EXTRACTOR_KeywordList));
  result->next = next;    
  result->keyword = keyword;
  result->keywordType = type;
  return result;
}

static struct {
  char * name;
  EXTRACTOR_KeywordType type;
} tagmap[] = {
   { "Author" , EXTRACTOR_AUTHOR}, 
   { "Description" , EXTRACTOR_DESCRIPTION},
   { "Comment", EXTRACTOR_COMMENT},
   { "Copyright", EXTRACTOR_COPYRIGHT}, 
   { "Source", EXTRACTOR_SOURCE},
   { "Creation Time", EXTRACTOR_DATE},
   { "Title", EXTRACTOR_TITLE},
   { "Software", EXTRACTOR_SOFTWARE},
   { "Disclaimer", EXTRACTOR_DISCLAIMER},
   { "Warning", EXTRACTOR_WARNING},
   { "Signature", EXTRACTOR_RESOURCE_IDENTIFIER}, 
   {NULL, EXTRACTOR_UNKNOWN},
};

static struct EXTRACTOR_Keywords * processtEXt(unsigned char * data,
					       unsigned int length,
					       struct EXTRACTOR_Keywords * prev) {
  unsigned char * type_of_information;
  char * keyword;
  unsigned int len;

  data+=4;
  type_of_information = data;
  len = length - (strlen(type_of_information)+1);
  keyword = malloc(len);
  memcpy(keyword,
	 &data[strlen(type_of_information)+1],
	 len-1);
  keyword[len-1] = 0;
  len = 0;
  while (tagmap[len].name != NULL) {
    if (0 == strcmp(tagmap[len].name, type_of_information))
      return addKeyword(tagmap[len].type,
			keyword,
			prev);

    len++;
  }
  return addKeyword(EXTRACTOR_UNKNOWN,
		    keyword,
		    prev);
}
static struct EXTRACTOR_Keywords * processiTXt(unsigned char * data,
					       unsigned int length,
					       struct EXTRACTOR_Keywords * prev) {
  unsigned char * type_of_information;
  char * keyword;
  unsigned int pos;
  unsigned int len;
  char * language;
  char * translated;
  struct EXTRACTOR_Keywords * result; 

  result = prev;
  type_of_information = data;
  pos = strlen(type_of_information)+3;
  language = &data[pos];
  if (strlen(language) > 0)
    result = addKeyword(EXTRACTOR_LANGUAGE,
			strdup(language),
			result);
  pos += strlen(language)+1;
  translated = &data[pos];
  if (strlen(language) > 0)
    result = addKeyword(EXTRACTOR_TRANSLATED,
			strdup(translated),
			result);
  pos += strlen(translated)+1;
  len = length - pos;

  keyword = malloc(len+1);
  memcpy(keyword,
	 &data[pos],
	 len);
  keyword[len] = 0;
  len = 0;
  while (tagmap[len].name != NULL) {
    if (0 == strcmp(tagmap[len].name, 
		    type_of_information))
      return addKeyword(tagmap[len].type,
			keyword,
			result);
    len++;
  }
  return addKeyword(EXTRACTOR_UNKNOWN,
		    keyword,
		    result);
}

static struct EXTRACTOR_Keywords * processIHDR(unsigned char * data,
					       unsigned int length,
					       struct EXTRACTOR_Keywords * prev) {
  struct EXTRACTOR_Keywords * result;
  char * tmp;
  if (length < 12)
    return prev;
  tmp = malloc(128);
  sprintf(tmp,
	  "%ux%u",
	  htonl(*(int*) &data[4]),
	  htonl(*(int*) &data[8])); 
  result = addKeyword(EXTRACTOR_SIZE,
		      strdup(tmp),
		      prev);
  free(tmp);
  return result;
}

/* not supported... */
static struct EXTRACTOR_Keywords * processzTXt(unsigned char * data,
					       unsigned int length,
					       struct EXTRACTOR_Keywords * prev) {
  struct EXTRACTOR_Keywords * result;
  result = prev;  
  return result;
}

#define PNG_HEADER "\211PNG\r\n\032\n"


struct EXTRACTOR_Keywords * libextractor_png_extract(char * filename,
                                                     unsigned char * data,
                                                     size_t size,
                                                     struct EXTRACTOR_Keywords * prev) {
  unsigned char * pos;
  unsigned char * end;
  struct EXTRACTOR_Keywords * result;
  unsigned int length;

  if (size < strlen(PNG_HEADER))
    return prev;
  if (0 != strncmp(data, PNG_HEADER, strlen(PNG_HEADER)))
    return prev;
  result = prev;
  end = &data[size];
  pos = &data[strlen(PNG_HEADER)];
  result = addKeyword(EXTRACTOR_MIMETYPE,
		      strdup("image/png"),
		      result);
  while(1) {
    if (pos+12 >= end)
      break;
    length = htonl(*(int*) pos);  pos+=4;    
    if (pos+4+length+4 > end)
      break;
    if (0 == strncmp(pos, "IHDR", 4))
      result = processIHDR(pos, length, result);
    if (0 == strncmp(pos, "iTXt", 4)) 
      result = processiTXt(pos, length, result);
    if (0 == strncmp(pos, "tEXt", 4)) 
      result = processtEXt(pos, length, result);
    if (0 == strncmp(pos, "zTXt", 4)) 
      result = processzTXt(pos, length, result);
    pos += 4+length+4; /* Chunk type, data, crc */
  }
  return result;  
}

#define HAVE_MAIN 0
#if HAVE_MAIN
int main (int argc, char **argv) {
  int file;
  char * buffer;
  struct stat fstatbuf;
  size_t size;

  if (argc != 2) {
    fprintf(stderr, 
	    "Call with filename as argument\n");
    return -1;
  }  
  file = OPEN(argv[1],O_RDONLY);
  if (-1 == file) 
    return -1;
  if (-1 == FSTAT(file, &fstatbuf)) {
    close(file);
    return -1;
  }
  size = fstatbuf.st_size;
  buffer = mmap(NULL, size, PROT_READ, MAP_SHARED, file, 0);  
  close(file);

  EXTRACTOR_printKeywords(stdout,
			  libextractor_png_extract(argv[1], 
						   buffer,
						   size,
						   NULL));
  munmap(buffer,size);
  return 0;
}
#endif
