/*
     This file is part of libextractor.
     (C) 2002, 2003 Vidyut Samanta and Christian Grothoff

     libextractor is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     libextractor is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with libextractor; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
 **/

#include "extractor.h"
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <stdlib.h>

#define M_SOI   0xD8		/* Start Of Image (beginning of datastream) */
#define M_EOI   0xD9		/* End Of Image (end of datastream) */
#define M_SOS   0xDA		/* Start Of Scan (begins compressed data) */
#define M_APP12	0xEC		
#define M_COM   0xFE		/* COMment */


static EXTRACTOR_KeywordList * addKeyword(EXTRACTOR_KeywordType type,
					  char * keyword,
					  EXTRACTOR_KeywordList * next) {
  EXTRACTOR_KeywordList * result;

  if (keyword == NULL)
    return next;
  result = malloc(sizeof(EXTRACTOR_KeywordList));
  result->next = next;    
  result->keyword = keyword;
  result->keywordType = type;
  return result;
}

/**
 * Get the next character in the sequence and advance
 * the pointer *data to the next location in the sequence.
 * If we're at the end, return EOF.
 **/
#define NEXTC(data,end) (unsigned char) ((*(data)<(end))?*((*(data))++):EOF)

/* The macro does:
unsigned int NEXTC(unsigned char ** data, char *  end) {
  if (*data < end) {
    char result = **data;
    (*data)++;
    return result;
  } else 
    return EOF;
}
*/

/**
 * Read length, convert to unsigned int.
 * All 2-byte quantities in JPEG markers are MSB first 
 * @return -1 on error
 */
static unsigned int readLength (char ** data, char * end) {
  unsigned int c1, c2;

  c1 = NEXTC(data, end);
  if (c1 == EOF)
    return -1;
  c2 = NEXTC(data, end);
  if (c2 == EOF)
    return -1;
  return ((((unsigned int) c1) << 8) + ((unsigned int) c2))-2;
}

/**
 * @return the next marker or EOF on error.
 **/
static int next_marker(char ** data, char * end) {
  int c;
  c = NEXTC(data, end);
  while ( (c != 0xFF) && (c != EOF) ) 
    c = NEXTC(data, end);  
  do {
    c = NEXTC(data, end);
  } while ( (c == 0xFF) && (c != EOF) );
  return c;
}

static void skip_variable(char ** data, char * end) {
  unsigned int length;

  length = readLength(data, end);
  if (length < 0) {
    while (EOF != NEXTC(data, end)) ;
    return;
  }
  /* Skip over the remaining bytes */
  while (length > 0) {
    (void) NEXTC(data, end);
    length--;
  }
}

static char * process_COM(char ** data, char * end) {
  unsigned int length;
  int ch;
  int pos;
  char * comment;

  length = readLength(data, end);
  if (length <= 0)
    return NULL; 
  comment = malloc(length+1);
  pos = 0;
  while (length > 0) {
    ch = NEXTC(data, end);
    if ( (ch == '\r')  ||
	 (ch == '\n') )
      comment[pos++] = '\n';
    else if (isprint(ch)) 
      comment[pos++]=ch;    
    length--;
  }
  comment[pos] = '\0';
  return comment;
}

struct EXTRACTOR_Keywords * libextractor_jpeg_extract(char * filename,
                                                      char * data,
                                                      size_t size,
                                                      struct EXTRACTOR_Keywords * prev) {
  unsigned char c1, c2;
  int marker;
  char * end;
  struct EXTRACTOR_Keywords * result;

  result = prev;
  end = &data[size];
  c1 = NEXTC(&data, end);
  c2 = NEXTC(&data, end);
  if ( (c1 != 0xFF) || (c2 != M_SOI) )
    return result; /* not a JPEG */
  result = addKeyword(EXTRACTOR_MIMETYPE,
		      strdup("image/jpeg"),
		      result);
  while(1) {
    marker = next_marker(&data, end);
    switch (marker) {
    case M_SOS:
    case M_EOI:
      goto RETURN; /* this used to be "return result", but this 
		      makes certain compilers unhappy...*/
    case M_COM:
    case M_APP12:
      result = addKeyword(EXTRACTOR_COMMENT,
			  process_COM(&data, end),
			  result);
      break;
    default:			
      skip_variable(&data, end);		
      break;
    }
  }
 RETURN:
  return result;
}

#define HAVE_MAIN 0
#if HAVE_MAIN
int main (int argc, char **argv) {
  int file;
  char * buffer;
  struct stat fstatbuf;
  size_t size;

  if (argc != 2) {
    fprintf(stderr, 
	    "Call with filename as argument\n");
    return -1;
  }  
  file = OPEN(filename,O_RDONLY);
  if (-1 == file) 
    return NULL;
  if (-1 == FSTAT(file, &fstatbuf)) {
    close(file);
    return NULL;
  }
  size = fstatbuf.st_size;
  buffer = mmap(NULL, size, PROT_READ, MAP_SHARED, file, 0);  
  close(file);

  EXTRACTOR_printKeywords(stdout,
			  libextractor_jpeg_extract(argv[1], 
						    NULL, 
						    buffer,
						    size,
						    NULL));
  free(size);
  return 0;
}
#endif
