/*
     This file is part of libextractor.
     (C) 2002, 2003, 2004 Vidyut Samanta and Christian Grothoff

     libextractor is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     libextractor is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with libextractor; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
 */

#include "extractor_util.h"
#ifdef USE_NS_MODULE
#include <mach-o/dyld.h>
#else
#if HAVE_LTDL_H
#include <ltdl.h>
#else
#include <../../libltdl/ltdl.h>
#endif
#endif
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#ifndef MINGW
  #include <sys/mman.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#define DEBUG 1

/**
 * The sources of keywords as strings.
 */
static const char *keywordTypes[] = {
  "unknown",
  "filename",
  "mimetype",
  "title",
  "author",
  "artist",
  "description",
  "comment",
  "date",
  "publisher",
  "language",
  "album",
  "genre",
  "location",
  "version",
  "organization",
  "copyright",
  "subject",
  "keywords",
  "contributor",
  "resource-type",
  "format",
  "resource-identifier",
  "source",
  "relation",
  "coverage",
  "software",
  "disclaimer",
  "warning",
  "translated",
  "creation date",
  "modification date",
  "creator",
  "producer",
  "page count",
  "page orientation",
  "paper size",
  "used fonts",
  "page order",
  "created for",
  "magnification",
  "release",
  "group",
  "size", 
  "summary",
  "packager",
  "vendor",
  "license",
  "distribution",
  "build-host",
  "os",
  "dependency",
  NULL,
};

/* the number of keyword types (for bounds-checking) */
#define HIGHEST_TYPE_NUMBER 52

#ifdef HAVE_LIBOGG
#ifdef HAVE_VORBIS
#define WITH_OGG 1
#endif
#endif

#ifdef HAVE_VORBISFILE
#define WITH_OGG 1
#endif

#ifdef WITH_OGG
#define OGGSO "libextractor_ogg:"
#else
#define OGGSO ""
#endif

#ifdef HAVE_ZLIB
#define QTSO "libextractor_qt:"
#else
#define QTSO ""
#endif


#define DEFSO \
"libextractor_html:\
libextractor_mp3:\
libextractor_png:\
libextractor_gif:\
libextractor_wav:\
libextractor_real:\
libextractor_pdf:\
libextractor_jpeg:\
libextractor_tiff:\
libextractor_ps:\
libextractor_zip:\
libextractor_rpm:\
libextractor_mime:\
libextractor_riff:\
libextractor_mpeg:\
libextractor_elf:\
libextractor_asf"

#define DEFAULT_LIBRARIES OGGSO QTSO DEFSO

/* ************library initialization ***************** */

#ifdef MINGW
void __attribute__ ((constructor)) le_win_init(void) {
  InitWinEnv();
}

void __attribute__ ((destructor)) le_win_fini(void) {
  ShutdownWinEnv();
}
#endif

#ifndef USE_NS_MODULE
static char * old_dlsearchpath = NULL;

/* using libtool, needs init! */
void __attribute__ ((constructor)) le_ltdl_init(void) {
  int err;
  err = lt_dlinit ();
  if (err > 0)
    {
#if DEBUG
      fprintf (stderr,
	       "initialization of plugin mechanism failed (%s)!\n",
	       lt_dlerror ());
#endif
      return;
    }
  if (lt_dlgetsearchpath() != NULL)
    old_dlsearchpath = strdup(lt_dlgetsearchpath());
  if (lt_dlgetsearchpath () == NULL)
    lt_dladdsearchdir ("/usr/lib");
  else if (strstr (lt_dlgetsearchpath (), "/usr/lib") == NULL)
    lt_dladdsearchdir ("/usr/lib");
  if (strstr (lt_dlgetsearchpath (), "/usr/local/lib") == NULL)
    lt_dladdsearchdir ("/usr/local/lib");
#ifdef LTDL_SYSSEARCHPATH
  if (strstr (lt_dlgetsearchpath (), LTDL_SYSSEARCHPATH) == NULL)
    lt_dladdsearchdir (LTDL_SYSSEARCHPATH);
#endif
#ifdef ELIBDIR
  if (strstr (lt_dlgetsearchpath (), ELIBDIR) == NULL)
    lt_dladdsearchdir (ELIBDIR);
#endif
#ifdef PLUGIN_PATH
  if (strstr (lt_dlgetsearchpath (), PLUGIN_PATH) == NULL)
    lt_dladdsearchdir (PLUGIN_PATH);
#endif
#ifdef LTDL_SHLLIBPATH_VAR
  env = getenv (LTDL_SHLIBPATH_VAR);
  if (env != NULL)
    if (strstr (lt_dlgetsearchpath (), env) == NULL)
      lt_dladdsearchdir (env);
#endif
}

void __attribute__ ((destructor)) le_ltdl_fini(void) {
  lt_dlsetsearchpath(old_dlsearchpath);
  if (old_dlsearchpath != NULL)
    free(old_dlsearchpath);
  lt_dlexit ();
}

#endif


/**
 * Load the default set of libraries. The default set of
 * libraries consists of the libraries that are part of 
 * the libextractor distribution (except split and filename
 * extractor) plus the extractors that are specified
 * in the environment variable "LIBEXTRACTOR_LIBRARIES".
 *
 * @return the default set of libraries.
 */
EXTRACTOR_ExtractorList *
EXTRACTOR_loadDefaultLibraries ()
{
  char *env;
  char *tmp;
  EXTRACTOR_ExtractorList *res;


  env = getenv ("LIBEXTRACTOR_LIBRARIES");
  if (env == NULL)
    {
      return EXTRACTOR_loadConfigLibraries (NULL, DEFAULT_LIBRARIES);
    }
  tmp = malloc (strlen (env) + strlen (DEFAULT_LIBRARIES) + 2);
  strcpy (tmp, env);
  strcat (tmp, ":");
  strcat (tmp, DEFAULT_LIBRARIES);
  res = EXTRACTOR_loadConfigLibraries (NULL, tmp);
  free (tmp);
  return res;
}

/**
 * Get the textual name of the keyword.
 * @return NULL if the type is not known
 */
const char *
EXTRACTOR_getKeywordTypeAsString(const EXTRACTOR_KeywordType type)
{
  if ((type >= 0) && (type < HIGHEST_TYPE_NUMBER))
    return keywordTypes[type];
  else
    return NULL;
}

/**
 * Load multiple libraries as specified by the user.
 * @param config a string given by the user that defines which
 *        libraries should be loaded. Has the format
 *        "[[-]LIBRARYNAME[:[-]LIBRARYNAME]*]". For example,
 *        libextractor_mp3.so:libextractor_ogg.so loads the
 *        mp3 and the ogg library. The '-' before the LIBRARYNAME
 *        indicates that the library should be added to the end
 *        of the library list (addLibraryLast).
 * @param prev the  previous list of libraries, may be NULL
 * @return the new list of libraries, equal to prev iff an error occured
 *         or if config was empty (or NULL).
 */
EXTRACTOR_ExtractorList *
EXTRACTOR_loadConfigLibraries (EXTRACTOR_ExtractorList * prev,
			       const char *config)
{
  char *cpy;
  int pos;
  int last;
  int len;

  if (config == NULL)
    return prev;
  len = strlen(config);
  cpy = strdup(config);
  pos = 0;
  last = 0;
  while (pos < len)
    {
      while ((cpy[pos] != ':') && (cpy[pos] != '\0'))
	pos++;
      cpy[pos++] = '\0';	/* replace ':' by termination */
      if (cpy[last] == '-')
	{
	  last++;
	  prev = EXTRACTOR_addLibraryLast (prev, &cpy[last]);
	}
      else
	prev = EXTRACTOR_addLibrary (prev, &cpy[last]);
      last = pos;
    }
  free (cpy);
  return prev;
}

static void *getSymbolWithPrefix(void *lib_handle,
                                 const char *lib_name,
                                 const char *sym_name)
{
  size_t name_size
    = strlen(lib_name)
    + strlen(sym_name)
    + 1 /* for the zero delim. */
    + 1 /* for the optional '_' prefix */;
  char *name=malloc(name_size),*first_error;
  void *symbol=NULL;
#ifdef USE_NS_MODULE
  NSSymbol *symbol_obj=NULL;
  NSLinkEditErrors ler;
  int lerno;
  const char *lerfile;
  const char *dyld_error;
#endif

  snprintf(name,
	   name_size,
	   "_%s%s",
	   lib_name,
	   sym_name);

#ifdef USE_NS_MODULE
  if (((struct mach_header*)lib_handle)->magic == MH_MAGIC ||
      ((struct mach_header*)lib_handle)->magic == MH_CIGAM) {
    if (NSIsSymbolNameDefinedInImage((struct mach_header*)lib_handle,name)) {
      symbol_obj=NSLookupSymbolInImage((struct mach_header*)lib_handle,name,
                                       NSLOOKUPSYMBOLINIMAGE_OPTION_BIND |
                                       NSLOOKUPSYMBOLINIMAGE_OPTION_RETURN_ON_ERROR);
    }
  } else {
    symbol_obj=NSLookupSymbolInModule(lib_handle,name);
  }
  if (symbol_obj==NULL) {
#if DEBUG
    NSLinkEditError(&ler,&lerno,&lerfile,&dyld_error);
    if (dyld_error && *dyld_error) {
      fprintf(stderr,"Could not load %s: NSLookupSymbolInModule() for '%s' failed: %s\n",
                     lib_name,name,dyld_error);
    } else {
      fprintf(stderr,"Could not load %s: NSLookupSymbolInModule() for '%s' failed.\n",
                     lib_name,name);
    }
#endif
  } else {
    symbol=NSAddressOfSymbol(symbol_obj);
  }
#else
  symbol=lt_dlsym(lib_handle,name+1 /* skip the '_' */);
  if (symbol==NULL) {
    first_error=strdup(lt_dlerror());
    symbol=lt_dlsym(lib_handle,name /* now try with the '_' */);
#if DEBUG
    fprintf (stderr,
             "Resolving symbol '%s' in library %s failed, "
             "so I tried '%s', but that failed also.  Errors are: "
             "%s; and then: %s\n",
             name+1,
             lib_name,
             name,
             first_error,
             lt_dlerror());
#endif
    free(first_error);
  }
#endif
  free(name);
  return symbol;
}

/**
 * Load a dynamic library.
 * @return 1 on success, -1 on error
 */
static int
loadLibrary (const char *name,
	     void **libHandle, 
	     ExtractMethod * method)
{
#ifdef USE_NS_MODULE
  NSObjectFileImage fileImage;
  NSObjectFileImageReturnCode returnCode;
  NSLinkEditErrors ler;
  int lerno;
  const char *lerfile;
  const char *dyld_error;
  char *file_name;

  file_name
     = malloc(strlen(PLUGIN_PATH) +
	      strlen("/") +
	      strlen(name) +
	      strlen(".so") + 1);
  sprintf(file_name,
	  "%s/%s.so",
	  PLUGIN_PATH,
	  name);
  returnCode
    = NSCreateObjectFileImageFromFile(file_name,
				      &fileImage);
  if (returnCode==NSObjectFileImageInappropriateFile) {
    NSDestroyObjectFileImage(fileImage);
    *libHandle=(void*)NSAddImage(file_name,
				 NSADDIMAGE_OPTION_RETURN_ON_ERROR);
  } else if (returnCode==NSObjectFileImageSuccess) {
    *libHandle=NSLinkModule(fileImage,
                            file_name,
                            NSLINKMODULE_OPTION_RETURN_ON_ERROR |
                            NSLINKMODULE_OPTION_PRIVATE |
                            NSLINKMODULE_OPTION_BINDNOW);
    
    NSDestroyObjectFileImage(fileImage);
  } else {
#if DEBUG
    switch (returnCode) {
    case NSObjectFileImageInappropriateFile:
      fprintf(stderr,
	      "Could not load %s: '%s' is not a bundle\n",
	      name,
	      file_name);
      break;
    case NSObjectFileImageFailure:
      fprintf(stderr,
	      "Could not load %s: Object file setup failure in '%s'\n",
	      name,
	      file_name);
      break;
    case NSObjectFileImageArch:
      fprintf(stderr,
	      "Could not load %s: No object for this architecture in '%s'\n",
	      name,
	      file_name);
      break;
    case NSObjectFileImageFormat:
      fprintf(stderr,
	      "Could not load %s: '%s' has bad object file format\n",
	      name,
	      file_name);
      break;
    case NSObjectFileImageAccess:
      fprintf(stderr,
	      "Could not load %s: Cannot read '%s'\n",
	      name,
	      file_name);
      break;
    default:
      fprintf(stderr,
	      "Could not load %s: Could not load '%s' because of some mystery error\n",
	      name,
	      file_name);
      break;
    }
#endif
    NSDestroyObjectFileImage(fileImage);
    free(file_name);
    return -1;
  }
  
  if (*libHandle==NULL) {
    NSLinkEditError(&ler,&lerno,&lerfile,&dyld_error);
    if (dyld_error && *dyld_error) {
      fprintf(stderr,
	      "Could not load %s: NSLinkModule() for '%s' failed: %s\n",
	      name,file_name,
	      dyld_error);
    } else {
      fprintf(stderr,
	      "Could not load %s: NSLinkModule() for '%s' failed.\n",
	      name,
	      file_name);
    }
    free(file_name);
    return -1;
  }
  
  free(file_name);
  /* end NS_MODULE */
#else  
  /* begin libtool */

  *libHandle = lt_dlopenext (name);
  if (*libHandle == NULL)
    {
#if DEBUG
      fprintf (stderr,
	       "Loading library %s failed (%s)!\n", 
	       name, 
	       lt_dlerror ());
#endif
      return -1;
    }
#endif

  *method = (ExtractMethod) getSymbolWithPrefix (*libHandle, name, "_extract");
  if (*method == NULL)
    {
#ifdef USE_NS_MODULE
      NSUnLinkModule(*libHandle,0);
#else
      lt_dlclose (*libHandle);
#endif
      return -1;
    }
  return 1;
}

/**
 * Add a library for keyword extraction.
 * @param prev the previous list of libraries, may be NULL
 * @param library the name of the library
 * @return the new list of libraries, equal to prev iff an error occured
 */
EXTRACTOR_ExtractorList *
EXTRACTOR_addLibrary (EXTRACTOR_ExtractorList * prev, 
		      const char *library)
{
  EXTRACTOR_ExtractorList *result;
  void *handle;
  ExtractMethod method;

  if (-1 == loadLibrary (library, &handle, &method))
    return prev;
  result = malloc (sizeof (EXTRACTOR_ExtractorList));
  result->next = prev;
  result->libraryHandle = handle;
  result->extractMethod = method;
  result->libname = strdup (library);
  return result;
}

/**
 * Add a library for keyword extraction at the END of the list.
 * @param prev the previous list of libraries, may be NULL
 * @param library the name of the library
 * @return the new list of libraries, always equal to prev
 *         except if prev was NULL and no error occurs
 */
EXTRACTOR_ExtractorList *
EXTRACTOR_addLibraryLast(EXTRACTOR_ExtractorList * prev, 
			 const char *library)
{
  EXTRACTOR_ExtractorList *result;
  EXTRACTOR_ExtractorList *pos;
  void *handle;
  ExtractMethod method;

  if (-1 == loadLibrary (library, &handle, &method))
    return prev;
  result = malloc (sizeof (EXTRACTOR_ExtractorList));
  result->next = NULL;
  result->libraryHandle = handle;
  result->extractMethod = method;
  result->libname = strdup (library);
  if (prev == NULL)
    return result;
  pos = prev;
  while (pos->next != NULL)
    pos = pos->next;
  pos->next = result;
  return prev;
}

/**
 * Remove a library for keyword extraction.
 * @param prev the current list of libraries
 * @param library the name of the library to remove
 * @return the reduced list, unchanged if the library was not loaded
 */
EXTRACTOR_ExtractorList *
EXTRACTOR_removeLibrary(EXTRACTOR_ExtractorList * prev, 
			const char *library)
{
  EXTRACTOR_ExtractorList *pos;
  EXTRACTOR_ExtractorList *first;
  pos = prev;
  first = prev;
  while ((pos != NULL) && (0 != strcmp (pos->libname, library)))
    {
      prev = pos;
      pos = pos->next;
    }
  if (pos != NULL)
    {
      /* found, close library */
      if (first == pos)
	first = pos->next;
      else
	prev->next = pos->next;
      /* found */
      free (pos->libname);
#ifdef USE_NS_MODULE
      NSUnLinkModule(pos->libraryHandle,0);
#else
      lt_dlclose (pos->libraryHandle);
#endif
      free (pos);
    }
#if DEBUG
  else
    fprintf(stderr, 
	    "Removing library %s failed!\n", 
	    library);
#endif
  return first;
}

/**
 * Remove all extractors.
 * @param libraries the list of extractors
 */
void
EXTRACTOR_removeAll (EXTRACTOR_ExtractorList * libraries)
{
  while (libraries != NULL)
    libraries = EXTRACTOR_removeLibrary (libraries, libraries->libname);
}

/**
 * Extract keywords from a file using the available extractors.
 * @param extractor the list of extractor libraries
 * @param filename the name of the file
 * @return the list of keywords found in the file, NULL if none
 *         were found (or other errors)
 */
EXTRACTOR_KeywordList *
EXTRACTOR_getKeywords (EXTRACTOR_ExtractorList * extractor,
		       const char *filename) {
  EXTRACTOR_KeywordList *result;
  int file;
  char *buffer;
  struct stat fstatbuf;
  size_t size;

  file = OPEN(filename, O_RDONLY);
  if (-1 == file)
    return NULL;
  if (-1 == FSTAT(file, &fstatbuf))
    {
      close (file);
      return NULL;
    }
  size = fstatbuf.st_size;
#ifndef MINGW
  if (size > 0)
    buffer = mmap(NULL, size, PROT_READ | PROT_WRITE, MAP_PRIVATE, file, 0);
  else {
    buffer = malloc(1);
    buffer[0] = '\0';
  }
#else
  buffer = malloc(size + 1);
  buffer[size] = 0;
  if (size > 0)
    read(file, buffer, size);
#endif
  close(file);
  if ( (buffer == NULL) || (buffer == (void *) -1) )
    return NULL;
  result = NULL;
  while (extractor != NULL) {
    result = extractor->extractMethod (filename, buffer, size, result);
    extractor = extractor->next;
  }
#ifndef MINGW
  if (size > 0)
    munmap (buffer, size);
  else
    free(buffer);
#else
  free(buffer);
#endif
  return result;
}

static void
removeKeyword (const char *keyword,
	       const EXTRACTOR_KeywordType type,
	       const unsigned int options,
	       EXTRACTOR_KeywordList ** list, 
	       EXTRACTOR_KeywordList * current)
{
  EXTRACTOR_KeywordList *first;
  EXTRACTOR_KeywordList *pos;
  EXTRACTOR_KeywordList *prev;
  EXTRACTOR_KeywordList *next;

  first = *list;
  pos = first;
  prev = NULL;
  while (pos != NULL)
    {
      if (pos == current)
	{
	  prev = pos;
	  pos = current->next;
	}
      if (pos == NULL)
	break;
      if ((0 == strcmp (pos->keyword, keyword)) &&
	  ((pos->keywordType == type) ||
	   (((options & EXTRACTOR_DUPLICATES_TYPELESS) > 0)) ||
	   (((options & EXTRACTOR_DUPLICATES_REMOVE_UNKNOWN) > 0) &&
	    (pos->keywordType == EXTRACTOR_UNKNOWN))))
	{
	  /* remove! */
	  if (prev == NULL)
	    first = pos->next;
	  else
	    prev->next = pos->next;
	  next = pos->next;
	  free (pos->keyword);
	  free (pos);
	  pos = next;
	}
      else
	{
	  prev = pos;
	  pos = pos->next;
	}
    }				/* end while */
  *list = first;
}

/**
 * Remove duplicate keywords from the list.
 * @param list the original keyword list (destroyed in the process!)
 * @param options a set of options (DUPLICATES_XXXX)
 * @return a list of keywords without duplicates
 */
EXTRACTOR_KeywordList *
EXTRACTOR_removeDuplicateKeywords (EXTRACTOR_KeywordList * list,
				   const unsigned int options)
{
  EXTRACTOR_KeywordList *pos;

  pos = list;
  while (pos != NULL)
    {
      removeKeyword (pos->keyword, pos->keywordType, options, &list, pos);
      pos = pos->next;
    }
  return list;
}

/**
 * Remove empty (all-whitespace) keywords from the list.
 * @param list the original keyword list (destroyed in the process!)
 * @return a list of keywords without duplicates
 */
EXTRACTOR_KeywordList *
EXTRACTOR_removeEmptyKeywords (EXTRACTOR_KeywordList * list)
{
  EXTRACTOR_KeywordList * pos;
  EXTRACTOR_KeywordList * last;

  last = NULL;
  pos = list;
  while (pos != NULL)
    {
      int allWhite;
      int i;
      allWhite = 1;
      for (i=strlen(pos->keyword)-1;i>=0;i--)
	if (! isspace(pos->keyword[i]))
	  allWhite = 0;
      if (allWhite) 
	{
	  EXTRACTOR_KeywordList * next;
	  next = pos->next;
	  if (last == NULL)
	    list = next;
	  else
	    last->next = next;
	  free(pos->keyword);
	  free(pos);
	  pos = next;
	}
      else 
	{
	  last = pos;
	  pos = pos->next;
	}
    }
  return list;
}


/**
 * Print a keyword list to a file.
 * For debugging.
 * @param handle the file to write to (stdout, stderr), may NOT be NULL
 * @param keywords the list of keywords to print, may be NULL
 */
void
EXTRACTOR_printKeywords (FILE * handle, 
			 EXTRACTOR_KeywordList * keywords)
{
  while (keywords != NULL)
    {
      if (keywords->keywordType >= HIGHEST_TYPE_NUMBER)
	fprintf (handle, "INVALID TYPE - %s\n", keywords->keyword);
      else
	fprintf (handle,
		 "%s - %s\n",
		 keywordTypes[keywords->keywordType], keywords->keyword);
      keywords = keywords->next;
    }
}

/**
 * Free the memory occupied by the keyword list (and the
 * keyword strings in it!)
 * @param keywords the list to free
 */
void
EXTRACTOR_freeKeywords (EXTRACTOR_KeywordList * keywords)
{
  EXTRACTOR_KeywordList *prev;
  while (keywords != NULL)
    {
      prev = keywords;
      keywords = keywords->next;
      free (prev->keyword);
      free (prev);
    }
}

/**
 * Return the highest type number, exclusive as in [0,highest).
 */
EXTRACTOR_KeywordType
EXTRACTOR_getHighestKeywordTypeNumber ()
{
  return HIGHEST_TYPE_NUMBER;
}

/**
 * Extract the last keyword that of the given type from the keyword list.
 * @param type the type of the keyword
 * @param keywords the keyword list
 * @return the last matching keyword, or NULL if none matches
 */
const char *
EXTRACTOR_extractLast (const EXTRACTOR_KeywordType type,
		       EXTRACTOR_KeywordList * keywords)
{
  char *result = NULL;
  while (keywords != NULL)
    {
      if (keywords->keywordType == type)
	result = keywords->keyword;
      keywords = keywords->next;
    }
  return result;
}

/**
 * Extract the last keyword of the given string from the keyword list.
 * @param type the string describing the type of the keyword
 * @param keywords the keyword list
 * @return the last matching keyword, or NULL if none matches
 */
const char *
EXTRACTOR_extractLastByString (const char * type, 
			       EXTRACTOR_KeywordList * keywords)
{
  char * result = NULL;
  if ( type == NULL )
    return result;
  while (keywords != NULL )
    {
      if ( !strcmp(keywordTypes[keywords->keywordType], type) )
	result = keywords->keyword;
      keywords = keywords->next;
    }
  return result;
}

/**
 * Count the number of keywords in the keyword list.
 * @param keywords the keyword list
 * @return the number of keywords in the list
 */
unsigned int
EXTRACTOR_countKeywords (EXTRACTOR_KeywordList * keywords)
{
  int count = 0;
  while (keywords != NULL)
    {
      count++;
      keywords = keywords->next;
    }
  return count;
}



/* ******************* Java support *********************** */

#ifdef HAVE_JNI_H

#include "org_ovmj_libextractor_Extractor.h"

/*
 * Class:     org_ovmj_libextractor_Extractor
 * Method:    loadDefaultInternal
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_org_ovmj_libextractor_Extractor_loadDefaultInternal(JNIEnv * env,
										 jclass c) {
  return (jlong) (long) EXTRACTOR_loadDefaultLibraries();
}

/*
 * Class:     org_ovmj_libextractor_Extractor
 * Method:    unloadInternal
 * Signature: (J)V
 */
JNIEXPORT void JNICALL Java_org_ovmj_libextractor_Extractor_unloadInternal(JNIEnv * env, 
									   jclass c,
									   jlong arg) {
  EXTRACTOR_removeAll((EXTRACTOR_ExtractorList*) (long) arg);
}

/*
 * Class:     org_ovmj_libextractor_Extractor
 * Method:    extractInternal
 * Signature: (JLjava/lang/String;)J
 */
JNIEXPORT jlong JNICALL Java_org_ovmj_libextractor_Extractor_extractInternal(JNIEnv * env, 
									     jclass c,
									     jlong arg, 
									     jstring f) {
  const char * fname;
  jboolean bo;
  jlong ret;

  bo = JNI_FALSE;
  fname = (*env)->GetStringUTFChars(env, f, &bo);
  ret = (jlong) (long) EXTRACTOR_getKeywords((EXTRACTOR_ExtractorList*) (long) arg,
					     fname);
  (*env)->ReleaseStringUTFChars(env, f, fname);
  return ret;
}

/*
 * Class:     org_ovmj_libextractor_Extractor
 * Method:    freeInternal
 * Signature: (J)V
 */
JNIEXPORT void JNICALL Java_org_ovmj_libextractor_Extractor_freeInternal(JNIEnv * env, 
									 jclass c, 
									 jlong list) {
  EXTRACTOR_freeKeywords( (EXTRACTOR_KeywordList*) (long) list);
}

/*
 * Class:     org_ovmj_libextractor_Extractor
 * Method:    typeInternal
 * Signature: (J)I
 */
JNIEXPORT jint JNICALL Java_org_ovmj_libextractor_Extractor_typeInternal(JNIEnv * env, 
									 jclass c, 
									 jlong list) {
  if (list == 0)
    return -1; /* error! */
  return ((EXTRACTOR_KeywordList*) (long) list)->keywordType;
}

/*
 * Class:     org_ovmj_libextractor_Extractor
 * Method:    keywordInternal
 * Signature: (J)Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_org_ovmj_libextractor_Extractor_keywordInternal(JNIEnv * env, 
									       jclass c, 
									       jlong list) {
  char * str;
  if (list == 0)
    return NULL; /* error! */
  str = ((EXTRACTOR_KeywordList*) (long) list)->keyword;
  return (*env)->NewStringUTF(env, str);
}

/*
 * Class:     org_ovmj_libextractor_Extractor
 * Method:    nextInternal
 * Signature: (J)J
 */
JNIEXPORT jlong JNICALL Java_org_ovmj_libextractor_Extractor_nextInternal(JNIEnv * env, 
									  jclass j, 
									  jlong list) {
  if (list == 0)
    return 0; /* error! */
  return (jlong) (long) ((EXTRACTOR_KeywordList*) (long) list)->next;
}

/*
 * Class:     org_ovmj_libextractor_Extractor
 * Method:    getTypeAsStringInternal
 * Signature: (I)Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_org_ovmj_libextractor_Extractor_getTypeAsStringInternal(JNIEnv * env, 
										       jclass c, 
										       jint type) {
  const char * str;
  if ( (type < 0) || (type > HIGHEST_TYPE_NUMBER) )
    return NULL; /* error! */  
  str = keywordTypes[type];
  if (str == NULL)
    return NULL;
  return (*env)->NewStringUTF(env, 
			      str);
}


/*
 * Class:     org_ovmj_libextractor_Extractor
 * Method:    getVersionInternal
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_org_ovmj_libextractor_Extractor_getVersionInternal(JNIEnv * env, 
									       jclass c) {
  return EXTRACTOR_VERSION;
}

/*
 * Class:     org_ovmj_libextractor_Extractor
 * Method:    getMaxTypeInternal
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_org_ovmj_libextractor_Extractor_getMaxTypeInternal(JNIEnv * env, 
									       jclass c) {
  return HIGHEST_TYPE_NUMBER;
}

/*
 * Class:     org_ovmj_libextractor_Extractor
 * Method:    unloadPlugin
 * Signature: (JLjava/lang/String;)J
 */
JNIEXPORT jlong JNICALL Java_org_ovmj_libextractor_Extractor_unloadPlugin(JNIEnv * env, 
									  jclass c, 
									  jlong handle, 
									  jstring name) {
  const char * lname;
  jboolean bo;
  jlong ret;

  bo = JNI_FALSE;
  lname = (*env)->GetStringUTFChars(env, name, &bo);
  ret = (jlong) (long) EXTRACTOR_removeLibrary((EXTRACTOR_ExtractorList*) (long) handle,
					       lname);
  (*env)->ReleaseStringUTFChars(env, name, lname);
  return ret; 
}

/*
 * Class:     org_ovmj_libextractor_Extractor
 * Method:    loadPlugin
 * Signature: (JLjava/lang/String;Z)J
 */
JNIEXPORT jlong JNICALL Java_org_ovmj_libextractor_Extractor_loadPlugin(JNIEnv * env, 
									jclass c, 
									jlong handle, 
									jstring name,
									jboolean place) {
  const char * lname;
  jboolean bo;
  jlong ret;

  bo = JNI_FALSE;
  lname = (*env)->GetStringUTFChars(env, name, &bo);
  if (place == JNI_TRUE) {
    ret = (jlong) (long) EXTRACTOR_addLibraryLast((EXTRACTOR_ExtractorList*) (long) handle,
						  lname);
  } else {
    ret = (jlong) (long) EXTRACTOR_addLibrary((EXTRACTOR_ExtractorList*) (long) handle,
					      lname);    
  }
  (*env)->ReleaseStringUTFChars(env, name, lname);
  return ret; 
}


#endif
/* #ifdef HAVE_JNI_H */
