/*
     This file is part of libextractor.
     (C) 2002, 2003 Vidyut Samanta and Christian Grothoff

     libextractor is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     libextractor is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with libextractor; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
 */

#include "platform.h"
#include "extractor.h"

#define UINT32 unsigned int
#define UINT16 unsigned short
#define UINT8 unsigned char

typedef struct{
  UINT32     object_id;
  UINT32     size;
  UINT16     object_version; /* must be 0 */
  UINT16                      stream_number;
  UINT32                      max_bit_rate;
  UINT32                      avg_bit_rate;
  UINT32                      max_packet_size;
  UINT32                      avg_packet_size;
  UINT32                      start_time;
  UINT32                      preroll;
  UINT32                      duration;
  UINT8                       stream_name_size;
  UINT8 data[0]; /* variable length section */
  /*
    UINT8[stream_name_size]     stream_name;
    UINT8                       mime_type_size;
    UINT8[mime_type_size]       mime_type;
    UINT32                      type_specific_len;
    UINT8[type_specific_len]    type_specific_data;
  */
} Media_Properties;

typedef struct {
  UINT32     object_id;
  UINT32     size;
  UINT16      object_version; /* must be 0 */
  UINT16    title_len;
  UINT8 data[0]; /* variable length section */
  /*
    UINT8[title_len]  title;
    UINT16    author_len;
    UINT8[author_len]  author;
    UINT16    copyright_len;
    UINT8[copyright_len]  copyright;
    UINT16    comment_len;
    UINT8[comment_len]  comment;
  */  
} Content_Description;


#define REAL_HEADER 0x2E524d46
#define MDPR_HEADER 0x4D445052
#define CONT_HEADER 0x434F4e54

static struct EXTRACTOR_Keywords * addKeyword(EXTRACTOR_KeywordType type,
				    char * keyword,
				    struct EXTRACTOR_Keywords * next) {
  EXTRACTOR_KeywordList * result;

  if (keyword == NULL)
    return next;
  result = malloc(sizeof(EXTRACTOR_KeywordList));
  result->next = next;    
  result->keyword = keyword;
  result->keywordType = type;
  return result;
}

static struct EXTRACTOR_Keywords * processMediaProperties(Media_Properties * prop,
							  struct EXTRACTOR_Keywords * prev) {

  UINT8 mime_type_size;
  UINT32 prop_size;
  char * data;

  prop_size = ntohl(prop->size);
  if (prop_size <= sizeof(Media_Properties))
    return prev;  
  if (0 != prop->object_version)
    return prev;
  if (prop_size <= prop->stream_name_size + sizeof(UINT8)
      + sizeof(Media_Properties))
    return prev;    
  
  mime_type_size = prop->data[prop->stream_name_size];
  if (prop_size <= prop->stream_name_size + sizeof(UINT8) +
      + mime_type_size + sizeof(Media_Properties))
    return prev;  
  
  data = malloc(mime_type_size+1);
  memcpy(data,&prop->data[prop->stream_name_size+1],mime_type_size);
  data[mime_type_size]='\0';
  
  return addKeyword(EXTRACTOR_MIMETYPE,
		    data,
		    prev);
}

static struct EXTRACTOR_Keywords * processContentDescription(Content_Description * prop,
							     struct EXTRACTOR_Keywords * prev) {
  

  UINT16 author_len;
  UINT16 copyright_len;
  UINT16 comment_len;
  UINT16 title_len;
  char * title;
  char * author;
  char * copyright;
  char * comment;
  UINT32 prop_size;

  prop_size = ntohl(prop->size);
  if (prop_size <= sizeof(Content_Description))
    return prev;  
  if (0 != prop->object_version)
    return prev;
  title_len = ntohs(prop->title_len);
  if (prop_size <= title_len + sizeof(UINT16)
      + sizeof(Content_Description))
    return prev;


  author_len = ntohs( *(UINT16*)&prop->data[title_len]);
   
  if (prop_size <= title_len + sizeof(UINT16) 
      + author_len + sizeof(Content_Description))
    return prev;

  copyright_len =ntohs(  *(UINT16*)&prop->data[title_len+
					author_len+
					sizeof(UINT16)]);
 
  if (prop_size <= title_len + 2*sizeof(UINT16) 
      + author_len + copyright_len + sizeof(Content_Description))
    return prev;
  
  comment_len = ntohs( *(UINT16*)&prop->data[title_len+
				      author_len+
				      copyright_len+
				      2*sizeof(UINT16)]);
  
  if (prop_size < title_len + 3*sizeof(UINT16) 
      + author_len + copyright_len + comment_len 
      + sizeof(Content_Description))
    return prev;
  
  title = malloc(title_len+1);
  memcpy(title,&prop->data[0],title_len);
  title[title_len]='\0';
 
  prev = addKeyword(EXTRACTOR_TITLE,
		    title,
		    prev);

  author = malloc(author_len+1);
  memcpy(author,&prop->data[title_len+sizeof(UINT16)],author_len);
  author[author_len]='\0';

  prev = addKeyword(EXTRACTOR_AUTHOR,
		    author,
		    prev);

  copyright=malloc(copyright_len+1);
  memcpy(copyright,
	 &prop->data[title_len + sizeof(UINT16)*2 + author_len],
	 copyright_len);
  copyright[copyright_len]='\0';


  prev = addKeyword(EXTRACTOR_COPYRIGHT,
		    copyright,
		    prev);


  comment=malloc(comment_len+1);
  memcpy(comment,
	 &prop->data[title_len + sizeof(UINT16)*3 + author_len + copyright_len],
	 comment_len);
  comment[comment_len]='\0';

  prev = addKeyword(EXTRACTOR_COMMENT,
		    comment,
		    prev);

  return prev;
}


struct EXTRACTOR_Keywords * libextractor_real_extract(char * filename,
                                                      unsigned char * data,
                                                      size_t size,
                                                      struct EXTRACTOR_Keywords * prev) {
  unsigned char * pos;
  unsigned char * end;
  struct EXTRACTOR_Keywords * result;
  unsigned int length;

  if (size < 2*sizeof(int)) {
    return prev;
  }
  if (REAL_HEADER != ntohl(*(int*)data)) {
    return prev; 
  }
  result = prev;
  end = &data[size];
  pos = &data[0];
  while(1) {
    if (pos+8 >= end)
      break;
    length = ntohl(*(((unsigned int*) pos)+1));
    if (length <= 0)
      break;
    if (pos + length >= end)
      break;
    switch (ntohl(*((unsigned int*) pos))) {
    case MDPR_HEADER:
      result = processMediaProperties((Media_Properties *)pos,
				      result);
      pos += length;
      break;
    case CONT_HEADER:
      result = processContentDescription((Content_Description *)pos,
					 result);
      pos += length;
      break;
    case REAL_HEADER: /* treat like default */
    default:
      pos += length;
      break;
    }
  }
  return result;  
}

#define HAVE_MAIN 0
#if HAVE_MAIN
int main (int argc, char **argv) {
  int file;
  char * buffer;
  struct stat fstatbuf;
  size_t size;

  if (argc != 2) {
    fprintf(stderr, 
	    "Call with filename as argument\n");
    return -1;
  }  
  file = OPEN(argv[1],O_RDONLY);
  if (-1 == file) 
    return -1;
  if (-1 == FSTAT(file, &fstatbuf)) {
    close(file);
    return -1;
  }
  size = fstatbuf.st_size;
  buffer = mmap(NULL, size, PROT_READ, MAP_SHARED, file, 0);  
  close(file);

  EXTRACTOR_printKeywords(stdout,
			  libextractor_real_extract(argv[1], 
						    buffer,
						    size,
						    NULL));
  munmap(buffer,size);
  return 0;
}
#endif
