/***************************************************************************
                          csocket.cpp  -  description
                             -------------------
    begin                : Sat Oct 6 2001
    copyright            : (C) 2001-2003 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <signal.h>

#ifndef WIN32
#include <netdb.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>

// includes for sun os
#ifdef HAVE_SYS_SOCKIO_H 
#include <sys/sockio.h>
#endif
#ifdef HAVE_SYS_TERMIOS_H
#include <sys/termios.h>
#endif
#ifdef HAVE_SYS_FILIO_H 
#include <sys/filio.h>
#endif
#endif

#include <sys/types.h>

#ifndef MSG_NOSIGNAL
// 0x2000  /* don't raise SIGPIPE */
#define MSG_NOSIGNAL      0
#endif

#include <dclib/dcos.h>
#include <dclib/core/casyncdns.h>
#include <dclib/core/clogfile.h>
#include <dclib/core/cssl.h>

#ifdef WIN32
//#include <winsock2.h>
//#include <ws2tcpip.h>
#include <iprtrmib.h>
#include <iphlpapi.h>
#endif

// Needed on OSF/1 
#ifndef hstrerror
#define hstrerror(x) ((const char*)"Resolver error.")
#endif

#include "csocket.h"

#ifdef HAVE_SOCKS
extern "C" {
#include <socks.h>
}
#endif

// init static socket vars
CTraffic CSocket::m_Traffic = CTraffic();
CThread CSocket::m_cs_ghbn = CThread();
eSocketLog CSocket::m_eSocketLog = eslNONE;

CSocket::CSocket( eSocketType type )
{
	SocketType    = type;
	iHandle       = INVALID_SOCKET;
	sError        = "";
	m_eSocketMode = esmSOCKET;
	m_bSSLInit    = FALSE;
#ifdef HAVE_SSL
	m_pCTX = 0;
	m_pSSL = 0;
#endif

#ifdef WIN32
	WORD wVersionRequested;
	WSADATA wsaData;
	int err;
 
	wVersionRequested = MAKEWORD( 2, 2 );
 
	err = WSAStartup( wVersionRequested, &wsaData );

	if ( err != 0 )
	{
		/* Tell the user that we could not find a usable */
		/* WinSock DLL.                                  */
		return;
	}
 
	/* Confirm that the WinSock DLL supports 2.2.*/
	/* Note that if the DLL supports versions greater    */
	/* than 2.2 in addition to 2.2, it will still return */
	/* 2.2 in wVersion since that is the version we      */
	/* requested.                                        */
 
	if ( LOBYTE( wsaData.wVersion ) != 2 ||
		HIBYTE( wsaData.wVersion ) != 2 )
	{
		/* Tell the user that we could not find a usable */
		/* WinSock DLL.                                  */
		WSACleanup( );
		return; 
	}
 
	/* The WinSock DLL is acceptable. Proceed. */
#endif
}

CSocket::~CSocket()
{
	if ( iHandle != INVALID_SOCKET )
	{
#ifdef WIN32
		closesocket(iHandle);
#else
		close(iHandle);
#endif
	}
}

/** */
int CSocket::SetSocket( int handle, eSocketType sockettype )
{
	if ( handle == INVALID_SOCKET )
	{
		return -1;
	}

	if ( (sockettype != estTCP) && (sockettype != estUDP) )
	{
		return -1;
	}

	SocketType = sockettype;
	iHandle    = handle;

	return 0;
}

/** rfc1918 */
bool CSocket::IsPrivateAddressSpace( const char * cp )
{
	unsigned long int i;
	 
	// sanity check
	if ( !cp )
		return FALSE;
	
	i = inet_addr(cp);

	if ( i == (unsigned long int)-1 )
		return FALSE;

	/** 10.0.0.0 - 10.255.255.255
	  * 172.16.0.0 - 172.31.255.255
	  * 192.168.0.0 - 192.168.255.255
	*/
	if ( ((i | inet_addr("10.255.255.255"))  == inet_addr("10.255.255.255")) ||
	     ((i | inet_addr("172.16.255.255"))  == inet_addr("172.31.255.255")) ||
	     ((i | inet_addr("192.168.255.255")) == inet_addr("192.168.255.255")) )
		return TRUE;

	return FALSE;
}

/** */
bool CSocket::GetHostByName( const char * name, struct sockaddr_in * sin )
{
	m_cs_ghbn.Lock();

	bool err = FALSE;
	struct hostent * hp;

	// sanity check
	if ( name && sin )
	{
		memset(sin,0,sizeof(struct sockaddr_in));

		if ( (hp = gethostbyname(name)) != 0 )
		{
			if ( hp->h_addr != 0 )
			{
		 		memcpy((void *)&sin->sin_addr, hp->h_addr, sizeof(sin->sin_addr) ); //hp->h_length);
				err = TRUE;
			}
		}
	}

	m_cs_ghbn.UnLock();

	return err;
}

/** */
CString CSocket::GetHostByName( CString Host )
{
	CString s = "";
	struct sockaddr_in sin;

	if ( Host != "" )
	{
		if ( GetHostByName( Host.Data(), &sin ) == TRUE )
		{
			s = inet_ntoa(sin.sin_addr);
		}
	}

	return s;
}

/** */
void CSocket::ParseHost( CString host, CString & ip, unsigned int & port )
{
	int i;

	ip = "";

	// remove all spaces
	host = host.Replace(" ","");

	i = host.Find(':');

	if ( i >= 0 )
	{
		ip = host.Mid(i+1,host.Length()-i-1);
	}

	if ( (i < 0) || (ip == "") )
	{
		port = 0;
		ip   = host;
	}
	else
	{
		port = ip.asINT();
		ip   = host.Mid(0,i);
	}	
}

/** */
int CSocket::GetInterfaceList( CList<CString> * interfacelist )
{
	if ( !interfacelist )
	{
		return -1;
	}

#ifdef WIN32
	// get the interfaces from win32
	LONG Status;
	ULONG Result;
	HKEY AdapKey,OneAdapKey;
	TCHAR AdapName[256];
	DWORD dim;
	UCHAR buffer[256];
	CString name,instance;
	int i;

	// ip-helper
	ULONG buf_size;
	DWORD getiftable_rc;
	char *buff = NULL;
	MIB_IFTABLE *ift;
	DWORD count;
	MIB_IFROW *ifr;
	IP_ADAPTER_INFO *ai;

	// iphelper interface table
	buf_size = 0;
	getiftable_rc = GetIfTable((PMIB_IFTABLE) buff, &buf_size, TRUE);

	if (getiftable_rc == ERROR_INSUFFICIENT_BUFFER)
	{
		buff = (char *) malloc(buf_size);
		memset(buff, 0, buf_size);

		getiftable_rc = GetIfTable((PMIB_IFTABLE) buff, &buf_size, TRUE);

		if (getiftable_rc == NO_ERROR)
		{
			ift = (MIB_IFTABLE *) buff;
			count = ift->dwNumEntries;

			for (i=0;i<count;i++)
			{
				ifr = (ift->table) + i;

				if ( ifr->dwDescrLen > 0 )
				{
					CString * s=0;
					while( (s=interfacelist->Next(s)) != 0 )
						if ( *s == (const char*)ifr->bDescr )
							break;
					if ( !s )
						interfacelist->Add( new CString((const char*)ifr->bDescr) );
				}
			}
		}

		free(buff);
	}

	// iphelper adapter info
	buf_size = 0;
	getiftable_rc = GetAdaptersInfo((PIP_ADAPTER_INFO) buff, &buf_size );

	if (getiftable_rc == ERROR_BUFFER_OVERFLOW)
	{
		buff = (char *) malloc(buf_size);
		memset(buff, 0, buf_size);

		getiftable_rc = GetAdaptersInfo((PIP_ADAPTER_INFO) buff, &buf_size );

		if (getiftable_rc == ERROR_SUCCESS)
		{
			ai = (IP_ADAPTER_INFO *) buff;

			for (;ai;ai=ai->Next)
			{
				CString * s=0;
				while( (s=interfacelist->Next(s)) != 0 )
					if ( *s == (const char*)ai->Description )
						break;
				if ( !s )
					interfacelist->Add( new CString((const char*)ai->Description));
			}
		}

		free(buff);
	}

	// win 95/98
	Status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
		TEXT("SYSTEM\\CurrentControlSet\\Services\\Class\\Net"),
		0,
		KEY_READ,
		&AdapKey);

	if ( Status == ERROR_SUCCESS )
	{
		i=0;

		// Get the size to allocate for the original device names
		while( (Result=RegEnumKey(AdapKey,i,AdapName,sizeof(AdapName)/2)) == ERROR_SUCCESS )
		{
			Status = RegOpenKeyEx(AdapKey,AdapName,0,KEY_READ,&OneAdapKey);

			if ( Status == ERROR_SUCCESS )
			{
				name     = "";
				instance = (char*)AdapName;
				dim = 256;

				Status = RegQueryValueEx(OneAdapKey,"DriverDesc",NULL,NULL,buffer,&dim);

				if ( Status == ERROR_SUCCESS )
				{
					name = (char*)buffer;
				}

				RegCloseKey(OneAdapKey);

				if ( (instance != "") && (name != "") )
				{
					// now we check if the interface exist
					instance  = "SYSTEM\\CurrentControlSet\\Services\\Class\\NetTrans\\"+instance;

					Status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
							TEXT(instance.Data()),
							0,
							KEY_READ,
							&OneAdapKey);

					if ( Status == ERROR_SUCCESS )
					{
						CString * s=0;
						while( (s=interfacelist->Next(s)) != 0 )
							if ( *s == name )
								break;
						if ( !s )
							interfacelist->Add( new CString(name) );
						RegCloseKey(OneAdapKey);
					}
				}
			}

			i++;
		}

		RegCloseKey(AdapKey);
	}

	// win nt/2k
	Status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
		TEXT("SYSTEM\\CurrentControlSet\\Control\\Class\\{4D36E972-E325-11CE-BFC1-08002BE10318}"),
		0,
		KEY_READ,
		&AdapKey);

	if ( Status == ERROR_SUCCESS )
	{
		i=0;

		// Get the size to allocate for the original device names
		while( (Result=RegEnumKey(AdapKey,i,AdapName,sizeof(AdapName)/2)) == ERROR_SUCCESS )
		{
			Status = RegOpenKeyEx(AdapKey,AdapName,0,KEY_READ,&OneAdapKey);

			if ( Status == ERROR_SUCCESS )
			{
				name     = "";
				instance = "";

				dim  = 256;
				Status = RegQueryValueEx(OneAdapKey,"NetCfgInstanceId",NULL,NULL,buffer,&dim);

				if ( Status == ERROR_SUCCESS )
				{
					dim = 256;
					instance = (char*)buffer;

					Status = RegQueryValueEx(OneAdapKey,"DriverDesc",NULL,NULL,buffer,&dim);

					if ( Status == ERROR_SUCCESS )
					{
						name = (char*)buffer;
					}
				}

				RegCloseKey(OneAdapKey);

				if ( (instance != "") && (name != "") )
				{
					// now we check if the interface exist
					instance  = "SYSTEM\\CurrentControlSet\\Services\\"+instance +"\\Parameters\\Tcpip";

					Status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
							TEXT(instance.Data()),
							0,
							KEY_READ,
							&OneAdapKey);

					if ( Status == ERROR_SUCCESS )
					{
						CString * s=0;
						while( (s=interfacelist->Next(s)) != 0 )
							if ( *s == name )
								break;
						if ( !s )
							interfacelist->Add( new CString(name) );
						RegCloseKey(OneAdapKey);
					}
				}
			}

			i++;
		}

		RegCloseKey(AdapKey);
	}
#else
	int sock = -1;
	struct ifreq ifreqa[256];
	int i;
	struct ifconf *pifconf;
	struct ifreq * pifreq;

	sock = socket( PF_INET, SOCK_DGRAM, 0 );

	if ( sock != -1 )
	{
		pifconf = (struct ifconf*) malloc(sizeof(ifconf));

		pifconf->ifc_len = 255*32;
		pifconf->ifc_ifcu.ifcu_req = ifreqa;

		i = ioctl( sock, SIOCGIFCONF, pifconf );

		if ( i != -1 )
		{
			if ( pifconf->ifc_len > 0 )
			{
				for( pifreq = pifconf->ifc_ifcu.ifcu_req, i = 0;
				     pifconf->ifc_len > i;)
				{
					CString * s=0;
					while( (s=interfacelist->Next(s)) != 0 )
						if ( *s == ((char*)pifreq->ifr_name) )
							break;
					if ( !s )
						interfacelist->Add( new CString((char*)pifreq->ifr_name) );
#ifdef HAVE_SA_LEN
					i += IFNAMSIZ + pifreq->ifr_addr.sa_len;
#else
					i += sizeof( struct ifreq );
#endif
					pifreq = (struct ifreq *)((caddr_t)pifconf->ifc_ifcu.ifcu_req + i);
				}
			}
		}

		close(sock);
		free(pifconf);
	}
#endif

	return interfacelist->Count();
}

/** */
CString CSocket::GetInterfaceIP( CString iface )
{
	CString s = "";
#ifdef WIN32
	// get the interfaces from win32
	LONG Status;
	ULONG Result;
	HKEY AdapKey,OneAdapKey;
	TCHAR AdapName[256];
	DWORD dim;
	DWORD RegType;
	DWORD DHCPEnabled; 
	UCHAR buffer[256];
	CString name,instance;
	int i;

	// ip-helper
	ULONG buf_size;
	DWORD getiftable_rc;
	char *buff = NULL;
	MIB_IFTABLE *ift;
	DWORD count;
	DWORD index=0xFFFFFFFF;
	MIB_IFROW *ifr;
	IP_ADAPTER_INFO *ai;
	MIB_IPADDRTABLE *ipt;
	MIB_IPADDRROW *ipr;

	// iphelper interface table
	buf_size = 0;
	buff = NULL;
	getiftable_rc = GetIfTable((PMIB_IFTABLE) buff, &buf_size, TRUE);

	if (getiftable_rc == ERROR_INSUFFICIENT_BUFFER)
	{
		buff = (char *) malloc(buf_size);
		memset(buff, 0, buf_size);

		getiftable_rc = GetIfTable((PMIB_IFTABLE) buff, &buf_size, TRUE);

		if (getiftable_rc == NO_ERROR)
		{
			ift = (MIB_IFTABLE *) buff;
			count = ift->dwNumEntries;

			for (i=0;i<count;i++)
			{
				ifr = (ift->table) + i;

				if ( iface == (const char*)ifr->bDescr )
				{
					index = ifr->dwIndex;
					break;
				}
			}
		}

		free(buff);
	}

	// get ip from nettable
	if ( index != 0xFFFFFFFF )
	{
		buf_size = 0;
		buff = NULL;
		getiftable_rc = GetIpAddrTable((PMIB_IPADDRTABLE) buff, &buf_size, TRUE );

		if (getiftable_rc == ERROR_INSUFFICIENT_BUFFER)
		{
			buff = (char *) malloc(buf_size);
			memset(buff, 0, buf_size);

			getiftable_rc = GetIpAddrTable((PMIB_IPADDRTABLE) buff, &buf_size, TRUE );

			if (getiftable_rc == NO_ERROR)
			{
				ipt = (MIB_IPADDRTABLE *) buff;
				count = ipt->dwNumEntries;

				for (i=0;i<count;i++)
				{
					ipr = (ipt->table) + i;

					if ( ipr->dwIndex == index )
					{
						struct in_addr in;

						in.s_addr = ipr->dwAddr;
						s = inet_ntoa(in);
						break;
					}
				}
			}
		}

		free(buff);
	}

	if ( s != "" )
	{
		return s;
	}

	// iphelper adapter info
	buf_size = 0;
	buff = NULL;
	getiftable_rc = GetAdaptersInfo((PIP_ADAPTER_INFO) buff, &buf_size );

	if (getiftable_rc == ERROR_BUFFER_OVERFLOW)
	{
		buff = (char *) malloc(buf_size);
		memset(buff, 0, buf_size);

		getiftable_rc = GetAdaptersInfo((PIP_ADAPTER_INFO) buff, &buf_size );

		if (getiftable_rc == ERROR_SUCCESS)
		{
			ai = (IP_ADAPTER_INFO *) buff;

			for (;ai;ai=ai->Next)
			{
				if ( iface == ai->Description )
				{
					s = ai->IpAddressList.IpAddress.String;
					break;
				}
			}
		}

		free(buff);
	}

	if ( s != "" )
	{
		return s;
	}

	// win 95/98
	Status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
		TEXT("SYSTEM\\CurrentControlSet\\Services\\Class\\Net"),
		0,
		KEY_READ,
		&AdapKey);

	if ( Status == ERROR_SUCCESS )
	{
		i=0;

		// Get the size to allocate for the original device names
		while( (Result=RegEnumKey(AdapKey,i,AdapName,sizeof(AdapName)/2)) == ERROR_SUCCESS )
		{
			Status = RegOpenKeyEx(AdapKey,AdapName,0,KEY_READ,&OneAdapKey);

			if ( Status == ERROR_SUCCESS )
			{
				name     = "";
				instance = (char*)AdapName;
				dim = 256;

				Status = RegQueryValueEx(OneAdapKey,"DriverDesc",NULL,NULL,buffer,&dim);

				if ( Status == ERROR_SUCCESS )
				{
					name = (char*)buffer;

					if ( name != iface )
					{
						name = "";
					}
				}

				RegCloseKey(OneAdapKey);

				if ( (instance != "") && (name != "") )
				{
					// now we check if the interface exist
					instance  = "SYSTEM\\CurrentControlSet\\Services\\Class\\NetTrans\\"+instance;

					Status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
							TEXT(instance.Data()),
							0,
							KEY_READ,
							&OneAdapKey);

					if ( Status == ERROR_SUCCESS )
					{
						dim = 4;
						Status = RegQueryValueEx(OneAdapKey,TEXT("EnableDHCP"),NULL,&RegType,(LPBYTE)&DHCPEnabled,&dim);

						if ( Status != ERROR_SUCCESS )
						{
							DHCPEnabled = 0;
						}

						if ( DHCPEnabled == 0 )
						{
							dim = 265;
							Status=RegQueryValueEx(OneAdapKey,"IPAddress",NULL,NULL,buffer,&dim);
							if ( Status == ERROR_SUCCESS )
							{
								s = (char*)buffer;
							}
						}
						else
						{
							dim = 256;
							Status = RegQueryValueEx(OneAdapKey,"DhcpIPAddress",NULL,NULL,buffer,&dim);

							if ( Status == ERROR_SUCCESS )
							{
								s = (char*)buffer;
							}
						}

						RegCloseKey(OneAdapKey);

						break;
					}

				}
			}

			i++;
		}

		RegCloseKey(AdapKey);
	}

	if ( s != "" )
	{
		return s;
	}

	// win nt/2k
	Status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
		TEXT("SYSTEM\\CurrentControlSet\\Control\\Class\\{4D36E972-E325-11CE-BFC1-08002BE10318}"),
		0,
		KEY_READ,
		&AdapKey);

	if ( Status == ERROR_SUCCESS )
	{
		i=0;

		// Get the size to allocate for the original device names
		while( (Result=RegEnumKey(AdapKey,i,AdapName,sizeof(AdapName)/2)) == ERROR_SUCCESS )
		{
			Status = RegOpenKeyEx(AdapKey,AdapName,0,KEY_READ,&OneAdapKey);

			if ( Status == ERROR_SUCCESS )
			{
				name     = "";
				instance = "";

				dim = 256;
				Status = RegQueryValueEx(OneAdapKey,"DriverDesc",NULL,NULL,buffer,&dim);

				if ( Status == ERROR_SUCCESS )
				{
					name = (char*)buffer;

					if ( name == iface )
					{
						// found
						dim = 256;
						Status = RegQueryValueEx(OneAdapKey,"NetCfgInstanceId",NULL,NULL,buffer,&dim);					

						if ( Status == ERROR_SUCCESS )
						{
							instance = (char*)buffer;
						}
					}
				}

				RegCloseKey(OneAdapKey);

				if ( (instance != "") && (name != "") )
				{
					// now we get the ip
					instance = "SYSTEM\\CurrentControlSet\\Services\\"+instance+"\\Parameters\\Tcpip";

					Status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
							TEXT(instance.Data()),
							0,
							KEY_READ,
							&OneAdapKey);

					if ( Status == ERROR_SUCCESS )
					{
						dim = 4;
						Status = RegQueryValueEx(OneAdapKey,TEXT("EnableDHCP"),NULL,&RegType,(LPBYTE)&DHCPEnabled,&dim);

						if ( Status != ERROR_SUCCESS )
						{
							DHCPEnabled = 0;
						}

						if ( DHCPEnabled == 0 )
						{
							dim = 265;
							Status=RegQueryValueEx(OneAdapKey,"IPAddress",NULL,NULL,buffer,&dim);
							if ( Status == ERROR_SUCCESS )
							{
								s = (char*)buffer;
							}
						}
						else
						{
							dim = 256;
							Status = RegQueryValueEx(OneAdapKey,"DhcpIPAddress",NULL,NULL,buffer,&dim);

							if ( Status == ERROR_SUCCESS )
							{
								s = (char*)buffer;
							}
						}

						RegCloseKey(OneAdapKey);

						break;
					}
				}
			}

			i++;
		}

		RegCloseKey(AdapKey);
	}
#else
	int sock = -1;
	struct ifreq ifreqa[256];
	int i;
	struct ifconf *pifconf;
	struct ifreq * pifreq;
	struct ifreq ifr;

	sock = socket( PF_INET, SOCK_DGRAM, 0 );

	if ( sock == -1 )
	{
		return s;
	}

	pifconf = (struct ifconf*) malloc(sizeof(ifconf));

	pifconf->ifc_len = 255*32;
	pifconf->ifc_ifcu.ifcu_req = ifreqa;

	i = ioctl( sock, SIOCGIFCONF, pifconf );

	if ( i == -1 )
	{
		close(sock);
		free(pifconf);
		return s;
	}

	if ( pifconf->ifc_len == 0 )
	{
		close(sock);
		free(pifconf);
		return s;
	}

	for( pifreq = pifconf->ifc_ifcu.ifcu_req, i = 0;
	     pifconf->ifc_len > i;)
	{
		if ( iface == (char*)pifreq->ifr_name )
		{
			memcpy( &ifr, pifreq, sizeof(struct ifreq));

			if ( ioctl( sock, SIOCGIFADDR, &ifr ) != -1 )
			{
				s = inet_ntoa(((sockaddr_in*)&ifr.ifr_broadaddr)->sin_addr);
			}

			break;
		}

#ifdef HAVE_SA_LEN
		i += IFNAMSIZ + pifreq->ifr_addr.sa_len;
#else
		i += sizeof( struct ifreq );
#endif
		pifreq = (struct ifreq *)((caddr_t)pifconf->ifc_ifcu.ifcu_req + i);
	}

	close(sock);
	free(pifconf);
#endif
	return s;
}

/** */
int CSocket::GetFreeSendBufferSize()
{
	int tot, free;
#ifndef WIN32
	int unsent;
#endif
	socklen_t ilen;

	free = 0;

	if ( iHandle == INVALID_SOCKET )
	{
		return free;
	}

	ilen = sizeof(int);

	if ( getsockopt(iHandle, SOL_SOCKET, SO_SNDBUF, (char*)&tot, &ilen) == 0 )
	{
		// quick & dirty fix for free bsd
		free = tot;
#ifndef WIN32
#ifndef __CYGWIN__
		if ( ioctl(iHandle, TIOCOUTQ, &unsent) == 0 )
		{
			free = tot - unsent;
		}
#endif
#endif
	}

	return free;
}

/** */
int CSocket::IsConnect()
{
	int i,err;
	struct timeval tv;
	fd_set rset,wset,eset;

	if ( iHandle == INVALID_SOCKET )
	{
		return -1;
	}

	FD_ZERO(&rset);
	FD_ZERO(&wset);
	FD_ZERO(&eset);
	FD_SET(iHandle, &rset);
	FD_SET(iHandle, &wset);
	FD_SET(iHandle, &eset);

	tv.tv_sec  = 0;
	tv.tv_usec = 1;

	err = -1;

	i = select(FD_SETSIZE, &rset, &wset, &eset, &tv);

	if ( (i > 0) && (!FD_ISSET(iHandle, &eset)) && (FD_ISSET(iHandle, &wset)) )
	{
		err = 1;
	}
	else if ( i == 0 )
	{
		err = 0;
	}
	else if ( err == -1 )
	{
		err = SocketError();

		if ( err != 0 )
		{
			sError = ext_strerror(err);
			err = -1;
		}
	}

	FD_CLR(iHandle, &rset);
	FD_CLR(iHandle, &wset);
	FD_CLR(iHandle, &eset);

#ifdef HAVE_SSL
	if ( (err != -1) && (m_eSocketMode != esmSOCKET) && (m_bSSLInit == FALSE))
	{
		if ( m_eSocketMode == esmSSLCLIENT )
		{
			err = SSL_connect(m_pSSL);
		}
		else if ( m_eSocketMode == esmSSLSERVER )
		{
			err = SSL_accept(m_pSSL);
		}

		if ( err != 1 )
		{
			ERR_print_errors_fp(stderr);
			err = SSL_get_error(m_pSSL,err);

			if ( (err != SSL_ERROR_WANT_READ) && (err != SSL_ERROR_WANT_WRITE) )
			{
				err = -1;
				ERR_print_errors_fp(stderr);
			}
			else
			{
				err = 0;
			}
		}
		else
		{
			printf("Connected with %s encryption '%s'\n",
				SSL_get_cipher(m_pSSL),
				SSL_get_cipher_version(m_pSSL));

			if ( SSL_get_peer_certificate(m_pSSL) == NULL )
			{
				printf("No certificate\n");
			}
			else if ( (err=SSL_get_verify_result(m_pSSL)) != X509_V_OK )
			{
				printf("Certificate verify failed: '%s'\n",X509_verify_cert_error_string(err));
			}
			m_bSSLInit = TRUE;
		}
	}
#endif

	return err;
}

/** */
int CSocket::SocketError()
{
	int err = 0;
	socklen_t ilen;

	if ( iHandle == -1 )
	{
		return err;
	}

	ilen = sizeof(int);

	if ( getsockopt(iHandle, SOL_SOCKET, SO_ERROR, (char*)&err, &ilen) != 0 )
	{
		err = 0;
	}

	return err;
}

/** */
eConnectState CSocket::Connect( CString Host, bool bAsync )
{
	unsigned int port;
	CString s = "";

	ParseHost( Host, s, port );

	// set default port
	if ( port == 0 )
	{
		port = 411;
	}

	return Connect( s, port, bAsync );
}

/** */
eConnectState CSocket::Connect( CString Host, int Port, bool bAsync )
{
	struct sockaddr_in sin,t_sin;
	SOCKET sockfd;
	eAsyncDns ead;
	int err;

	if ( iHandle != INVALID_SOCKET )
	{
		Disconnect();
	}

	memset((void *)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;

	if ( Host == "" )
	{
		if ( SocketType == estTCP )
		{
			return ecsERROR;
		}
		else
		{
			sin.sin_addr.s_addr = htonl(INADDR_ANY);
		}
	}
	else
	{
		if ( bAsync == FALSE )
		{
			// use blocked gethostbyname
			if ( GetHostByName( Host.Data(), &t_sin ) == FALSE )
			{
#ifdef WIN32
				sError = ext_strerror(WSAGetLastError());
#else
				sError = hstrerror(h_errno);
#endif
				return ecsERROR;
			}
			else
			{
		 		memcpy((void *)&sin.sin_addr, (void *)&t_sin.sin_addr, sizeof(t_sin.sin_addr));
			}
		}
		else
		{
			// use async dns class
			if ( CAsyncDns::Instance() )
			{
				ead = CAsyncDns::Instance()->GetHostByName( Host, &t_sin, &err );
			}
			else
			{
				return ecsERROR;
			}

			if ( ead == eadAGAIN )
			{
				return ecsAGAIN;
			}
			else if ( ead == eadERROR )
			{
#ifdef WIN32
				sError = ext_strerror(err);
#else
				sError = hstrerror(err);
#endif
				return ecsERROR;
			}
			else
			{
				memcpy((void *)&sin.sin_addr, (void *)&t_sin.sin_addr, sizeof(t_sin.sin_addr));
			}
		}
	}

	sin.sin_port = htons((unsigned short)Port);

	// socket
	if ( SocketType == estTCP )
	{
		if ((sockfd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
		{
			sError = ext_strerror(errno);
			return ecsERROR;
		}
	}
	else
	{
		if( (sockfd = socket( PF_INET, SOCK_DGRAM, 0 )) < 0 )
		{
			sError = ext_strerror(errno);
			return ecsERROR;
		}
	}

	// set async flag
#ifdef WIN32
	unsigned long flag = bAsync;
	if ( ioctlsocket(sockfd, FIONBIO, &flag ) != 0 )
#else
	int flag = bAsync;
	if ( ioctl(sockfd, FIONBIO, &flag ) != 0 )
#endif
	{
		sError = ext_strerror(errno);
		return ecsERROR;
	}

	if ( Host != "" )
	{
		// connect
		if ( connect(sockfd, (struct sockaddr *)&sin, sizeof(sin)) != 0 )
		{
#ifdef WIN32
			int e = WSAGetLastError();
			if ( (e != WSAEINPROGRESS) && (e != WSAEWOULDBLOCK) )
#else
			if ( errno != EINPROGRESS )
#endif
			{
				sError = ext_strerror(errno);
#ifdef WIN32
				closesocket(sockfd);
#else
				close(sockfd);
#endif
				return ecsERROR;
			}
		}
	}
	else
	{
		if ( bind(sockfd, (struct sockaddr *) &sin, sizeof(sin)) < 0 )
		{
			sError = ext_strerror(errno);
#ifdef WIN32
			closesocket(sockfd);
#else
			close(sockfd);
#endif
			return ecsERROR;
		}
	}

	iHandle = sockfd;

	return ecsSUCCESS;
}

/** */
int CSocket::Disconnect()
{
	if ( iHandle != INVALID_SOCKET )
	{
#ifdef HAVE_SSL
		if ( (m_eSocketMode != esmSOCKET) && (m_pSSL) )
		{
			SSL_free(m_pSSL);
			m_pSSL = 0;
		}
#endif

#ifdef WIN32
		closesocket(iHandle);
#else
		close(iHandle);
#endif
#ifdef HAVE_SSL
		if ( (m_eSocketMode != esmSOCKET) && (m_pCTX) )
		{
			SSL_CTX_free(m_pCTX);
			m_pCTX = 0;
		}
#endif
		m_eSocketMode = esmSOCKET;
		iHandle       = INVALID_SOCKET;
		m_bSSLInit    = FALSE;
	}

	return 0;
}

/** */
int CSocket::Read( char * buffer, int len, int sec_timeout, int usec_timeout )
{
	int l,i;
	struct timeval tv;
	fd_set readset;
	socklen_t cli_len;
	struct sockaddr cli_addr;

	cli_len = sizeof(cli_addr);

	l = 0;
	i = 0;

	if ( (iHandle == INVALID_SOCKET) || (!buffer) || (len <= 0) )
	{
		return -1;
	}
	
#ifdef HAVE_SSL
	if ( (m_eSocketMode == esmSSLCLIENT) || (m_eSocketMode == esmSSLSERVER) )
	{
		l = SSL_read(m_pSSL, buffer, len);

		if ( (l <= 0) && ((i=IsConnect()) != -1) )
		{
//			printf("READ: %d %d\n",l,i);

			l = SSL_get_error(m_pSSL,l);

			if ( (l != SSL_ERROR_WANT_READ) && (l != SSL_ERROR_WANT_WRITE) )
			{
				l = -1;
				ERR_print_errors_fp(stderr);
			}
			else
			{
				l = i = 0;
			}
		}
	}
	else
#endif
	{
		if ( IsConnect() < 0 )
		{
			i = 1;
			l = 0;
		}
		else
		{
		FD_ZERO(&readset);
		FD_SET(iHandle, &readset);

		tv.tv_sec  = sec_timeout;
		tv.tv_usec = usec_timeout;

		i = select(FD_SETSIZE, &readset, NULL, NULL, &tv);

		if ( (i > 0) && (FD_ISSET(iHandle,&readset)) )
		{
			// handle udp socket
			if ( SocketType == estUDP )
			{
				if ( (l=recvfrom(iHandle,buffer,len,0,&cli_addr,&cli_len)) < 0 )
				{
#ifdef WIN32
					int e = WSAGetLastError();
					if ( (e != WSAEINPROGRESS) && (e != WSAEWOULDBLOCK) )
					{
#else
					int e = errno;
					if ( (e != EINPROGRESS) && (e != EAGAIN) )
					{
#endif
						sError = ext_strerror(e);
					}
					else
					{
						i = l = 0;
					}
				}
			}
			// handle tcp socket
			else if ( SocketType == estTCP )
			{
				if ( (l=recv(iHandle,buffer,len,0)) < 0 )
				{
#ifdef WIN32
					int e = WSAGetLastError();
					if ( (e != WSAEINPROGRESS) && (e != WSAEWOULDBLOCK) )
					{
#else
					int e = errno;
					if ( (e != EINPROGRESS) && (e != EAGAIN) )
					{
#endif
						sError = ext_strerror(e);
					}
					else
					{
						i = l = 0;
					}
				}
			}
		}
		else if ( i < 0 )
		{
			i = SocketError();

			if ( i == 0 )
			{
				l = i = 0;
			}
			else
			{
				sError = ext_strerror(i);
				i = l = -1;
			}
		}

		FD_CLR(iHandle, &readset);
		}
	}

	// log options TODO: fix
	if ( (CSocket::m_eSocketLog == eslRECV) ||
	     (CSocket::m_eSocketLog == eslBOTH) )
	{
		if ( l > 0 )
		{
			CString s;
			s = "RECV:" + CString().setNum(l);
			if( l <= len )
			{
				s+="\n";
				buffer[l]=0;
				s += buffer;
			}
			CLogFile::Write("dcsocket.log",eltINFO,s);
		}
	}

	/** we are disconnected */
	if ( (i==1) && (l==0) )
	{
		i = errno;
		l = SocketError();

//		printf("1: %s\n",ext_strerror(i).Data());
//		printf("2: %s\n",ext_strerror(l).Data());

		if ( l == 0 )
			l = i;

//		if ( (l != EINPROGRESS) && (l != EAGAIN) )
//		{
			sError = ext_strerror(l);
			l = -1;
//		}
//		else
//		{
//			l = 0;
//		}
	}

	if ( l > 0 )
		CSocket::m_Traffic.AddTraffic( ettRX, l );

	return l;
}

/** */
int CSocket::Write( const unsigned char * buffer, int len, int sec_timeout, int usec_timeout )
{
	int i;
	struct timeval tv;
	fd_set writeset;

	if ( (iHandle == INVALID_SOCKET) || (!buffer) || (len <= 0) )
	{
		return -1;
	}

#ifdef HAVE_SSL
	if ( (m_eSocketMode == esmSSLCLIENT) || (m_eSocketMode == esmSSLSERVER) )
	{
		i = SSL_write(m_pSSL, buffer, len);

		if ( (i == -1) && (IsConnect() != -1) )
		{
//			printf("WRITE: %d %d\n",i,IsConnect());

			i = SSL_get_error(m_pSSL,i);

			if ( (i != SSL_ERROR_WANT_READ) && (i != SSL_ERROR_WANT_WRITE) )
			{
				i = -1;
				ERR_print_errors_fp(stderr);
			}
			else
			{
				i = 0;
			}
		}
	}
	else
#endif
	{
		if ( IsConnect() < 0 )
		{
			i = -1;
		}
		else
		{
		FD_ZERO(&writeset);
		FD_SET(iHandle, &writeset);

		tv.tv_sec  = sec_timeout;
		tv.tv_usec = usec_timeout;

		i = select(FD_SETSIZE, NULL, &writeset, NULL, &tv);

		FD_CLR(iHandle, &writeset);
		}

		if ( i > 0 )
		{
			// handle tcp socket
			if ( SocketType == estTCP )
			{
				i = send( iHandle, (const char*)buffer, len, MSG_NOSIGNAL );
			}
			else
			{
				i = send( iHandle, (const char*)buffer, len, MSG_NOSIGNAL );
			}

			// send error or wrong socket type
			if ( i < 0 )
			{
#ifdef WIN32
				int e = WSAGetLastError();
				if ( (e != WSAEINPROGRESS) && (e != WSAEWOULDBLOCK) )
				{
#else
				int e = errno;
				if ( (e != EINPROGRESS) && (e != 0) && (e != EAGAIN) )
				{
#endif
					sError = ext_strerror(e);
				}
				else
				{
					i = 0;
				}
			}
			else if ( i == 0 )
			{
				i = -1;
			}

		}
		else if ( i < 0 )
		{
			i = SocketError();

			if ( i == 0 )
			{
				i = 0;
			}
			else
			{
				sError = ext_strerror(i);
				i = -1;
			}
		}
	}

	// log options TODO: fix
	if ( (CSocket::m_eSocketLog == eslSEND) ||
	     (CSocket::m_eSocketLog == eslBOTH) )
	{
		if ( i > 0 )
		{
			CString s;
			s = "SEND:" + CString().setNum(len)+"\n";
			((char*)buffer)[i]=0;
			s += (char*)buffer;
			CLogFile::Write("dcsocket.log",eltINFO,s);
		}
	}

	if ( i > 0 )
		CSocket::m_Traffic.AddTraffic( ettTX, i );

	return i;
}

/** */
int CSocket::Accept()
{
	struct timeval tv;
	fd_set readset;
	struct sockaddr_in serv_addr;
	SOCKET s = -1;
	int i;
	socklen_t sin_size;

	sin_size = sizeof(struct sockaddr_in);

	if ( iHandle == INVALID_SOCKET )
	{
		return s;
	}

	FD_ZERO(&readset);
	FD_SET(iHandle, &readset);

	tv.tv_sec  = 0;
	tv.tv_usec = 1;

	i = select(FD_SETSIZE, &readset, NULL, NULL, &tv);

	FD_CLR(iHandle, &readset);

	if ( i > 0 )
	{
		if ((s = accept(iHandle, (struct sockaddr *)&serv_addr, &sin_size)) == INVALID_SOCKET)
		{
			sError = ext_strerror(SocketError());
			return -1;
		}
		else
		{
			// set async flag
#ifdef WIN32
			unsigned long flag = 1;
			if ( ioctlsocket(s, FIONBIO, &flag ) != 0 )
#else
			int flag = 1;
			if ( ioctl(s, FIONBIO, &flag ) != 0 )
#endif
			{
				sError = ext_strerror(errno);
				return -1;
			}
		}
	}
	
	return s;
}

/** */
int CSocket::Listen( int port, CString ip )
{
	SOCKET sockfd;
	struct sockaddr_in serv_addr;

	if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET)
	{
#ifdef WIN32
		sError = ext_strerror(WSAGetLastError());
#else
		sError = ext_strerror(errno);
#endif
		return -1;
	}

	/* Let the kernel reuse the socket address. This lets us run
	   twice in a row, without waiting for the (ip, port) tuple
	   to time out. */
#ifdef WIN32
	const char i = 1;
#else
	int i = 1;
#endif
	if ( setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &i, sizeof(i)) != 0 )
	{
#ifdef WIN32
		sError = ext_strerror(WSAGetLastError());
		closesocket(sockfd);
#else
		sError = ext_strerror(errno);
		close(sockfd);
#endif
		return -1;
	}

	serv_addr.sin_family = AF_INET;
	serv_addr.sin_port = htons((unsigned short)port);

	if ( ip != "" )
	{
		serv_addr.sin_addr.s_addr = inet_addr(ip.Data());

		if ( serv_addr.sin_addr.s_addr == 255 )
		{
			return -1;
		}
	}
	else
	{
		serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	}

	if (bind(sockfd, (struct sockaddr *)&serv_addr, sizeof(struct sockaddr)) == -1)
	{
#ifdef WIN32
		sError = ext_strerror(WSAGetLastError());
		closesocket(sockfd);
#else
		sError = ext_strerror(errno);
		close(sockfd);
#endif
		return -1;
	}

	if (listen(sockfd, 5) == -1)
	{
#ifdef WIN32
		sError = ext_strerror(WSAGetLastError());
		closesocket(sockfd);
#else
		sError = ext_strerror(errno);
		close(sockfd);
#endif
		return -1;
	}

	iHandle = sockfd;

	return 0;
}

/** */
bool CSocket::GetPeerName( CString * host, int * port )
{
	struct sockaddr_in addr;
	socklen_t sin_size;

	if ( iHandle == INVALID_SOCKET )
	{
		return FALSE;
	}

	if ( (!host) || (!port) )
	{
		return FALSE;
	}

	sin_size = sizeof(struct sockaddr_in);

	if ( getpeername( iHandle, (struct sockaddr*)&addr, &sin_size ) == -1 )
	{
		sError = ext_strerror(SocketError());
		return FALSE;
	}

	*host = inet_ntoa(addr.sin_addr);
	*port = ntohs(addr.sin_port);

	return TRUE;
}

/** */
bool CSocket::ChangeSocketMode( eSocketMode mode, CString cert, CString key )
{
	bool res = FALSE;

	switch(mode)
	{
		case esmSOCKET:
		{
			m_eSocketMode = mode;
			res = TRUE;
			break;
		}
#ifdef HAVE_SSL
		case esmSSLCLIENT:
		case esmSSLSERVER:
		{
			if ( ((cert == "") || (key == "")) && (mode == esmSSLSERVER) )
			{
				printf("no cert/key available\n");
				return res;
			}

			if ( m_eSocketMode == esmSOCKET )
			{
				CSSL cssl;

				if ( mode == esmSSLCLIENT )
				{
					if ( (m_pCTX = cssl.InitClientCTX()) == 0 )
					{
						printf("InitClientCTX failed\n");
						return res;
					}
				}
				else
				{
					if ( (m_pCTX = cssl.InitServerCTX()) == 0 )
					{
						printf("InitServerCTX failed\n");
						return res;
					}
					
					if ( cssl.LoadCertificates(m_pCTX, cert.Data(), key.Data() ) == FALSE )
					{
						SSL_CTX_free(m_pCTX);
						m_pCTX = 0;

						printf("load cert/key failed\n");
						return res;
					}
				}

				SSL_CTX_set_mode(m_pCTX, SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER|SSL_MODE_ENABLE_PARTIAL_WRITE);

				if ( (m_pSSL = SSL_new(m_pCTX)) == 0 )
				{
					printf("SSL_new failed\n");
					SSL_CTX_free(m_pCTX);
					m_pCTX = 0;
					SSL_free(m_pSSL);
					m_pSSL = 0;

					return res;
				}

				if ( SSL_set_fd(m_pSSL, iHandle) == 0 )
				{
					printf("SSL_set_fd failed\n");
					SSL_CTX_free(m_pCTX);
					m_pCTX = 0;
					SSL_free(m_pSSL);
					m_pSSL = 0;

					return res;
				}

				m_eSocketMode = mode;
				res = TRUE;
			}
			else
			{
				printf("CSocket: wrong socket mode to change\n");
			}

			break;
		}
#endif
		default:
			break;
	}

	return res;
}

/** http://synergy2.sourceforge.net/ */
CString CSocket::ext_strerror( int err )
{
#ifdef WIN32
	// built-in windows function for looking up error message strings
	// may not look up network error messages correctly.  we'll have
	// to do it ourself.
	static const struct { int m_code; const char* m_msg; } s_netErrorCodes[] = {
		/* 10004 */{WSAEINTR,			"The (blocking) call was canceled via WSACancelBlockingCall"},
		/* 10009 */{WSAEBADF,			"Bad file handle"},
		/* 10013 */{WSAEACCES,			"The requested address is a broadcast address, but the appropriate flag was not set"},
		/* 10014 */{WSAEFAULT,			"WSAEFAULT"},
		/* 10022 */{WSAEINVAL,			"WSAEINVAL"},
		/* 10024 */{WSAEMFILE,			"No more file descriptors available"},
		/* 10035 */{WSAEWOULDBLOCK,		"Socket is marked as non-blocking and no connections are present or the receive operation would block"},
		/* 10036 */{WSAEINPROGRESS,		"A blocking Windows Sockets operation is in progress"},
		/* 10037 */{WSAEALREADY,		"The asynchronous routine being canceled has already completed"},
		/* 10038 */{WSAENOTSOCK,		"At least on descriptor is not a socket"},
		/* 10039 */{WSAEDESTADDRREQ,	"A destination address is required"},
		/* 10040 */{WSAEMSGSIZE,		"The datagram was too large to fit into the specified buffer and was truncated"},
		/* 10041 */{WSAEPROTOTYPE,		"The specified protocol is the wrong type for this socket"},
		/* 10042 */{WSAENOPROTOOPT,		"The option is unknown or unsupported"},
		/* 10043 */{WSAEPROTONOSUPPORT,"The specified protocol is not supported"},
		/* 10044 */{WSAESOCKTNOSUPPORT,"The specified socket type is not supported by this address family"},
		/* 10045 */{WSAEOPNOTSUPP,		"The referenced socket is not a type that supports that operation"},
		/* 10046 */{WSAEPFNOSUPPORT,	"BSD: Protocol family not supported"},
		/* 10047 */{WSAEAFNOSUPPORT,	"The specified address family is not supported"},
		/* 10048 */{WSAEADDRINUSE,		"The specified address is already in use"},
		/* 10049 */{WSAEADDRNOTAVAIL,	"The specified address is not available from the local machine"},
		/* 10050 */{WSAENETDOWN,		"The Windows Sockets implementation has detected that the network subsystem has failed"},
		/* 10051 */{WSAENETUNREACH,		"The network can't be reached from this hos at this time"},
		/* 10052 */{WSAENETRESET,		"The connection must be reset because the Windows Sockets implementation dropped it"},
		/* 10053 */{WSAECONNABORTED,	"The virtual circuit was aborted due to timeout or other failure"},
		/* 10054 */{WSAECONNRESET,		"The virtual circuit was reset by the remote side"},
		/* 10055 */{WSAENOBUFS,			"No buffer space is available or a buffer deadlock has occured. The socket cannot be created"},
		/* 10056 */{WSAEISCONN,			"The socket is already connected"},
		/* 10057 */{WSAENOTCONN,		"The socket is not connected"},
		/* 10058 */{WSAESHUTDOWN,		"The socket has been shutdown"},
		/* 10059 */{WSAETOOMANYREFS,	"BSD: Too many references"},
		/* 10060 */{WSAETIMEDOUT,		"Attempt to connect timed out without establishing a connection"},
		/* 10061 */{WSAECONNREFUSED,	"The attempt to connect was forcefully rejected"},
		/* 10062 */{WSAELOOP,			"Undocumented WinSock error code used in BSD"},
		/* 10063 */{WSAENAMETOOLONG,	"Undocumented WinSock error code used in BSD"},
		/* 10064 */{WSAEHOSTDOWN,		"Undocumented WinSock error code used in BSD"},
		/* 10065 */{WSAEHOSTUNREACH,	"No route to host"},
		/* 10066 */{WSAENOTEMPTY,		"Undocumented WinSock error code"},
		/* 10067 */{WSAEPROCLIM,		"Undocumented WinSock error code"},
		/* 10068 */{WSAEUSERS,			"Undocumented WinSock error code"},
		/* 10069 */{WSAEDQUOT,			"Undocumented WinSock error code"},
		/* 10070 */{WSAESTALE,			"Undocumented WinSock error code"},
		/* 10071 */{WSAEREMOTE,			"Undocumented WinSock error code"},
		/* 10091 */{WSASYSNOTREADY,		"Underlying network subsytem is not ready for network communication"},
		/* 10092 */{WSAVERNOTSUPPORTED,	"The version of WinSock API support requested is not provided in this implementation"},
		/* 10093 */{WSANOTINITIALISED,	"WinSock subsystem not properly initialized"},
		/* 10101 */{WSAEDISCON,			"Virtual circuit has gracefully terminated connection"},
		/* 11001 */{WSAHOST_NOT_FOUND,	"The specified host is unknown"},
		/* 11002 */{WSATRY_AGAIN,		"A temporary error occurred on an authoritative name server"},
		/* 11003 */{WSANO_RECOVERY,		"A non-recoverable name server error occurred"},
		/* 11004 */{WSANO_DATA,			"The requested name is valid but does not have an IP address"},
		/* end   */{0,					NULL}
	};

	for (unsigned int i = 0; s_netErrorCodes[i].m_code != 0; ++i)
	{
		if ( s_netErrorCodes[i].m_code == err )
		{
			return s_netErrorCodes[i].m_msg;
		}
	}
#endif
	return strerror(err);
}
