/***************************************************************************
                          chubsearch.cpp  -  description
                             -------------------
    begin                : Fri Mar 15 2002
    copyright            : (C) 2002-2004 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>

#ifndef WIN32
#include <unistd.h>
#else
#include <io.h>
#endif

#include <dclib/dcos.h>
#include <dclib/dclib.h>
#include <dclib/dcobject.h>
#include <dclib/cconfig.h>
#include <dclib/cclient.h>
#include <dclib/core/cmanager.h>
#include <dclib/cmessagehandler.h>
#include <dclib/cconnectionmanager.h>
#include <dclib/cfilemanager.h>

#include "chubsearch.h"

/** */
CHubSearch::CHubSearch()
{
	m_pHubServerList = 0;
	m_pSearchList    = 0;
	m_ehSearchType   = ehstNONE;
	m_bEnableTag     = FALSE;

	m_pHubSearchClientList = new CList<CObject>();

	m_pCallback = new CCallback<CHubSearch>( this, &CHubSearch::DC_ManagerCallBack );
	CManager::Instance()->Add( m_pCallback );

	m_SearchSocket.SetCallBackFunction( new CCallback<CHubSearch>( this, &CHubSearch::DC_SearchSocketCallBack ) );
	
	SetInstance(this);
}

/** */
CHubSearch::~CHubSearch()
{
	m_SearchSocket.Disconnect();

	m_HubSearchCallBackLock.Lock();

	if ( m_pCallback )
	{
		if ( CManager::Instance() )
			CManager::Instance()->Remove( m_pCallback );
		delete m_pCallback;
		m_pCallback = 0;
	}

	m_HubSearchCallBackLock.UnLock();

	// clear the client list
	HubSearchClientListThread.Lock();

	sHubSearchClient * HubSearchClient;

	if ( m_pHubSearchClientList != 0 )
	{
		while ( ( (HubSearchClient = (sHubSearchClient *)m_pHubSearchClientList->Next(0)) != NULL) )
		{
			HubSearchClient->SetCallBackFunction(0);
			HubSearchClient->Disconnect(TRUE);
			m_pHubSearchClientList->Remove(HubSearchClient);
			delete HubSearchClient;
		}

		delete m_pHubSearchClientList;
		m_pHubSearchClientList = 0;
	}

	HubSearchClientListThread.UnLock();

	// clear hubserverlist
	if ( m_pHubServerList != 0 )
	{
		delete m_pHubServerList;
		m_pHubServerList = 0;
	}
}

/** passive search results */
bool CHubSearch::HandleMessage( CMessageSearchResult * MessageSearchResult )
{
	bool res = FALSE;

	if ( m_ehSearchType == ehstEXTERNAL )
	{
		if ( CDownloadManager::Instance() )
		{
			res = CDownloadManager::Instance()->DLM_HandleSearch(MessageSearchResult);
		}
	}
	else
	{
		SendSearchResult(MessageSearchResult);
		res = TRUE;
	}

	return res;
}

/** */
bool CHubSearch::ExternalSearch( bool enabled )
{
	bool res = FALSE;

	/* TODO:
	 * add active searches ... we need to listen to the socket ...
	 * like (copy/past from dcgui ...) if ( Connect( "", Config->GetUDPListenPort(), estUDP ) != 0 )
         * and if it failed we fallback to passive search ...
	 * TODO:
	 * we need to disconnect if we add the active search ;-) see StopSearch ...
	*/
	
	if ( (m_ehSearchType == ehstNONE) && (enabled) )
	{
		m_ehSearchType = ehstEXTERNAL;
		res = TRUE;
	}
	else if ( (m_ehSearchType == ehstEXTERNAL) && (!enabled) )
	{
		m_ehSearchType = ehstNONE;
		res = TRUE;
	}

	return res;
}

/** */
int CHubSearch::StartSearch( eHubSearchHubs hubs, bool multi, CList<CObject> * searchlist, CStringList * hublist, CString s )
{
	CObject * object;
	sHubSearchClient * HubSearchClient;

	m_bMultiSearch = multi;
	m_ehSearchType = ehstNONE;

	// reset current server
	m_sCurrentServer = 0;
	m_nCurrentHub    = 0;
	m_nError         = 0;

	// set start time
	m_tStartTime = time(0);

	// clear client list
	HubSearchClientListThread.Lock();

	while ( ( (HubSearchClient = (sHubSearchClient *)m_pHubSearchClientList->Next(0)) != NULL) )
	{
		m_pHubSearchClientList->Remove(HubSearchClient);

		HubSearchClient->SetCallBackFunction(0);
		HubSearchClient->Disconnect();
		delete HubSearchClient;
	}

	HubSearchClientListThread.UnLock();

	if ( m_pHubServerList != 0 )
	{
		delete m_pHubServerList;
		m_pHubServerList = 0;
	}

	m_pSearchList = searchlist;

	// check if a user search in the list
	m_bHandleUserList = FALSE;
	object = 0;

	while ( (object=m_pSearchList->Next(object)) != 0 )
	{
		if ( ((CDCMessage*)object)->m_eType == DC_MESSAGE_SEARCH_USER )
		{
			m_bHandleUserList = TRUE;
			break;
		}
	}

	if ( CConfig::Instance()->GetMode() != ecmPASSIVE )
	{
		if (  m_SearchSocket.Connect( "", CConfig::Instance()->GetUDPListenPort(), estUDP ) != 0 )
		{
			return -1;
		}
	}

	// connected hubs
	if ( (hubs == ehshCONNECTEDALL) ||
	     (hubs == ehshCONNECTEDSINGLE) )
	{
		// get connected hub count
		if ( CConnectionManager::Instance()->GetConnectedHubCount() == 0 )
		{
			m_SearchSocket.Disconnect();
			return -1;
		}

		m_sHubName = s;
		m_pCurrentSearchObject = 0;
		m_tHubSearchTimeout = time(0);

		m_ehSearchType = ehstLOCAL;

		// send first search
		if ( SendSearch(m_sHubName) <= 0 )
		{
			m_ehSearchType = ehstNONE;
			m_SearchSocket.Disconnect();
			return -1;
		}
	}
	else
	{
		// all public hubs
		if ( hubs == ehshPUBLIC )
		{
			// get public hub server list
			if ( hublist != 0 )
				m_pHubServerList = hublist;
			else
			{
				m_pHubServerList = CConfig::Instance()->GetPublicHubServerList();
			}
		}
		// all bookmark hubs
		else if ( hubs == ehshBOOKMARK )
		{
			// get bookmark hub server list
			m_pHubServerList = CConfig::Instance()->GetBookmarkHubServerList();
		}

		if ( (m_pHubServerList == 0) || (m_pHubServerList->Count() <= 0) )
		{
			m_SearchSocket.Disconnect();
			return -1;
		}

		m_ehSearchType = ehstGLOBAL;
	}

	return 0;
}

/** */
void CHubSearch::StopSearch()
{
	if ( m_ehSearchType != ehstNONE )
	{
		m_ehSearchType = ehstSTOP;

		m_SearchSocket.Disconnect(TRUE);
	}
}

/** */
void CHubSearch::SendDebug( CString s )
{
	CMessageLog * MessageLog = new CMessageLog();

	MessageLog->sMessage = s;
	MessageLog->m_eType  = DC_MESSAGE_LOG;

	if ( DC_CallBack( MessageLog ) == -1 )
	{
		printf("CallBack failed (state)...\n");
		delete MessageLog;
	}
}

/** */
void CHubSearch::SendSearchResult( CObject * object )
{
	if ( DC_CallBack( object ) == -1 )
	{
		printf("CallBack failed (state)...\n");
		delete object;
	}
}

/** */
void CHubSearch::UpdateClients()
{
	if ( m_pHubSearchClientList != 0 )
	{
		sHubSearchClient * HubSearchClient = 0;

		while ( ( (HubSearchClient = (sHubSearchClient *)m_pHubSearchClientList->Next(HubSearchClient)) != NULL) )
		{
			HubSearchClient->Thread(0);
		}
	}
}

/** */
int CHubSearch::DC_ManagerCallBack( CObject *, CObject * )
{
	if ( m_ehSearchType == ehstNONE )
	{
		return 0;
	}

	// handle search socket
	m_SearchSocket.Thread(0);
	
	m_HubSearchCallBackLock.Lock();

	UpdateClients();

	if ( (m_ehSearchType == ehstWAITTIMEOUT) &&
	     ((time(0)-m_tHubSearchTimeout) >= 120) )
	{
		// stop search
		StopSearch();
	}
	else if ( (m_ehSearchType == ehstGLOBAL) ||
	     (m_ehSearchType == ehstWAITTIMEOUT) ||
	     (m_ehSearchType == ehstSTOP) )
	{
		CheckClient();

		if ( m_ehSearchType == ehstGLOBAL )
		{
			NewClient();
		}

		if ( m_ehSearchType == ehstSTOP )
		{
			HubSearchClientListThread.Lock();

			if ( m_pHubSearchClientList->Count() == 0 )
			{
				printf("search end\n");
				m_ehSearchType = ehstNONE;
			}

			HubSearchClientListThread.UnLock();
		}
	}
	else if ( m_ehSearchType == ehstLOCAL )
	{
		if ( (time(0)-m_tHubSearchTimeout) >= 11 )
		{
			// send first search
			if ( SendSearch(m_sHubName) <= 0 )
			{
				m_ehSearchType = ehstWAITTIMEOUT;
			}

			m_tHubSearchTimeout = time(0);
		}
	}

	m_HubSearchCallBackLock.UnLock();

	return 0;
}

/** */
void CHubSearch::CheckClient()
{
	sHubSearchClient * HubSearchClient,* OldHubSearchClient;

	HubSearchClientListThread.Lock();

	if ( m_pHubSearchClientList != 0 )
	{
		HubSearchClient = OldHubSearchClient = 0;

		// check all hubs in the list
		while ( ( (HubSearchClient = (sHubSearchClient *)m_pHubSearchClientList->Next(HubSearchClient)) != NULL) )
		{
			if ( (m_ehSearchType == ehstSTOP) && (HubSearchClient->m_bSearchRemove != TRUE) )
			{
				// set flag to remove all clients
				HubSearchClient->SetCallBackFunction(0);
				HubSearchClient->m_bSearchRemove = TRUE;

				OldHubSearchClient = HubSearchClient;
			}
			// remove the client or if timeout
			else if ( (HubSearchClient->m_bSearchRemove == TRUE) || ((time(0)-HubSearchClient->m_tSearchHubTimeout) >= 300) )
			{
				// check remove timeout
				if ( (HubSearchClient->m_bSearchRemove == TRUE) &&
				     ((time(0)-HubSearchClient->m_tSearchHubTimeout) < 6) &&
				     (m_ehSearchType != ehstSTOP) )
				{
					// hold this client in the list
					OldHubSearchClient = HubSearchClient;
				}
				else
				{
					HubSearchClient->SetCallBackFunction(0);

					if ( HubSearchClient->GetConnectionState() != estNONE )
					{
						if ( HubSearchClient->GetConnectionState() != estDISCONNECTING )
						{
							HubSearchClient->Disconnect(TRUE);
						}
						OldHubSearchClient = HubSearchClient;
					}
					else
					{
						HubSearchClient->Stop();
						m_pHubSearchClientList->Remove(HubSearchClient);
						delete HubSearchClient;
						HubSearchClient = OldHubSearchClient;
					}
				}
			}
			else
			{
				// send search
				if ( (HubSearchClient->m_bSearchEnable == TRUE) && ((time(0)-HubSearchClient->m_tSearchTimeout) >= 11) )
				{
					if ( SendSearch(HubSearchClient) == FALSE )
					{
						// no more searches, remove the client
						HubSearchClient->m_bSearchRemove = TRUE;
					}
					else
					{
						// reset hubtimeout
						HubSearchClient->m_tSearchHubTimeout = time(0);
					}
				}

				OldHubSearchClient = HubSearchClient;
			}
		}
	}

	HubSearchClientListThread.UnLock();
}

/** */
void CHubSearch::NewClient()
{
	int l;
	CString server,addr,s;
	sHubSearchClient * HubSearchClient;

	if ( m_pHubServerList == 0 )
	{
		return;
	}

	// done ?
	if ( m_pHubServerList->Count() == m_nCurrentHub )
	{
		SendDebug("Serverlist done\n");

		m_ehSearchType = ehstWAITTIMEOUT;
		m_tHubSearchTimeout = time(0);

		return;
	}

	HubSearchClientListThread.Lock();

	l = m_pHubSearchClientList->Count();

	HubSearchClientListThread.UnLock();

	for(;(l<m_nMaxThreads)&&(m_nCurrentHub<m_pHubServerList->Count());l++)
	{
		HubSearchClient = new sHubSearchClient;

		HubSearchClient->m_bSearchRemove = FALSE;
		HubSearchClient->m_tSearchHubTimeout = time(0);

		if ( CConfig::Instance()->GetMode() == ecmPASSIVE )
		{
			HubSearchClient->m_tSearchHubTimeout += 30;
		}

		HubSearchClient->SetNick(CConfig::Instance()->GetSearchNick());
		HubSearchClient->SetComment(CConfig::Instance()->GetDescription(!m_bEnableTag));
		HubSearchClient->SetConnectionType(CConfig::Instance()->GetSpeed());
		HubSearchClient->SetEMail(CConfig::Instance()->GetEMail());
		HubSearchClient->SetVersion(VERSION);
		HubSearchClient->SetShareSize(CString().setNum(CFileManager::Instance()->GetShareSize()));
		HubSearchClient->SetMode(CConfig::Instance()->GetMode());
		HubSearchClient->SetHandleUserList(m_bHandleUserList);
		HubSearchClient->SetHandleSearch(FALSE);
		HubSearchClient->SetHandleMyinfo(FALSE);
		HubSearchClient->SetHandleForceMove(FALSE);
		HubSearchClient->SetHandleTransfer(FALSE);
		HubSearchClient->SetCallBackFunction( new CCallback<CHubSearch>( this, &CHubSearch::DC_ClientCallBack ) );

		if ( m_pHubServerList->Next((CObject*&)m_sCurrentServer) != 0 )
		{
			server = *m_sCurrentServer;

			m_nCurrentHub++;

			SendDebug( "Search on: [" + CString().setNum(m_nCurrentHub) + "] " + server );

			HubSearchClientListThread.Lock();

			m_pHubSearchClientList->Add((CObject*)HubSearchClient);

			HubSearchClient->Connect(server);
		}
		else
		{
			// end on error
			m_nCurrentHub = m_pHubServerList->Count();
		}

		HubSearchClientListThread.UnLock();
	}
}

/** */
CString CHubSearch::GetSearchString( CMessageSearchFile * msg )
{
	CString s = "";

	if ( msg->m_bLocal == TRUE )
		s += "Hub:";
	s += msg->m_sSource;
	s += " ";

	if ( msg->m_bSizeLimit == FALSE )
		s += "F?";
	else
		s += "T?";

	if ( msg->m_bSizeAtMost == FALSE )
		s += "F?";
	else
		s += "T?";

	s += CString().setNum(msg->m_nSize) + "?";
	s += CString().setNum(msg->m_eFileType) + "?";
	s += msg->m_sString.Replace(' ',"$") + "|";

	// passive searches with multisearch
	if ( msg->m_bLocal == TRUE )
		s = "$Search " + s;
	else if ( !msg->m_bMulti )
		s = "$Search " + s;
	else
		s = "$MultiSearch " + s;

	return s;
}

/** send the search to the clients */
int CHubSearch::SendSearch( CString s )
{
	int i = 0;
	CString search;
	CList<DCHubObject> list;
	DCHubObject * HubObject;

	if ( (m_pCurrentSearchObject=m_pSearchList->Next(m_pCurrentSearchObject)) != 0 )
	{
		switch(((CDCMessage*)m_pCurrentSearchObject)->m_eType)
		{
			case DC_MESSAGE_SEARCH_USER:
			{
				CMessageSearchUser * msg = (CMessageSearchUser*)m_pCurrentSearchObject;

				if ( CConnectionManager::Instance()->IsUserOnline( msg->m_sNick, s, "", &list ) == TRUE )
				{
					HubObject = 0;

					while ( (HubObject=list.Next(HubObject)) != 0 )
					{
						CMessageSearchResultUser * MessageSearchResultUser = new CMessageSearchResultUser();

						MessageSearchResultUser->m_eType    = DC_MESSAGE_SEARCHRESULT_USER;
						MessageSearchResultUser->m_sHubName = HubObject->m_sHubName;
						MessageSearchResultUser->m_sNick    = msg->m_sNick;

						SendSearchResult(MessageSearchResultUser);
					}

					list.Clear();
				}

				i = 1;

				break;
			}

			case DC_MESSAGE_SEARCH_FILE:
			{
				CMessageSearchFile * msg = (CMessageSearchFile*)m_pCurrentSearchObject;

				msg->m_bMulti = m_bMultiSearch;
				i = CConnectionManager::Instance()->SendStringToConnectedServers(msg,s);

				break;
			}

			default:
			{
				break;
			}
		}
	}

	return i;
}

/** send the search to this client */
bool CHubSearch::SendSearch( sHubSearchClient * HubSearchClient )
{
	CString search;
	CObject *object;
	bool res = FALSE;

	object = HubSearchClient->m_pSearchCurrent;

	while ( (res == FALSE) && ((object=m_pSearchList->Next(object)) != 0) )
	{
		switch(((CDCMessage*)object)->m_eType)
		{
			case DC_MESSAGE_SEARCH_USER:
			{
				CMessageSearchUser * msg = (CMessageSearchUser*)object;

				if ( HubSearchClient->UserList()->IsUserOnline( msg->m_sNick ) == TRUE )
				{
					CMessageSearchResultUser * MessageSearchResultUser = new CMessageSearchResultUser();

					MessageSearchResultUser->m_eType    = DC_MESSAGE_SEARCHRESULT_USER;
					MessageSearchResultUser->m_sHubName = HubSearchClient->GetHubName();
					MessageSearchResultUser->m_sNick    = msg->m_sNick;

					SendSearchResult(MessageSearchResultUser);
				}

				break;
			}

			case DC_MESSAGE_SEARCH_FILE:
			{
				CMessageSearchFile * msg = (CMessageSearchFile*)object;

				msg->m_bMulti = m_bMultiSearch;
				search = GetSearchString(msg);

				if ( search != "" )
					HubSearchClient->SendString(search);

				// update current object
				HubSearchClient->m_pSearchCurrent = object;
				// update timeout
				HubSearchClient->m_tSearchTimeout = time(0);

				res = TRUE;

				break;
			}

			default:
			{
				break;
			}
		}
	}

	return res;
}

/** */
int CHubSearch::DC_ClientCallBack( CObject * Client, CObject * Object )
{
	ClientThread.Lock();

	CClient * c = (CClient*)Client;
	CObject * o = Object;

	if ( (c == 0) || (o == 0) )
	{
		ClientThread.UnLock();
		return -1;
	}

	CDCMessage *DCMsg;

	DCMsg = (CDCMessage*) o;

	switch ( DCMsg->m_eType )
	{
		case DC_MESSAGE_CONNECTION_STATE:
		{
			CMessageConnectionState *msg = (CMessageConnectionState*) o;

			switch(msg->m_eState)
			{
				case estCONNECTED:
				{
					break;
				}

				case estSOCKETERROR:
				{
					SendDebug( "SocketError on " + c->GetIP() + CString(":") + CString().setNum(c->GetPort()) + " [" + msg->m_sMessage + "]");
					m_nError++;
					break;
				}

				case estDISCONNECTED:
				{
					((sHubSearchClient*)c)->m_tSearchHubTimeout = time(0);
					((sHubSearchClient*)c)->SetCallBackFunction(0);
					((sHubSearchClient*)c)->m_bSearchRemove = TRUE;
				}

				default:
				{
					break;
				}
			}

			break;
		}

		case DC_MESSAGE_VALIDATEDENIDE:
		{
			SendDebug( "Validate denide " + c->GetIP() + CString(":") + CString().setNum(c->GetPort()) );
			c->Disconnect(TRUE);
			break;
		}

		case DC_MESSAGE_HUBISFULL:
		{
			SendDebug( "Hub is full on " + c->GetIP() + CString(":") + CString().setNum(c->GetPort()) );
			c->Disconnect(TRUE);
			break;
		}

		case DC_MESSAGE_FORCEMOVE:
		{
			SendDebug( "Force move on " + c->GetIP() + CString(":") + CString().setNum(c->GetPort()) );
			c->Disconnect(TRUE);
			break;
		}

		case DC_MESSAGE_LOCK:
		case DC_MESSAGE_HUBNAME:
		case DC_MESSAGE_CHAT:
		case DC_MESSAGE_MYNICK:
		case DC_MESSAGE_PRIVATECHAT:
		case DC_MESSAGE_SEARCH:
		case DC_MESSAGE_HELLO:
		case DC_MESSAGE_QUIT:
		case DC_MESSAGE_GETINFO:
		{
			break;
		}

		case DC_MESSAGE_MYINFO:
		{
			CString s;
			CMessageMyInfo * MessageMyInfo = (CMessageMyInfo *)o;

			if ( MessageMyInfo->m_sNick == CConfig::Instance()->GetSearchNick() )
			{
				if ( m_bHandleUserList == FALSE )
				{
					// enable search
					((sHubSearchClient*)c)->m_bSearchEnable = TRUE;

					// send search
					//SendSearch(c);

					// remove the client only for active searches
					//if ( CConfig::Instance()->GetMode() != ecmPassive )
					//	RemoveClient(c);
				}
			}
			break;
		}

		case DC_MESSAGE_NICKLIST:
		{
			if ( m_bHandleUserList == TRUE )
			{
				// enable search
				((sHubSearchClient*)c)->m_bSearchEnable = TRUE;

				// send search
				//SendSearch(c);

				// remove the client only for active searches
				//if ( CConfig::Instance()->GetMode() != ecmPassive )
				//	RemoveClient(c);
			}
			break;
		}

		case DC_MESSAGE_SEARCHRESULT:
		{
			if ( CConfig::Instance()->GetMode() == ecmPASSIVE )
			{
				SendSearchResult(o);
				o = 0;
			}
			break;
		}

		default:
		{
			if ( dclibVerbose() > 1 )
				printf("callback: %d\n",DCMsg->m_eType);
			break;
		}
	}

	if ( o )
		delete o;

	ClientThread.UnLock();

	return 0;
}

/** */
int CHubSearch::DC_SearchSocketCallBack( CObject *, CObject * object )
{
	if ( HandleMessage( (CMessageSearchResult*)object ) == FALSE )
	{
		return -1;
	}
	
	return 0;
}
