/* 
 * $Id: ctksignal.c,v 1.37 2000/07/24 19:28:58 terpstra Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <glib.h>
#include <stdlib.h>
#include <arr.h>

#include "ctk.h"

typedef struct CtkSignalHash_S CtkSignalHash;

gboolean ctk_signal_return;
gboolean ctk_signal_run;

struct CtkSignalHash_S 
{
      CtkType object_type;
      GQuark  quark;
      guint   signal_id;
};

static GHashTable *ctk_signal_hash_table = NULL;
static GQuark ctk_signal_handler_quark = 0;

static guint ctk_connection_id = 1;
static guint ctk_private_n_signals = 1;

static guint ctk_signal_hash (gconstpointer h);
static gint ctk_signal_compare(gconstpointer h1, gconstpointer h2);


/* Signal Init */
void ctk_signal_init()
{
      if (!ctk_signal_handler_quark)
      {
	    ctk_signal_handler_quark = g_quark_from_static_string("ctk-signal-handlers");
	    
	    ctk_signal_hash_table = g_hash_table_new(ctk_signal_hash,
						     ctk_signal_compare);
      }
}

/* Execute Signals */
gboolean ctk_signal_execute_signal(CtkObject* object, CtkSignal* sig)
{
	gboolean ret_val;
	
	/* We don't execute disabled signals */
	if (!sig->sig_func)
		return FALSE;
	
	if (sig->object_data)
	{
	      ret_val = sig->sig_func(sig->object_data);
	}
       	/* CtkWidget Signal Prototypes */
	else if (!strcmp(sig->event, "button_press_event") ||
		 !strcmp(sig->event, "button_release_event"))
	{
	      CdkEventButton data;
	      ctk_assert(CTK_CHECK_TYPE(object, CtkTypeWidget), 
	      	"Non-widget getting mouse events!");
	      data.type = CDK_BUTTON_PRESS;
	      data.x    = ctk_mouse_x;
	      data.y    = ctk_mouse_y;
	      data.dx   = ctk_mouse_dx;
	      data.dy   = ctk_mouse_dy;
	      ret_val = sig->sig_func(sig->object, &data, sig->data);
	}
	else if (!strcmp(sig->event, "ctk_drag"))
	{
	      // FIXME: need to do dragging right later
	      CdkEventButton data;
	      ctk_assert(CTK_CHECK_TYPE(object, CtkTypeWidget),
	        "Non-widget getting drag event!");
	      data.type = CDK_BUTTON_PRESS;
	      data.x    = ctk_mouse_x;
	      data.y    = ctk_mouse_y;
	      data.dx   = ctk_mouse_dx;
	      data.dy   = ctk_mouse_dy;
	      ret_val = sig->sig_func(sig->object, &data, sig->data);
	}
	else if (!strcmp(sig->event,"focus_in_event") ||
		 !strcmp(sig->event,"focus_out_event"))
	{
	      CdkEventFocus data;
	      ctk_assert(CTK_CHECK_TYPE(object, CtkTypeWidget),
	      	"Non-widget getting focus events");
	      data.type = CDK_FOCUS_CHANGE;
	      ret_val = sig->sig_func(sig->object, &data, sig->data);
	}
	else if (!strcmp(sig->event,"delete_event"))
	{
	      CdkEvent data;
	      data.type = CDK_DELETE;
	      ret_val = sig->sig_func(sig->object, &data, sig->data);
	}
	else if (!strcmp(sig->event,"destroy_event"))
	{
	      CdkEvent data;
	      data.type = CDK_DESTROY;
	      ret_val = sig->sig_func(sig->object, &data, sig->data);
	}
	else if (!strcmp(sig->event,"enter_notify_event"))
	{
	      CdkEventCrossing data;
	      ctk_assert(CTK_CHECK_TYPE(object, CtkTypeWidget),
	      	"Enter event on a non-widget");
	      data.type = CDK_ENTER_NOTIFY;
	      data.x = ctk_mouse_x;
	      data.y = ctk_mouse_y;
	      ret_val = sig->sig_func(sig->object, &data, sig->data);
	}
	else if (!strcmp(sig->event,"leave_notify_event"))
	{
	      CdkEventCrossing data;
	      ctk_assert(CTK_CHECK_TYPE(object, CtkTypeWidget),
	      	"Leave event on a non-widget");
	      data.type = CDK_LEAVE_NOTIFY;
	      data.x = ctk_mouse_x;
	      data.y = ctk_mouse_y;
	      ret_val = sig->sig_func(sig->object, &data, sig->data);
	}
	else if (!strcmp(sig->event,"key_press_event"))
	{
	      CdkEventKey data;
	      gchar       buf[2];

	      ctk_assert(CTK_CHECK_TYPE(object, CtkTypeWidget),
	      	"Key press event on non-widget");

	      buf[0] = ctk_keyval;
	      buf[1] = 0;

	      data.keyval = ctk_keyval;
	      data.length = 1;
	      data.string = &buf[0];
	      ret_val = sig->sig_func(sig->object, &data, sig->data);
	}
	/* CtkButton Signal Prototypes */
	else if (!strcmp(sig->event,"clicked") ||
		 !strcmp(sig->event,"released") ||
		 !strcmp(sig->event,"pressed") ||
		 !strcmp(sig->event,"enter") ||
		 !strcmp(sig->event,"leave"))
	{
	      ctk_assert(CTK_CHECK_TYPE(object, CtkTypeButton),
	      	"Button signal on non-button (click/release/enter/etc)");
	      ret_val = FALSE;
	      sig->sig_func(CTK_BUTTON(object),sig->data);
	}
	// FIXME: a hack for each widget in here is not really OO...
	// What we should do is have another function ptr in object: emit
	// Then call the ancestors version in each...
	/* CtkCList Signal Prototypes */
	else if (!strcmp(sig->event,"select_row") ||
		 !strcmp(sig->event,"unselect_row"))
	{
	      ctk_assert(CTK_CHECK_TYPE(object, CtkTypeCList),
	      	"(Un)select signal on a non-clist");
	      ret_val = FALSE;
	      sig->sig_func(CTK_CLIST(object),
			    CTK_CLIST(object)->selected_row,
			    CTK_CLIST(object)->selected_col,
			    NULL,
			    sig->data);
	}
	else if (!strcmp(sig->event,"click_column"))
	{
	      ctk_assert(CTK_CHECK_TYPE(object, CtkTypeCList),
	      	"click_column signal on a non-clist");
	      ret_val = FALSE;
	      sig->sig_func(CTK_CLIST(object),
			    CTK_CLIST(object)->selected_col,
			    sig->data);
	}
	else if (!strcmp(sig->event,"changed") &&
		CTK_CHECK_TYPE(object, CtkTypeEditable))
	{
		ret_val = FALSE;
		sig->sig_func(CTK_EDITABLE(object), sig->data);
	}
	/* CtkAdjustment Signal Prototypes */
	else if (!strcmp(sig->event,"changed") ||
		 !strcmp(sig->event,"value_changed"))
	{
	      ctk_assert(CTK_CHECK_TYPE(object, CtkTypeAdjustment),
	      	"changed/value_changed signal on non-adjustment/ediatble");
	      ret_val = FALSE;
	      sig->sig_func(CTK_ADJUSTMENT(object),sig->data);
	}
	/* CtkToggleBUtton Signal Prototypes */
	else if (!strcmp(sig->event,"toggled"))
	{
	      ctk_assert(CTK_CHECK_TYPE(object, CtkTypeToggleButton),
	      	"toggled signal on non-togglebutton");
	      ret_val = FALSE;
	      sig->sig_func(CTK_TOGGLE_BUTTON(object),sig->data);
	}
	else if (CTK_CHECK_TYPE(object,CtkTypeMenuItem) &&
		 !strcmp(sig->event,"activate"))
	{
	      ret_val = FALSE;
	      sig->sig_func(CTK_MENU_ITEM(object),sig->data);
	}
	else if (CTK_CHECK_TYPE(sig->object,CtkTypeEditable) &&
		 !strcmp(sig->event,"activate"))
	{
	      ret_val = FALSE;
	      sig->sig_func(CTK_EDITABLE(object),sig->data);
	}
	else
	      ret_val = FALSE;
	
	return ret_val;
}

/* Lookup Signal ID */
guint ctk_signal_lookup(const gchar* event, CtkType object_type)
{
      CtkType       scan;
      CtkSignalHash hash;
      guint         signal_id;
      
      if (!event)
	    return FALSE;

      hash.quark = g_quark_from_static_string(event);
	
      for (scan = object_type; scan != CtkTypeEND; scan = CTK_PARENT_TYPE(scan))
      {
	    hash.object_type = scan;
	    signal_id = GPOINTER_TO_UINT(g_hash_table_lookup(ctk_signal_hash_table, &hash));

	    if (signal_id)
		  return signal_id;
      }
      
      return 0;
}

/* Create a new Signal */

guint ctk_signal_new(const gchar* event, CtkType object_type)
{
      CtkSignalHash* hash;
      guint          signal_id;

      if (!event)
	    return 0;
	
      if (!ctk_signal_handler_quark)
	    ctk_signal_init();
	
      signal_id = ctk_signal_lookup(event, object_type);
	
      /* If a signal already exists just ignore it */
      if (signal_id)
	    return signal_id;

      hash = g_malloc(sizeof(CtkSignalHash));

      hash->object_type = object_type;
      hash->quark       = g_quark_from_string(event);
      hash->signal_id   = ctk_private_n_signals++;
	
/*	ctk_close();
	printf("E: %s\nQ: %d\nS: %d\n",name,hash->quark,hash->signal_id);
	exit(0); */

      g_hash_table_insert(ctk_signal_hash_table, hash, GUINT_TO_POINTER(hash->signal_id));
	
      return hash->signal_id;
}

/* Attach a signal to an object */
gint ctk_signal_connect(CtkObject* object, gchar *event, CtkSignalFunc func, gpointer data)
{
      CtkSignal* sig;
      guint      signal_id;

      if (!object) 
      {
	    ctk_close();
	    g_error("ctk_signal_connect: CtkObject *object == NULL!");
      }
	
      signal_id = ctk_signal_lookup(event, object->type);

      if (!signal_id) 
      {
	    ctk_close();
	    g_error("ctk_signal_connect: Invalid signal event value '%s'! Type: %d",
		    event, object->type);
      }

      sig = g_malloc(sizeof(CtkSignal));
      sig->object        = object;
      sig->sig_func      = func;
      sig->data          = data;
      sig->object_data   = NULL;
      sig->event         = g_strdup(event);
      sig->connection_id = ctk_connection_id++;
      sig->signal_id     = signal_id;

      object->signals = g_slist_prepend(object->signals,
					(gpointer)sig);

      return sig->connection_id;
}

/* Attach a signal to an object */
gint ctk_signal_connect_after(CtkObject* object, gchar *event, CtkSignalFunc func, gpointer data)
{
      CtkSignal* sig;
      guint      signal_id;

      if (!object) 
      {
	    ctk_close();
	    g_error("ctk_signal_connect: CtkObject *object == NULL!");
      }
	
      signal_id = ctk_signal_lookup(event, object->type);

      if (!signal_id) 
      {
	    ctk_close();
	    g_error("ctk_signal_connect: Invalid signal event value '%s'! Type: %d",
		    event, object->type);
      }

      sig = g_malloc(sizeof(CtkSignal));
      sig->object        = object;
      sig->sig_func      = func;
      sig->data          = data;
      sig->object_data   = NULL;
      sig->event         = g_strdup(event);
      sig->connection_id = ctk_connection_id++;
      sig->signal_id     = signal_id;

      object->signals = g_slist_append(object->signals,
					(gpointer)sig);

      return sig->connection_id;
}

/* Signal Connect Object */
gint ctk_signal_connect_object(CtkObject *object, gchar *event, CtkSignalFunc func, 
			       CtkObject *slot_object)
{
	CtkSignal *sig;
	guint signal_id;
	CtkWidget *widget;		

	if (!object) 
	{
		ctk_close();
		g_error("ctk_signal_connect: CtkObject *object == NULL!");
	}
	
	if (!slot_object) 
	{
		ctk_close();
		g_error("ctk_signal_connect: CtkObject *slot_object == NULL!");
	}
	
	widget = CTK_WIDGET(object);
	
	signal_id = ctk_signal_lookup(event,((CtkObject *)widget)->type);

	if (!signal_id) {
		ctk_close();
		g_error("ctk_signal_connect: Invalid signal event value '%s'!",event);
	}

	sig = g_malloc(sizeof(CtkSignal));
	sig->object = object;
	sig->sig_func = (gpointer)func;
	sig->data = NULL;
	sig->object_data = slot_object;
	sig->event = g_strdup(event);
	sig->connection_id = ctk_connection_id++;
	sig->signal_id = signal_id;

	((CtkObject *)widget)->signals = g_slist_append(((CtkObject *)widget)->signals,(gpointer)sig);

	return(sig->connection_id);
}

void ctk_signal_disconnect(CtkObject* object, gint connection_id)
{
      GSList*    scan;
      CtkSignal* sig;

      for (scan = object->signals; scan; scan = scan->next)
      {
	    sig = (CtkSignal*)scan->data;
	    if (sig->connection_id == connection_id)
		  break;
      }
      
      if (scan)
      {
	      /* Disable signal - removing might modify the list within the signal! */
	      sig->sig_func = NULL;
      }
      
      // FIXME: leaks - write signal garbage collector compactor
}

void ctk_signal_emit_by_name(CtkObject* object, gchar* event)
{
	guint      signal_id;
	GSList*    signal_scan;
	CtkSignal* signal;

	if (!object || !object->signals)
	    return;
	
	ctk_assert(CTK_CHECK_TYPE(object, CtkTypeObject),
		"signal emit on non-object");

	signal_id = ctk_signal_lookup(event, CTK_OBJECT(object)->type);
		
	if (!signal_id) return;

	for (signal_scan = CTK_OBJECT(object)->signals; 
	     signal_scan; signal_scan = signal_scan->next)
	{
	      signal = (CtkSignal *)signal_scan->data;
	      
	      if (signal->signal_id == signal_id)
	      {
		    ctk_signal_return = ctk_signal_execute_signal(object, signal);
		    ctk_signal_run = TRUE;
	      }
	}
}

gboolean ctk_event_emit_by_name(CtkObject* object, gchar* event)
{
      CtkWidget* widget_scan;
      CtkWidget* widget_next;

      gboolean propogate;

      if (!strcmp(event, "delete_event") ||
	  !strcmp(event, "destroy_event") ||
	  !strcmp(event, "enter_notify_event") ||
	  !strcmp(event, "leave_notify_event") ||
	  !strcmp(event, "focus_in_event") ||
	  !strcmp(event, "focus_out_event"))
      {
	    propogate = FALSE;
      }
      else if (!strcmp(event, "button_press_event") ||
	       !strcmp(event, "button_release_event") ||
	       !strcmp(event, "key_press_event") ||
	       !strcmp(event, "key_release_event") ||
	       !strcmp(event, "ctk_drag"))
      {
	    propogate = TRUE;
      }
      else
      {
	    ctk_assert_not_reached("Events should all be eaten by now");
	    return FALSE; //  stop warnings
      }

      ctk_signal_return = FALSE;
      ctk_signal_run    = FALSE;
      for (widget_scan = CTK_WIDGET(object); widget_scan; widget_scan = widget_next)
      {
	    if (widget_scan->node && widget_scan->node->parent)
	    {
		  widget_next = CTK_WIDGET(widget_scan->node->parent->data);
	    }
	    else
	    {
		  widget_next = NULL;
	    }
	    
	    if (widget_scan->sensitive)
		  ctk_signal_emit_by_name(CTK_OBJECT(widget_scan), event);

	    if (ctk_signal_return == TRUE) break;
	    if (ctk_signal_run == TRUE && propogate == FALSE) break;
      }

      return ctk_signal_return;
}

static guint ctk_signal_hash (gconstpointer h)
{
	const CtkSignalHash *hash = h;
	
	return hash->object_type ^ hash->quark;
}

static gint ctk_signal_compare(gconstpointer h1, gconstpointer h2)
{
      const CtkSignalHash* hash1 = h1;
      const CtkSignalHash* hash2 = h2;
      
      return (hash1->quark       == hash2->quark &&
	      hash1->object_type == hash2->object_type);
}

/* Destroy All signals for a widget */
void ctk_signal_destroy_all_signals(CtkWidget* widget)
{
	GSList*    list;
	CtkSignal* sig;
	
	for (list = CTK_OBJECT(widget)->signals; list; list = list->next)
	{
		sig = (CtkSignal *)list->data;
		if (sig->event)
		    free(sig->event);
		free(sig);
	}
	
	g_slist_free(CTK_OBJECT(widget)->signals);
	CTK_OBJECT(widget)->signals = NULL;
}
