/*--------------------------------------------------------------------------+
$Id: CommandLineTest.java 26283 2010-02-18 11:18:57Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.options;

import java.io.OutputStreamWriter;
import java.io.PrintWriter;

import junit.framework.TestCase;

/**
 * Tests for <code>CommandLine</code>
 * 
 * @author Benjamin Hummel
 * @author $Author: juergens $
 * 
 * @version $Rev: 26283 $
 * @levd.rating GREEN Hash: D29DE224D7112D2756321E048F1769A1
 */
public class CommandLineTest extends TestCase {

	/** The dummy handler. */
	private SimpleOptionHandler soh;

	/** The command line object used in each test. */
	private CommandLine cl;

	/** Perform basic setup. */
	@Override
	public void setUp() {
		soh = new SimpleOptionHandler();
		OptionRegistry r = new OptionRegistry(soh);
		cl = new CommandLine(r);
	}

	/**
	 * Test the usage method. Just dumps everything to the console to see if any
	 * exceptions are thrown.
	 */
	public void testUsage() {
		cl.printUsage(new PrintWriter(new OutputStreamWriter(System.out)));
	}

	/** Test simple arguments. */
	public void testSimpleOptions() throws Exception {
		String[] args = new String[] { "-i", "--store", "42" };
		assertEquals(0, cl.parse(args).length);
		assertEquals(1, soh.counter);
		assertTrue(Math.abs(42 - soh.storedDouble) < 1e-6);
	}

	/** Test simple arguments. */
	public void testComplexOptions() throws Exception {
		String[] args = new String[] { "-iiiis", "42", "foo", "bar" };
		assertEquals(2, cl.parse(args).length);
		assertEquals(4, soh.counter);
		assertTrue(Math.abs(42 - soh.storedDouble) < 1e-6);
	}

	/** Test arguments given via = sign. */
	public void testEqualArg() throws Exception {
		String[] args = new String[] { "-i", "--store=42" };
		assertEquals(0, cl.parse(args).length);
		assertTrue(Math.abs(42 - soh.storedDouble) < 1e-6);
	}

	/** Tests error behaviour in case of missing argument. */
	public void testMissingArgument() {
		boolean ok = false;
		String[] args = new String[] { "-i", "--store" };
		try {
			cl.parse(args);
		} catch (OptionException e) {
			ok = true;
		}
		assertTrue(ok);
	}

	/** Tests error behaviour in case of missing argument in multi options. */
	public void testMissingArgumentInMulti() {
		boolean ok = false;
		String[] args = new String[] { "-iiisi", "42" };
		try {
			cl.parse(args);
		} catch (OptionException e) {
			ok = true;
		}
		assertTrue(ok);
	}

	/** Tests error behaviour in case of missing argument. */
	public void testWrongType() {
		boolean ok = false;
		String[] args = new String[] { "-s", "noNumber" };
		try {
			cl.parse(args);
		} catch (OptionException e) {
			ok = true;
		}
		assertTrue(ok);
	}

	/** Tests error behaviour in case of argument that should not exist. */
	public void testSuperfluousArgument() {
		boolean ok = false;
		String[] args = new String[] { "--inc=fail" };
		try {
			assertEquals(0, cl.parse(args).length);
		} catch (OptionException e) {
			ok = true;
		}
		assertTrue(ok);
	}

	/** Tests greedy options. */
	public void testGreedyOption() throws OptionException {
		String[] args = new String[] { "--greedy", "4", "3", "7", "-g", "9",
				"23", "54", "--inc", "test" };
		assertEquals(1, cl.parse(args).length);
		assertEquals(4 + 3 + 7 + 9 + 23 + 54 + 1, soh.counter);
	}
}