/*--------------------------------------------------------------------------+
$Id: HTMLWriter.java 28096 2010-06-09 13:12:48Z hummelb $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.html;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;

import edu.tum.cs.commons.filesystem.FileSystemUtils;
import edu.tum.cs.commons.string.StringUtils;
import edu.tum.cs.commons.xml.IXMLResolver;
import edu.tum.cs.commons.xml.XMLWriter;

/**
 * This class is used for writing HTML.
 * 
 * @author Benjamin Hummel
 * @author $Author: hummelb $
 * @version $Rev: 28096 $
 * @levd.rating GREEN Hash: 8D23F1A3EAAA692885E5A25667EF254D
 */
public class HTMLWriter extends XMLWriter<EHTMLElement, EHTMLAttribute> {

	/** The CSS manager class used. */
	private final CSSManagerBase cssManager;

	/**
	 * Creates a new writer for HTML documents.
	 * 
	 * @param file
	 *            the file to write to.
	 */
	public HTMLWriter(File file, CSSManagerBase cssManager) throws IOException {
		this(new PrintStream(file, FileSystemUtils.UTF8_ENCODING), cssManager);
	}

	/**
	 * Creates a new writer for HTML documents.
	 * 
	 * @param stream
	 *            the stream to print to.
	 */
	public HTMLWriter(OutputStream stream, CSSManagerBase cssManager) {
		super(new PrintWriter(wrapStream(stream)), new HTMLResolver());
		this.cssManager = cssManager;
	}

	/**
	 * Helper method for {@link #HTMLWriter(OutputStream, CSSManagerBase)} to
	 * deal with the exception.
	 */
	private static OutputStreamWriter wrapStream(OutputStream stream) {
		try {
			return new OutputStreamWriter(stream, FileSystemUtils.UTF8_ENCODING);
		} catch (UnsupportedEncodingException e) {
			throw new AssertionError("UTF-8 should be supported!");
		}
	}

	/**
	 * Creates a new writer for HTML documents.
	 * 
	 * @param writer
	 *            the writer to print to.
	 */
	public HTMLWriter(PrintWriter writer, CSSManagerBase cssManager) {
		super(writer, new HTMLResolver());
		this.cssManager = cssManager;
	}

	/**
	 * This adds a default header for HTML files consisting of the XML header
	 * and a DOCTYPE of the xhtml frameset DTD.
	 * <p>
	 * XML version is set to "1.0", encoding provided by a parameter, and doc
	 * type definition to XHTML 1.0 Frameset.
	 */
	public void addStdHeader(String encoding) {
		addHeader("1.0", encoding);
		addPublicDocTypeDefintion(EHTMLElement.HTML,
				"-//W3C//DTD XHTML 1.0 Frameset//EN",
				"http://www.w3.org/TR/xhtml1/DTD/xhtml1-frameset.dtd");
	}

	/**
	 * This adds a default header for HTML files consisting of the XML header
	 * and a DOCTYPE of the xhtml frameset DTD.
	 * <p>
	 * XML version is set to "1.0", encoding to "UTF-8", and doc type definition
	 * to XHTML 1.0 Frameset.
	 */
	public void addStdHeader() {
		addStdHeader(FileSystemUtils.UTF8_ENCODING);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * Made this public here.
	 */
	@Override
	public void addRawString(String html) {
		super.addRawString(html);
	}

	/**
	 * Adds a line separator with closing and open tag (see
	 * {@link #addNewLine()}.
	 */
	public void addRawNewLine() {
		addRawString(StringUtils.CR);
	}

	/**
	 * Adds an attribute to the currently open element but checks in addition if
	 * the attribute may be added at all.
	 * 
	 * @throws HTMLWriterException
	 *             if the attribute is not allowed for the current element.
	 */
	@Override
	public void addAttribute(EHTMLAttribute attribute, Object value) {
		if (!getCurrentElement().allowsAttribute(attribute)) {
			throw new HTMLWriterException("Attribute " + attribute
					+ " not allowed for element " + getCurrentElement());
		}

		if (attribute == EHTMLAttribute.STYLE
				|| attribute == EHTMLAttribute.CLASS) {
			if (!(value instanceof CSSDeclarationBlock)) {
				throw new HTMLWriterException(
						"The argument for STYLE and CLASS attributes must be a "
								+ CSSDeclarationBlock.class.getSimpleName()
								+ "!");
			}
		}

		if (attribute == EHTMLAttribute.STYLE) {
			super.addAttribute(attribute, ((CSSDeclarationBlock) value)
					.toInlineStyle());
		} else if (attribute == EHTMLAttribute.CLASS) {
			super.addAttribute(attribute, cssManager
					.getCSSClassName((CSSDeclarationBlock) value));
		} else {
			super.addAttribute(attribute, value);
		}
	}

	/** The resolver used for the {@link HTMLWriter}. */
	public static class HTMLResolver implements
			IXMLResolver<EHTMLElement, EHTMLAttribute> {

		/** {@inheritDoc} */
		public String resolveAttributeName(EHTMLAttribute attribute) {
			return attribute.toString();
		}

		/** {@inheritDoc} */
		public String resolveElementName(EHTMLElement element) {
			return element.toString();
		}

		/** {@inheritDoc} */
		public Class<EHTMLAttribute> getAttributeClass() {
			return EHTMLAttribute.class;
		}
	}
}