/*
* Copyright (c) 2006 Rudi Cilibrasi, Rulers of the RHouse
* All rights reserved.     cilibrar@cilibrar.com
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions are met:
*
*     * Redistributions of source code must retain the above copyright
*       notice, this list of conditions and the following disclaimer.
*     * Redistributions in binary form must reproduce the above copyright
*       notice, this list of conditions and the following disclaimer in the
*       documentation and/or other materials provided with the distribution.
*     * Neither the name of the RHouse nor the
*       names of its contributors may be used to endorse or promote products
*       derived from this software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE RULERS AND CONTRIBUTORS "AS IS" AND ANY
* EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE RULERS AND CONTRIBUTORS BE LIABLE FOR ANY
* DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
* ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include <string.h>
#include <assert.h>
#include <stdio.h>

#include <complearn/complearn.h>

struct StringStack {
  struct DRA *da;
};

struct StringStack *clStringstackNewSingle(const char *str)
{
	struct StringStack *ss;
  ss = clStringstackNew();
  clStringstackPush(ss, str);
  return ss;
}

struct StringStack *clStringstackNew()
{
	struct StringStack *ss;
	ss = (struct StringStack*)clCalloc(sizeof(struct StringStack), 1);
  ss->da = clDraNew();
	// ss->size = 0;
	return ss;
}

struct StringStack *clStringstackClone(struct StringStack *ss)
{
  struct StringStack *nss;
  int i, sz;
  sz = clStringstackSize(ss);
  nss = clStringstackNew();
  for (i = 0; i < sz; ++i) {
    union PCTypes p;
    p.str = clStrdup(clDraGetValueAt(ss->da, i).str);
    clDraSetValueAt(nss->da, i, p);
  }
  return nss;
}

int clStringstackFree(struct StringStack *ss)
{
  int i;
  for (i = 0; i < clDraSize(ss->da); i += 1) {
    clFree(clDraGetValueAt(ss->da, i).str);
  }
  clDraFree(ss->da);
  ss->da = NULL;
	clFreeandclear(ss);
	return CL_OK;
}

int clStringstackUnshift(struct StringStack *ss, const char *str)
{
  union PCTypes p;
  clAssert(ss != NULL);
  clAssert(str != NULL);
  p.str = clStrdup(str);
  clDraUnshift(ss->da, p);
  return CL_OK;
}

int clStringstackPush(struct StringStack *ss, const char *str)
{
  union PCTypes p;
  clAssert(ss != NULL);
  clAssert(str != NULL);
  p.str = clStrdup(str);
  clDraPush(ss->da, p);
  return CL_OK;
}

int clStringstackIsEmpty(struct StringStack *ss)
{
	return clDraSize(ss->da) == 0;
}

int clStringstackSize(const struct StringStack *ss)
{
	return clDraSize(ss->da);
}

/* After calling clShiftSS, it is the responsibility of the programmer to free
 * the shifted string
 */
char *clShiftSS(struct StringStack *ss)
{
  union PCTypes p;
  clAssert(clStringstackSize(ss) > 0);
  memset(&p, 0, sizeof(p));
	if (clStringstackSize(ss) == 0) return p.str;
	p = clDraShift(ss->da);
  return p.str;
}

/* After calling clStringstackPop, it is the responsibility of the programmer to free
 * the popped string
 */
char *clStringstackPop(struct StringStack *ss)
{
  union PCTypes p;
  memset(&p, 0, sizeof(p));
	if (clStringstackSize(ss) == 0) return p.str;
	p = clDraPop(ss->da);
  return p.str;
}

char *clStringstackReadAt(struct StringStack *ss, int i)
{
	return clDraGetValueAt(ss->da, i).str;
}

int clStringstackSort(struct StringStack *ss)
{
  int flipped, i;
  int sz = clStringstackSize(ss);
  do {
    flipped = 0;
    for (i = 1; i < sz; ++i) {
      if (strcmp(clDraGetValueAt(ss->da, i-1).str, clDraGetValueAt(ss->da, i).str) > 0) {
        clDraSwapAt(ss->da, i-1, i);
        flipped = 1;
      }
    }
  } while (flipped);
  return CL_OK;
}

struct StringStack *clStringstackMerge(struct StringStack *ssa, struct StringStack *ssb)
{
  struct StringStack *result;
  int i;
  result = clStringstackClone(ssa);
  for (i = 0; i < clStringstackSize(ssb); i += 1)
    clStringstackPush(result, clStringstackReadAt(ssb, i));
  return result;
}

void clStringstackPrint(struct StringStack *ss)
{
  int i;
  for (i = 0; i < clStringstackSize(ss); i += 1)
    printf("%s\n", clStringstackReadAt(ss, i));
}

int clStringstackLongestLength(struct StringStack *ss)
{
  int i, bestlen = -1;
  for (i = 0; i < clStringstackSize(ss); i += 1) {
    int len;
    len = strlen(clStringstackReadAt(ss, i));
    if (len > bestlen)
      bestlen = len;
  }
  return bestlen;
}
