/*
 * Maildir folder access
 *
 * Copyright (C) 2004  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma implementation

#include "MaildirMailFolder.h"
#include "Exception.h"

#include <sys/types.h>  // stat, opendir, readdir, utimes
#include <sys/stat.h>   // stat
#include <dirent.h>		// opendir, readdir
#include <sys/time.h>	// utimes

#include <errno.h>

using namespace std;

MaildirMailFolder::MaildirMailFolder(const string& path) throw ()
	: _path(path)
{
	_name = _path;

	/// Normalize the folder name

	// Remove trailing '/'
	while (_name[_name.size() - 1] == '/')
		_name.resize(_name.size() - 1);

	// Remove leading path
	size_t lastslash = _name.find_last_of('/');
	if (lastslash != string::npos)
		_name = _name.substr(lastslash + 1);

	// Remove leading dot
	if (_name[0] == '.')
		_name = _name.substr(1);
}

MaildirMailFolder::MaildirMailFolder(const string& name, const string& path) throw ()
	: _name(name), _path(path), _stat_total(-1), _stat_unread(-1), _stat_new(-1), _stat_flagged(-1),
	  _new_mtime(0), _cur_mtime(0) {}


bool MaildirMailFolder::changed()
{
	// Compute 'new' and 'cur' directory names
	string path_new = _path + "/new";
	string path_cur = _path + "/cur";

	struct stat st_new;
	if (stat(path_new.c_str(), &st_new) != 0)
		throw SystemException(errno, "getting informations on " + path_new);
	struct stat st_cur;
	if (stat(path_cur.c_str(), &st_cur) != 0)
		throw SystemException(errno, "getting informations on " + path_cur);

	return st_new.st_mtime > _new_mtime || st_cur.st_mtime > _cur_mtime;
}

void MaildirMailFolder::updateStatistics()
{
	int res_total = 0;
	int res_unread = 0;
	int res_new = 0;
	int res_flagged = 0;

	// Compute 'new' and 'cur' directory names
	string path_new = _path + "/new";
	string path_cur = _path + "/cur";

	// Perform consistency checks on the 'new' directory
	struct stat st_new;
	if (stat(path_new.c_str(), &st_new) != 0)
		throw SystemException(errno, "getting informations on " + path_new);
	if (S_ISDIR(st_new.st_mode) == 0)
		throw ConsistencyCheckException(path_new + " is not a directory");

	// Perform consistency checks on the 'cur' directory
	struct stat st_cur;
	if (stat(path_cur.c_str(), &st_cur) != 0)
		throw SystemException(errno, "getting informations on " + path_cur);
	if (S_ISDIR(st_cur.st_mode) == 0)
		throw ConsistencyCheckException(path_cur + " is not a directory");

	_new_mtime = st_new.st_mtime;
	_cur_mtime = st_cur.st_mtime;

	/// Count messages in the 'new' directory

	// Count the files in the 'new' directory
	DIR *dir = opendir(path_new.c_str());
	if (dir == NULL)
		throw SystemException(errno, "opening " + path_new);

	struct dirent *d;
	while ((d = readdir (dir)) != NULL)
	{
		if (d->d_name[0] == '.') 
			continue;
		res_total++;
		res_new++;
	}
	closedir(dir);

	// Restore the access time of the mailbox for other checking programs
	struct timeval t[2];
	t[0].tv_sec = st_new.st_atime;
	t[0].tv_usec = 0;
	t[1].tv_sec = st_new.st_mtime;
	t[1].tv_usec = 0;
	utimes(path_new.c_str(), t);


	/// Count messages in the 'cur' directory

	// Count the files in the 'cur' directory
	dir = opendir(path_cur.c_str());
	if (dir == NULL)
		throw SystemException(errno, "opening " + path_cur);

	while ((d = readdir (dir)) != NULL)
	{
		if (d->d_name[0] == '.') 
			continue;

		res_total++;

		// Look for an `info' block in the name
		char* info = strrchr(d->d_name, ':');
		if (info == 0)
			continue;

		// Ensure that the info block is in the right format
		if (strncmp(info, ":2,", 3) != 0)
			continue;

		// Look for the 'S' flag (it should not be there)
		info += 3;
		if (strchr(info, 'S') == 0)
			res_unread++;
		if (strchr(info, 'F') != 0)
			res_flagged++;
	}
	closedir(dir);

	// Restore the access time of the mailbox for other checking programs
	t[0].tv_sec = st_cur.st_atime;
	t[0].tv_usec = 0;
	t[1].tv_sec = st_cur.st_mtime;
	t[1].tv_usec = 0;
	utimes(path_cur.c_str(), t);


	// Return the values
	_stat_total = res_total;
	_stat_unread = res_unread + res_new;
	_stat_new = res_new;
	_stat_flagged = res_flagged;
}


static bool isMaildir(const std::string& pathname)
{
	// It must be a directory
	struct stat st;
	if (stat(pathname.c_str(), &st) != 0)
		throw SystemException(errno, "getting informations on " + pathname);
	if (S_ISDIR(st.st_mode) == 0)
		return false;

	// It must contain cur, new and tmp subdirectories
	char* subdirs[3] = { "cur", "new", "tmp" };
	for (int i = 0; i < 3; i++)
	{
		string subdir = pathname + "/" + subdirs[i];

		struct stat st;
		if (stat(subdir.c_str(), &st) != 0)
			return false;
		if (S_ISDIR(st.st_mode) == 0)
			return false;
	}

	// It appears to be a maildir directory
	return true;
}

void MaildirMailFolder::enumerateFolders(const std::string& parent, MailFolderConsumer& cons)
{
	// Perform consistency checks on the parent directory
	struct stat st;
	if (stat(parent.c_str(), &st) != 0)
		return;
//		throw SystemException(errno, "getting informations on " + parent);
	if (S_ISDIR(st.st_mode) == 0)
		return;
//		throw ConsistencyCheckException(parent + " is not a directory");

	if (isMaildir(parent))
	{
//		fprintf(stderr, "MMF: parent is folder\n");
		MailFolder f(new MaildirMailFolder(parent));
		cons.consume(f);
	}

	// Enumerate the Maildirs in it
	DIR *dir = opendir(parent.c_str());
	if (dir == NULL)
		throw SystemException(errno, "opening " + parent);

	struct dirent *d;
	while ((d = readdir (dir)) != NULL)
	{
		if (strcmp(d->d_name, ".") == 0)
			continue;
		if (strcmp(d->d_name, "..") == 0)
			continue;
		string candidate = parent + "/" + d->d_name;
		if (isMaildir(candidate))
		{
//			fprintf(stderr, "MMF: %.*s is folder\n", PFSTR(candidate));
			MailFolder f(new MaildirMailFolder(candidate));
			cons.consume(f);
		}// else
//			fprintf(stderr, "MMF: %.*s is not a folder\n", PFSTR(candidate));
	}
	closedir(dir);
}

// vim:set ts=4 sw=4:
