/**
 * Copyright (c) 2011-2015 libbitcoin developers (see AUTHORS)
 *
 * This file is part of libbitcoin.
 *
 * libbitcoin is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License with
 * additional permissions to the one published by the Free Software
 * Foundation, either version 3 of the License, or (at your option)
 * any later version. For more information see LICENSE.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include <bitcoin/bitcoin/utility/binary.hpp>

#include <sstream>
#include <bitcoin/bitcoin/constants.hpp>
#include <bitcoin/bitcoin/utility/assert.hpp>
#include <iostream>

namespace libbitcoin {

binary_type::size_type binary_type::blocks_size(const size_type bitsize)
{
    if (bitsize == 0)
        return 0;
    return (bitsize - 1) / bits_per_block + 1;
}

binary_type::binary_type()
{
}
binary_type::binary_type(const std::string& bitstring)
{
    std::stringstream(bitstring) >> *this;
}
binary_type::binary_type(size_type size, data_slice blocks)
{
    // Copy blocks
    blocks_.resize(blocks.size());
    std::copy(blocks.begin(), blocks.end(), blocks_.begin());
    // Pad with 00 for safety.
    while (blocks_.size() * bits_per_block < size)
        blocks_.push_back(0x00);
    resize(size);
}

void binary_type::resize(size_type size)
{
    final_block_excess_ = 0;
    blocks_.resize(blocks_size(size), 0);

    size_type offset = size % bits_per_block;

    if (offset > 0)
    {
        BITCOIN_ASSERT((bits_per_block - offset) <= max_uint8);
        final_block_excess_ = static_cast<uint8_t>(bits_per_block - offset);

        uint8_t mask = 0xFF << final_block_excess_;
        blocks_[blocks_.size() - 1] = blocks_[blocks_.size() - 1] & mask;
    }
}

bool binary_type::operator[](size_type index) const
{
    BITCOIN_ASSERT(index < size());
    const size_type block_index = index / bits_per_block;
    const uint8_t block = blocks_[block_index];
    const size_type offset = index - (block_index * bits_per_block);
    const uint8_t bitmask = 1 << (bits_per_block - offset - 1);
    return (block & bitmask) > 0;
}

const data_chunk& binary_type::blocks() const
{
    return blocks_;
}

binary_type::size_type binary_type::size() const
{
    return (blocks_.size() * bits_per_block) - final_block_excess_;
}

void binary_type::append(const binary_type& post)
{
    const size_type block_offset = size() / bits_per_block;
    const size_type offset = size() % bits_per_block;

    // overkill for byte alignment
    binary_type duplicate(post.size(), post.blocks());
    duplicate.shift_right(offset);

    resize(size() + post.size());

    data_chunk post_shift_blocks = duplicate.blocks();

    for (size_type i = 0; i < post_shift_blocks.size(); i++)
    {
        blocks_[block_offset + i] = blocks_[block_offset + i] | post_shift_blocks[i];
    }
}

void binary_type::prepend(const binary_type& prior)
{
    shift_right(prior.size());

    data_chunk prior_blocks = prior.blocks();

    for (size_type i = 0; i < prior_blocks.size(); i++)
    {
        blocks_[i] = blocks_[i] | prior_blocks[i];
    }
}

void binary_type::shift_left(size_type distance)
{
    const size_type initial_size = size();
    const size_type initial_block_count = blocks_.size();

    size_type destination_size = 0;

    if (distance < initial_size)
        destination_size = initial_size - distance;

    const size_type block_offset = distance / bits_per_block;
    const size_type offset = distance % bits_per_block;

    // shift
    for (size_type i = 0; i < initial_block_count; i++)
    {
        uint8_t leading_bits = 0x00;
        uint8_t trailing_bits = 0x00;

        if ((offset != 0) && ((block_offset + i + 1) < initial_block_count))
        {
            trailing_bits = blocks_[block_offset + i + 1] >> (bits_per_block - offset);
        }

        if ((block_offset + i) < initial_block_count)
        {
            leading_bits = blocks_[block_offset + i] << offset;
        }

        blocks_[i] = leading_bits | trailing_bits;
    }

    resize(destination_size);
}

void binary_type::shift_right(size_type distance)
{
    const size_type initial_size = size();
    const size_type initial_block_count = blocks_.size();

    const size_type offset = distance % bits_per_block;
    const size_type offset_blocks = distance / bits_per_block;
    const size_type destination_size = initial_size + distance;

    for (size_type i = 0; i < offset_blocks; i++)
    {
        blocks_.insert(blocks_.begin(), 0x00);
    }

    uint8_t previous = 0x00;

    for (size_type i = 0; i < initial_block_count; i++)
    {
        uint8_t current = blocks_[offset_blocks + i];

        uint8_t leading_bits = previous << (bits_per_block - offset);
        uint8_t trailing_bits = current >> offset;

        blocks_[offset_blocks + i] = leading_bits | trailing_bits;

        previous = current;
    }

    resize(destination_size);

    if (offset_blocks + initial_block_count < blocks_.size())
    {
        blocks_[blocks_.size() - 1] = previous << (bits_per_block - offset);
    }
}

binary_type binary_type::get_substring(size_type start, size_type length) const
{
    size_type current_size = size();

    if (start > current_size)
    {
        start = current_size;
    }

    if ((length == max_size_t) || ((start + length) > current_size))
    {
        length = current_size - start;
    }

    binary_type result(current_size, blocks_);

    result.shift_left(start);

    result.resize(length);

    return result;
}

bool operator==(const binary_type& lhs, const binary_type& rhs)
{
    for (binary_type::size_type i = 0; i < lhs.size() && i < rhs.size(); ++i)
        if (lhs[i] != rhs[i])
            return false;

    return true;
}
bool operator!=(const binary_type& lhs, const binary_type& rhs)
{
    return !(lhs == rhs);
}
bool operator<(const binary_type& lhs, const binary_type& rhs)
{
    for (binary_type::size_type i = 0; i < lhs.size() && i < rhs.size(); ++i)
        if (lhs[i] < rhs[i])
            return true;
        else if (lhs[i] > rhs[i])
            return false;

    return false;
}

std::istream& operator>>(std::istream& stream, binary_type& prefix)
{
    std::string bitstring;
    stream >> bitstring;

    prefix.resize(0);

    uint8_t block = 0;
    binary_type::size_type bit_iter = binary_type::bits_per_block;

    for (const char repr : bitstring)
    {
        if (repr != '0' && repr != '1')
        {
            prefix.blocks_.clear();
            return stream;
        }

        // Set bit to 1
        if (repr == '1')
        {
            const uint8_t bitmask = 1 << (bit_iter - 1);
            block |= bitmask;
        }

        // Next bit
        --bit_iter;

        if (bit_iter == 0)
        {
            // Move to the next block.
            prefix.blocks_.push_back(block);
            block = 0;
            bit_iter = binary_type::bits_per_block;
        }
    }

    // Block wasn't finished but push it back.
    if (bit_iter != binary_type::bits_per_block)
        prefix.blocks_.push_back(block);

    prefix.resize(bitstring.size());
    return stream;
}
std::ostream& operator<<(std::ostream& stream, const binary_type& prefix)
{
    for (binary_type::size_type i = 0; i < prefix.size(); ++i)
        if (prefix[i])
            stream << '1';
        else
            stream << '0';
    return stream;
}

} // namespace libbitcoin

