/*
 *  http.c
 *  mod_musicindex
 *
 *  $Id: http.c 609 2006-05-14 19:53:34Z boudinr $
 *
 *  Created by Regis BOUDIN on Sun Jun 13 2004.
 *  Copyright (c) 2003-2004 Regis BOUDIN
 *  Copyright (c) 2003-2005 Thibaut VARENE
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 2.1,
 *  as published by the Free Software Foundation.
 *
 */

/**
 * @file 
 * HTTP features handling.
 *
 * @author Regis Boudin
 * @author Thibaut Varene
 * @version $Revision: 609 $
 * @date 2004-2005
 *
 * This file is here to deal with HTTP related
 * features which are not directly handled by
 * Apache. It includes parsing of arguments
 * received via GET or POST requests and
 * cookie management for cross-directories
 * playlists.
 *
 * @todo Test, test, test.
 * @todo Find a way to handle the bloody unescaped characters (like '&')
 *
 * @bug Beware of "big lists"
 * @todo Store custom list locally, and store only a session reference in the cookie.
 *
 */

#include "http.h"
#include "sort.h"
#include "playlist.h"

/**
 * Treats the GET arguments
 *
 * This function searches for keywords passed as URL arguments (with "?xxx")
 * and sets the handler options accordingly.
 * 
 * @param r Apache request_rec struct to handle connection details.
 * @param conf MusicIndex configuration paramaters struct.
 */
void treat_get_args(request_rec *r, mu_config *const conf)
{
	const char *s = r->args;
	const char *p;
	register unsigned short i;

	if (s == NULL)
		return;
	
	conf->custom_list = r->args;

	while (s[0]) {
		p = ap_getword(r->pool, &s, '&');
		if (!strncmp(p, "action=", 7)) {
			p += 7;
			
			if (!strcmp(p, "randomdir")) {
				conf->options |= MI_RANDOMDIR;
			}
			
			if (conf->options & MI_ALLOWTARBALL) {
				if (!strcmp(p, "tarball")) {
					conf->options |= (MI_DWNLDALL | MI_QUICKPL) ;
					conf->order[0] = SB_DIR;
					conf->order[1] = SB_URI;
					set_fctptrs(conf);
				}
			}
			
			if (conf->options & MI_ALLOWSTREAM) {
				if (!strcmp(p, "playall"))
					conf->options |= MI_STREAMALL;
			}
			
			if (conf->rss_items > 0) {
				if (!strcmp(p, "RSS")) {
					conf->options |= MI_RSS;
					conf->order[0] = SB_MTIME;
					conf->order[1] = SB_URI;
					conf->options &= ~MI_RECURSIVE;
				}

				if (!strcmp(p, "podcast")) {
					conf->options |= MI_RSS|MI_PODCAST;
					conf->order[0] = SB_MTIME;
					conf->order[1] = SB_URI;
					conf->options &= ~MI_RECURSIVE;
				}
			}
			
			return;
		}
		else if (!strncmp(p, "sort=", 5)) {
			p += 5;
		/* Temporary workaround : only have one sort param */
			conf->order[ARG_NUMBER-1] = '\0';
			for (i = ARG_NUMBER-2; i > 0; i--)
				conf->order[i] = conf->order[i-1];
			conf->order[0] = *p;
			set_fctptrs(conf);
		}
		else if (!strncmp(p, "option=", 7)) {
			p += 7;
			if (!strcmp(p, "recursive"))
				conf->options |= MI_RECURSIVE;
			else if (!strcmp(p, "shuffle")) {
				conf->order[0] = SB_RANDOM;
				conf->order[1] = SB_DEFAULT;
				set_fctptrs(conf);
			}
			else if (!strcmp(p, "quick"))
				conf->options |= MI_QUICKPL;
		}
		else if (!strncmp(p, "limit=", 6)) {
			if (conf->rss_items > 0)
				conf->rss_items = atoi(p+6);
		}
	}
}

/**
 * Treats the POST arguments
 *
 * This function searches for keywords in arguments sent in a POST
 * request. It only sets option flags and the search string when they are
 * needed.
 * 
 * @param r Apache request_rec struct to handle connection details.
 * @param conf MusicIndex configuration paramaters struct.
 */
void treat_post_args(request_rec *r, mu_config *const conf)
{
	char buffer[MAX_STRING];
	const char *s = NULL;
	const char *p = s;
	short i = 0;

	ap_setup_client_block(r, REQUEST_CHUNKED_DECHUNK);

	do {
		i = ap_get_client_block(r, buffer, MAX_STRING-1);
		buffer[i] = '\0';
		if (s == NULL)
			s = ap_pstrdup(r->pool, buffer);
		else
			s = ap_pstrcat(r->pool, s, buffer, NULL);
	} while (i == (MAX_STRING-1));

	conf->custom_list = s;

	/* Find the search string */
	p = strstr(s, "&search=");

	if (p != NULL)
		p += 1;
	else if (strncmp(s, "search=", 7) == 0)
		p = s;
	
	if (p != NULL) {
		p += 7;
		conf->search = ap_getword(r->pool, &p, '&');
		for (i=0; p[i]; i++) {
			if (conf->search[i] == '+')
				conf->search[i] = ' ';
		}
		ap_unescape_url(conf->search);
	}
	
	/* Find the sort string */
	p = strstr(s, "&sort=");

	if (p != NULL)
		p += 1;
	else if (strncmp(s, "sort=", 5) == 0)
		p = s;
	
	if (p != NULL) {
		p += 5;
		conf->order[ARG_NUMBER-1] = '\0';
		for (i = ARG_NUMBER-2; i > 0; i--)
			conf->order[i] = conf->order[i-1];
		conf->order[0] = *p;
		set_fctptrs(conf);
	}
	
	/* Find the action string */
	p = strstr(s, "&action=");

	if (p != NULL)
		p += 1;
	else if (strncmp(s, "action=", 7) == 0)
		p = s;
	
	if (p != NULL) {
		p += 7;
	
		p = ap_getword(r->pool, &p, '&');
		
		if (conf->options & MI_ALLOWSTREAM) {
			if (!strcmp(p, "Play+Selected")) {
				conf->options |= MI_STREAMLST;
			} else if (!strcmp(p, "Play+All")) {
				conf->options |= MI_STREAMALL;
			} else if (!strcmp(p, "Shuffle+All")) {
				conf->options |= MI_STREAMALL;
				conf->order[0] = SB_RANDOM;
				conf->order[1] = SB_DEFAULT;
				set_fctptrs(conf);
			} else if (!strcmp(p, "Remove+from+Playlist")) {
				conf->options |= MI_COOKIEDELLST;
			} else if (!strcmp(p, "Clear+Playlist")) {
				conf->options |= MI_COOKIEPURGE;
			} else if (!strcmp(p, "Stream+Playlist")) {
				conf->options |= MI_COOKIESTREAM;
			} else if (!strcmp(p, "Add+To+Playlist")) {
				conf->options |= MI_COOKIEADDLST;
			} else if (!strcmp(p, "Add+All+To+Playlist")) {
				conf->options |= MI_COOKIEADDALL;
			}
			
			/* In any situation, no recursive directories
			scan for Cookie operations */
			if (conf->options & MI_COOKIEOP)
				conf->options &= ~MI_RECURSIVE;
		}
		
		if (conf->options & MI_ALLOWTARBALL) {

			if (!strcmp(p, "Download+All")) {
				conf->options |= MI_DWNLDALL;
			} else if (!strcmp(p, "Download+Selected")) {
				conf->options |= MI_DWNLDLST;
			}else if (!strcmp(p, "Download+Playlist")) {
				conf->options |= MI_COOKIEDWNLD;
			}
			
			/* In any situation, no recursive directories
			scan for Cookie operations */
			if (conf->options & MI_COOKIEOP)
				conf->options &= ~MI_RECURSIVE;
		}
		
		if ((conf->options & MI_ALLOWSEARCH) && conf->search && conf->search[0]) {
			if (!strcmp(p, "Search")) {
				if (!conf->cache)
					conf->options |= MI_QUICKPL;
			}
			else if (!strcmp(p, "Recursive+Search")) {
				conf->options |= MI_RECURSIVE;
				conf->order[0] = SB_DIR;
				conf->order[1] = SB_URI;
				set_fctptrs(conf);
				if (!conf->cache)
					conf->options |= MI_QUICKPL;
			}
		}
	}
}

static const char *cookie_add(request_rec *r, apr_pool_t *subpool, const mu_config *const conf)
{
	const char *new_cookie_string = NULL;
	const char *args = conf->custom_list, *p = NULL;
	char *uri = NULL;
	register unsigned short i;

	p = ap_table_get(r->headers_in, "Cookie");

	if (p)
		new_cookie_string = ap_getword(subpool, &p, ';');
	else
		new_cookie_string = ap_pstrdup(subpool, "playlist=");
	
	while (*args) {
		p = ap_getword(subpool, &args, '&');
		if (strncmp(p, "file=", 5))
			continue;
		
		/* I should check if the file exists */
		uri = ap_pstrcat(subpool, r->uri, p+5, NULL);
		for (i=0; uri[i]; i++) {
			if (uri[i] == '+')
				uri[i] = ' ';
		}
		ap_unescape_url(uri);
		uri = ap_escape_uri(subpool, uri);
		
		if (strstr(new_cookie_string, uri) == NULL)
			new_cookie_string = ap_pstrcat(subpool, new_cookie_string, uri, "&", NULL);
	}

	return new_cookie_string;
}

static const char *cookie_addall(request_rec *r, apr_pool_t *subpool, mu_config *const conf)
{
	mu_ent		*head = NULL, *custom = NULL;
	const char 	*new_cookie_string = NULL, *p = NULL;
	const char	*uri = NULL;

	p = ap_table_get(r->headers_in, "Cookie");

	if (p)
		new_cookie_string = ap_getword(subpool, &p, ';');
	else
		new_cookie_string = ap_pstrdup(subpool, "playlist=");
	
	head = make_music_entry(r, subpool, NULL, conf, NULL, MI_RECURSIVE);
	head = quicksort(head, NULL, conf);

	for (custom = head; custom; custom = custom->next) {
		if (custom->filetype < 0)
			continue;
		
		uri = ap_escape_uri(subpool, custom->uri);
		if (strstr(new_cookie_string, uri) == NULL)
			new_cookie_string = ap_pstrcat(subpool, new_cookie_string, uri, "&", NULL);
	}

	return new_cookie_string;
}

static const char *cookie_remove(request_rec *r, apr_pool_t *subpool, const mu_config *const conf)
{
	const char *new_cookie_string = NULL, *p = NULL, *args;
	char *escaped_postargs = NULL;
	register unsigned short i;

	args = ap_table_get(r->headers_in, "Cookie");

	if (args == NULL)
		return NULL;
	
	args = strstr(args, "playlist=");
	
	if (args == NULL)
		return NULL;
	
	escaped_postargs = ap_pstrdup(subpool, conf->custom_list);
	
	for (i=0; escaped_postargs[i]; i++) {
		if (escaped_postargs[i] == '+')
			escaped_postargs[i] = ' ';
	}
	ap_unescape_url(escaped_postargs);
	escaped_postargs = ap_escape_uri(subpool, escaped_postargs);
	
	args += 9;
	new_cookie_string = ap_pstrdup(subpool, "playlist=");

	/* If our token was found, copy from the client string
	to our cookie_header */
	while ((*args != '\0') && (*args != ';')) {
		p = ap_getword(subpool, &args, '&');
		
		/* check the incoming track is not in a request to remove */
		if (strstr(escaped_postargs, p) == NULL)
			new_cookie_string = ap_pstrcat(subpool, new_cookie_string, p, "&", NULL);
	}

	return new_cookie_string;
}

void cookie_and_stream_work(request_rec *r, mu_config *const conf)
{
	const char *end_string = NULL, *args = NULL;
	apr_pool_t *subpool = NULL;

	/* Create a subpool we will clear sometimes. This is to save memory, as
	some operations are based on many ap_{palloc,strcat,strdup,...} for 
	playlists */
	apr_pool_create(&subpool, r->pool);

	if (subpool == NULL)
		subpool = r->pool;
	
	switch (conf->options & MI_ALLOPS) {
		case MI_COOKIEADDALL:
			args = cookie_addall(r, subpool, conf);
			break;
		case MI_COOKIEADDLST:
			args = cookie_add(r, subpool, conf);
			break;
		case MI_COOKIEDELLST:
			args = cookie_remove(r, subpool, conf);
			break;
		case MI_COOKIEPURGE:
			args = ap_pstrdup(subpool, "playlist=");
			break;
		case MI_COOKIESTREAM:
		default:
			args = ap_table_get(r->headers_in, "Cookie");
			if (args)
				args = ap_getword(subpool, &args, ';');
			else
				conf->custom_list = NULL;
			break;
	}

	if (args) {
		end_string = ap_psprintf(subpool, ";Version=1; Max-Age=%d; Path=/",
			(args[9] == '\0') ? 0 : conf->cookie_life);
		conf->custom_list = ap_pstrcat(r->pool, args, end_string, NULL);
	}
	else
		conf->custom_list = NULL;

	/* We do not need the subpool any more */
	ap_destroy_pool(subpool);
}
