/*
 *  playlist-mp3.c
 *  mod_musicindex
 *
 *  $Id: playlist-mp3.c,v 1.3 2004/03/11 19:32:48 boudinr Exp $
 *
 *  Created by Regis BOUDIN on Thu Jan 22 2004.
 *  Copyright (c) 2003-2004 Regis BOUDIN
 *  Copyright (c) 2003-2004 Thibaut VARENE
 *   
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1, or (at your option)
 *  any later version.
 *
 */

/**
 * @file 
 * MP3 files management system.
 *
 * @author Thibaut Varene
 * @version $Revision: 1.3 $
 * @date 2003-2004
 *
 * This file contains everything needed to produce music entries from
 * MP3 files.
 *
 * @todo Find out if we trash "detect by content".
 */
 
#include "playlist.h"
#include "playlist-mp3.h"

/**
 * MP3 content detection doesn't actually work.
 */
#undef DETECT_BY_CONTENT_NOT_BORKEN
#ifdef DETECT_BY_CONTENT_NOT_BORKEN
/* The following pieces of code are taken from xmms, in:
 *	Input/mpg123/common.c
 *	Input/mpg123/mpg123.c
 */
static unsigned long mpg123_convert_to_header(unsigned char *buf)
{
        return (buf[0] << 24) + (buf[1] << 16) + (buf[2] << 8) + buf[3];
}

/**
 * Checks for valid MP3 header.
 *
 * doesn't seem to work :P
 *
 * @param head The file header to check
 *
 * @return FALSE when the header is correct, TRUE otherwise.
 */
static short mpg123_head_check(unsigned long head)
{
        if ((head & 0xffe00000) != 0xffe00000)
                return FALSE;
        if (!((head >> 17) & 3))
                return FALSE;
        if (((head >> 12) & 0xf) == 0xf)
                return FALSE;
        if (!((head >> 12) & 0xf))
                return FALSE;
        if (((head >> 10) & 0x3) == 0x3)
                return FALSE;
        if (((head >> 19) & 1) == 1 && ((head >> 17) & 3) == 3 && ((head >> 16) & 1) == 1)
                return FALSE;
        if ((head & 0xffff0000) == 0xfffe0000)
                return FALSE;

        return TRUE;
}
#endif

/**
 * Reference table for digit coded genres.
 */
const char *mpg123_id3_genres[GENRE_MAX] = {
	"Blues", "Classic Rock", "Country", "Dance",
	"Disco", "Funk", "Grunge", "Hip-Hop",
	"Jazz", "Metal", "New Age", "Oldies",
	"Other", "Pop", "R&B", "Rap", "Reggae",
	"Rock", "Techno", "Industrial", "Alternative",
	"Ska", "Death Metal", "Pranks", "Soundtrack",
	"Euro-Techno", "Ambient", "Trip-Hop", "Vocal",
	"Jazz+Funk", "Fusion", "Trance", "Classical",
	"Instrumental", "Acid", "House", "Game",
	"Sound Clip", "Gospel", "Noise", "Alt",
	"Bass", "Soul", "Punk", "Space",
	"Meditative", "Instrumental Pop",
	"Instrumental Rock", "Ethnic", "Gothic",
	"Darkwave", "Techno-Industrial", "Electronic",
	"Pop-Folk", "Eurodance", "Dream",
	"Southern Rock", "Comedy", "Cult",
	"Gangsta Rap", "Top 40", "Christian Rap",
	"Pop/Funk", "Jungle", "Native American",
	"Cabaret", "New Wave", "Psychedelic", "Rave",
	"Showtunes", "Trailer", "Lo-Fi", "Tribal",
	"Acid Punk", "Acid Jazz", "Polka", "Retro",
	"Musical", "Rock & Roll", "Hard Rock", "Folk",
	"Folk/Rock", "National Folk", "Swing",
	"Fast-Fusion", "Bebob", "Latin", "Revival",
	"Celtic", "Bluegrass", "Avantgarde",
	"Gothic Rock", "Progressive Rock",
	"Psychedelic Rock", "Symphonic Rock", "Slow Rock",
	"Big Band", "Chorus", "Easy Listening",
	"Acoustic", "Humour", "Speech", "Chanson",
	"Opera", "Chamber Music", "Sonata", "Symphony",
	"Booty Bass", "Primus", "Porn Groove",
	"Satire", "Slow Jam", "Club", "Tango",
	"Samba", "Folklore", "Ballad", "Power Ballad",
	"Rhythmic Soul", "Freestyle", "Duet",
	"Punk Rock", "Drum Solo", "A Cappella",
	"Euro-House", "Dance Hall", "Goa",
	"Drum & Bass", "Club-House", "Hardcore",
	"Terror", "Indie", "BritPop", "Negerpunk",
	"Polsk Punk", "Beat", "Christian Gangsta Rap",
	"Heavy Metal", "Black Metal", "Crossover",
	"Contemporary Christian", "Christian Rock",
	"Merengue", "Salsa", "Thrash Metal",
	"Anime", "JPop", "Synthpop"
};

/**
 * Checks for valid MP3 filename extension.
 *
 * @param filename The filename to check.
 *
 * @return FALSE when the extension is correct, TRUE otherwise.
 */
static short mpg123_mp3_ext_check(char *filename)
{
	char *ext = strrchr(filename, '.');
	if (ext	&& (!strncasecmp(ext, ".mp2", 4) || !strncasecmp(ext, ".mp3", 4)))
		return FALSE;
	return TRUE;
}

/**
 * Converts an id3tag string in utf8.
 *
 * This function reads the id3tag from an MP3 file (if any) and converts
 * the result in the utf8 standard format.
 *
 * @warning Allocates memory. Don't forget to free().
 * 
 * @param tag The tag struct to parse
 * @param frameid The type of tag data searched
 * @param index The id3tag index value
 *
 * @return When possible, an utf8_t string, NULL otherwise.
 */
static id3_utf8_t *utf8_id3tag_findframe(struct id3_tag *tag, char const *frameid,
	unsigned short index)
{
	const struct id3_frame	*frame = NULL;
	const union id3_field	*field = NULL;
	const id3_ucs4_t	*ucs4 = NULL;

	if ((frame = id3_tag_findframe (tag, frameid, index))) {
		field = &frame->fields[1];
		if (id3_field_getnstrings(field)
			&& (ucs4 = id3_field_getstrings(field, index))) {
			return(id3_ucs4_utf8duplicate(ucs4));
			/* id3_ucs4_utf8duplicate() allocates memory.
			   Don't forget to free() after the function call. */
		}
	}
	return NULL;
}

/**
 * Fills in the information fields about MP3 data.
 *
 * This function reads the id3 tags (using libid3tag0) from the MP3 file
 * and fills in the struct mu_ent fields accordingly.
 * 
 * @param pool Pool
 * @param head Head
 * @param in MP3 file to parse (closed on normal exit)
 * @param conf MusicIndex configuration paramaters struct
 * @param names Names
 * @param r Apache request_rec struct to handle log writings (debugging)
 *
 * @return When possible, struct mu_ent correctly set up, file stream closed.
 *
 * @bug Reports wrong bitrate/length with VBR files.
 * @todo VBR handling (Xing headers decoding).
 */
mu_ent *make_mp3_entry(apr_pool_t *pool, mu_ent *head,
	FILE *in, mu_config *conf, mu_ent_names *names, request_rec *r) 
{
	mu_ent 			*p = head;

	struct id3_file		*id3struct = NULL;
	struct id3_tag		*tag = NULL;
	id3_utf8_t 		*utf8 = NULL;

	struct stat		filestat;
	
	struct mad_stream 	madstream;
	struct mad_frame	madframe;
	unsigned char		madinput_buffer[INPUT_BUFFER_SIZE];
	unsigned short 		mp3gid = GENRE_MAX;
	unsigned short		isd = 0;

#ifdef DETECT_BY_CONTENT_NOT_BORKEN
	unsigned char		tmp[4];

	if (fread(tmp, 1, 4, in) != 4)
		goto exit;

	if (mpg123_head_check(mpg123_convert_to_header(tmp)))
#else
	if (mpg123_mp3_ext_check(names->filename))
#endif
		goto exit;
	
	mad_stream_init(&madstream);
	mad_frame_init(&madframe);
	
	fstat(fileno(in), &filestat);
	
	p = new_ent(pool, head);
	p->filetype = FT_MP3;
	p->size = filestat.st_size;
				
	if (conf->options & (MI_QUICKPL))
		p->bitrate = p->length = 0;
	else {
		/* First of all we try to find out the bitrate and length of
		 * the file */
		do {
			if (madstream.buffer == NULL || madstream.error == MAD_ERROR_BUFLEN) {
				size_t		madread_size, remaining;
				unsigned char	*madread_start;
				
				if (madstream.next_frame != NULL) {
					remaining = madstream.bufend-madstream.next_frame;
					memmove(madinput_buffer, madstream.next_frame, remaining);
					madread_start = madinput_buffer + remaining;
					madread_size = INPUT_BUFFER_SIZE - remaining;
				}
				else {
					madread_size = INPUT_BUFFER_SIZE;
					madread_start = madinput_buffer;
					remaining = 0;
				}
				
				madread_size = fread(madread_start, 1, madread_size, in);
				if (madread_size <= 0) {
					ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] DBG: maderror madread_size <= 0 on %s", names->filename);
					goto exit;
					/* TODO traitement eventuel erreur lecture */
				}
	
				mad_stream_buffer(&madstream, madinput_buffer, madread_size + remaining);
				madstream.error = 0;
			}
	
	
				
			if (mad_frame_decode(&madframe, &madstream)) {
				if (MAD_RECOVERABLE(madstream.error)) {
					/* TODO traitement eventuel erreur recoverable */
					continue;
				}
				else {
					if (madstream.error == MAD_ERROR_BUFLEN)
						continue;
					else {
						ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] DBG: maderror madstream.error unrecoverable on %s", names->filename);
						/* TODO traitement eventuel erreur unrecoverable */
						goto exit;
					}
				}
			}
		
			p->bitrate = madframe.header.bitrate * 1.024;
			p->length = filestat.st_size / (madframe.header.bitrate / 8);
			break;
		} while (1);
	}
	
	/* Then we check for an ID3 tag and read it if any */
	if ((id3struct = id3_file_open(names->filename, ID3_FILE_MODE_READONLY))) {
		if ((tag = id3_file_tag(id3struct)) && (tag->frames)) {
			if ((utf8 = utf8_id3tag_findframe(tag, ID3_FRAME_TITLE, 0))) {
				p->title = ap_pstrdup(pool, (char *)utf8);
				free(utf8);
			}

			if ((utf8 = utf8_id3tag_findframe(tag, ID3_FRAME_ARTIST, 0))) {
				p->artist = ap_pstrdup(pool, (char *)utf8);
				free(utf8);
			}
			
			if ((utf8 = utf8_id3tag_findframe(tag, ID3_FRAME_ALBUM, 0))) {
				p->album = ap_pstrdup(pool, (char *)utf8);
				free(utf8);
			}

			if ((utf8 = utf8_id3tag_findframe(tag, ID3_FRAME_YEAR, 0))) {
				p->date = atoi((char *)utf8);
				free(utf8);
			}

			if ((utf8 = utf8_id3tag_findframe(tag, ID3_FRAME_TRACK, 0))) {
				p->track = atoi((char *)utf8);
				free(utf8);
			}
			/* TPOS is Part Of a Set (aka disc number) */
			if ((utf8 = utf8_id3tag_findframe(tag, "TPOS", 0))) {
				p->posn = atoi((char *)utf8);
				free(utf8);
			}
			/* Some MP3s (especially VBRs) are nice enough to tag their length */
			if ((utf8 = utf8_id3tag_findframe(tag, "TLEN", 0))) {
				if ((atoi((char *)utf8) / 1000) > 0)
					p->length = atoi((char *)utf8) / 1000;
				free(utf8);
			}
			if ((utf8 = utf8_id3tag_findframe(tag, ID3_FRAME_GENRE, 0))) {
				if (*utf8 == '(') {	/* some encoders encaps id nb in brackets */
					if ((isd = isdigit(*(((char *)utf8)+1))))
						mp3gid = atoi(((char *)utf8)+1);
				}
				else if ((isd = isdigit(*utf8)))
					mp3gid = atoi((char *)utf8);
				
				if (isd && (mp3gid < GENRE_MAX))
					p->genre = ap_pstrdup(pool, mpg123_id3_genres[mp3gid]);
				else
					p->genre = ap_pstrdup(pool, (char *)utf8);

				free(utf8);
			}
		}
		id3_file_close (id3struct);
	}
	fclose(in);

exit:
	return p;
}
