/*
 *  html.c
 *  mod_musicindex
 *
 *  $Id: html.c,v 1.75 2004/05/29 13:17:44 varenet Exp $
 *
 *  Created by Thibaut VARENE on Thu Mar 20 2003.
 *  Copyright (c) 2003-2004 Regis BOUDIN
 *  Copyright (c) 2003-2004 Thibaut VARENE
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1, or (at your option)
 *  any later version.
 *
 */

/**
 * @file 
 * HTML I/O handling.
 *
 * @author Regis Boudin
 * @author Thibaut Varene
 * @version $Revision: 1.75 $
 * @date 2003-2004
 *
 * That file groups all functions related to HTML generation
 * and parsing, that is to say functions that are used to
 * send HTML code to the client and functions that are in
 * the process of parsing URI requests.
 * Here are also the functions related to m3u playlist creation.
 *
 * @todo still some work on the custom table
 * @todo maybe move treat_args somewhere else ?
 */
 
#include "html.h"
#include "inf.h"

/**
 * Formats the list of songs to be sent.
 *
 * This function formats a list of tracks (if any) to be sent
 * to the client, as found in the p list passed in argument.
 * Depending on config options, it allows streaming/shuffle etc.
 * 
 * @todo Sort option for recursive search?
 * @todo Keep sort conf on userside (using cookie(s)).
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param p struct mu_ent.
 * @param conf MusicIndex configuration paramaters struct.
 */
static void list_songs(request_rec *r, struct mu_ent *p, mu_config *conf)
{
	struct mu_ent *q = p;
	unsigned short i = 0, j = 0;
	char *current = NULL, *new = NULL;
	unsigned short local_options = conf->options;
	
	ap_rputs("  <tr class=\"title\">\n", r);
	
	if ((conf->options & (MI_LISTCUSTOM | MI_ALLOWDWNLD | MI_ALLOWSTREAM)) ||
	    ((conf->search) && (conf->options & MI_RECURSIVE)))
		ap_rvputs(r, "   <th class=\"Select\">", _("Select"), "</th>\n", NULL);

	/* The sort option is disabled for search result, and this
	   implementation handles consecutive sort requests. TODO: keep
	   memory of sort configuration by the user (either on a per-dir basis,
	   or globally), probably using cookie(s) */
	
	for (i = 0; conf->fields[i] != '\0'; i++) {	/* Display title line */
		switch (conf->fields[i]) { 
			case SB_TITLE:
				current = "Title";	/* the style class */
				new = _("Title");	/* the field name */
				break;
			case SB_TRACK:
				current = "Track";
				new = _("Track");
				break;
			case SB_POSN:
				current = "Disc";
				new = _("Disc");
				break;
			case SB_ARTIST:
				current = "Artist";
				new = _("Artist");
				break;
			case SB_LENGTH:
				current = "Length";
				new = _("Length");
				break;
			case SB_BITRATE:
				current = "Bitrate";
				new = _("Bitrate");
				break;
			case SB_ALBUM:
				current = "Album";
				new = _("Album");
				break;
			case SB_DATE:
				current = "Date";
				new = _("Date");
				break;
			case SB_FILETYPE:
				current = "Filetype";
				new = _("Filetype");
				break;
			case SB_GENRE:
				current = "Genre";
				new = _("Genre");
				break;
			default:
				continue;
		}
		
		/* This test should not be necessary. But, just to be sure... */
		if ((current == NULL) || (new == NULL))
			continue;
		
		/* With the quickplay option, do not show Length and bitrate */
		if ((local_options & MI_QUICKPL) && ((conf->fields[i] == SB_LENGTH) || (conf->fields[i] == SB_BITRATE)))
			continue;
		
		ap_rvputs(r, "   <th class=\"", current, "\">", NULL);
		if (!(local_options & MI_LISTCUSTOM) && !(conf->search))
			ap_rprintf(r, "<a href=\"?sort=%c\">%s</a>", conf->fields[i], new);
		else if ((conf->search) && !(local_options & MI_RECURSIVE))
			ap_rprintf(r, "<a href=\"?sort=%c&amp;option=%s&amp;action=Search\">%s</a>", conf->fields[i], conf->search, new);
		else
			ap_rputs(new, r);
		ap_rputs("</th>\n", r);
	}
	ap_rputs("  </tr>\n", r);

	current = "*";
	new = current;

	while (q) {	/* XXX doc */
		if ((conf->search != NULL) &&
			(conf->options & (MI_LISTCUSTOM|MI_RECURSIVE)) == MI_RECURSIVE) {
			new = ap_make_dirstr_parent(r->pool, q->file);
			
			if (strcmp(current, new)) {
				request_rec *subreq = NULL;
				current = new;
				if ((current[0] == '\0') || (current[1] == '\0')) {
					ap_rvputs(r, "  <tr class=\"title\"><th align=\"left\" colspan=\"10\">",
						_("in Current Directory"), "</th>\n", NULL);
					local_options = conf->options;	/* apache doesn't know directory '\0' ;P */
				}
				else {
					ap_rvputs(r, "  <tr class=\"title\"><th align=\"left\" colspan=\"10\">in <a href=\"",
						current, "\">", current, "</a></th>\n", NULL);
					subreq = ap_sub_req_lookup_uri(current, r, NULL);
					local_options = ((mu_config*)ap_get_module_config(subreq->per_dir_config, &musicindex_module))->options;
					ap_destroy_sub_req(subreq);
				}
				j = 0;
			}
				
		}
		
		if ((j++) & 1)
			ap_rputs("  <tr class=\"odd\">\n", r);
		else
			ap_rputs("  <tr class=\"even\">\n", r);
		
		/* prepare the "Select" panel, if any */
		if ((conf->options & (MI_LISTCUSTOM | MI_ALLOWDWNLD | MI_ALLOWSTREAM)) ||
			((conf->search) && (conf->options & MI_RECURSIVE))) {		
			ap_rputs("   <td class=\"Select\">\n", r);
			
			if (local_options & MI_ALLOWSTREAM)	/* Display checkbox */
				ap_rvputs(r, "    <input type=\"checkbox\" name=\"file\" value=\"",
					q->file, "\" />\n", NULL);
			
			if (!(conf->options & MI_LISTCUSTOM)) {
				if (local_options & MI_ALLOWDWNLD)	/* Display [download] */
					ap_rvputs(r, "    <a href=\"", ap_escape_html(r->pool, ap_escape_uri(r->pool, q->file)), "\">"
						"<img alt=\"[D]\" src=\"", conf->directory, "/", conf->fetch_icon, "\" /></a>\n", NULL);
				
				if (local_options & MI_ALLOWSTREAM)	/* Display [stream] */
					ap_rvputs(r, "    <a href=\"", ap_escape_html(r->pool, ap_escape_uri(r->pool, q->file)), "?stream\">"
						"<img alt=\"[S]\" src=\"", conf->directory, "/", conf->sound_icon, "\" /></a>\n", NULL);
			}
			
			ap_rputs("   </td>\n", r);
		}
		
		/* Now fill in the fields */
		for (i = 0; conf->fields[i] != '\0'; i++) {
			switch (conf->fields[i]) {
				case SB_TITLE:
					ap_rvputs(r, "   <td class=\"Title\">", ap_escape_html(r->pool, q->title), "</td>\n", NULL);
					break;
				case SB_TRACK:
					if (q->track)
						ap_rprintf(r, "   <td class=\"Track\">%d</td>\n", q->track);
					else
						ap_rputs( "   <td></td>\n", r);
					break;
				case SB_POSN:
					if (q->posn)
						ap_rprintf(r, "  <td class=\"Disc\">%d</td>\n", q->posn);
					else
						ap_rputs( "   <td></td>\n", r);
					break;
				case SB_ARTIST:
					ap_rprintf(r, "   <td class=\"Artist\">%s</td>\n", q->artist ? ap_escape_html(r->pool, q->artist) : "");
					break;
				case SB_LENGTH:
					if (conf->options & MI_QUICKPL)
						break;
					if (q->length)
						ap_rprintf(r, "   <td class=\"Length\">%ld:%.2ld</td>\n", q->length / 60, q->length % 60);
					else
						ap_rputs( "   <td></td>\n", r);
					break;
				case SB_BITRATE:
					if (conf->options & MI_QUICKPL)
						break;
					if (q->bitrate)
						ap_rprintf(r, "   <td class=\"Bitrate\">%ld</td>\n", q->bitrate >> 10);
					else
						ap_rputs( "   <td></td>\n", r);
					break;
				case SB_DATE:
					if (q->date)
						ap_rprintf(r, "   <td class=\"Date\">%d&nbsp;</td>\n", q->date);
					else
						ap_rputs( "   <td></td>\n", r);
					break;
				case SB_ALBUM:
					ap_rprintf(r, "   <td class=\"Album\">%s</td>\n", q->album ? ap_escape_html(r->pool, q->album) : "");
					break;
				case SB_FILETYPE:
					switch(q->filetype) {
						case FT_MP3:
							ap_rvputs(r, "   <td>MP3</td>\n", NULL);
							break;
						case FT_OGG:
							ap_rvputs(r, "   <td>OGG</td>\n", NULL);
							break;
					}
					break;
				case SB_GENRE:
					ap_rprintf(r, "   <td class=\"Genre\">%s</td>\n", q->genre ? ap_escape_html(r->pool, q->genre) : "");
					break;
				default:
					break;
			}
		}
		ap_rputs("  </tr>\n", r);
		q = q->next;
	}

}

/**
 * Treats the URL arguments.
 *
 * This function searches for keywords passed as URL arguments (with "?xxx")
 * and sets the handler options accordingly.
 * 
 * @param r Apache request_rec struct to handle connection details.
 * @param conf MusicIndex configuration paramaters struct.
 *
 * @return The corresponding option letter.
 */
void treat_args(request_rec *r, mu_config *conf)
{
	const char *s = r->args;
	const char *p;
	unsigned short i;

	if (s == NULL)
		return;

	while (s[0]) {
		p = ap_getword(r->pool, &s, '&');
		if (!strncmp(p, "action=", 7)) {
			p += 7;
			if ((!strcmp(p, "Play+Selected")) && (conf->options & MI_ALLOWSTREAM)) {
				conf->options |= MI_PLAYLIST;
			} else if ((!strcmp(p, "playall") || !strcmp(p, "Play+All")) && (conf->options & MI_ALLOWSTREAM)) {
				conf->options |= MI_PLAYALL; /* generate a playlist */
			} else if (!strcmp(p, "Shuffle+All") && (conf->options & MI_ALLOWSTREAM)) {
				conf->options |= MI_PLAYALL; /* generate a playlist */
				conf->order[0] = SB_RANDOM;
				conf->order[1] = SB_DEFAULT;
				set_fctptrs(conf);
			} else if (!strcmp(p, "Search") && (conf->options & MI_ALLOWSEARCH) && (conf->search[0] != '\0')) {
				set_fctptrs(conf);
				if (!conf->cache_path)
					conf->options |= MI_QUICKPL;
			} else if (!strcmp(p, "Recursive+Search") && (conf->options & MI_ALLOWSEARCH) && (conf->search[0] != '\0')) {
				conf->play_recursive |= MI_RECURSIVE;
				conf->order[0] = SB_DIR;
				conf->order[1] = SB_URI;
				set_fctptrs(conf);
				if (!conf->cache_path)
					conf->options |= MI_QUICKPL;
			} else if (!strcmp(p, "Remove+from+Playlist")) {
				conf->options |= MI_COOKIEDEL;
			} else if (!strcmp(p, "Clear+Playlist")) {
				conf->options |= MI_COOKIEPURGE;
			} else if (!strcmp(p, "Stream+Playlist")) {
				conf->options |= MI_COOKIESTREAM;
			} else if (!strcmp(p, "Add+To+Playlist")) {
				conf->options |= MI_COOKIEADD;
			} else if (!strcmp(p, "Add+All+To+Playlist")) {
				conf->options |= MI_COOKIEALL;
				conf->play_recursive &= ~MI_RECURSIVE;
			}
			return;
		}
		else if (!strncmp(p, "sort=", 5)) {
			p += 5;
		/* Temporary workaround : only have one sort param */
			conf->order[ARG_NUMBER-1] = '\0';
			for (i = ARG_NUMBER-2; i > 0; i--)
				conf->order[i] = conf->order[i-1];
			conf->order[0] = *p;
			set_fctptrs(conf);
		/*  XXX bon la ya 2 options: soit on arrive a partir d'un conf->order vide au premier clic de tri perso,
		et on le remplit par la droite; soit on y arrive pas, et alors on ne change que la premiere clef de
		tri en collant directement la clef par defaut (celle de la conf) derriere, ce que j'ai essaye de faire
		mais pas reussi paske la clef de la conf est aussi celle dans laquelle on ecrit... 
		Or moi je voudrais etre toujours sur de n'ecrire a la suite de la clef de tri que la partie definie
		en conf, pas le tri precedent (qui est incremental, et donc risquerait d'overflower).
		Comme on le voit dans les 2 cas, il faudrait une variable de tri intermediaire, qui ne modifie pas la conf... 
		Au final on veut: "je clic sur successivement Artist Album Title", et j'obtiens un tri par Artiste, puis par
		Albums de l'artiste, puis par chansons de l'album, pas le contraire. */
		}
		else if (!strncmp(p, "option=", 7)) {
			p += 7;
			if (!strcmp(p, "recursive"))
				conf->play_recursive |= MI_RECURSIVE;
			else if (!strcmp(p, "shuffle")) {
				conf->order[0] = SB_RANDOM;
				conf->order[1] = SB_DEFAULT;
				set_fctptrs(conf);
			}
			else if (!strcmp(p, "quick"))
				conf->options |= MI_QUICKPL;
			else {
				conf->search = ap_pstrdup(r->pool, p);
				for (i=0; p[i]; i++) {
					if (p[i] == '+')
						conf->search[i] = ' ';
				}
				ap_unescape_url(conf->search);
			}
		}
	}
}

/**
 * Formats a complete url string to be sent to client.
 *
 * This function does all the URL formatting, in the form:
 * http://[user:][passwd\@]hostname[:port]/uri/
 * 
 * @param r Apache request_rec struct to handle connection details.
 * @param uri A string (basically a path) to be sent to form a complete url.
 * @param command An eventual command (like ?action=playall).
 * @param conf MusicIndex configuration paramaters struct.
 */
static void send_url(request_rec *r, char *uri, char *command, mu_config *conf)
{
	char prefix[MAX_PREFIX];
	char str_port[6];	/* 65536 + '\0' */
	char *bufcoded, *decoded;
	unsigned short l;
	
	strcpy(prefix, "http://");

	/* if we have an icecast server, it takes precedence, in a 'staticdir' fashion */
	if (conf->iceserver) {
		/* if we only have a port number, we assume icecast is running on the same host */
		if (conf->iceserver[0] == ':')
			strcat(prefix, r->hostname);
		
		strcat(prefix, conf->iceserver);
	}
	else {
		if (REQUEST_USER(r)) {
			/* grab the auth credentials base64 encoded */
			const char *auth = ap_table_get(r->headers_in, "Authorization");
			if (auth) {
				bufcoded = strrchr(auth, ' ');
				bufcoded++;
				decoded = (char *)ap_palloc(r->pool, 1 + ap_base64decode_len(bufcoded));
				l = ap_base64decode(decoded, bufcoded);
				strncat(prefix, decoded, l);	/* we have "user:pass" */
			}
			strcat(prefix, "@");
		}
			
		/* add the hostname */
		strcat(prefix, r->hostname);
		
		/* add the port number if needed */
		if (r->server->port != 80) {
			sprintf(str_port, "%u", r->server->port);
			strcat(prefix, ":");
			strcat(prefix, str_port);
		}
	}
	
	/* add the uri and potential command */
	ap_rvputs(r, prefix, ap_escape_uri(r->pool, uri), NULL);
	if (command)
		ap_rvputs(r, command, NULL);
	ap_rvputs(r, "\n", NULL);
}	
		
/**
 * Sends HTML page headers and top of the page.
 *
 * This function takes care of the formating of the HTML page headers.
 * It sends the data that is common to all pages generated by the handler.
 * 
 * @bug No ending "/" on dir URLs.
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_head(request_rec *r, mu_config *conf)
{
	char *s, *t, *u, bk;
	request_rec *subreq;

	s = ap_pstrdup(r->pool, r->uri);

	ap_rputs("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
		"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n"
		"<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"en\" lang=\"en\">\n"
		"<head>\n"
		" <meta name=\"generator\" content=\"mod_musicindex\" />\n", r);

	if ((subreq = ap_sub_req_lookup_uri(conf->directory, r, NULL)) != NULL) {
		DIR 		*dir;
		struct dirent	*dstruct;
		
		if ((dir = opendir(subreq->filename))) {
			while ((dstruct = readdir(dir))) {
				if ((dstruct->d_name[0] != '.') &&
					(strlen(dstruct->d_name) > 4) &&
					(!strcmp(".css", dstruct->d_name + strlen(dstruct->d_name) - 4))){
				
					if (!strcmp(dstruct->d_name, conf->css))
						ap_rputs(" <link rel=\"stylesheet\" title=\"default\"", r);
					else
						ap_rvputs(r, " <link rel=\"alternate stylesheet\" title=\"", dstruct->d_name, "\"", NULL);
					
					ap_rvputs(r, " type=\"text/css\" href=\"", conf->directory, "/", dstruct->d_name, "\" />\n", NULL);
				}
			}
			closedir(dir);
		}
	}

	ap_rvputs(r, " <link rel=\"shortcut icon\" href=\"", conf->directory, "/", conf->favicon, "\" />\n"
		" <link rel=\"icon\" href=\"", conf->directory, "/", conf->favicon, "\" type=\"image/ico\" />\n"
		" <title>", _("Musical index of"), " ", r->uri, "</title>\n"
		"</head>\n\n"
		"<body>\n"
		"<!-- begin header -->\n", NULL);

	ap_rputs("<div id=\"header\">\n"
		" <div id=\"mainicon\">\n"
		"  <img alt=\"Dir\" src=\"", r);

	/* XXX case insensitive ca serait mieux, mais faut pas rever ;) */
	if (access(ap_pstrcat(r->pool, r->filename, "/cover.png", NULL), R_OK) == 0)
		ap_rputs("cover.png", r);
	else if (access(ap_pstrcat(r->pool, r->filename, "/cover.jpg", NULL), R_OK) == 0)
		ap_rputs("cover.jpg", r);
	else if (access(ap_pstrcat(r->pool, r->filename, "/cover.gif", NULL), R_OK) == 0)
		ap_rputs("cover.gif", r);
	else
		ap_rvputs(r, conf->directory, "/", conf->cd_icon, NULL);

	ap_rputs("\" />\n"
		" </div>\n", r);

	ap_rputs(" <div id=\"maintitle\">\n"
		"  <h1>\n", r);

	u = s;
	t = u + 1;
	do{
		while ((*u != '/') && (*u != '\0'))
			u++;
		
		bk = *(++u);
		*u ='\0';
		
#if 0 /* This code is here and works, it only displays directories in which the
	 module is active... Only need to decide wether we use it or
	 not... */
	 /* XXX tv: I feel this will lead us & the user to headaches... */
		subreq = ap_sub_req_lookup_uri(s, r, NULL);
		
		if (((mu_config*)ap_get_module_config(subreq->per_dir_config, &musicindex_module))->options & MI_ACTIVE) {
			if (u == (s+1))
				t = conf->title;
			else
				*(u-1) = '\0';
			
			ap_rvputs(r, "   <a href=\"", s, "\">", t, "</a>\n", NULL);
			*(u-1) = '/';
			t = u;
			
			if (*t)
				ap_rvputs(r, "     <img src=\"", conf->directory, "/",
					conf->arrow, "\" alt=\"=>\" />\n", NULL);
		}
		*u = bk;
		
		ap_destroy_sub_req(subreq);
#else
		if (u == (s+1))
			t = conf->title;
		else
			*(u-1) = '\0';
		
		ap_rvputs(r, "   <a href=\"", s, "\">", t, "</a>\n", NULL);
		*(u-1) = '/';	/* XXX ca ca marche pas, l'url se termine sans '/' final ce qui est mal */
		t = u;
		
		*u = bk;

		if (*t)
			ap_rvputs(r, "     <img src=\"", conf->directory, "/",
				conf->arrow, "\" alt=\"=>\" />\n", NULL);
#endif
	} while (*u != '\0');

	ap_rputs("  </h1>\n", r);

	if (conf->options & MI_ALLOWSTREAM) {
		ap_rvputs(r, "  <span>\n"
			"   <a class=\"shuffle\" "
			"href=\"?option=recursive&amp;option=shuffle&amp;action=playall\">[",
			_("Shuffle All"), "]</a>\n"
			"   <a class=\"stream\" "
			"href=\"?option=recursive&amp;action=playall\">[",
			_("Stream All"), "]</a>\n"
			"  </span>\n", NULL);
	}
	
	ap_rputs(" </div>\n", r);

	/* displays a search box if option is activated */
 	if (conf->options & MI_ALLOWSEARCH){
		ap_rvputs(r,
			" <form method=\"get\" action=\"", ap_escape_uri(r->pool, r->uri), "\""
			"enctype=\"application/x-www-form-urlencoded\" id=\"searching\">\n"
			"  <p>\n"
			"   <input type=\"text\" name=\"option\" />\n"
			"   <br />\n"
			"   <input type=\"submit\" name=\"action\" value=\"Search\" />\n"
			"   <input type=\"submit\" name=\"action\" value=\"Recursive Search\" />\n"	/* XXX youpi on peut pas i18n-iser... */
			"   <input type=\"hidden\" name=\"action\" value=\"Search\" />\n"
			"  </p>\n"
			" </form>\n",
			NULL);
	}

	ap_rputs("</div>\n"
		"<hr />\n"
		"<!-- end header -->\n\n", r);

}

/**
 * Sends directory listing for the current folder.
 *
 * This function takes care of preparing and sending to the client the
 * list (if any) of the available directories in the current folder.
 * Depending on config options, it allows streaming/shuffle etc.
 * 
 * @param r Apache request_rec struct to handle connection details.
 * @param p struct mu_ent.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_directories(request_rec *r, struct mu_ent *p, mu_config *conf)
{
	struct mu_ent *q = p;
	unsigned short dircnt = 0, nb = 0;
	char *c;
	char temp[MAX_STRING];

	if (!q)
		return;

	if (q->filetype != FT_DIR)
		return;
	
	/* We count the number of directories for later use */
	do {
		nb++;
	} while ((q = q->next) && (q->filetype == FT_DIR));
	
	q = p;
		
	ap_rputs("<!-- begin subdirs -->\n<h2>",r);
	ap_rprintf(r, _("Music Directories (%d)"), nb);
	ap_rputs("</h2>\n\n<table id=\"directories\">\n", r);

	while (q && (q->filetype == FT_DIR)) {
		c = ap_cpystrn(temp, q->file, MAX_STRING)-1;
		*c = '\0';
		
		if (dircnt++ == 0)
			ap_rputs(" <tr>\n", r);
		
		ap_rvputs(r, "  <td>\n"
			"   <a href=\"", ap_escape_uri(r->pool, q->file), NULL);
		
		if (conf->options & MI_ALLOWSTREAM)
			ap_rputs("?option=recursive&amp;action=playall", r);
		
		ap_rputs("\"><img alt=\"\" src=\"", r);
#ifdef SHOW_THUMBNAILS
/* XXX tv: This is just wrong. It doesn't rescale the picture, hence, loading 20 dirs with a 200kB pic each is AWFUL
Not mentionning the fact that letting the browser do the scaling leads often to ugly and unreadable results.
That's IMHO a bad feature */		
		/* TODO : optimize to use less memory */
		if (access(ap_pstrcat(r->pool, r->filename, "/", q->file, "cover.png", NULL), R_OK) == 0)
			ap_rvputs(r, q->file, "cover.png", NULL);
		else if (access(ap_pstrcat(r->pool, r->filename, "/", q->file, "cover.jpg", NULL), R_OK) == 0)
			ap_rvputs(r, q->file, "cover.jpg", NULL);
		else if (access(ap_pstrcat(r->pool, r->filename, "/", q->file, "cover.gif", NULL), R_OK) == 0)
			ap_rvputs(r, q->file, "cover.gif", NULL);
		else
#endif
			ap_rvputs(r, conf->directory, "/", conf->small_cd_icon, NULL);
		
		ap_rputs("\" /></a>\n", r);
		
		ap_rvputs(r, "   <div>\n"
			"    <a href=\"", ap_escape_uri(r->pool, q->file), "\">",
	    		temp, "</a><br />\n", NULL);
		
		if (conf->options & MI_ALLOWSTREAM) {
			ap_rvputs(r, "     <a class=\"shuffle\" href=\"",
				ap_escape_uri(r->pool, q->file),
				"?option=recursive&amp;option=shuffle&amp;action=playall\">[",
				_("Shuffle"), "]</a>\n", NULL);
			
			ap_rvputs(r, "     <a class=\"stream\" href=\"",
				ap_escape_uri(r->pool, q->file),
				"?option=recursive&amp;action=playall\">[",
				_("Stream"), "]</a>\n", NULL);
		}
		
		ap_rputs("   </div>\n"
			"  </td>\n",r);
		
		if (dircnt == DIRPERLINE) {
			dircnt = 0;
			ap_rputs(" </tr>\n", r);
		}
		q = q->next;
	}
	
	if (dircnt != 0)
		ap_rputs( "</tr>\n", r);

	ap_rputs("</table>\n<hr />\n<!-- end subdirs -->\n\n", r);
}

/**
 * Sends track listing for the current folder.
 *
 * This function takes care of preparing and sending to the client the
 * list (if any) of the available songs in the current p list.
 * Depending on config options, it allows streaming/shuffle etc.
 *
 * @todo better handling of per-directory configuration for select panel
 * 
 * @param r Apache request_rec struct to handle connection details.
 * @param p struct mu_ent.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_tracks(request_rec *r, struct mu_ent *p, mu_config *conf)
{
	struct mu_ent *q = p;
	unsigned short nb = 0;
	
	/* We count the number of songs for later use */
	for (q=p; (q!=NULL); q = q->next) {
		if (q->filetype != FT_DIR)
			nb++;
	}

	/* Go to the first song entry (just after dirs) */
	for (q=p; (q!=NULL); q = q->next) {
		if (q->filetype != FT_DIR)
			break;
	}

	if (!(q))
		return;

	ap_rputs("<!-- begin tracks -->\n<h2>", r);
	
	if (conf->search)
		ap_rprintf(r, _("Result List (%d)"), nb);
	else
		ap_rprintf(r, _("Song List (%d)"), nb);
	
	ap_rputs("</h2>\n\n", r);

	ap_rvputs(r, "<form method=\"get\" action=\"", ap_escape_uri(r->pool, r->uri), "\" "
		"enctype=\"application/x-www-form-urlencoded\" id=\"tracks\">\n", NULL);
	
	ap_rputs(" <table>\n", r);

	list_songs(r, q, conf);
	
	/* Do NOT show 'select all' if there's nothing to select... TODO: better
	   handling of per-directory configuration */
	if ((conf->search) && (conf->options & MI_ALLOWSTREAM))
		ap_rputs("  <tr class=\"title\"><th align=\"left\" colspan=\"10\">\n"
			"   <input type=\"checkbox\" name=\"all\" onClick=\"for(var i=0;i<this.form.elements.length;i++){var inpt=this.form.elements[i];"
			"var m=inpt.name.match(/-/g);if((inpt.name.substr(0,4)=='file') && (m<1)) inpt.checked=this.form.all.checked}\" />\n"
			"Select All</th>\n"
			"</tr>\n", r);
			
	ap_rputs(" </table>\n", r);

	/* Do NOT show buttons if we do not allow anything but listing or downloading! */
	if (conf->options & MI_ALLOWSTREAM) {
		ap_rvputs(r, " <div>\n"
			"  <input type=\"hidden\" name=\"sort\" value=\"", conf->order,"\" />\n"
			"  <input type=\"submit\" name=\"action\" value=\"Add To Playlist\" class=\"playlist\" />\n", NULL); /* XXX i18n DTC */
	
		if (conf->search == NULL)
			ap_rputs("  <input type=\"submit\" name=\"action\" value=\"Add All To Playlist\" class=\"playlist\" />\n"
				"  <input type=\"submit\" name=\"action\" value=\"Shuffle All\" />\n"
				"  <input type=\"submit\" name=\"action\" value=\"Play All\" />\n", r);	/* XXX idem */
	
		ap_rputs("  <input type=\"submit\" name=\"action\" value=\"Play Selected\" />\n"
			" </div>\n", r);	/* XXX et encore */
	}
	
	ap_rputs("</form>\n"
		"<hr />\n"
		"<!-- end tracks -->\n\n", r);
}

void send_customlist(request_rec *r, mu_ent *p, mu_config *conf)
{
	struct mu_ent *q = p;
	unsigned short nb = 0;
	
	if (p == NULL)
		return;
	
	/* We count the number of songs for later use */
	for (q=p; (q!=NULL); q = q->next) {
		nb++;
	}

	ap_rputs("<!-- begin custom -->\n<h2>", r);
	ap_rprintf(r, _("Custom Playlist (%d)"), nb);
	ap_rputs("</h2>\n\n", r);
	
	ap_rvputs(r, " <form method=\"get\" action=\"", ap_escape_uri(r->pool, r->uri), "\" "
		"enctype=\"application/x-www-form-urlencoded\" id=\"custom\">\n", NULL);

	ap_rputs("  <table>\n", r);
	
	conf->options |= MI_LISTCUSTOM;
	list_songs(r, p, conf);
	conf->options &= ~(MI_LISTCUSTOM);

	ap_rputs("  </table>\n"
		"  <div>\n"
		"   <input type=\"submit\" name=\"action\" value=\"Remove from Playlist\" class=\"playlist\" />\n"
		"   <input type=\"submit\" name=\"action\" value=\"Clear Playlist\" class=\"playlist\" />\n"
		"   <input type=\"submit\" name=\"action\" value=\"Stream Playlist\" class=\"playlist\" />\n"
		"  </div>\n"
		" </form>\n"
		"<hr />\n"
		"<!-- end custom -->\n\n", r);
}

/**
 * Sends playlist with url and EXTM3U info to the client.
 *
 * This function prepares an m3u playlist (with extra information where
 * available: length, artist, title, album) and sends it to the client.
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param p struct mu_ent.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_playlist(request_rec *r, mu_ent *p, mu_config *conf)
{
	mu_ent *q = p;
	if (!p)
		return;

	ap_rputs("#EXTM3U\n", r);

	while (q)
	{
		ap_rprintf(r, "#EXTINF:%li,", q->length);
		if (q->artist)
			ap_rvputs(r, q->artist, " - ", NULL);
		ap_rvputs(r, q->title, NULL);
		if (q->album)
			ap_rvputs(r, " (", q->album, ")", NULL);
		ap_rvputs(r, "\n", NULL);
		send_url(r, q->uri, NULL, conf);
		q = q->next;
	}
}

/**
 * Sends HTML page footers.
 *
 * This function takes care of the formating of the HTML page footers.
 * It sends the data that is common to all pages generated by the handler.
 * Currently it relies on apache_mp3 original stylesheets.
 * 
 * @param r Apache request_rec struct to handle connection details.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_foot(request_rec *r, mu_config *conf)
{
	ap_rprintf(r, "<!-- begin footer -->\n<!-- mod_musicindex v.%s -->\n", MUSIC_VERSION_STRING);
	ap_rprintf(r, "<!-- Authors: %s -->\n", MUSIC_AUTHORS_STRING);
	ap_rputs("<div id=\"footer\">\n"
		" <div id=\"valid\">\n"
#if 1
		"  <a href=\"http://validator.w3.org/check/referer\">\n"
#else
		"  <a href=\"http://localhost/w3c-markup-validator/check/referer\">\n"
#endif
		"   <img src=\"http://www.w3.org/Icons/valid-xhtml10\"\n"
        	"    alt=\"Valid XHTML 1.0!\" height=\"31\" width=\"88\" />\n"
		"  </a>\n"
		"  <a href=\"http://jigsaw.w3.org/css-validator/\">\n"
		"   <img src=\"http://jigsaw.w3.org/css-validator/images/vcss\"\n"
		"    alt=\"Valid CSS!\" height=\"31\" width=\"88\" />\n"
		"  </a>\n"
		" </div>\n", r);
	ap_rprintf(r, " <div id=\"name\">"
		"<a href=\"http://freshmeat.net/projects/musicindex/\">"
		"MusicIndex v.%s</a></div>\n", MUSIC_VERSION_STRING);
	ap_rputs("</div>\n<!-- end footer -->\n\n</body>\n</html>", r);
}
