/*
 *  config.c
 *  mod_musicindex
 *
 *  $Id: config.c,v 1.47 2004/05/29 11:26:54 varenet Exp $
 *
 *  Created by Thibaut VARENE on Thu Mar 20 2003.
 *  Copyright (c) 2003-2004 Regis BOUDIN
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1, or (at your option)
 *  any later version.
 *
 */

/**
 * @file 
 * Configuration handling and default settings.
 *
 * @author Regis Boudin
 * @version $Revision: 1.47 $
 * @date 2003-2004
 *
 * That file takes care of the module configuration, either via setting some
 * known default values, and/or by getting them from environmental configuration
 * variables passed to apache in its config file.
 *
 * @todo Regis, please document.
 */

#include "config.h"
#include "inf.h"

#define COOKIE_DEFAULT_LIFE 300

static const char default_sort[] = "APNBTLRFMU";	/**< Default sort order */
static const char default_fields[] = "TBLR";		/**< Default fields displayed */

static char default_directory[] = "/musicindex";	/**< Default musicindex directory */

/**
 * Generates a per-directory configuration.
 *
 * This function creates a new configuration structure and fills it with
 * default values.
 * 
 * @param p Apache pool for memory allocation.
 * @param dummy String given by apache we don't use.
 *
 * @return The newly created configuration structure.
 */
void *create_musicindex_config(apr_pool_t *p, char *dummy)
{
	unsigned short i;
	mu_config *new = (mu_config *)ap_pcalloc(p, sizeof(mu_config));

	for (i = 0; i < ARG_NUMBER; i++) {
		new->order[i] = SB_DEFAULT;
		new->fields[i] = '\0';
	}

	strcpy(new->fields, default_fields);
	strcpy(new->order, default_sort);

	new->title = ap_pstrdup(p, "Music");

	new->directory = default_directory;
	new->favicon = ap_pstrdup(p, "sound.png");
	new->cd_icon = ap_pstrdup(p, "general.png");
	new->small_cd_icon = ap_pstrdup(p, "directory.png");
	new->sound_icon = ap_pstrdup(p, "sound.png");
	new->fetch_icon = ap_pstrdup(p, "fetch.png");
	new->arrow = ap_pstrdup(p, "right_arrow.gif");
	new->css = ap_pstrdup(p, "musicindex.css");
	new->search = NULL;
	new->cache_path = NULL;
	new->iceserver = NULL;

	new->cookie_life = COOKIE_DEFAULT_LIFE;

	new->play_recursive = ~MI_RECURSIVE;
	new->options = MI_RECURSIVE;
	new->options_not = 0;
	
	set_fctptrs(new);

	return (void *)new;
}

/**
 * Merge per-directory configurations.
 *
 * This function is supposed to merge two per-directory configurations. We
 * simply copy it from the parent directory.
 * 
 * @param p Apache pool for memory allocation.
 * @param basev Pointer to configuration structure.
 * @param addv Pointer to parent configuration structure.
 *
 * @return The newly created configuration structure.
 */
void *merge_musicindex_configs(apr_pool_t *p, void *basev, void *addv)
{
	mu_config *new = (mu_config *)ap_pcalloc(p, sizeof(mu_config));
	mu_config *base = (mu_config *)basev;
	mu_config *add = (mu_config *)addv;

	if (strcmp(add->order, default_sort))
		strcpy(new->order, add->order);
	else
		strcpy(new->order, base->order);

	if (strcmp(add->fields, default_fields))
		strcpy(new->fields, add->fields);
	else
		strcpy(new->fields, base->fields);

	new->title = ap_pstrdup(p, add->title);

/* This is an example for later configuration use
	if (add->directory != default_directory)
		new->directory = ap_pstrdup(p, add->directory);
	else if (base->directory != default_directory)
		new->directory = ap_pstrdup(p, base->directory);
	else
*/
	new->directory = default_directory;

	new->favicon = ap_pstrdup(p, add->favicon);
	new->cd_icon = ap_pstrdup(p, add->cd_icon);
	new->small_cd_icon = ap_pstrdup(p, add->small_cd_icon);
	new->sound_icon = ap_pstrdup(p, add->sound_icon);
	new->fetch_icon = ap_pstrdup(p, add->fetch_icon);
	new->arrow = ap_pstrdup(p, add->arrow);
	new->css = ap_pstrdup(p, add->css);
	new->search = ap_pstrdup(p, add->search);
	new->cache_path = ap_pstrdup(p, add->cache_path);

	if (add->iceserver) /* new server defined ? */
		new->iceserver = ap_pstrdup(p, add->iceserver);
	else if (base->iceserver) /* previous sever defined ? */
		new->iceserver = ap_pstrdup(p, base->iceserver);
	else /* OK, no server defined... */
		new->iceserver = NULL;

	new->play_recursive = add->play_recursive;
	new->options = base->options | add->options;
	new->options &= ~(add->options_not);
	
	new->options_not = add->options_not;
	
	if (add->cookie_life != COOKIE_DEFAULT_LIFE)
		new->cookie_life = add->cookie_life;
	else
		new->cookie_life = base->cookie_life;
	
	set_fctptrs(new);

	return new;
}

typedef struct {
	const char *string;
	char value;
} option_string;

option_string options[] = {
	{"track",	SB_TRACK},
	{"disc",	SB_POSN},
	{"length",	SB_LENGTH},
	{"bitrate",	SB_BITRATE},
	{"artist",	SB_ARTIST},
	{"album",	SB_ALBUM},
	{"title",	SB_TITLE},
	{"filename",	SB_FILENAME},
	{"date",	SB_DATE},
	{"filetype",	SB_FILETYPE},
	{"genre",	SB_GENRE},
	{"uri",	SB_URI},
	{NULL, '\0'}
};

void sort_or_fields(cmd_parms *cmd, char *list, const char *optstr)
{
	char *r;
	unsigned short i = 0, j;

	while (optstr[0] && (i < ARG_NUMBER)) {
		r = ap_getword_conf(cmd->pool, &optstr);
		for (j=0; options[j].string; j++) {
			if (!strcasecmp(r, options[j].string)) {
				list[i++] = options[j].value;
				break;
			}
		}	
	}
	if (i == ARG_NUMBER)
		i--;
	list[i] = '\0';
}

const char *sort_order(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *d_cfg = (mu_config *)d;
	sort_or_fields(cmd, d_cfg->order, optstr);
	set_fctptrs(d_cfg);
	return NULL;
}

const char *set_fields(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *d_cfg = (mu_config *)d;
	sort_or_fields(cmd, d_cfg->fields, optstr);
	return NULL;
}

/**
 * Enables the module.
 *
 * This function simply sets the module as active.
 * 
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param bool
 *
 * @return NULL don't ask why.
 */
const char *music_lister(cmd_parms *cmd, void *d, int bool)
{
	mu_config *cfg = (mu_config *)d;
	if (bool)
		cfg->options |= MI_ACTIVE;
	else {
		cfg->options &= ~MI_ACTIVE;
		cfg->options_not |= MI_ACTIVE;
	}

	return NULL;
}

/**
 * Enables the stream in the module.
 *
 * This function simply sets the streaming function of the module as active.
 * 
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param bool
 *
 * @return NULL don't ask why.
 */
const char *allow_stream(cmd_parms *cmd, void *d, int bool)
{
	mu_config *cfg = (mu_config *)d;
	if (bool)
		cfg->options |= MI_ALLOWSTREAM;
	else {
		cfg->options &= ~MI_ALLOWSTREAM;
		cfg->options_not |= MI_ALLOWSTREAM;
	}

	return NULL;
}

/**
 * Enables the download in the module.
 *
 * This function simply sets the downloading function of the module as active.
 * 
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param bool
 *
 * @return NULL don't ask why.
 */
const char *allow_download(cmd_parms *cmd, void *d, int bool)
{
	mu_config *cfg = (mu_config *)d;
	if (bool)
		cfg->options |= MI_ALLOWDWNLD;
	else {
		cfg->options &= ~MI_ALLOWDWNLD;
		cfg->options_not |= MI_ALLOWDWNLD;
	}

	return NULL;
}

/**
 * Enables the search in the module.
 *
 * This function simply sets the searching function of the module as active.
 * 
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param bool
 *
 * @return NULL don't ask why.
 */
const char *allow_search(cmd_parms *cmd, void *d, int bool)
{
	mu_config *cfg = (mu_config *)d;
	if (bool)
		cfg->options |= MI_ALLOWSEARCH;
	else {
		cfg->options &= ~MI_ALLOWSEARCH;
		cfg->options_not |= MI_ALLOWSEARCH;
	}
	
	return NULL;
}

/**
 * Configures the cache location in the filesystem.
 *
 * This function sets where the cache should be stored in the system.
 * 
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
const char *set_cache_path(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *cfg = (mu_config *)d;
	cfg->cache_path = ap_pstrdup(cmd->pool, optstr);
	return NULL;
}

/**
 * Sets the title of the page.
 *
 * This function simply chages the name of the root.
 * 
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
const char *set_page_title(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *cfg = (mu_config *)d;
	cfg->title = ap_pstrdup(cmd->pool, optstr);
	return NULL;
}

/**
 * Sets the icecast server address.
 * This function sets an icecast server for streaming.
 * 
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
const char *set_ice_server(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *cfg = (mu_config *)d;
	cfg->iceserver = ap_pstrdup(cmd->pool, optstr);
	cfg->options |= MI_ALLOWSTREAM; /* Should we let the admin set it ? */
	return NULL;
}

/**
 * Sets the default CSS file
 * This function sets which CSS file will be used by default
 * 
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
const char *set_css_default(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *cfg = (mu_config *)d;
	cfg->css = ap_pstrdup(cmd->pool, optstr);
	return NULL;
}

/**
 * Sets the default CSS file
 * This function sets which CSS file will be used by default
 * 
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
const char *set_cookie_life(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *cfg = (mu_config *)d;
	cfg->cookie_life = atoi(optstr);
	return NULL;
}
