#!/usr/bin/perl -w

#$Header: /home2/cvsroot/LogTrend/Common/LogDie.pm,v 1.6 2001/06/07 14:08:54 slhullier Exp $
##******************************************************************************
## Class LogDie
##  Description  : A package for sysloging and dieing
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Sylvain Lhullier s.lhullier@atrid.fr (06.03.2001)
##******************************************************************************
#$Log: LogDie.pm,v $
#Revision 1.6  2001/06/07 14:08:54  slhullier
#
#Passage de  unshift @INC,'..';  aux packages Logtrend::....
#
#Revision 1.5  2001/05/29 16:43:32  fdesar
#
#Corrected loop problem in OpenLog
#
#Revision 1.4  2001/05/29 12:58:27  fdesar
#
#Changes to log multiple lines messages to syslog
#
#Revision 1.2  2001/03/22 13:32:24  slhullier
#
#Petit bug du serveur du au parseur
#
#Revision 1.1  2001/03/07 15:15:30  slhullier
#
#LogDie mis en place sur le serveur et l'agent Linux
#
#Revision 1.1  2001/03/07 10:53:59  slhullier
#
#Extrait de l'agent
#

package LogTrend::Common::LogDie;

use Sys::Syslog qw(:DEFAULT setlogsock);

use Exporter;
use vars qw( @ISA @EXPORT );
@ISA = qw(Exporter);
@EXPORT = qw( &OpenLog &SysLog &Message &CloseLog &Die );

use strict;

my $daemon_mode;
my $program_name;

##******************************************************************************
## Method OpenLog  public
##  Description  : prepare the logger : opens a connection to the system logger
##                 if necessary
##  Parameters   : the name of the program, the demon mode ("daemon"/"nodaemon")
##  Return value : none
##******************************************************************************
sub OpenLog
{
   my ($program, $daemon) = @_;

   if(defined($daemon_mode) )
   { return }

   if( $daemon eq "daemon" )
   {
      setlogsock('unix');
      openlog( "$program", 'pid', 'user' );
      $daemon_mode = 1;
   }
   elsif( $daemon eq "nodaemon" )
   {
      $daemon_mode = 0;
   }
   else
   {
      Die( "No such daemon mode ($daemon)" );
   }

   $program_name = $program;

}

##******************************************************************************
## Method Internal_SysLog  private
##  Description  : generates a log message to the system logger or to stderr
##  Parameters   : the message
##  Return value : none
##******************************************************************************
sub Internal_SysLog
{
   my ($message) = @_;

   if( $daemon_mode == 1 )
   {
      syslog( "user|info", "%s", $message );
   }
   else
   {
      print STDERR "$program_name: $message";
   }

}

##******************************************************************************
## Method SysLog  public
##  Description  : generates a log message to the system logger or to stderr
##  Parameters   : the message
##  Return value : none
##******************************************************************************
sub SysLog
{
   my ($message) = @_;
   $message = $! if( !defined($message) );

   my ($package,$file,$line) = caller();

   if( !defined($daemon_mode) )
   { OpenLog( $0, "nodaemon" ); }
   for (split("\n", $message)) {
       Internal_SysLog( "[$file:$line] $_\n" );
   }
}

##******************************************************************************
## Method Message  public
##  Description  : generates a message to stdout if not in daemon mode
##  Parameters   : the message
##  Return value : none
##******************************************************************************
sub Message
{
   my ($message) = @_;
   if( !defined($daemon_mode) ||
       $daemon_mode == 0 )
   {
      print "$message\n";
   }
}

##******************************************************************************
## Method CloseLog  public
##  Description  : closes the descriptor to the system logger
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CloseLog
{
   return if( !defined($daemon_mode) );

   closelog() if( $daemon_mode == 1 );
 
   undef($daemon_mode);

}

##******************************************************************************
## Method Die  public
##  Description  : makes the program die with message logging
##  Parameters   : the message (default:$!), the exit status value (default:1)
##  Return value : none (the program die...)
##******************************************************************************
sub Die
{
   my ($message,$status) = @_;
   $message = $! if( !defined($message) );
   $status  = 1  if( !defined($status) );

   my ($package,$file,$line) = caller();

   if( !defined($daemon_mode) )
   { OpenLog( $0, "nodaemon" ); }
   for (split("\n", $message)) {
       Internal_SysLog( "[$file:$line] $_\n" );
   }
   CloseLog();
   exit( $status );
}

##******************************************************************************
1;
