#!/usr/bin/perl

#
# Common.pl: Holds routines common to a few scripts of
# localization-config
# Based on the example apt-cache in libapt-pkg-perl
#

use AptPkg::Config '$_config';
use AptPkg::System '$_system';
use AptPkg::Cache;
use AptPkg::Version;

my $cache;
my $vs;

sub debug {
    print @_ if $opts{'d'};
}

sub init {
    # initialise the global config object with the default values and
    # setup the $_system object
    $_config->init;
    $_system = $_config->system;

    # supress cache building messages
    $_config->{quiet} = 2;

    # set up the cache
    $cache = AptPkg::Cache->new;
    $vs = $_system->versioning;
}

sub is_installed($) {
    my ($pack) = @_;
    my $p = $cache->{$pack};
    unless ($p)
    {
        return 0;
    }
    
    if ($p->{CurrentVer}) {
        return 1;
    } else {
        return 0;
    }    
}

# We only use this subroutine from the scripts.
# It takes the package name and it's version map
# and it returns the most probable release name for
# this package.
# According to this release name, we execute the
# appropriate configuration script.

sub get_release($) {
    my ($pack, %vermap) = @_;
    my $p = $cache->{$pack};
    my $rel;
    unless ($p)
    {
        debug("Fallback to /etc/debian_version\n");
        $rel = get_debian_version();
        debug("Release to be used: $release\n");
        return $rel;
    }

    debug("Package: $pack\n");

    if ($p->{CurrentVer}) {
        my $curver = $p->{CurrentVer}{VerStr};
        $rel = check_vermap($curver, %vermap);
        debug("CurrentVer: $curver\n");
    } else {
        debug("Not Installed\n");
        debug("Checking Available versions\n");
    
        if (my $available = $p->{VersionList})
        {
            # Check the first available version in package version list
            # we only care about the latest package anyway.
            $curver = @$available[0]->{VerStr};
            $rel = check_vermap($curver, %vermap);
        }
    }
    
    if ($rel eq "") {
        $rel = get_debian_version();
    }

    debug("Release to be used: $rel\n");
    return $rel;
}

# This subroutine does the actual retrieval of the 
# release name in the version map.
sub check_vermap($) {
    my ($ver, %vermap) = @_;
    my $rel;
    for $v (sort keys %vermap) {
        if ($ver ge $v) {
            # print "$v\t".$vermap{$v}->{'RELEASE'}."\n";
            $rel = $vermap{$v}->{'RELEASE'};
    }
    }
    return $rel;
}

# Will return the first correct supported Debian version
# according to the value in /etc/debian_version
# This is used only as last resort.
sub get_debian_version {
    my %supported_versions = (  'woody'            => { RELEASE => 'woody' },
                                'sarge'            => { RELEASE => 'sarge' },
                                'testing/unstable' => { RELEASE => 'sarge' },
				'3.1'              => { RELEASE => 'sarge' }
                             );
    my $debver_file= "/etc/debian_version";
    my $version, my $found, my $folder;
    open(DAT, $debver_file) || die("Could not open /etc/debian_version!");
    my @raw_data=<DAT>;
    close(DAT);
    foreach $version (@raw_data) {
        chop($version);
        if (defined($found = $supported_versions{$version})) {
        return $found->{'RELEASE'};
        }
    }
}

sub get_subarch() {
   my $subarch = 'none';
   my $proc_arch = `grep ^machine /proc/cpuinfo`;
   
   if ($proc_arch =~ /PReP.Blackhawk/) {
        $subarch = 'ppcbug';
   } elsif ($proc_arch =~ /PReP/ ) {
        $subarch = 'prep';
   } elsif ($proc_arch =~ /CHRP/ ) {
        $subarch = 'chrp';
   } elsif ($proc_arch =~ /NewWorld/ ) {
        $subarch = 'pmac_newworld';
   } elsif ($proc_arch =~ /OldWorld/ ) {
        $subarch = 'pmac_oldworld';
   } elsif ($proc_arch =~ /Amiga/ ) {
        $subarch = 'amiga';
   }

   return $subarch;
}

1;
