/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */
 
#include "include/support.h"
#include "include/program.h"
#include "include/parser.h"
#include "include/interpreter.h"

static gboolean keypress_cb (GtkWidget *widget,
GdkEventKey *event, LWInterpreter *interp)
{
	(void) widget;
	
	LWContext *context = interp->getContext();
	
	g_queue_push_tail (context->queue_keys,	GUINT_TO_POINTER(event->keyval));
	
	if (context->wait_for & LW_RESUME_KEYPRESS)
		interp->resume ();
	
	return TRUE;
}

static gboolean timeout_cb (LWInterpreter *interp)
{
	interp->resume ();
	return FALSE;
}

static gboolean idle_cb (LWInterpreter *interp)
{
	LWContext *context = interp->getContext();
		
	if (FALSE == interp->resume ())
		return FALSE;

	if (context->wait_for & LW_RESUME_KEYPRESS)
		if (FALSE == g_queue_is_empty (context->queue_keys))
			return TRUE;
		
	return (context->wait_for & LW_RESUME_IDLE) ? TRUE : FALSE;
}

static void free_value (LWValue *value, gpointer dummy)
{
	(void) dummy;
	
	if (FALSE == value->isVariable())
		delete value;	
}

/* Constructor of interpreter */
LWInterpreter::LWInterpreter (LWProgram *a_program, LWProgramData *a_program_data)
: program(a_program), program_data (a_program_data)
{
	context.instrPtr = NULL;
	buildList();
}

/* Destructor */
LWInterpreter::~LWInterpreter ()
{							   
	while (TRUE == g_source_remove_by_user_data ((gpointer) this));

	if (NULL != program->getWorld()) {
		GtkWidget *window = gtk_widget_get_ancestor (program->getWorld()->getWidget(), GTK_TYPE_WINDOW);
		g_signal_handlers_disconnect_by_func(G_OBJECT(window), (gpointer) keypress_cb, (gpointer) this);
	}
	
	g_slist_free (list);
	g_queue_foreach (context.stack, (GFunc) free_value, NULL);
	
	g_queue_free (context.stack);
	g_queue_free (context.queue_keys );
}

/* Build a list from a tree */
void LWInterpreter::buildList()
{
	GQueue *q = g_queue_new();
	
	list = NULL;
		
	g_queue_push_head (q, (gpointer) program_data->getTree());
				
	while (FALSE == g_queue_is_empty(q)) {
		GNode *n = (GNode *) g_queue_pop_head (q);
		list = g_slist_prepend (list, (gpointer) n);
		
		for (n = n->children; n != NULL; n = n->next)
			g_queue_push_head (q, (gpointer) n);
	}
				
	g_queue_free (q);
}

void LWInterpreter::execute ()
{
	g_return_if_fail (context.instrPtr == NULL);
	g_return_if_fail (program->getWorld() != NULL);
	GtkWidget *window = gtk_widget_get_ancestor (program->getWorld()->getWidget(), GTK_TYPE_WINDOW);
	gtk_window_set_title (GTK_WINDOW (window), _("Executing program..."));
	gtk_window_set_modal (GTK_WINDOW(window), TRUE);
	GTK_WIDGET_SET_FLAGS (window, GTK_CAN_FOCUS);
	gtk_widget_grab_focus (window);
	
	g_signal_connect (G_OBJECT (window), "key_press_event", 
		G_CALLBACK (keypress_cb), (gpointer) this);
	
	context.wizard = program->getWorld()->getWizard();
	context.stack = g_queue_new ();
	context.queue_keys = g_queue_new ();
	context.instrPtr = (GNode *) list->data;
	context.wait_for = 0;	
	context.resume = FALSE;
	
	list_entry = list;
	lastip = NULL;
	resume_point = NULL;
	
	resume ();
}

gboolean LWInterpreter::resume ()
{
	typedef GNode *GNodePtr;
	GNodePtr &ip = context.instrPtr;
	
	if (resume_point != lastip) {
		list_entry = (lastip == ip) ? list_entry->next : g_slist_find (list, (gpointer) ip);
		
		if (list_entry == NULL) {
			finish();
			return FALSE;
		}
		
		ip = (GNode *)list_entry->data;
	}
	
	lastip = ip;

	LWCommand *cmd = LWParser::getNodeCommand (ip);

	guint oldwaitfor = context.wait_for;

	context.wait_for = LW_RESUME_IDLE;
	context.resume = FALSE;
	
	try {
		
		if (resume_point == ip) {
			resume_point = NULL;
			cmd->execute (&context, TRUE);
		} else
			cmd->execute (&context, FALSE);
		
	} catch (LWMessage *msg) {
		msg->setPiece((LWPiece *) lastip->data);
		program->showMessage(msg);
		return FALSE;
	}
	
	if (context.resume == TRUE) {
		g_return_val_if_fail (resume_point == NULL, FALSE);
		resume_point = ip;
	}
	
	if (ip == NULL)
		ip = lastip->parent;
	else
		if (lastip != ip) {
			gboolean children_before_parent = TRUE;
			
			//Check if command request one of the parents
			for (GNode *n = lastip; n != NULL; n = n->parent)
				if (ip == n) {
					children_before_parent = FALSE;
					break;
				}
		
			//If not, we just entering new commands, don't forget about children
			if (children_before_parent == TRUE)
				for (GNode *n = ip; n != NULL; n = n->children)
					ip = n;	
		}

	//Optimization, it's not necessary to reintialize all sources in that case
	if (oldwaitfor == LW_RESUME_IDLE && context.wait_for == LW_RESUME_IDLE)
		return TRUE;

	while (TRUE == g_source_remove_by_user_data ((gpointer) this));

	if (context.wait_for & LW_RESUME_TIMEOUT)
		g_timeout_add (context.wait_time, (GSourceFunc) timeout_cb, (gpointer) this);

	if ((context.wait_for & LW_RESUME_IDLE) || 
		(FALSE == g_queue_is_empty (context.queue_keys) && (context.wait_for & LW_RESUME_KEYPRESS)))
			g_idle_add ( (GSourceFunc) idle_cb, (gpointer) this);
	
	return TRUE;
}
	
void LWInterpreter::finish()
{
	while (TRUE == g_source_remove_by_user_data ((gpointer) this));
	
	context.instrPtr = NULL;
	context.wait_for = 0;
	
	GtkWidget *window = gtk_widget_get_ancestor (program->getWorld()->getWidget(), GTK_TYPE_WINDOW);
	gtk_window_set_title (GTK_WINDOW (window), _("Program finished..."));
	gtk_window_set_modal (GTK_WINDOW(window), FALSE);
	
	program->finish ();
}

void LWInterpreter::stop()
{
	g_return_if_fail (context.instrPtr != NULL);
	
	LWMessage *msg = new LWMessage (LW_INFO_ProgramInterrupted);
	msg->setPiece((LWPiece *) context.instrPtr->data);
	program->showMessage(msg);
}

gboolean LWInterpreter::isRunning()	
{
	return (context.instrPtr != NULL) ? TRUE : FALSE;
}

LWContext *LWInterpreter::getContext()
{
	return &context;
}
