/*
 *
 * Linup - Uptimes Project Client for Linux
 * Copyright (C) 2002  Joao Sena Ribeiro
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "linup.h"
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>

/* send_info_tcp(char*, char*, int, char*, int, char*, int) -> (void)
 *
 * Sends the uptimes info to the server (info is in a string,
 * which pointer the function receives as an argument), and
 * receives the server response.
 *
 */
void send_info_tcp(char *progname, char *info, int useproxy, char *proxy,
        int useproxyauth, char *proxyauth, int proxyport) {
  FILE *ifile, *ofile;
  int sock, i;
  char s_buff[BUFFSIZE], r_buff[BUFFSIZE], buff[BUFFSIZE], c;
  struct sockaddr_in address;
  struct hostent *host;
  char hostname[BUFFSIZE];
  int port;
  char proxyauth_b64[BUFFSIZE];

  if(useproxy) {
    strncpy(hostname, proxy, sizeof(hostname));
    if(!proxyport)
      proxyport = DEFAULTPROXYPORT;
    port = proxyport;
    if(useproxyauth)
      base64_encode(proxyauth_b64, proxyauth, strlen(proxyauth));
  } else {
    strncpy(hostname, HOST, sizeof(hostname));
    port = PORT;
  }

  if(!(host = gethostbyname(hostname))) {
    fprintf(stderr, "%s: %s: Couldn't resolve hostname\n", hostname, progname);
    exit(EXIT_FAILURE);
  }
  
  sock = socket(PF_INET, SOCK_STREAM, 0);

  address.sin_family = PF_INET;
  address.sin_port = htons(port);
  bcopy(host->h_addr, (char *) &address.sin_addr, host->h_length);
  
  if((connect(sock, (struct sockaddr *) &address, sizeof(address))) < 0)
    crash(progname, hostname);

  ifile = fdopen(sock, "r");
  ofile = fdopen(sock, "w");

  if(useproxy)
    snprintf(s_buff, sizeof(s_buff), "POST http://%s:%d%s HTTP/1.1\r\n",
	HOST, PORT, LOCATION);
  else
    snprintf(s_buff, sizeof(s_buff), "POST %s HTTP/1.1\r\n", LOCATION);
  snprintf(buff, sizeof(buff), "Host: %s:%d\r\n", HOST, PORT);
  strncat(s_buff, buff, (sizeof(s_buff) - strlen(s_buff)));
  snprintf(buff, sizeof(buff), "Connection: close\r\n");
  strncat(s_buff, buff, (sizeof(s_buff) - strlen(s_buff)));
  snprintf(buff, sizeof(buff), "User-Agent: upclient/4.20/%s-%s\r\n",
      PROGNAME, VERSION);
  strncat(s_buff, buff, (sizeof(s_buff) - strlen(s_buff)));
  snprintf(buff, sizeof(buff),
      "Content-type: application/x-www-form-urlencoded\r\n");
  strncat(s_buff, buff, (sizeof(s_buff) - strlen(s_buff)));
  snprintf(buff, sizeof(buff), "Content-length: %d\r\n", (int) strlen(info));
  strncat(s_buff, buff, (sizeof(s_buff) - strlen(s_buff)));
  if(useproxy && useproxyauth) {
    snprintf(buff, sizeof(buff),
	"Proxy-Authorization: Basic %s\r\n", proxyauth_b64);
    strncat(s_buff, buff, (sizeof(s_buff) - strlen(s_buff)));
  }
  snprintf(buff, sizeof(buff), "\r\n%s\r\n", info);
  strncat(s_buff, buff, (sizeof(s_buff) - strlen(s_buff)));

  fprintf(ofile, s_buff);
  fflush(ofile);

  for(i = 0; ((c = fgetc(ifile)) != EOF) && (i < (sizeof(r_buff) - 1)); i++)
    r_buff[i] = c;
  r_buff[i] = '\0';

  recon_resp(progname, r_buff);
}

/* send_info_udp(char*, char*) -> (void)
 *
 * Sends the uptimes info to the server (info is in a string,
 * which pointer the function receives as an argument), through
 * the UDP protocol.
 *
 */
void send_info_udp(char *progname, char *info) {
  int sock;
  struct hostent *host;
  struct sockaddr_in address;

  if(!(host = gethostbyname(HOST))) {
    fprintf(stderr, "%s: %s: Couldn't resolve hostname\n", HOST, progname);
    exit(EXIT_FAILURE);
  }

  sock = socket(PF_INET, SOCK_DGRAM, 0);

  address.sin_family = PF_INET;
  address.sin_port = htons(UDP_PORT);
  bcopy(host->h_addr, (char *) &address.sin_addr, host->h_length);

  if((connect(sock, (struct sockaddr *) &address, sizeof(address))) < 0)
    crash(progname, HOST);

  send(sock, info, strlen(info), 0);
}

/* recon_resp(char*, char*) -> (void)
 *
 * Sees if the server response was "Success!" and prints an error
 * message if not.
 *
 */
void recon_resp(char *progname, char *response) {
  char *message;

  if(strstr(response, "UP4: 000 ") == NULL) {
    message = strstr(response, "UP4: ");
    if(message != NULL) {
      message += 5;
      fprintf(stderr, "%s: Unable to update the uptime information\n", progname);
      fprintf(stderr, "Server response: %s", message);
      exit(EXIT_FAILURE);
    } else {
      fprintf(stderr, "%s: Unable to update the uptime information\n", progname);
      fprintf(stderr, "Server response:\n\n%s\n", response);
      exit(EXIT_FAILURE);
    }
  }
}
