/*  This file is part of LingoTeach, the Language Teaching program 
 *  Copyright (C) 2001-2003 The LingoTeach Team
 *
 *  This program is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License as published by 
 *  the Free Software Foundation; either version 2 of the License, or 
 *  (at your option) any later version. 
 *
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *  GNU General Public License for more details. 
 * 
 *  You should have received a copy of the GNU General Public License 
 *  along with this program; if not, write to the Free Software 
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */

#include <string.h>
#include <glib/gprintf.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "lingoteach-i18n.h"
#include "errors.h"
#include "learningpref.h"
#include "util.h"

/* external global variables */
extern lingLesson *lessons;
extern struct lpreferences genLprefs;

int err = 0; /* error indication */

static GtkWidget *dlg_fileselect;

/* prototypes */
void on_ok_button1_clicked (lingfunc function);


/*********************
 * private functions *
 *********************/

void 
on_ok_button1_clicked (lingfunc function)
{
  const gchar *name;
  name = gtk_file_selection_get_filename (GTK_FILE_SELECTION (dlg_fileselect));
  function ((gchar *) name);

  return;
}


/********************
 * public functions *
 ********************/

GtkWidget*
util_lookup_widget (GtkWidget *widget, const gchar *widget_name)
{
  GtkWidget *parent;
  GtkWidget *found_widget = NULL;

  for (;;)
    {
      if (GTK_IS_MENU (widget))
        parent = gtk_menu_get_attach_widget (GTK_MENU (widget));
      else
        parent = widget->parent;
      if (!parent)
        parent = (GtkWidget*) g_object_get_data (G_OBJECT (widget), 
						 "GladeParentKey");
      if (parent == NULL)
        break;
      widget = parent;
    }
  
  found_widget = (GtkWidget*) g_object_get_data (G_OBJECT (widget),
                                                 widget_name);
  if (!found_widget)
    g_warning ("Widget not found: %s", widget_name);
  return found_widget;
}

/* 
 * globally used file selection 
 */
GtkWidget*
util_create_dlg_fileselect (const gchar *title, lingfunc function)
{
  GtkWidget *ok_button1;
  GtkWidget *cancel_button1;

  dlg_fileselect = gtk_file_selection_new (title);
  gtk_container_set_border_width (GTK_CONTAINER (dlg_fileselect), 10);
  gtk_window_set_modal (GTK_WINDOW (dlg_fileselect), TRUE);
  gtk_window_set_resizable (GTK_WINDOW (dlg_fileselect), FALSE);
  gtk_window_set_destroy_with_parent (GTK_WINDOW (dlg_fileselect), TRUE);
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dlg_fileselect));

  ok_button1 = GTK_FILE_SELECTION (dlg_fileselect)->ok_button;
  gtk_widget_show (ok_button1);
  GTK_WIDGET_SET_FLAGS (ok_button1, GTK_CAN_DEFAULT);

  cancel_button1 = GTK_FILE_SELECTION (dlg_fileselect)->cancel_button;
  gtk_widget_show (cancel_button1);
  GTK_WIDGET_SET_FLAGS (cancel_button1, GTK_CAN_DEFAULT);

  g_signal_connect_swapped ((gpointer) ok_button1, "clicked",
			    G_CALLBACK (on_ok_button1_clicked),
			    (gpointer) function);
  g_signal_connect_swapped ((gpointer) ok_button1, "clicked",
			    G_CALLBACK (gtk_widget_destroy),
			    GTK_OBJECT (dlg_fileselect));
  g_signal_connect_swapped ((gpointer) cancel_button1, "clicked",
                            G_CALLBACK (gtk_widget_destroy),
                            GTK_OBJECT (dlg_fileselect));

  return dlg_fileselect;
}

/* 
 * creates a menu 
 */
GtkWidget*
util_get_menu (lingchar **menu_list, int entries)
{
  GtkWidget* menu;
  GtkWidget* mitem;
  gint i;

  menu = gtk_menu_new ();

  for (i = 0; i < entries; i++)
    {
      mitem = gtk_menu_item_new_with_label (menu_list[i]);
      gtk_widget_show (mitem);
      gtk_menu_shell_append  ((GtkMenuShell *)(menu), mitem);
    }
  gtk_menu_set_active (GTK_MENU (menu), 0);
  return menu;
}

/* 
 * fetches the label text from a menu item of the passed option menu 
 */
gchar* 
util_get_menu_label (GtkOptionMenu *menu)
{
  gchar *text = NULL;
  
  if (GTK_BIN (menu)->child)
    {
      GtkWidget *child = GTK_BIN(menu)->child;
      if (GTK_IS_LABEL (child))
        text = (gchar *) gtk_label_get_text (GTK_LABEL (child));
    }
  return text;
}

/*
 * returns a lesson based on the method
 */
lingLesson*
util_return_lesson (lingLesson *lesson, Method type)
{
  lingLesson *tmp = NULL;

  switch (type)
    {
    case RANDOM:
      tmp = ling_lesson_return_rand_lesson (lesson);
      if (tmp == NULL)
	err_error_cb (_("No lesson found! Please check, if you activated at " 
		      "least one lesson in your configuration!"));
      return tmp;
      break;
    case SEQUENCE:
      return lesson;
      break;
    case REVIEW:
      return lesson;
      break;
    case LEARN:
      return lesson;
      break;
    default:
      return lesson;
      break;
    }
}

/*
 * checks the settings of the user
 */
gboolean
util_check_settings (void)
{
  if (err != 1)
    {
      if (genLprefs.mlanguage == NULL)
	{
	  err_error_cb 
	    (_("Please select your main translation language first!"));
	  err = 1;
	  return FALSE;
	}
      if (genLprefs.languages == NULL)
	{
	  err_error_cb 
	    (_("Please select your wanted translation languages first!"));
	  err = 1;
	  return FALSE;
	}
      if (lessons == NULL)
	{
	  err_error_cb (_("Please load your wanted lesson files first!"));
	  err = 1;
	  return FALSE;
	}
    }
  if (err == 1)
    return FALSE;
  return TRUE;
}

/*
 * sound playing
 */
void
util_on_btn_play_clicked (lingMeaning *meaning)
{
  gchar  *cmd;
  GError *err = NULL;
  char   *file;

  if (genLprefs.soundpath != NULL && genLprefs.player != NULL)
    {
      file = ling_meaning_find_sound (genLprefs.soundpath, meaning);
      cmd = g_malloc (strlen (genLprefs.player) + strlen (file) + 1);
      if (cmd == NULL)
        {
	  free (file);
	  err_error_cb (_("Could not allocate enough memory "
	                "for playing the sound!"));
	  return;
	}
      g_sprintf (cmd, "%s %s", genLprefs.player, file);
      if (!g_spawn_command_line_async (cmd, &err))
	err_error_cb (err->message);
      g_free (cmd);
      free (file);
    }
  return;
}

