/* $Id: preferences.c,v 1.19 2005/03/31 15:54:51 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>

#include "../li18n.h"
#include "../errors.h"
#include "hig-widgets.h"
#include "gtkdefs.h"

#include "util.h"
#include "util-lang.h"
#include "gtkconfig.h"
#include "gtkplugin.h"
#include "preferences.h"

extern lingGtkMainWindow *main_app;
extern lingGtkPrefs *settings;

static lingGtkPrefs *current_set;
static const gchar *methods[] = { N_("Random"), N_("Sequence"), N_("Review"),
                                  N_("Learn") };

#define BROWSE N_("_Browse...")

enum {
     PLUGIN_NAME,
     PLUGIN_DESCR,
     PLUGIN_PATH,
     PLUGIN_COLUMNS
};

static gboolean backup_settings (void);
static void merge_new_config (void);
static GtkWidget *create_page_general (void);
static GtkWidget *create_page_learning (void);
static GtkWidget *create_page_training (void);
static GtkWidget *create_page_editor (void);
static GtkWidget *create_page_plugins (void);
static GtkWidget *create_notebook_list (GtkWidget *notebook);
static GtkListStore *create_plugin_list (void);
static void add_plugin (GtkTreeView *tree);
static void remove_plugin (GtkTreeView *tree);
static void activate_page (GtkTreeView *list, GtkNotebook *notebook);

static gboolean
backup_settings (void)
{
     learnLanguage *tmp = NULL;

     current_set = gtkconfig_init_settings (conf_init_new ());
     if (!current_set)
          return FALSE;

     current_set->prefs->config = settings->prefs->config;
     current_set->prefs->languages = settings->prefs->languages;

     tmp = settings->learn_lang;
     while (tmp)
     {
          current_set->learn_lang =
               learn_language_append_language (current_set->learn_lang,
                                               tmp->language, tmp->used,
                                               tmp->main);
          tmp = tmp->next;
     }
     tmp = settings->train_lang;
     while (tmp)
     {
          current_set->train_lang =
               learn_language_append_language (current_set->train_lang,
                                               tmp->language, tmp->used,
                                               tmp->main);
          tmp = tmp->next;
     }          
     current_set->learn_words = settings->learn_words;
     current_set->learn_hints = settings->learn_hints;
     current_set->learn_method = settings->learn_method;
     current_set->train_words = settings->train_words;
     current_set->train_hints = settings->train_hints;
     current_set->train_method = settings->train_method;
     current_set->edit_lang = settings->edit_lang;
     
     return TRUE;
}

static void
merge_new_config (void)
{
     debug ("Applying new configuration...\n");

     /* first check the gui independant stuff */
     conf_set_preference_val (settings->prefs, CONF_VAL_SOUND,
                              current_set->prefs->soundpath);
     conf_set_preference_val (settings->prefs, CONF_VAL_BROWSER,
                              current_set->prefs->browser);
     conf_set_preference_val (settings->prefs, CONF_VAL_PLAYER,
                              current_set->prefs->player);

     /* now the gui dependant */
     settings->learn_words = current_set->learn_words;
     settings->learn_hints = current_set->learn_hints;
     settings->learn_method = current_set->learn_method;
     settings->train_words = current_set->train_words;
     settings->train_hints = current_set->train_hints;
     settings->train_method = current_set->train_method;
     learn_language_free (settings->learn_lang);
     settings->learn_lang = current_set->learn_lang;
     settings->train_lang = current_set->train_lang;
     settings->edit_lang = current_set->edit_lang;

     /* raise events */
     event_emit_signal (main_app->objects, "edit-lang-changed");

     /* clean current_set up now */
     g_free (current_set->prefs->soundpath);
     g_free (current_set->prefs->browser);
     g_free (current_set->prefs->player);
     free (current_set->prefs);
     g_free (current_set);

     gtkconfig_save_settings (settings);

     return;
}

static GtkWidget*
create_page_general (void)
{
     GtkWidget *frm_general;
     GtkWidget *align;
     GtkWidget *tbl_main;
     GtkWidget *lbl_sound;
     GtkWidget *txt_sound;
     GtkWidget *btn_sound;
     GtkWidget *lbl_browser;
     GtkWidget *txt_browser;
     GtkWidget *btn_browser;
     GtkWidget *lbl_player;
     GtkWidget *txt_player;
     GtkWidget *btn_player;

     frm_general = hig_frame_new (_("General"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_general), 0);
     align = gtk_bin_get_child (GTK_BIN (frm_general));

     tbl_main = hig_table_new (3, 3);
     gtk_container_set_border_width (GTK_CONTAINER (tbl_main), 0);
     gtk_container_add (GTK_CONTAINER (align), tbl_main);

     /* sound path entry */
     lbl_sound = gtk_label_new_with_mnemonic (_("_Soundpath:"));
     gtk_table_attach (GTK_TABLE (tbl_main), lbl_sound, 0, 1, 0, 1,
                       GTK_FILL, GTK_FILL, 0, 0);
     gtk_misc_set_alignment (GTK_MISC (lbl_sound), 0.0, 0.5);
     
     txt_sound = gtk_entry_new ();
     gtk_table_attach (GTK_TABLE (tbl_main), txt_sound, 1, 2, 0, 1,
                       GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
     gtk_tooltips_set_tip (main_app->tips, txt_sound,
                           _("The path the sound snippets can be found in"),
                           NULL);
     gtk_label_set_mnemonic_widget (GTK_LABEL (lbl_sound), txt_sound);
     g_signal_connect (G_OBJECT (txt_sound), "destroy",
                       G_CALLBACK (util_copy_string),
                       &current_set->prefs->soundpath);

     btn_sound = gtk_button_new_with_mnemonic (BROWSE);
     g_signal_connect_swapped (G_OBJECT (btn_sound), "clicked",
                               G_CALLBACK (util_get_folderselection),
                               txt_sound);
     gtk_table_attach (GTK_TABLE (tbl_main), btn_sound, 2, 3, 0, 1,
                       GTK_FILL, GTK_FILL, 0, 0);

     /* player entry */
     lbl_player = gtk_label_new_with_mnemonic (_("Sound _player:"));
     gtk_table_attach (GTK_TABLE (tbl_main), lbl_player, 0, 1, 1, 2,
                       GTK_FILL, GTK_FILL, 0, 0);
     gtk_misc_set_alignment (GTK_MISC (lbl_player), 0.0, 0.5);

     txt_player = gtk_entry_new ();
     gtk_table_attach (GTK_TABLE (tbl_main), txt_player, 1, 2, 1, 2,
                       GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
     gtk_tooltips_set_tip (main_app->tips, txt_player,
                           _("The sound player to use (with opt. parameters)"),
                           NULL);
     gtk_label_set_mnemonic_widget (GTK_LABEL (lbl_player), txt_player);
     g_signal_connect (G_OBJECT (txt_player), "destroy",
                       G_CALLBACK (util_copy_string),
                       &current_set->prefs->player);

     btn_player = gtk_button_new_with_mnemonic (BROWSE);
     g_signal_connect_swapped (G_OBJECT (btn_player), "clicked",
                               G_CALLBACK (util_get_fileselection),
                               txt_player);
     gtk_table_attach (GTK_TABLE (tbl_main), btn_player, 2, 3, 1, 2,
                       GTK_FILL, GTK_FILL, 0, 0);

     /* the web browser */
     lbl_browser = gtk_label_new_with_mnemonic (_("_Web browser:"));
     gtk_table_attach (GTK_TABLE (tbl_main), lbl_browser, 0, 1, 2, 3,
                       GTK_FILL, GTK_FILL, 0, 0);
     gtk_misc_set_alignment (GTK_MISC (lbl_browser), 0.0, 0.5);

     txt_browser = gtk_entry_new ();
     gtk_table_attach (GTK_TABLE (tbl_main), txt_browser, 1, 2, 2, 3,
                       GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
     gtk_tooltips_set_tip (main_app->tips, txt_browser,
                           _("The web browser to use for displaying websites"),
                           NULL);
     gtk_label_set_mnemonic_widget (GTK_LABEL (lbl_browser), txt_browser);
     g_signal_connect (G_OBJECT (txt_browser), "destroy",
                       G_CALLBACK (util_copy_string),
                       &current_set->prefs->browser);

     btn_browser = gtk_button_new_with_mnemonic (BROWSE);
     g_signal_connect_swapped (G_OBJECT (btn_browser), "clicked",
                               G_CALLBACK (util_get_fileselection),
                               txt_browser);
     gtk_table_attach (GTK_TABLE (tbl_main), btn_browser, 2, 3, 2, 3,
                       GTK_FILL, GTK_FILL, 0, 0);

     /* fill all with life now */
     if (settings->prefs->soundpath)
          gtk_entry_set_text (GTK_ENTRY (txt_sound),
                              settings->prefs->soundpath);
     if (settings->prefs->browser)
          gtk_entry_set_text (GTK_ENTRY (txt_browser),
                              settings->prefs->browser);
     if (settings->prefs->player)
          gtk_entry_set_text (GTK_ENTRY (txt_player),
                              settings->prefs->player);
     
     return frm_general;
}

static GtkWidget*
create_page_learning (void)
{
     GtkWidget *box_main;

     GtkWidget *frm_lesson;
     GtkWidget *align_lesson;
     GtkWidget *box_lesson;
     GtkWidget *lbl_words;
     GtkWidget *spin_words;
     GtkObject *spin_words_adj;

     GtkWidget *frm_learn;
     GtkWidget *align_learn;
     GtkWidget *box_learn;
     GtkWidget *btn_hints;
     GtkWidget *box_access;
     GtkWidget *lbl_access;
     GtkWidget *combo_access;
     guint i = 0;

     GtkWidget *frm_languages;
     GtkWidget *align_languages;
     GtkWidget *scr_win;

     const gchar *titles[3] = { N_("Use"), N_("Learn from?"), N_("Language") };

     box_main = hig_vbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_main), 3 * BOX_SPACING);
     gtk_container_set_border_width (GTK_CONTAINER (box_main), 0);

     frm_lesson = hig_frame_new (_("Lesson defaults"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_lesson), 0);
     gtk_box_pack_start (GTK_BOX (box_main), frm_lesson, FALSE, TRUE, 0);
     align_lesson = gtk_bin_get_child (GTK_BIN (frm_lesson));

     box_lesson = hig_hbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_lesson), 0);
     gtk_container_add (GTK_CONTAINER (align_lesson), box_lesson);

     lbl_words = gtk_label_new_with_mnemonic (_("Words to _learn:"));
     gtk_box_pack_start (GTK_BOX (box_lesson), lbl_words, FALSE, FALSE, 0);

     spin_words_adj = gtk_adjustment_new (1.0, 0.0, 100.0, 1.0, 10.0, 10.0);
     spin_words = gtk_spin_button_new (GTK_ADJUSTMENT (spin_words_adj), 1, 0);
     gtk_label_set_mnemonic_widget (GTK_LABEL (lbl_words), spin_words);
     gtk_box_pack_start (GTK_BOX (box_lesson), spin_words, FALSE, FALSE, 0);
     gtk_tooltips_set_tip (main_app->tips, spin_words,
                           _("Indicates, how many words should be learned"),
                           NULL);
     g_signal_connect (G_OBJECT (spin_words), "destroy",
                       G_CALLBACK (util_copy_int),
                       &current_set->learn_words);

     frm_learn = hig_frame_new (_("Learn"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_learn), 0);
     gtk_box_pack_start (GTK_BOX (box_main), frm_learn, FALSE, TRUE, 0);
     align_learn = gtk_bin_get_child (GTK_BIN (frm_learn));

     box_learn = hig_vbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_learn), 0);
     gtk_container_add (GTK_CONTAINER (align_learn), box_learn);

     btn_hints = gtk_check_button_new_with_mnemonic (_("_Enable hints"));
     gtk_box_pack_start (GTK_BOX (box_learn), btn_hints, FALSE, FALSE, 0);
     gtk_tooltips_set_tip (main_app->tips, btn_hints,
                           _("Enables the description and picture of the "
                             "meanings"), NULL);
     g_signal_connect (G_OBJECT (btn_hints), "destroy",
                       G_CALLBACK (util_copy_boolean),
                       &current_set->learn_hints);

     /* access config */
     box_access = hig_hbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_access), 0);
     gtk_box_pack_start (GTK_BOX (box_learn), box_access, FALSE, FALSE, 0);
     
     lbl_access = gtk_label_new_with_mnemonic (_("A_ccess method:"));
     gtk_box_pack_start (GTK_BOX (box_access), lbl_access, FALSE, FALSE, 0);

     combo_access = gtk_combo_box_new_text ();
     while (i < G_N_ELEMENTS (methods))
     {
          gtk_combo_box_append_text (GTK_COMBO_BOX (combo_access), methods[i]);
          i++;
     }
     gtk_label_set_mnemonic_widget (GTK_LABEL (lbl_access), combo_access);
     gtk_box_pack_start (GTK_BOX (box_access), combo_access, FALSE, FALSE, 0);
     gtk_tooltips_set_tip (main_app->tips, combo_access,
                           _("The default access method for choosing "
                             "meanings from the lesson"), NULL);
     g_signal_connect (G_OBJECT (combo_access), "changed",
                       G_CALLBACK (util_copy_int),
                       &current_set->learn_method);

     /* languages */
     frm_languages = hig_frame_new (_("Default languages to use"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_languages), 0);
     gtk_box_pack_start (GTK_BOX (box_main), frm_languages, TRUE, TRUE, 0);
     align_languages = gtk_bin_get_child (GTK_BIN (frm_languages));

     /* create the list view */
     scr_win = util_lang_create_language_tree_view (current_set,
                                                    current_set->learn_lang,
                                                    titles);
     gtk_container_set_border_width (GTK_CONTAINER (scr_win), 0);
     gtk_container_add (GTK_CONTAINER (align_languages), scr_win);

     /* fill all the with life now */
     gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_words),
                                (double) settings->learn_words);
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn_hints),
                                   settings->learn_hints);
     gtk_combo_box_set_active (GTK_COMBO_BOX (combo_access),
                               (gint) settings->learn_method);

     return box_main;
}

static GtkWidget*
create_page_training (void)
{
     GtkWidget *box_main;

     GtkWidget *frm_lesson;
     GtkWidget *align_lesson;
     GtkWidget *box_lesson;
     GtkWidget *lbl_words;
     GtkWidget *spin_words;
     GtkObject *spin_words_adj;

     GtkWidget *frm_train;
     GtkWidget *align_train;
     GtkWidget *box_train;
     GtkWidget *btn_hints;
     GtkWidget *box_access;
     GtkWidget *lbl_access;
     GtkWidget *combo_access;
     guint i = 0;
     
     GtkWidget *frm_languages;
     GtkWidget *align_languages;
     GtkWidget *scr_win;

     const gchar *titles[3] = { N_("Use"), N_("Train with?"), N_("Language") };

     box_main = hig_vbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_main), 3 * BOX_SPACING);
     gtk_container_set_border_width (GTK_CONTAINER (box_main), 0);

     frm_lesson = hig_frame_new (_("Lesson defaults"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_lesson), 0);
     gtk_box_pack_start (GTK_BOX (box_main), frm_lesson, FALSE, TRUE, 0);
     align_lesson = gtk_bin_get_child (GTK_BIN (frm_lesson));

     box_lesson = hig_hbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_lesson), 0);
     gtk_container_add (GTK_CONTAINER (align_lesson), box_lesson);
     
     lbl_words = gtk_label_new_with_mnemonic (_("Words to _test:"));
     gtk_box_pack_start (GTK_BOX (box_lesson), lbl_words, FALSE, FALSE, 0);

     spin_words_adj = gtk_adjustment_new (1.0, 0.0, 100.0, 1.0, 10.0, 10.0);
     spin_words = gtk_spin_button_new (GTK_ADJUSTMENT (spin_words_adj), 1, 0);
     gtk_label_set_mnemonic_widget (GTK_LABEL (lbl_words), spin_words);
     gtk_box_pack_start (GTK_BOX (box_lesson), spin_words, FALSE, FALSE, 0);
     gtk_tooltips_set_tip (main_app->tips, spin_words,
                           _("Indicates, how many words should be tested"),
                           NULL);
     g_signal_connect (G_OBJECT (spin_words), "destroy",
                       G_CALLBACK (util_copy_int), &current_set->train_words);

     frm_train = hig_frame_new (_("Training"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_train), 0);
     gtk_box_pack_start (GTK_BOX (box_main), frm_train, FALSE, TRUE, 0);
     align_train = gtk_bin_get_child (GTK_BIN (frm_train));

     box_train = hig_vbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_train), 0);
     gtk_container_add (GTK_CONTAINER (align_train), box_train);

     btn_hints = gtk_check_button_new_with_mnemonic (_("_Enable hints"));
     gtk_box_pack_start (GTK_BOX (box_train), btn_hints, FALSE, FALSE, 0);
     gtk_tooltips_set_tip (main_app->tips, btn_hints,
                           _("Enables the description and picture of the "
                             "meanings"), NULL);
     g_signal_connect (G_OBJECT (btn_hints), "destroy",
                       G_CALLBACK (util_copy_boolean),
                       &current_set->train_hints);

     /* access config */
     box_access = hig_hbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_access), 0);
     gtk_box_pack_start (GTK_BOX (box_train), box_access, FALSE, FALSE, 0);
     
     lbl_access = gtk_label_new_with_mnemonic (_("A_ccess method:"));
     gtk_box_pack_start (GTK_BOX (box_access), lbl_access, FALSE, FALSE, 0);

     combo_access = gtk_combo_box_new_text ();
     while (i < G_N_ELEMENTS (methods))
     {
          gtk_combo_box_append_text (GTK_COMBO_BOX (combo_access), methods[i]);
          i++;
     }
     gtk_label_set_mnemonic_widget (GTK_LABEL (lbl_access), combo_access);
     gtk_box_pack_start (GTK_BOX (box_access), combo_access, FALSE, FALSE, 0);
     gtk_tooltips_set_tip (main_app->tips, combo_access,
                           _("The default access method for choosing "
                             "meanings from the lesson"), NULL);
     g_signal_connect (G_OBJECT (combo_access), "changed",
                       G_CALLBACK (util_copy_int), &current_set->train_method);

     /* languages */
     frm_languages = hig_frame_new (_("Default languages to use"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_languages), 0);
     gtk_box_pack_start (GTK_BOX (box_main), frm_languages, TRUE, TRUE, 0);
     align_languages = gtk_bin_get_child (GTK_BIN (frm_languages));

     /* create the list view */
     scr_win = util_lang_create_language_tree_view (current_set,
                                                    current_set->train_lang,
                                                    titles);
     gtk_container_set_border_width (GTK_CONTAINER (scr_win), 0);
     gtk_container_add (GTK_CONTAINER (align_languages), scr_win);

     /* fill all the with life now */
     gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_words),
                                (double) current_set->train_words);
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn_hints),
                                   current_set->train_hints);
     gtk_combo_box_set_active (GTK_COMBO_BOX (combo_access),
                               (gint) settings->train_method);

     return box_main;
}

static GtkWidget*
create_page_editor (void)
{
     GtkWidget *frm_editor;
     GtkWidget *align;
     GtkWidget *box_main;

     GtkWidget *scr_win;

     frm_editor = hig_frame_new (_("Editor"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_editor), 0);
     align = gtk_bin_get_child (GTK_BIN (frm_editor));

     box_main = hig_vbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_main), 0);
     gtk_container_add (GTK_CONTAINER (align), box_main);

     /* translation language */
     scr_win = util_lang_create_language_single_selection (current_set);
     gtk_container_set_border_width (GTK_CONTAINER (scr_win), 0);
     gtk_box_pack_start (GTK_BOX (box_main), scr_win, TRUE, TRUE, 0);
     
     return frm_editor;
}

static GtkWidget*
create_page_plugins (void)
{
     GtkWidget *frm_plugins;
     GtkWidget *align;
     GtkWidget *box_main;
     GtkWidget *box_buttons;
     GtkWidget *btn_add;
     GtkWidget *btn_remove;
     GtkWidget *scr_win;

     GtkWidget *view_plugins;
     GtkCellRenderer *renderer;
     GtkListStore *list;
     GtkTreeSelection *select;

     frm_plugins = hig_frame_new (_("Plugins"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_plugins), 0);
     align = gtk_bin_get_child (GTK_BIN (frm_plugins));

     box_main = hig_vbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_main), 0);
     gtk_container_add (GTK_CONTAINER (align), box_main);

     list = create_plugin_list ();
     view_plugins = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list));
     g_object_unref (G_OBJECT (list));

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_plugins),
                                                  PLUGIN_NAME, ("Name"),
                                                  renderer, "text",
                                                  PLUGIN_NAME, NULL);
     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_plugins),
                                                  PLUGIN_DESCR,
                                                  _("Description"),
                                                  renderer, "text",
                                                  PLUGIN_DESCR, NULL);

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_plugins),
                                                  PLUGIN_PATH,
                                                  _("Path"),
                                                  renderer, "text",
                                                  PLUGIN_PATH, NULL);

     select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view_plugins));
     gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);

     scr_win = hig_scrolled_window_new ();
     gtk_container_set_border_width (GTK_CONTAINER (scr_win), 0);
     gtk_container_add (GTK_CONTAINER (scr_win), view_plugins);
     gtk_box_pack_start (GTK_BOX (box_main), scr_win, TRUE, TRUE, 0);

     box_buttons = gtk_hbutton_box_new ();
     gtk_button_box_set_layout (GTK_BUTTON_BOX (box_buttons),
                                GTK_BUTTONBOX_SPREAD);
     gtk_box_pack_start (GTK_BOX (box_main), box_buttons, FALSE, FALSE, 0);

     btn_add = gtk_button_new_from_stock (GTK_STOCK_ADD);
     g_signal_connect_swapped (G_OBJECT (btn_add), "clicked",
                               G_CALLBACK (add_plugin), view_plugins);
     gtk_container_add (GTK_CONTAINER (box_buttons), btn_add);

     btn_remove = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
     g_signal_connect_swapped (G_OBJECT (btn_remove), "clicked",
                               G_CALLBACK (remove_plugin), view_plugins);
     gtk_container_add (GTK_CONTAINER (box_buttons), btn_remove);

     return frm_plugins;
}

static GtkWidget* 
create_notebook_list (GtkWidget *notebook)
{
     GtkWidget *view_list;
     GtkListStore *list;
     GtkCellRenderer *renderer;
     int i = 0;

     /* preference menu entries */
     gchar *entry[] = { 
          N_("General"), N_("Learning"), N_("Training"), 
          N_("Editor"), N_("Plugins"), NULL 
     };

     /* preference page entries */
     GtkWidget *page[] = {
          create_page_general (), create_page_learning (),
          create_page_training (), create_page_editor (),
          create_page_plugins (), NULL
     };

     list = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_INT);

     /* create the entries */
     while (entry[i])
     {
          preferences_add_page_entry (list, notebook, entry[i], page[i]);
          i++;
     }
     
     /* create the list view */
     view_list = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list));
     g_signal_connect (G_OBJECT (view_list), "cursor-changed",
                       G_CALLBACK (activate_page), notebook);
    
     g_object_unref (G_OBJECT (list));
     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_list),
                                                  -1, _("Preferences"),
                                                  renderer, "text", 0, NULL);
     return view_list;
}

static GtkListStore*
create_plugin_list (void)
{
     GtkListStore *list;
     GtkTreeIter iter;
     GSList *tmp = settings->plugins;

     list = gtk_list_store_new (PLUGIN_COLUMNS, G_TYPE_STRING, G_TYPE_STRING,
                                G_TYPE_STRING);
     while (tmp)
     {
          gtk_list_store_append (list, &iter);
          gtk_list_store_set (list, &iter,
                              PLUGIN_NAME,
                              ((lingGtkPlugin *) tmp->data)->name (),
                              PLUGIN_DESCR,
                              ((lingGtkPlugin *) tmp->data)->desc (),
                              PLUGIN_PATH, ((lingGtkPlugin *) tmp->data)->path,
                              -1);
          tmp = tmp->next;
     }
     return list;
}


static void
add_plugin (GtkTreeView *tree)
{
     lingGtkPlugin *plugin = NULL;
     GSList *tmp = settings->plugins;
     GtkListStore *list;
     
     gchar *file = util_run_fileselection (_("Choose Plugin..."),
                                           GTK_FILE_CHOOSER_ACTION_OPEN);
     if (!file)
          return;

     debug ("Loading plugin...\n");

     /* check, if it is not already loaded */
     while (tmp)
     {
          if (g_utf8_collate (file, ((lingGtkPlugin *) tmp->data)->path) == 0)
          {
               util_info (_("The plugin is already loaded!"));
               g_free (file);
               return;
          }
          tmp = tmp->next;
     }

     plugin = gtkplugin_plugin_load (file);
     g_free (file);
     if (!plugin)
          return;

     settings->plugins = g_slist_append (settings->plugins, plugin);

     list = create_plugin_list ();
     gtk_tree_view_set_model (tree, GTK_TREE_MODEL (list));
     g_object_unref (list);

     /* now call init */
     plugin->init ();

     return;
}

static void
remove_plugin (GtkTreeView *tree)
{
     GtkTreeSelection *select = gtk_tree_view_get_selection (tree);
     GtkTreeModel *model = gtk_tree_view_get_model (tree);
     GtkTreeIter iter;
     GtkListStore *list;
     gchar *path = NULL;
     GSList *tmp = settings->plugins;
     lingGtkPlugin *plugin = NULL;

     if (gtk_tree_selection_get_selected (select, &model, &iter))
     {
          gtk_tree_model_get (model, &iter, PLUGIN_PATH, &path, -1);
          while (tmp)
          {
               plugin = (lingGtkPlugin *) tmp->data;

               if (g_utf8_collate (path, plugin->path) == 0)
               {
                    /* remove plugin */
                    debug ("Removing plugin...\n");

                    settings->plugins = g_slist_remove_link (settings->plugins,
                                                             tmp);
                    gtkplugin_plugin_free (plugin);
                    g_slist_free_1 (tmp);
                    break;
               }
               tmp = tmp->next;
          }
          g_free (path);
     }
     
     list = create_plugin_list ();
     gtk_tree_view_set_model (tree, GTK_TREE_MODEL (list));
     g_object_unref (list);

     return;
}

static void
activate_page (GtkTreeView *list, GtkNotebook *notebook)
{
     gint page = 0;
     GtkTreeIter iter;
     GtkTreeModel *model = gtk_tree_view_get_model (list);
     GtkTreeSelection *sel = gtk_tree_view_get_selection (list);
     
     if (gtk_tree_selection_get_selected (sel, &model, &iter))
     {
          gtk_tree_model_get (model, &iter, 1, &page, -1);
          debug ("Switching to page %d\n", page);
          gtk_notebook_set_current_page (notebook, page);
     }
     return;
}

void
preferences_activate (void)
{
     GtkWidget *dlg_prefs;
     GtkWidget *box_prefs;
     GtkWidget *notebook;
     GtkWidget *view_list;

     gint response = 0;

     /* create temporary settings */
     backup_settings ();

     /* the dialog */
     dlg_prefs = gtk_dialog_new_with_buttons (_("Preferences"),
                                              GTK_WINDOW (main_app->window),
                                              GTK_DIALOG_MODAL |
                                              GTK_DIALOG_DESTROY_WITH_PARENT,
                                              GTK_STOCK_CANCEL,
                                              GTK_RESPONSE_CANCEL,
                                              GTK_STOCK_OK,
                                              GTK_RESPONSE_ACCEPT,
                                              NULL);
     gtk_window_set_default_size (GTK_WINDOW (dlg_prefs), 480, 400);
     gtk_dialog_set_has_separator (GTK_DIALOG (dlg_prefs), FALSE);
     gtk_container_set_border_width (GTK_CONTAINER (dlg_prefs), WIDGET_BORDER);

     /* preferences box */
     box_prefs = hig_hbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_prefs), 2 * BOX_SPACING);
     gtk_container_set_border_width (GTK_CONTAINER (box_prefs), 6);
     gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg_prefs)->vbox), box_prefs,
                         TRUE, TRUE, 0);

     /* the main notebook - it has to be on the right side, so use pack_end */
     notebook = gtk_notebook_new ();
     gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), FALSE);
     gtk_notebook_set_show_border (GTK_NOTEBOOK (notebook), FALSE);
     gtk_box_pack_end (GTK_BOX (box_prefs), notebook, TRUE, TRUE, 0); 

     view_list = create_notebook_list (notebook);
     gtk_box_pack_start (GTK_BOX (box_prefs), view_list, FALSE, FALSE, 0);

     /* run the dialog */
     gtk_widget_show_all (box_prefs);
     response = gtk_dialog_run (GTK_DIALOG (dlg_prefs));
     switch (response)
     {
     case GTK_RESPONSE_ACCEPT:
          gtk_widget_destroy (dlg_prefs);
          merge_new_config ();
          break;
     case GTK_RESPONSE_CANCEL:
     default:
          gtk_widget_destroy (dlg_prefs);
          break;
     }
     return;
}

void
preferences_add_page_entry (GtkListStore *list, GtkWidget *notebook,
                            gchar *name, GtkWidget *page)
{
     GtkTreeIter iter;
     int i = gtk_notebook_get_n_pages (GTK_NOTEBOOK (notebook));

     debug ("Adding page entry %s at page %d\n", name, i);

     gtk_list_store_append (list, &iter);
     gtk_list_store_set (list, &iter, 0, name, 1, i, -1);

     gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, NULL);
     
     return;
}
