#############################################################################
#
#  Linux Desktop Testing Project http://ldtp.freedesktop.org
# 
#  Author:
#     M Nagashree <mnagashree@novell.com>
#     Veerapuram Varadhan <v.varadhan@gmail.com>
#     Nagappan Alagappan <nagappan@gmail.com>
# 
#  Copyright 2004 - 2006 Novell, Inc.
#  Copyright 2007 - 2009 Nagappan Alagappan
# 
#  This library is free software; you can redistribute it and/or
#  modify it under the terms of the GNU Library General Public
#  License as published by the Free Software Foundation; either
#  version 2 of the License, or (at your option) any later version.
# 
#  This library is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  Library General Public License for more details.
# 
#  You should have received a copy of the GNU Library General Public
#  License along with this library; if not, write to the
#  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
#  Boston, MA 02110, USA.
#
#############################################################################

import gc
import os
import re
import sys
import time
import wnck
import string
import commands
import tempfile
import threading
import xml.dom.minidom

import ldtp

_ldtpDebug = os.getenv ('LDTP_DEBUG')

gtkImport = True
try:
    import gtk.gdk
except ImportError:
    if _ldtpDebug:
        print 'pygtk package not installed'
    gtkImport = False

statGrabMsg = None
try:
    import statgrab
except ImportError:
    statGrabMsg = 'pystatgrab package not installed'

def tuplelist2list (lst):
        d = []
        for x in range (1, len (lst) + 1):
                for y in range (1, len (lst[x-1]) + 1):
                        d.append (lst[x-1][y-1])
        return d

def getFullPath (path):
    if path [0] == "~":
        path = os.path.expanduser (path)
    elif path [0] == ".":
        path = os.path.abspath (path)

    return path

def imagecompare (imgfile1, imgfile2):
    try:
        import ImageChops, Image
    except ImportError:
        ldtp.log ('Python-Imaging package not installed', 'error')
        raise ldtp.LdtpExecutionError ('Python-Imaging package not installed')
    try:
        diffcount = 0.0
        im1 = Image.open (imgfile1)
        im2 = Image.open (imgfile2)

        imgcompdiff = ImageChops.difference (im1, im2)
        diffboundrect = imgcompdiff.getbbox ()
        imgdiffcrop = imgcompdiff.crop (diffboundrect)

        seq = list (imgdiffcrop.getdata ())
        seq = tuplelist2list (seq)
        #print seq
        for i in range (0, imgdiffcrop.size[0] * imgdiffcrop.size[1] * 3, 3):
            if seq[i] != 0 or seq[i+1] != 0 or seq[i+2] != 0:
                diffcount = diffcount + 1.0
        
        diffImgLen = imgcompdiff.size[0] * imgcompdiff.size[1] * 1.0
        diffpercent = (diffcount * 100) / diffImgLen
        ldtp.log ('length ' + str (diffImgLen))
        ldtp.log ('diffcount ' + str (diffcount))
        ldtp.log ('diff percent ' + str (diffpercent))
        return diffpercent
    except IOError:
        ldtp.log ('Input file does not exist', 'error')
        raise ldtp.LdtpExecutionError ('Input file does not exist')

def imagecapture (winName = None, outFile = None, resolution1 = None,
                  resolution2 = None, x = 0, y = 0):
    # winname == None, let us capture the entire screen
    # if output file name is not passed, then a random file name will be generated in
    # /tmp and it will be returned
    winid = None
    if winName:
        screen = wnck.screen_get_default()
        # Process pending gtk+ events so that wnck can find out about 
        # existing windows.
        while gtk.events_pending():
            gtk.main_iteration()
        for window in screen.get_windows():
            if winName == window.get_name():
                winid = window.get_xid()
                break
    if outFile == None:
        fp = tempfile.NamedTemporaryFile ()
        tmpFile = fp.name + '.png'
        fp.close ()
    else:
        tmpFile = outFile
    pb = None
    if gtkImport is True and winid is None:
        # Based on http://ubuntuforums.org/showthread.php?t=448160
        window = gtk.gdk.get_default_root_window ()
        size = window.get_size ()
        pb = gtk.gdk.Pixbuf (gtk.gdk.COLORSPACE_RGB, False, 8, 
                             resolution1 or size [0], 
                             resolution2 or size [1])
        pb = pb.get_from_drawable (window, window.get_colormap (),
                                   x, y, 0, 0, 
                                   resolution1 or size [0], 
                                   resolution2 or size [1])
    if pb:
        pb.save (tmpFile, "png")
        # based on http://faq.pygtk.org/index.py?req=show&file=faq08.004.htp
        del pb
        gc.collect ()
    else:
        if  winid == None:
            winid = 'root'
        cmd = "import -window %s" % winid
        if resolution1 and resolution2:
            cmd = "%s -crop %dx%d+%d+%d" % (cmd, resolution1,
                                            resolution2, x, y)
        cmd = "%s %s" % (cmd, tmpFile)
        status = commands.getstatusoutput (cmd)
        if status [0] != 0:
            ldtp.log ('Unable to capture screenshot: %s' % status [1], 'error')
            return None
    return tmpFile

def blackoutregion (infile, outfile, topx, topy, botx, boty):
    try:
        import ImageChops, Image
    except ImportError:
        ldtp.log ('Python-Imaging package not installed', 'error')
        raise ldtp.LdtpExecutionError ('Python-Imaging package not installed')
    im = Image.open (infile)
    box = (topx, topy, botx, boty)
    region = im.crop (box)
    region = region.point (lambda i: i * 0)
    im.paste (region, box)
    im.save (outfile)

# XML Data file parser
class LdtpDataFileParser:
    def __init__ (self, filename = None):
        self.ldtpdataxml = []
        if filename:
            try:
                dom = xml.dom.minidom.parse (filename)
                self.ldtpdataxml = dom.getElementsByTagName ("data")
                if self.ldtpdataxml == []:
                    ldtp.log ('data xml tag not present')
                    if _ldtpDebug and _ldtpDebug == '2':
                        print 'data xml tag not present'
            except xml.parsers.expat.ExpatError, msg:
                ldtp.log ('XML Error: ' + str (msg), 'error')
            except IOError:
                ldtp.log ('XML \"' + filename + '\" file not found', 'error')
    def setfilename (self, filename):
        self.ldtpdataxml = []
        if filename:
            try:
                dom = xml.dom.minidom.parse (filename)
                self.ldtpdataxml = dom.getElementsByTagName ("data")
                if self.ldtpdataxml == []:
                    ldtp.log ('data xml tag not present')
                    if _ldtpDebug and _ldtpDebug == '2':
                        print 'data xml tag not present'
            except xml.parsers.expat.ExpatError, msg:
                if _ldtpDebug and _ldtpDebug == '2':
                    print 'XML Error: ' + str (msg)
                ldtp.log ('XML Error: ' + str (msg), 'error')
            except IOError:
                if _ldtpDebug and _ldtpDebug == '2':
                    print 'XML \"' + filename + '\" file not found'
                ldtp.log ('XML \"' + filename + '\" file not found', 'error')
    def getText (self, nodelist):
        rc = ""
        for node in nodelist:
            if node.nodeType == node.TEXT_NODE:
                rc = rc + node.data
        return rc
    def gettagvalue (self, tagname):
        self.taglist = []
        if self.ldtpdataxml == []:
            return self.taglist
        for dataelements in self.ldtpdataxml:
            for data in dataelements.getElementsByTagName (tagname):
                self.taglist.append (self.getText (data.childNodes))
        return self.taglist

class pstats (threading.Thread):
    """Capturing Memory and CPU Utilization statistics for an application and its related processes
    EXAMPLE USAGE:
        xstats = pstats('evolution', 2)
        # Start Logging by calling start
        xstats.start()
        # Stop the process statistics gathering thread by calling the stopstats method
        xstats.stop()"""

    def __init__ (self, appname, inter = 2):
        if statGrabMsg is not None:
            ldtp.log (statGrabMsg, 'error')
            raise ldtp.LdtpExecutionError (statGrabMsg)
        threading.Thread.__init__ (self)
        self.processname = appname
        self.interval = inter
        self.stop_flag = 0
    def run (self):
        while (self.stop_flag == 0):
            for i in statgrab.sg_get_process_stats ():
                if self.stop_flag == 1:
                    break
                result = re.search (self.processname, str(i['process_name']))
                if (result):
                    title = str (i['proctitle'])
                    proctitle = re.split ("\s", title)
                    procname = re.split ("\/", proctitle[0])
                    # Put the stats into ldtp log file    
                    ldtp.log (procname[-1] + ' - ' + str (i['proc_resident'] / (1024*1024)) + 'M',
                          'meminfo')
                    ldtp.log (procname[-1] + ' - ' + str (round(i['cpu_percent'], 2)),
                          'cpuinfo')
            # Wait for interval seconds before gathering stats again
            time.sleep (self.interval)

    def stop (self):
        self.stop_flag = 1

############# Application Functions #################

def execute (cmd):
    status = commands.getstatusoutput (cmd)
    if status[0] != 0:
        ldtp.log (status[1], 'error')
        raise ldtp.LdtpExecutionError (status[1])
    return 1
    
########### LTFX Functions ###############
def getactivewin ():
    #Get currently active window title name
    cmd = "ltfx -e 'get_active_win'"
    status = commands.getstatusoutput (cmd)
    if status[0] != 0:
        ldtp.log (status[1], 'error')
        raise ldtp.LdtpExecutionError (status[1])
    return status[1]
    
def windowexists (window_name):
    #Check window name exists with the given name
    cmd = "ltfx -e \'win_exists \"" +  window_name + "\"\'"
    return execute (cmd)

def partialexists (window_name):
    #Check window name exists with the given partial name
    cmd = "ltfx -e \'win_exists_partial \"" +  window_name + "\"\'"
    return execute (cmd)

def activatewinpartialname (window_name):
    # Set active window based on the given partial name"
    cmd = "ltfx -e \'activate_win_partial \"" +  window_name + "\"\'"
    return execute (cmd)

def activatewin (window_name):
    #Set active window based on the given name
    cmd = "ltfx -e \'activate_win \"" +  window_name + "\"\'"    
    return execute (cmd)

def activatewinid (window_id):
    #Set active window based on the given window-id
    cmd = "ltfx -e \'activate_win_id \"" +  window_id + "\"\'"    
    return execute (cmd)

def closewindow  (window_name):
    #Close the window with the given title
    return 0

def waitwinname (window_name):
    #Wait for window with name to appear
    cmd = "ltfx -e 'wait_for_win \"" +  window_name + "\"\'"    
    return execute (cmd)

def waitwinpartialname (window_name):
    #Wait for window with partial name to appear
    cmd = "ltfx -e 'wait_for_win_partial \"" +  window_name + "\"\'"    
    return execute (cmd)

def waitwinclose (window_name):
    #Wait for window to close with the given name
    cmd = "ltfx -e 'wait_for_close \"" +  window_name + "\"\'"    
    return execute (cmd)

def waitwinpartialclose (window_name):
    #Wait for window to close with the given partial name
    cmd = "ltfx -e 'wait_for_close_partial \"" +  window_name + "\"\'"        
    return execute (cmd)

def typekey (window_name):
    #Type the given text in the focused window
    cmd = "ltfx -e 'type \"" +  window_name + "\"\'"    
    return execute (cmd)
