/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <dlfcn.h>
#include <syslog.h>
#include <sys/stat.h>
#include <lcas/lcas.h>

#include "llgt_config.h"

#include "llgt_lcas.h"
#include "llgt_utils.h"

#ifndef LCAS_LIBDIR
    #define LCAS_LIBDIR ""
#endif
#ifndef LIBLCAS_SO
    #define LIBLCAS_SO "liblcas" LIBSUFF
#endif

static int set_liblcas_path(char ** liblcas_path);

static int set_liblcas_path(char ** liblcas_path)
{
    char * lcas_libdir       = NULL;
    char * lcas_moddir_sfx   = NULL;
    char * lcas_modules_dir  = NULL;
    struct stat st;

    /* First get the $LLGT_LCAS_LIBDIR environment variable as a base setting */
    lcas_libdir = getenv("LLGT_LCAS_LIBDIR");
    if (lcas_libdir && ((lcas_libdir)[0] != '\0')) {
        /* If it's non-empty, check it is an absolute and valid directory */
        if ((lcas_libdir)[0] != '/' || stat(lcas_libdir, &st)!=0 || !S_ISDIR(st.st_mode))  {
            llgt_logmsg(LOG_WARNING, "%s: Ignoring $LLGT_LCAS_LIBDIR as \"%s\" is not an absolute path to a valid directory\n", __func__, lcas_libdir);
            lcas_libdir = LCAS_LIBDIR;
        }
        /* Get the potentially different $LCAS_MODULE_SFX, based on the customized libdir */
        lcas_moddir_sfx = getenv("LLGT_LCAS_MODULEDIR_SFX");
        if (!lcas_moddir_sfx) {
            lcas_moddir_sfx = LCAS_MODULEDIR_SFX; /* Take the input from the LLGT configure */
        }

        /* Concatting LCAS_LIBDIR and LCAS_MODULEDIR_SFX to be exported as $LCAS_MODULES_DIR */
        lcas_modules_dir = malloc(AP_MAXPATH);
        if (!lcas_modules_dir) {
            llgt_logmsg(LOG_ERR, "%s: Could not allocate memory: %s\n", __func__, strerror(errno));
            return 1;
        } else {
            if (snprintf(lcas_modules_dir,AP_MAXPATH,"%s%s", lcas_libdir, lcas_moddir_sfx) >= AP_MAXPATH) {
                llgt_logmsg(LOG_WARNING, "Full modulespath '%s%s' would be too long, not setting LCAS_MODULES_DIR\n", lcas_libdir,lcas_moddir_sfx);
            } else {
                llgt_logmsg(LOG_DEBUG, "Setting LCAS_MODULES_DIR to '%s'\n", lcas_modules_dir);
                setenv("LCAS_MODULES_DIR", lcas_modules_dir, 1);
            }
            free(lcas_modules_dir);
        }

        /* Setting the full path to the liblcas.so object */
        *liblcas_path = malloc(AP_MAXPATH);
        if (snprintf(*liblcas_path, AP_MAXPATH, "%s/%s", lcas_libdir, LIBLCAS_SO) >= AP_MAXPATH) {
            llgt_logmsg(LOG_ERR, "Full path to %s \"%s/%s\" is too long\n", LIBLCAS_SO, lcas_libdir, LIBLCAS_SO);
            return 1;
        }
    } else {
        /* Just set the name of the shared object file, let the system figure it out */
        *liblcas_path = strdup(LCAS_LIBDIR LIBLCAS_SO);
    }
    llgt_logmsg(LOG_DEBUG, "LCAS library path : \"%s\"\n", *liblcas_path);
    return 0;
}


void llgt_setup_lcas_environment(void)
{
    if (!getenv ("LCAS_DEBUG_LEVEL"))
        setenv("LCAS_DEBUG_LEVEL", "3", 1);

    if (!getenv ("LCAS_DB_FILE"))
        setenv("LCAS_DB_FILE", SYSCONFDIR"/lcas/lcas.db", 1);

/* Not required by default, because the setting of the LCMAPS_LOG_FILE will declare to log to file, or use Syslog instead */
#ifdef LCAS_LCMAPS_FORCE_LOG_TO_FILE
    if (!getenv ("LCAS_LOG_FILE"))
        setenv ("LCAS_LOG_FILE",  "/var/log/gt_lcas_lcmaps.log", 1);
#endif

    if (!getenv ("LCAS_DIR"))
        setenv("LCAS_DIR", SYSCONFDIR, 1);

    /* if (!getenv ("LCAS_MOD_HOME")) */
        /* setenv("LCAS_MOD_HOME", LCAS_MODULEDIR, 1); */
}


/* AUTHORISATION WITH GLOBUS INTERFACE */
int llgt_run_lcas(gss_cred_id_t user_cred_handle, char * client_name, FILE * logfile)
{
    int    retval       = 0;
    char * error        = NULL;
    void * lcas_handle  = NULL;
    char * liblcas_path = NULL;
    int (*LcasInit)(FILE *);
    int (*LcasGetFabricAuthorization)(char*, gss_cred_id_t, char*);
    int (*LcasTerm)();


    /* Get the path to the to be opened LCAS main library - Must be free'd */
    if (set_liblcas_path(&liblcas_path)) {
        llgt_logmsg(LOG_ERR, "Couldn't set the path to \"%s\"\n", LIBLCAS_SO);
        return 1;
    }

    /* If there is no PEM String, nothing to do */
    if (!user_cred_handle)
    {
        llgt_logmsg(LOG_ERR, "Couldn't extract the client certificate credentials.\n");
        return 1;
    }

    /* I must have a path set */
    if (!liblcas_path) {
        llgt_logmsg(LOG_ERR, "Failed set a name or path to find liblcas.so\n");
        return 1;
    }

    lcas_handle = dlopen(liblcas_path, RTLD_LAZY|RTLD_GLOBAL);
    if (!lcas_handle)
    {
        llgt_logmsg(LOG_ERR, "Failed to dynamically load the library for LCAS: \"%s\"\n", liblcas_path);
        return 1;
    }
    /* Free liblcas_path */
    free(liblcas_path);

    /* Check the symbols */
    LcasInit=dlsym(lcas_handle,"lcas_init");
    if ((error = dlerror()) != NULL)
    {
        llgt_logmsg(LOG_ERR, "LCAS module not compliant: Symbol \"lcas_init\" not found: %s\n", error);
        dlclose(lcas_handle);
        return 1;
    }
    LcasGetFabricAuthorization=dlsym(lcas_handle,"lcas_get_fabric_authorization");
    if ((error = dlerror()) != NULL)
    {
        llgt_logmsg(LOG_ERR, "LCAS module not compliant: Symbol \"lcas_get_fabric_authorization\" not found: %s\n", error);
        dlclose(lcas_handle);
        return 1;
    }
    LcasTerm=dlsym(lcas_handle,"lcas_term");
    if ((error = dlerror()) != NULL)
    {
        llgt_logmsg(LOG_ERR, "LCAS module not compliant: Symbol \"lcas_term\" not found: %s\n", error);
        dlclose(lcas_handle);
        return 1;
    }

    /* Initialize, send authorization request to and terminate the LCAS */
    /* retval=lcas_init(config->usrlog_fp); */
    if ((retval = (*LcasInit)(logfile)))
    {
        llgt_logmsg(LOG_ERR, "LCAS initialization failure.\n");
        dlclose(lcas_handle);
        return 1;
    }

    /* Authorization based on the GSI input */
    if ((retval=(*LcasGetFabricAuthorization)(client_name, user_cred_handle, "RSL") != 0))
    {
        llgt_logmsg(LOG_WARNING, "%s: The user is not authorized by LCAS.\n", __func__);
    }
    else
    {
        llgt_logmsg(LOG_WARNING, "%s: The user is authorized by LCAS.\n", __func__);
    }

    /* Terminate LCAS framework */
    if ((*LcasTerm)())
    {
        llgt_logmsg(LOG_ERR, "LCAS termination failure.\n");
        dlclose(lcas_handle);
        return 1;
    }

    /* Release the library handle */
    dlclose(lcas_handle);

    /* All is ok - but you could still be not authorized! */
    return retval;
}

