//LabPlot : HistListDialog.cc

#include "HistListDialog.h"
#include "defs.h"

#ifdef HAVE_GSL
#include <gsl/gsl_errno.h>
#include <gsl/gsl_histogram.h>
#include <gsl/gsl_histogram2d.h>
#endif

HistListDialog::HistListDialog(MainWin *m, const char *name)
	: ListDialog(m, name)
{
	setCaption(i18n("Histogram Dialog"));
	KConfig *config = mw->Config();
	config->setGroup( "Histogram" );

	QTabWidget *tw = new QTabWidget(vbox);
	QVBox *tab1 = new QVBox(tw);

	QHBox *hb = new QHBox(tab1);
	new QLabel(i18n(" From : "),hb);
	ale = new KLineEdit(config->readEntry("From","0"),hb);
	ale->setValidator(new QDoubleValidator(ale));
	new QLabel(i18n(" To : "),hb);
	double b=config->readDoubleNumEntry("To",0.0);
	if(s)
		b=s->Table()->numRows();
	ble = new KLineEdit(QString::number(b),hb);
	ble->setValidator(new QDoubleValidator(ble));

	hb = new QHBox(tab1);
	new QLabel(i18n(" Bins : "),hb);
	binni = new KIntNumInput(config->readNumEntry("Bins",100),hb);
	binni->setRange(1,INF,1,false);

	if (p) {
		updateRange(1);
		QObject::connect(lv,SIGNAL(selectionChanged()),SLOT(updateRange()));
	}

	Style *style=0;
	Symbol *symbol=0;
	QVBox *styletab;
	if(p && p->getPlot(p->API())->Type() == PSURFACE)
		styletab = surfaceStyle(tw,true);
	else
		styletab = simpleStyle(tw, style, symbol);

	tw->addTab(tab1,i18n("Parameter"));
	tw->addTab(styletab,i18n("Style"));

	QObject::connect(ok,SIGNAL(clicked()),SLOT(ok_clicked()));
	QObject::connect(apply,SIGNAL(clicked()),SLOT(apply_clicked()));
	QObject::connect(save,SIGNAL(clicked()),SLOT(saveSettings()));

	setMinimumWidth(vbox->minimumSizeHint().width());
	setMinimumHeight(gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height());
	resize(minimumSize());
}

void HistListDialog::saveSettings() {
	KConfig *config = mw->Config();
	config->setGroup( "Histogram" );

	config->writeEntry("From", ale->text());
	config->writeEntry("To", ble->text());
	config->writeEntry("Bins", binni->value());
}

int HistListDialog::apply_clicked() {
#ifdef HAVE_GSL
	int bins = binni->value();
	double a=(ale->text()).toDouble();
	double b=(ble->text()).toDouble();

	if(s) {
		QTable *table = s->Table();
		int col = table->currentColumn();

		// add column if needed
		bool empty=true;
		for(int i=0;i<table->numRows();i++) {
			if(!table->text(i,table->numCols()-1).isEmpty())
				empty = false;
		}
		if (!empty)
			s->addColumn();

		gsl_histogram *h = gsl_histogram_alloc (bins);
		gsl_histogram_set_ranges_uniform (h, a, b);

		// insert new data
		int newcol = table->numCols()-1;
		for (int i = 0;i<table->numRows();i++) {
			if(table->numSelections() == 0 || table->isRowSelected(i)) {
				int status = gsl_histogram_increment(h,table->text(i,col).toDouble());
				if(status == GSL_EDOM)
					gsl_histogram_increment(h,table->text(i,col).toDouble()-(b-a)/bins);
			}
		}

		// insert into last column
		for (int i = 0;i<bins;i++) {
			table->setText(i,newcol,QString::number(gsl_histogram_get (h,i)));
		}
		gsl_histogram_free (h);

		return 0;
	}

	int item = (int) (lv->itemPos(lv->currentItem())/lv->currentItem()->height());

	GraphList *gl = p->getPlot(p->API())->getGraphList();
	if(gl->Number()==0) {
		KMessageBox::error(this,i18n("No graph found!"));
		return -2;
	}

	GRAPHType st = gl->getType(item);

	Style *style=0;
	Symbol *symbol=0;
	if(st != GRAPHM) {
		style = new Style((StylesType)cb2->currentItem(),color->color(),filled->isChecked(),fcolor->color(),
			width->value(),pencb->currentItem(),brushcb->currentItem());
		style->setBoxWidth(boxwidth->value());
		style->setAutoBoxWidth(autobox->isChecked());
		style->setPointsSorting(sortpointscb->isChecked());
		symbol = new Symbol((SType)symbolcb->currentItem(),scolor->color(),ssize->value(),
			(FType)symbolfillcb->currentItem(),sfcolor->color(),sbrushcb->currentItem());
	}

	double xmin=0, xmax=1, ymin=0, ymax=1, zmin=0, zmax=1;
	if (st == GRAPH2D) {
		Graph2D *g = gl->getGraph2D(item);
		int nx = g->Number();
		Point *data = g->Data();
		Point *ptr = new Point[bins];

		gsl_histogram *h = gsl_histogram_alloc (bins);
		gsl_histogram_set_ranges_uniform (h, a, b);
		for (int i = 0;i<nx;i++)
			gsl_histogram_increment(h,data[i].Y());

		for (int i = 0;i<bins;i++) {
			double x = i;
			double y = gsl_histogram_get (h,i);

			ptr[i].setPoint(x,y);
		}
		gsl_histogram_free (h);

		mw->calculateRanges2D(ptr,bins,&xmin,&xmax,&ymin,&ymax);

		// create the new graph
		LRange range[2];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);

		QString fun = QString(i18n("histogram (")+QString::number(bins)+i18n(") of ")+g->getLabel()->simpleTitle());

		Graph2D *ng = new Graph2D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,ptr,bins);
		mw->addGraph2D(ng,sheetcb->currentItem());
	}
	else if (st == GRAPH3D) {
		Graph3D *g = gl->getGraph3D(item);
		int nx = g->Number();
		Point3D *data = g->Data();
		Point *ptr = new Point[bins];

		// TODO : PQWT3D	-> 2d histogram
		if(type != P2D) {
			KMessageBox::error(this,i18n("Sorry. This function is not yet implemented!"));
			return -3;
		}

		gsl_histogram *h = gsl_histogram_alloc (bins);
		gsl_histogram_set_ranges_uniform (h, a, b);
		for (int i = 0;i<nx;i++)
			gsl_histogram_increment(h,data[i].Y());

		for (int i = 0;i<bins;i++) {
			double x = i;
			double y = gsl_histogram_get (h,i);

			ptr[i].setPoint(x,y);
		}
		gsl_histogram_free (h);

		mw->calculateRanges2D(ptr,bins,&xmin,&xmax,&ymin,&ymax);

		LRange range[2];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);

		QString fun = QString(i18n("histogram (")+QString::number(bins)+i18n(") of ")+g->getLabel()->simpleTitle());

		// 3D->2D graph
		Graph2D *ng = new Graph2D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,ptr,bins);
		mw->addGraph2D(ng,sheetcb->currentItem());
	}
	else if (st == GRAPH4D) {
		Graph4D *g = gl->getGraph4D(item);
		int nx = g->Number();
		Point4D *data = g->Data();
		Point *ptr = new Point[bins];

		gsl_histogram *h = gsl_histogram_alloc (bins);
		gsl_histogram_set_ranges_uniform (h, a, b);
		for (int i = 0;i<nx;i++)
			gsl_histogram_increment(h,data[i].Y());

		for (int i = 0;i<bins;i++) {
			double x = i;
			double y = gsl_histogram_get (h,i);

			ptr[i].setPoint(x,y);
		}
		gsl_histogram_free (h);

		mw->calculateRanges2D(ptr,bins,&xmin,&xmax,&ymin,&ymax);

		LRange range[2];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);

		QString fun = QString(i18n("histogram (")+QString::number(bins)+i18n(") of ")+g->getLabel()->simpleTitle());

		// 4D->2D graph
		Graph2D *ng = new Graph2D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,ptr,bins);
		mw->addGraph2D(ng,sheetcb->currentItem());
	}
	else if (st == GRAPHM) {
		//	-> construct from 2 x 1d histograms
		// TODO : xbins, ybins
		//int xbins=bins;
		int ybins=bins;
		GraphM *g = gl->getGraphM(item);
		int nx = g->NX(), ny = g->NY();
		double *data = g->Data();
		Point3D *ptr = new Point3D[ny*ybins];

		for(int i=0;i<nx;i++) {
			gsl_histogram *h = gsl_histogram_alloc (ybins);
			gsl_histogram_set_ranges_uniform (h, a, b);

			for (int j = 0;j<ny;j++)
				gsl_histogram_increment(h,data[i*ny+j]);

			for (int j = 0;j<ybins;j++) {
				double x = i;
				double y = j;
				double z = gsl_histogram_get (h,j);

				ptr[i*ybins+j].setPoint(x,y,z);
			}
			gsl_histogram_free (h);
		}

		mw->calculateRanges3D(ptr,nx*ybins,&xmin,&xmax,&ymin,&ymax,&zmin, &zmax);

		LRange range[3];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);
		range[2] = LRange(zmin,zmax);

		QString fun = QString(i18n("histogram 2d (")+QString::number(nx*ybins)+i18n(") of ")+g->getLabel()->simpleTitle());

		// M->3D graph
		Graph3D *ng = new Graph3D(fun,fun,range,SSPREADSHEET,PSURFACE,style,symbol,ptr,nx,ybins);
		mw->addGraph3D(ng,sheetcb->currentItem(),type);
	}

	updateList();
#else
	KMessageBox::error(this, i18n("Sorry. Your installation doesn't support the GSL!"));
	return -3;
#endif

	return 0;
}
