/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifndef FLOSSPALETTE_H
#define FLOSSPALETTE_H

#include <qmap.h>
#include <qvaluelist.h>

class Floss;

/** FlossPalette stores and manipulates the floss colors that have been added to the pattern
    @author Stephen Allewell
  */
class FlossPalette
{
public:
/** Constructor - creates an empty floss palette */
  FlossPalette();

/** Sets the floss scheme
    @param name Name of the scheme to be used.
  */
  void setScheme(QString name);
/** Gets the floss scheme of the current palette
    @return The name of the FlossScheme
  */
  QString getScheme();
/** Change the floss scheme of the current palette
    @return TRUE if changed correctly, FALSE otherwise
    @param name Name of the scheme to be used
  */
  bool changeScheme(QString name);
/** Adds a specific color to the palette, first checks the palette to see if the color already exists
    @return color index of added color
    @param c @ref QColor to be added
  */
  int addColor(QColor color);
/** Remove unused colors from the palette
    @return TRUE if colors removed, FALSE otherwise
    @param usedColors A list of the indexes used in the pattern
  */
  bool removeUnused(QValueList<int> usedColors);
/** swap stitches of the current color with the color specified
    @param color The color index to be swapped with the current color
  */
  void swapColors(int color);
/** Checks if a named color is present in the current palette, used internally by @ref addColor so that is doesn't
    add colors to the list box that are already present in the palette.
    @return TRUE if the color is in the palette
    @param name QString of the colors name
  */
  bool contains(QString name);
/** Checks if a QColor color is present in the current palette, used when adding colors from file imports
    @return index of the color if the color is in the palette, or -1 if not
    @param color QColor of the colors RGB value
  */
  int contains(QColor color);
/** Allocates the next available symbol for the palette floss color
    @return A character code for the symbol
  */
  QChar freeSymbol();
/** Allocates the next available index for the palette floss color
    @return index value to be used for the map index
  */
  int freeIndex();
/** Sets the current color within the palette, initially set at -1 for no color selected
    @param i The index of the selected color
  */
  void setCurrentColor(int i);
/** Retrieves the currently selected color in the palette
    @return The index of the current color, -1 if no color is currently selected
  */
  int currentColor();
/** The number of colors in the palette
    @return The number of colors
  */
  int colors();
/** Clear the contents of the palette
  */
  void clear();
/** Gets the floss information at a specified index number
    @return Pointer to a Floss entry
    @param i index of the Floss entry required
  */
  Floss *flossAt(int i);
/** Get the symbol associated with a floss index
    @return QChar symbol
    @param i index of the Floss entry required
  */
  QChar symbolAt(int i);
/** Get the number of strands to be used for this index
    @return Number of strands
    @param i index of the Floss entry required
  */
  int stitchStrandsAt(int i);
/** Get the number of backstitch strands to be used for this index
    @return Number of strands
    @param i index of the Floss entry required
  */
  int backstitchStrandsAt(int i);
/** Get the colormap
    @return a QMap of the floss palette
  */
  QMap<int,Floss*>& colorMap() { return m_palette; }
/** write the floss palette to the data stream
    @return TRUE if the palette was written, FALSE if an error occured.
    @param s The datastream to write to.
  */
  bool writePalette(QDataStream& s);
/** read the floss palette from the data stream
    @return TRUE if the palette was read, FALSE if an error occured.
    @param s The datastream to read from.
    @param v The version of the file being read.
  */
  bool readKXStitchPalette(QDataStream& s,int v);
  bool readPCStitch5Palette(QDataStream& s);
/** load a palette from a file
    @return TRUE if the palette was read, FALSE if an error occured.
    @param schemeName The name of the scheme to be read
    */
  bool loadPalette(QString schemeName);
/** save the current palette to a file
    @return TRUE if the palette was written, FALSE if an error occured
    @param schemeName The name of the scheme to be written
  */
  bool savePalette(QString schemeName);

  friend class PaletteManagerDialog;

private:
  int               m_current;
  QString           m_schemeName;
  QMap<int,Floss*>  m_palette;
  QMap<int,QChar>   m_symbols;
  QMap<int,int>     m_stitchStrands;
  QMap<int,int>     m_backstitchStrands;
};

#endif
