#!/usr/bin/ruby

$: << '..'
require 'kwartz'
require 'test/unit'
require 'test/unit/ui/console/testrunner'


Dirname = 'test.d'


## ====================
## test for Scanner
## ====================
class ScannerTest < Test::Unit::TestCase

    def _test_scan(input, expected, flag_verbose=false)
	scanner = Kwartz::Scanner.new(input)
	if flag_verbose then
	    assert_equal(expected, scanner.scan_all)
	else
	    assert(expected == scanner.scan_all)
	end
    end

    ## --------------------
    ## basic test
    ## --------------------
    def test_scan1	# basic scanner test
	input = <<END
# sample program
:if(arg>=1)
  :set(i=0)
  :while(i<arg.length)
    :print(arg[i])
    :set(i=i+1)
  :end
:end
END
	expected = <<END
:if
(
arg
>=
1
)
:set
(
i
=
0
)
:while
(
i
<
arg
.
length
)
:print
(
arg
[
i
]
)
:set
(
i
=
i
+
1
)
:end
:end
END
	_test_scan(input, expected)
    end


    ## --------------------
    ## string
    ## --------------------
    def test_scan2a	# string
	input = <<'END'
"hoobar\a\t\r\n"
'hoobar\a\t\r\n'
END
	expected = <<'END'
"hoobara\t\r\n"
"hoobar\\a\\t\\r\\n"
END
	_test_scan(input, expected)
    end


    ## --------------------
    ## string #2
    ## --------------------
    def test_scan2b	# string2
	input = <<'END'
"hoobar\"\'"
'hoobar\"\''
END
	expected = <<'END'
"hoobar\"'"
"hoobar\\\"'"
END
	_test_scan(input, expected)
    end


    ## --------------------
    ## symbol
    ## --------------------
    def test_scan3	## symbol
	input = <<END
:set(foo = bar ? 2 * R * 3.14 : 3.14 * R ^ 2)
END
	expected = <<END
:set
(
foo
=
bar
?
2
*
R
*
3.14
:
3.14
*
R
^
2
)
END
	scanner = Kwartz::Scanner.new(input)
	assert(scanner.scan_all == expected)
    end

    ## --------------------
    ## raw code
    ## --------------------
    def test_scan4	## ':::'
	input  = "::: for (i=0; i<arg.length; i++) {\n"
	expected = <<END
::: for (i=0; i<arg.length; i++) {
END
	_test_scan(input, expected, true)
    end

    
    ## --------------------
    ## macro and expand
    ## --------------------
    def test_scan5	## macro and expand
	input = <<END
:macro(elem_user)
  :expand(stag_user)
  :expand(cont_user)
  :expand(etag_user)
:end
:expand(elem_user)
END
	expected = <<END
:macro
(
elem_user
)
:expand
(
stag_user
)
:expand
(
cont_user
)
:expand
(
etag_user
)
:end
:expand
(
elem_user
)
END
	a = 0
	_test_scan(input, expected, true)
    end


    ## --------------------
    ##  :elem and @macroname
    ## --------------------
    def test_scan6	## :elem and @macroname
	input = <<END
:elem(user)
  @stag
  :foreach(user=list)
    @cont
  :end
  @etag
:end
@elem_user
END
   	expected = <<END
:elem
(
user
)
:stag
:foreach
(
user
=
list
)
:cont
:end
:etag
:end
@elem_user
END
	_test_scan(input, expected, true)
    end

##    ## --------------------
##    ## invalid token
##    ## --------------------
##    def test_scan7	## invalid token
##    	input = "@elem_foo\n"
##	expected = nil
##	assert_raises(Kwartz::SyntaxError) do
##	    _test_scan(input, expected, true)
##	end
##    end


    ## --------------------
    ## invalid token
    ## --------------------
    def test_scan8	## invalid token
    	input = <<END
:set(flag = obj == null ? true : false)
END
	expected = <<END
:set
(
flag
=
obj
==
null
?
true
:
false
)
END
	_test_scan(input, expected, true)
    end
    

end


## ====================
## test for Parser
## ====================
class ParserTest < Test::Unit::TestCase
    def _test_expr(input, expected, toppings={})
	scanner = Kwartz::Scanner.new(input, toppings)
	parser  = Kwartz::Parser.new(scanner, toppings)
	node = parser.parse_assignment()
	unless parser.token() == nil then
	    raise Kwartz::SyntaxError.new("<_text_expr>", scanner)
	end
	#node = parser.parse_expression()
	#assert_not_nil(node)
	assert(node.print_all == expected)
    end

    def _test_stmt(input, expected, toppings={})
	scanner = Kwartz::Scanner.new(input, toppings)
	parser  = Kwartz::Parser.new(scanner, toppings)
	nodelist = parser.parse()
	#assert(expected, nodelist.print_all)
	assert_equal(expected, nodelist.print_all)
    end

    ## --------------------
    ## variable
    ## --------------------
    def test_expr01	## variable
	input = 'aaa'
	expected = "aaa\n"
	_test_expr(input, expected)
    end

    ## --------------------
    ## number (integer)
    ## --------------------
    def test_expr02	## number (integer)
	input = '123'
	expected = "123\n"
	_test_expr(input, expected)
    end

    ## --------------------
    ## number (float)
    ## --------------------
    def test_expr03	## number (float)
	input = '3.14'
	expected = "3.14\n"
	_test_expr(input, expected)
    end

    ## --------------------
    ## string
    ## --------------------
    def test_expr04	## string
	input = '"foobar\n"'
	expected = '"foobar\n"' + "\n"
	_test_expr(input, expected)
    end


    ## --------------------
    ## array
    ## --------------------
    def test_expr11	## array
	input = 'foo[i+1]'
	expected = <<END
[]
  foo
  +
    i
    1
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## hash{key} (obsolete)
    ## --------------------
    def test_expr12	## hash{key} (obsolete)
	input = "foo{'bar'}"
	expected = <<'END'
{}
  foo
  "bar"
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## hash[:key]
    ## --------------------
    def test_expr13	## hash[:key]
	input = 'foo[:bar]'
	expected = <<END
[:]
  foo
  bar
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## .property
    ## --------------------
    def test_expr14	## .property
	input = 'foo.bar'
	expected = <<END
.
  foo
  bar
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## nested array, hash, property
    ## --------------------
    def test_expr15	## nested array, hash, property
	input = "a[0]{'foo'}[:bar].baz"
	expected = <<END
.
  [:]
    {}
      []
        a
        0
      "foo"
    bar
  baz
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## term
    ## --------------------
    def test_expr21	## term
	input = '3.14 * r * 2'
	expected = <<END
*
  *
    3.14
    r
  2
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## arithmetic
    ## --------------------
    def test_expr22	## arithmetic
	input = 'a + 2 - b'
	expected = <<END
-
  +
    a
    2
  b
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## term and arithmetic
    ## --------------------
    def test_expr23	## term and arithmetic
	input = 'a + b % c - d * e'
	expected = <<END
-
  +
    a
    %
      b
      c
  *
    d
    e
END
	_test_expr(input, expected)
    end



    ## --------------------
    ## unary minus operator
    ## --------------------
    def test_expr24	## unary minus operator
	input = '- a * b % c + d * e'
	expected = <<END
+
  -
    %
      *
        a
        b
      c
  *
    d
    e
END
	_test_expr(input, expected)
    end

    
    ## --------------------
    ## invalid unary operator
    ## --------------------
    def test_expr25	## invalid unary operator
	input = '- a * - b'
	expected = ''
	assert_raises(Kwartz::SyntaxError) do
	    _test_expr(input, expected)
	end
    end


    ## --------------------
    ## paren
    ## --------------------
    def test_expr26	## paren
	input = 'a * (b+c * (d-e))'
	expected = <<END
*
  a
  +
    b
    *
      c
      -
        d
        e
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## conditional operator
    ## --------------------
    def test_expr27	## conditional operator
	input = 'a ? true : false'
	expected = <<END
?
  a
  true
  false
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## string concat
    ## --------------------
    def test_expr28	## string concat
	input = "'foo' .+ '.txt'"
	expected = <<END
.+
  "foo"
  ".txt"
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## compare (equal)
    ## --------------------
    def test_expr31	## compare (equal)
	input = 'a == i+1'
	expected = <<END
==
  a
  +
    i
    1
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## compare (less than)
    ## --------------------
    def test_expr32	## compare (less than)
	input = '10 + a < 30 - a'
	expected = <<END
<
  +
    10
    a
  -
    30
    a
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## compare with conditional operator
    ## --------------------
    def test_expr33	## compare with conditional operator
	input = 'a!=100 ? b>=0 : c<=0'
	expected = <<END
?
  !=
    a
    100
  >=
    b
    0
  <=
    c
    0
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## logical and/or
    ## --------------------
    def test_expr34	## logical and/or
	input = 'a && b || c && d'
	expected = <<END
||
  &&
    a
    b
  &&
    c
    d
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## complex logical expression
    ## --------------------
    def test_expr35	## complex logical expression
	input = '-a<i && i<a*2 || i==a'
	expected = <<END
||
  &&
    <
      -
        a
      i
    <
      i
      *
        a
        2
  ==
    i
    a
END
	_test_expr(input, expected)
    end


    ## --------------------
    ## unary not operator
    ## --------------------
    def test_expr36	## unary not operator
	input = '! a == b || ! a && ! b && ! c'
	expected = <<END
||
  ==
    !
      a
    b
  &&
    &&
      !
        a
      !
        b
    !
      c
END
	_test_expr(input, expected)
    end


    ## --------------------
    ## assignment
    ## --------------------
    def test_expr41	## assignment
	input = 'foo = bar % 2'
	expected = <<END
=
  foo
  %
    bar
    2
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## assignment and conditional operator
    ## --------------------
    def test_expr42	## assignment and conditional operator
	input = "color = ctr % 2 == 0 ? '#FFCCCC' : '#CCCCFF'"
	expected = <<END
=
  color
  ?
    ==
      %
        ctr
        2
      0
    "#FFCCCC"
    "#CCCCFF"
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## assignment with string
    ## --------------------
    def test_expr43	## assignment with string
	input = 'str .+= suffix'
	expected = <<END
.+=
  str
  suffix
END
	_test_expr(input, expected)
    end

    ## --------------------
    ## true/false/null
    ## --------------------
    def test_expr51	## true/false/null
	input = 'flag = obj == null ? true : false'
	expected = <<END
=
  flag
  ?
    ==
      obj
      null
    true
    false
END
	_test_expr(input, expected)
    end


    ## --------------------
    ## true/false/null (2)
    ## --------------------
    def test_expr52	## true/false/null (2)
	#input = 'array[0] = true[0]'
	input = 'array = true[0]'
	expected = ''
	assert_raises(Kwartz::SyntaxError) do
	    _test_expr(input, expected)
	end
    end


    ## --------------------
    ## empty
    ## --------------------
    def test_expr53	## empty
	input = 'params[:name] == empty && params[:email] != empty'
	expected = <<END
&&
  :empty
    [:]
      params
      name
  :notempty
    [:]
      params
      email
END
	_test_expr(input, expected)
    end

    
    ## --------------------
    ## :set stmt
    ## --------------------
    def test_stmt01	## :set stmt
	input = ':set(v=1)'
	expected = <<END
<nodelist>
  :set
    =
      v
      1
END
	_test_stmt(input, expected)
    end


    ## --------------------
    ## :set stmt (2)
    ## --------------------
    def test_stmt02	## :set stmt (2)
	input = ':set(false = 0)'
	expected = nil
	assert_raises(Kwartz::SemanticError) do
	    _test_stmt(input, expected)
	end
    end


    ## --------------------
    ## :print stmt
    ## --------------------
    def test_stmt11	## :print stmt
	input = ":print(foo, '.txt')"
	expected = <<END
<nodelist>
  :print
    ,
      foo
      ".txt"
END
	_test_stmt(input, expected)
    end


    ## --------------------
    ## :while stmt
    ## --------------------
    def test_stmt12	## :while stmt
	input = <<END
:while(i=i+1)
  :print(a[i])
:end
END
	expected = <<END
<nodelist>
  :while
    =
      i
      +
        i
        1
    <nodelist>
      :print
        []
          a
          i
END
	_test_stmt(input, expected)
    end

    
    ## --------------------
    ## :foreach stmt
    ## --------------------
    def test_stmt13	## :foreach stmt
	input = <<END
:foreach(item=list)
  :set(ctr+=1)
  :print(ctr, item)
:end
END
	expected = <<END
<nodelist>
  :foreach
    =
      item
      list
    <nodelist>
      :set
        +=
          ctr
          1
      :print
        ,
          ctr
          item
END
	_test_stmt(input, expected)
    end


    ## --------------------
    ## :expand stmt
    ## --------------------
    def test_stmt21	## :expand stmt
	input = ':expand(foo)'
	expected = <<END
<nodelist>
  :expand
    foo
END
	_test_stmt(input, expected)
    end

    ## --------------------
    ## :expand2 stmt
    ## --------------------
    def test_stmt22	## :expand2 stmt
	input = '@foo'
	expected = <<END
<nodelist>
  :expand
    foo
END
	_test_stmt(input, expected)
    end


    ## --------------------
    ## :macro stmt
    ## --------------------
    def test_stmt31	## :macro stmt
	input = <<END
:macro(foo)
  :set(v=1)
  :print(v)
:end
END
	expected = <<END
<nodelist>
  :macro
    foo
    <nodelist>
      :set
        =
          v
          1
      :print
        v
END
	_test_stmt(input, expected)
    end

    ## --------------------
    ## :elem stmt and special expantion(@stag, @cont, @etag)
    ## --------------------
    def test_stmt32	## :elem stmt and special expantion(@stag, @cont, @etag)
	input = <<END
:elem(user)
  @stag
  :foreach(user=list)
    @cont
  :end
  @etag
:end
:expand(elem_user)
END
	expected = <<END
<nodelist>
  :macro
    elem_user
    <nodelist>
      :expand
        stag_user
      :foreach
        =
          user
          list
        <nodelist>
          :expand
            cont_user
      :expand
        etag_user
  :expand
    elem_user
END
	_test_stmt(input, expected)
    end

    ## --------------------
    ## @stag, @cont or @end must be in :elem() statement
    ## --------------------
    def test_stmt33	## @stag, @cont or @end must be in :elem() statement
    	input = <<END
:macro(elem_user)
  @stag
  @cont
  @etag
:end
END
	assert_raises(Kwartz::SemanticError) do
	    _test_stmt(input, "")
	end
    end

    
    ## --------------------
    ## :if-stmt (then only)
    ## --------------------
    def test_stmt61	## :if-stmt (then only)
	input = <<END
:if(cond)
  :print(foo)
:end
END
	expected = <<END
<nodelist>
  :if
    cond
    <nodelist>
      :print
        foo
END
	_test_stmt(input, expected)
    end

    ## --------------------
    ## :if-stmt (with :else)
    ## --------------------
    def test_stmt62	## :if-stmt (with :else)
	input = <<END
:if(cond)
  :print(true)
:else
  :print(false)
:end
END
	expected = <<END
<nodelist>
  :if
    cond
    <nodelist>
      :print
        true
    <nodelist>
      :print
        false
END
	_test_stmt(input, expected)
    end

    ## --------------------
    ## :if-stmt (with :elsif :else)
    ## --------------------
    def test_stmt63	## :if-stmt (with :elsif :else)
	input = <<'END'
:if(cond1)
  :print(var1)
:elsif(cond2)
  :print(var2)
:elsif(cond3)
  :print(var3)
:else
  :print(var9)
:end
END
	expected = <<END
<nodelist>
  :if
    cond1
    <nodelist>
      :print
        var1
    :if
      cond2
      <nodelist>
        :print
          var2
      :if
        cond3
        <nodelist>
          :print
            var3
        <nodelist>
          :print
            var9
END
	_test_stmt(input, expected)
    end

    ## --------------------
    ## :::
    ## --------------------
    def test_stmt71	## :::
	input = " ::: for (i=0; i<args.length; i++) {\n"
	expected = <<END
<nodelist>
  :rawcode
     for (i=0; i<args.length; i++) {
END
	_test_stmt(input, expected)
    end

    ## --------------------
    ## :rawcode
    ## --------------------
    def test_stmt72	## :rawcode
	input = " :rawcode('for (i=0; i<args.length; i++) {')"
	expected = <<END
<nodelist>
  :rawcode
    for (i=0; i<args.length; i++) {
END
	_test_stmt(input, expected)
    end

    ## --------------------
    ## :value
    ## --------------------
    def test_stmt73	## :value
	input = ":value(name=hash[:key])"
	expected = <<END
<nodelist>
  :macro
    cont_name
    <nodelist>
      :print
        [:]
          hash
          key
END
	_test_stmt(input, expected)
    end


    ## --------------------
    ## --enable_eruby=true
    ## --------------------
    def test_stmt74	## --enable_eruby=true
	input = <<'END'
% if __lang__ == 'ruby'
::: s = '%03d' % i
% elsif __lang__ == 'php'
::: <?php $s = sprintf('%03d', $i); ?>
% else
:set(s = '' .+ i)
% end
END
	expected = <<END
<nodelist>
  :rawcode
     s = '%03d' % i
END
        _test_stmt(input, expected, {:enable_eruby=>true, :lang=>'ruby'})
	
	expected = <<END
<nodelist>
  :rawcode
     <?php $s = sprintf('%03d', $i); ?>
END
        _test_stmt(input, expected, {:enable_eruby=>true, :lang=>'php'})
	
	expected = <<END
<nodelist>
  :set
    =
      s
      .+
        ""
        i
END
        _test_stmt(input, expected, {:enable_eruby=>true, :lang=>'jsp'})
    end


    ## --------------------
    ## :load('filename')
    ## --------------------
    def test_stmt75	## :load('filename')
	load1 = <<END
:print('--- load1 start ---\n')
:load('load2.plogic')
:print('--- load1 end -----\n')
END
	load2 = <<END
:print('--- load2 start ---\n')
:foreach(item = list)
  :print(item)
:end
:print('--- load2 end -----\n')
END
	input = <<END
:load('load1.plogic')
END
	expected = <<END
<nodelist>
  :print
    \"--- load1 start ---\\n\"
  :print
    \"--- load2 start ---\\n\"
  :foreach
    =
      item
      list
    <nodelist>
      :print
        item
  :print
    \"--- load2 end -----\\n\"
  :print
    \"--- load1 end -----\\n\"
END
	Dir.mkdir('test.d') if !test(?d, 'test.d')
	File.open('test.d/load1.plogic', 'w') { |f| f.write(load1) }
	File.open('test.d/load2.plogic', 'w') { |f| f.write(load2) }
	File.open('test.d/load.plogic',  'w') { |f| f.write(input) }
	File.open('test.d/load.expected','w') { |f| f.write(expected) }
	_test_stmt(input, expected, { :load_path=>['.', 'test.d'] })
	
	input2 = ":load(load1.plogic)\n"
	assert_raises(Kwartz::SyntaxError) do
	    _test_stmt(input2, '', { :load_path=>['.', 'test.d'] })
	end
	
	input3 = ":load('foo.plogic')\n"
	assert_raises(Kwartz::ParseError) do
	    _test_stmt(input3, '', { :load_path=>['.', 'test.d'] })
	end
	
    end



    ## --------------------
    ## synstax error - array assignment
    ## --------------------
#    def test_stmt81	## synstax error - array assignment
#	input = ":set(array[0] = value)"
#	expected = ""
#	assert_raises(Kwartz::SemanticError) do
#	    _test_stmt(input, expected)
#	end
#    end


    ## --------------------
    ## synstax error - hash assignment
    ## --------------------
#    def test_stmt82	## synstax error - hash assignment
#	input = ":set(hash['key'] = value)"
#	expected = ""
#	assert_raises(Kwartz::SemanticError) do
#	    _test_stmt(input, expected)
#	end
#    end


    ## --------------------
    ## synstax error - property assignment
    ## --------------------
#    def test_stmt83	## synstax error - property assignment
#	input = ":set(obj.property = value)"
#	expected = ""
#	assert_raises(Kwartz::SemanticError) do
#	    _test_stmt(input, expected)
#	end
#    end



    ## --------------------
    ## synstax error - assignment into expression
    ## --------------------
    def test_stmt84	## synstax error - assignment into expression
	input = ":set(a + b = value)"
	expected = ""
	assert_raises(Kwartz::SemanticError) do
	    _test_stmt(input, expected)
	end
    end



    ## --------------------
    ## synstax error - :foreach(array[0] = list)
    ## --------------------
#    def test_stmt85	## synstax error - :foreach(array[0] = list)
#	input = ":foreach(array[0] = value) :end"
#	expected = ""
#	assert_raises(Kwartz::SemanticError) do
#	    _test_stmt(input, expected)
#	end
#    end


    ## --------------------
    ## synstax error - :foreach(hash['key'] = list)
    ## --------------------
#    def test_stmt86	## synstax error - :foreach(hash['key'] = list)
#	input = ":foreach(hash[:key] = value) :end"
#	expected = ""
#	assert_raises(Kwartz::SemanticError) do
#	    _test_stmt(input, expected)
#	end
#    end


    ## --------------------
    ## synstax error - :foreach(obj.property = list)
    ## --------------------
#    def test_stmt87	## synstax error - :foreach(obj.property = list)
#	input = ":foreach(obj.property = value) :end"
#	expected = ""
#	assert_raises(Kwartz::SemanticError) do
#	    _test_stmt(input, expected)
#	end
#    end


    ## --------------------
    ## synstax error - :foreach(a+b = list)
    ## --------------------
    def test_stmt88	##  synstax error - :foreach(a+b = list)
	input = ":foreach(a+b = value) :end"
	expected = ""
	assert_raises(Kwartz::SemanticError) do
	    _test_stmt(input, expected)
	end
    end

    
    ## --------------------
    ## synstax error - empty
    ## --------------------
    def test_stmt89	##  synstax error - empty
	input = ":print(empty)"
	expected = ""
	assert_raises(Kwartz::SyntaxError) do
	    _test_stmt(input, expected)
	end
    end



    ## --------------------
    ## complex statement list #1
    ## --------------------
    def test_stmt91	##  complex statement list #1
	input = <<END
:macro(elem_foo)
  :foreach(user=user_list)
    :expand(stag_foo)
    :expand(cont_foo)
    :expand(etag_foo)
  :end
:end

:macro(stag_foo)
  :print(' <tr>\n')
:end

:macro(etag_foo)
  :print(' </tr>\n')
:end

:macro(cont_foo)
  :print('  <td>', user, '</td>\n')
:end

:print('<table>\n')
:expand(elem_foo)
:print('</table>\n')
END
	expected = <<'END'
<nodelist>
  :macro
    elem_foo
    <nodelist>
      :foreach
        =
          user
          user_list
        <nodelist>
          :expand
            stag_foo
          :expand
            cont_foo
          :expand
            etag_foo
  :macro
    stag_foo
    <nodelist>
      :print
        " <tr>\n"
  :macro
    etag_foo
    <nodelist>
      :print
        " </tr>\n"
  :macro
    cont_foo
    <nodelist>
      :print
        ,
          ,
            "  <td>"
            user
          "</td>\n"
  :print
    "<table>\n"
  :expand
    elem_foo
  :print
    "</table>\n"
END
	_test_stmt(input, expected)
    end

    ## --------------------
    ## complex statement list #1
    ## --------------------
    def test_stmt92	##  complex statement list #1
	input = <<'END'
:macro(elem_foo)
  :set(user_ctr=0)
  :foreach(user=user_list)
    :set(user_ctr+=1)
    :set(odd_even = user_ctr % 2 == 0 ? 'odd' : 'even')
    :expand(stag_foo)
    :expand(cont_foo)
    :expand(etag_foo)
  :end
:end

:macro(stag_foo)
  :print(" <tr class=\"", odd_even, "\">\n")
:end

:macro(etag_foo)
  :print(" </tr>\n")
:end

:macro(cont_foo)
  :print("  <td>", user, "</td>\n")
:end

:print("<table>\n")
:expand(elem_foo)
:print("</table>\n")

END
	expected = <<'END'
<nodelist>
  :macro
    elem_foo
    <nodelist>
      :set
        =
          user_ctr
          0
      :foreach
        =
          user
          user_list
        <nodelist>
          :set
            +=
              user_ctr
              1
          :set
            =
              odd_even
              ?
                ==
                  %
                    user_ctr
                    2
                  0
                "odd"
                "even"
          :expand
            stag_foo
          :expand
            cont_foo
          :expand
            etag_foo
  :macro
    stag_foo
    <nodelist>
      :print
        ,
          ,
            " <tr class=\""
            odd_even
          "\">\n"
  :macro
    etag_foo
    <nodelist>
      :print
        " </tr>\n"
  :macro
    cont_foo
    <nodelist>
      :print
        ,
          ,
            "  <td>"
            user
          "</td>\n"
  :print
    "<table>\n"
  :expand
    elem_foo
  :print
    "</table>\n"
END
	_test_stmt(input, expected)
    end


=begin
    ## --------------------
    ## 
    ## --------------------
    def test_stmtN	##  
	input = ''
	expected = <<END
END
	_test_stmt(input, expected)
    end
=end

end


## ====================
## TranslatorTest
## ====================
class TranslatorTest < Test::Unit::TestCase
    def _test_translate(input, expected, lang, flag_verbose=true)
	scanner = Kwartz::Scanner.new(input)
	parser  = Kwartz::Parser.new(scanner)
	translator = Kwartz::Translator.instance(lang, {}, scanner.newline)
	nodelist = parser.parse()
	code = translator.translate_all(nodelist)
	if flag_verbose then
	    assert_equal(expected, code)
	else
	    assert(code == expected)
	end
    end

    ## --------------------
    ## variable and number
    ## --------------------
    @@input01 = ":set(ctr=0)"
    def test_ruby01
	expected = "ctr = 0\n"
	_test_translate(@@input01, expected, 'ruby')
    end
    def test_php01
	expected = "<?php $ctr = 0; ?>\n"
	_test_translate(@@input01, expected, 'php')
    end
    def test_jsp01
        expected = '<c:set var="ctr" value="0"/>' + "\n"
        _test_translate(@@input01, expected, 'jsp')
    end
    def test_velocity01
        expected = "#set ($ctr = 0)\n"
        _test_translate(@@input01, expected, 'velocity')
    end

    ## --------------------
    ## term
    ## --------------------
    @@input02 = ":set(v=a+b*c/d-e)\n:set(v=-1*(-2))"
    def test_ruby02
	expected = "v = a + b * c / d - e\nv = ( - 1 * (( - 2)))\n"
	_test_translate(@@input02, expected, 'ruby')
    end
    def test_php02
	expected = <<END
<?php $v = $a + $b * $c / $d - $e; ?>
<?php $v = ( - 1 * (( - 2))); ?>
END
	_test_translate(@@input02, expected, 'php')
    end
    def test_jsp02
	expected = <<END
<c:set var="v" value="${a + b * c / d - e}"/>
<c:set var="v" value="${( - 1 * (( - 2)))}"/>
END
	_test_translate(@@input02, expected, 'jsp')
    end
    def test_velocity02
	expected = <<END
#set ($v = $a + $b * $c / $d - $e)
#set ($v = ( - 1 * (( - 2))))
END
	_test_translate(@@input02, expected, 'velocity')
    end

    ## --------------------
    ## array
    ## --------------------
    @@input03 = ":set(v=a[i]{'foo'}[:bar].baz)"
    def test_ruby03
	expected = 'v = a[i]["foo"][:bar].baz' + "\n"
	_test_translate(@@input03, expected, 'ruby')
    end
    def test_php03
	expected = "<?php $v = $a[$i]['foo']['bar']->baz; ?>\n"
	_test_translate(@@input03, expected, 'php')
    end
    def test_jsp03
	expected = '<c:set var="v" value="${a[i][\'foo\'][\'bar\'].baz}"/>' + "\n"
	_test_translate(@@input03, expected, 'jsp')
    end
    def test_velocity03
	expected = "#set ($v = $a.get($i).get('foo').bar.baz)\n"
	_test_translate(@@input03, expected, 'velocity')
    end

    ## --------------------
    ## string concat
    ## --------------------
    @@input04 = ":set(v=a[i] .+ b{'foo'} .+ c[:bar] .+ d.baz)"
    def test_ruby04
	expected = "v = a[i] + b[\"foo\"] + c[:bar] + d.baz\n"
	_test_translate(@@input04, expected, 'ruby')
    end
    def test_php04
	expected = "<?php $v = $a[$i] . $b['foo'] . $c['bar'] . $d->baz; ?>\n"
	_test_translate(@@input04, expected, 'php')
    end
    def test_jsp04
	expected = '<c:set var="v" value="${a[i]}${b[\'foo\']}${c[\'bar\']}${d.baz}"/>' + "\n"
	_test_translate(@@input04, expected, 'jsp')
    end
    def test_velocity04
	expected = '#set ($v = "${a.get($i)}${b.get(\'foo\')}${c.bar}${d.baz}")' + "\n"
	_test_translate(@@input04, expected, 'velocity')
    end

    ## --------------------
    ## conditional operator
    ## --------------------
    @@input05 = ":set(v = ctr%2==0 ? 'even' : 'odd')"
    def test_ruby05
	expected = 'v = (ctr % 2 == 0 ? "even" : "odd")' + "\n"
	_test_translate(@@input05, expected, 'ruby')
    end
    def test_php05
	expected = "<?php $v = ($ctr % 2 == 0 ? 'even' : 'odd'); ?>\n"
	_test_translate(@@input05, expected, 'php')
    end
    def test_jsp05
        expected = <<END
<c:choose>
<c:when test="${ctr % 2 == 0}">
 <c:set var="v" value="even"/>
</c:when>
<c:otherwise>
 <c:set var="v" value="odd"/>
</c:otherwise>
</c:choose>
END
	_test_translate(@@input05, expected, 'jsp')
    end
    def test_velocity05
	expected = <<END
#if ($ctr % 2 == 0)
  #set ($v = 'even')
#else
  #set ($v = 'odd')
#end
END
	_test_translate(@@input05, expected, 'velocity')
    end

    ## --------------------
    ## true/false/null
    ## --------------------
    #@@input06 = ":set(hash['key'] = obj==null? true : false)"
    @@input06 = ":set(hash = obj==null? true : false)"
    def test_ruby06
	#expected = "hash[\"key\"] = (obj == nil ? true : false)\n"
	expected = "hash = (obj == nil ? true : false)\n"
	_test_translate(@@input06, expected, 'ruby')
    end
    def test_php06
	#expected = "<?php $hash['key'] = ($obj == NULL ? TRUE : FALSE); ?>\n"
	expected = "<?php $hash = ($obj == NULL ? TRUE : FALSE); ?>\n"
	_test_translate(@@input06, expected, 'php')
    end
    def test_jsp06
#	expected = <<END
#<c:choose>
#<c:when test="${obj == null}">
# <c:set var="hash['key']" value="${true}"/>
#</c:when>
#<c:otherwise>
# <c:set var="hash['key']" value="${false}"/>
#</c:otherwise>
#</c:choose>
#END
	expected = <<END
<c:choose>
<c:when test="${obj == null}">
 <c:set var="hash" value="${true}"/>
</c:when>
<c:otherwise>
 <c:set var="hash" value="${false}"/>
</c:otherwise>
</c:choose>
END
	_test_translate(@@input06, expected, 'jsp')
    end
    def test_velocity06
#	expected = <<END
##if ((! $obj))
#  #set ($hash.key = true)
##else
#  #set ($hash.key = false)
##end
#END
	expected = <<END
#if (! $obj)
  #set ($hash = true)
#else
  #set ($hash = false)
#end
END
	#assert_raises(Kwartz::TranslationError) do
	#    _test_translate(@@input06, expected, 'velocity')
	#end
	_test_translate(@@input06, expected, 'velocity')
    end


    ## --------------------
    ## && ||
    ## --------------------
    @@input07 = ":set(v = !flag || s != null && s != '' ? 1 : 0)"
    def test_ruby07
	#expected = "v = (((! flag) || (s != nil && s != \"\")) ? 1 : 0)\n"
	expected = "v = (! flag || (s != nil && s != \"\") ? 1 : 0)\n"
	_test_translate(@@input07, expected, 'ruby')
    end
    def test_php07
	#expected = "<?php $v = (((! $flag) || ($s != NULL && $s != '')) ? 1 : 0); ?>\n"
	expected = "<?php $v = (! $flag || ($s != NULL && $s != '') ? 1 : 0); ?>\n"
	_test_translate(@@input07, expected, 'php')
    end
    def test_jsp07
	expected = <<END
<c:choose>
<c:when test="${not flag or (s != null and s != '')}">
 <c:set var="v" value="1"/>
</c:when>
<c:otherwise>
 <c:set var="v" value="0"/>
</c:otherwise>
</c:choose>
END
	_test_translate(@@input07, expected, 'jsp')
    end
    def test_velocity07
	expected = <<END
#if (! $flag || ($s && $s != ''))
  #set ($v = 1)
#else
  #set ($v = 0)
#end
END
	_test_translate(@@input07, expected, 'velocity')
    end



    ## --------------------
    ## operator priority
    ## --------------------
    @@input08 = ":print((1+2)*(3-4))"
    def test_ruby08
	expected = "print (1 + 2) * (3 - 4)\n"
	_test_translate(@@input08, expected, 'ruby')
    end
    def test_php08
	expected = "<?php echo (1 + 2) * (3 - 4); ?>"
	_test_translate(@@input08, expected, 'php')
    end
    def test_jsp08
	expected = '<c:out value="${(1 + 2) * (3 - 4)}" escapeXml="false"/>'
	_test_translate(@@input08, expected, 'jsp')
    end
    def test_velocity08
	#expected = '${(1 + 2) * (3 - 4)}'
	expected = nil
	assert_raises(Kwartz::TranslationError) do
	    _test_translate(@@input08, expected, 'velocity')
	end
    end

    ## --------------------
    ## operator priority
    ## --------------------
    @@input09 = <<END
:if(! (a || b) && c || d)
  :print('true\n')
:end
END
    def test_ruby09
#if (((! (a || b)) && c) || d) then
	expected = <<'END'
if (! (a || b) && c) || d then
  print "true\n"
end
END
	_test_translate(@@input09, expected, 'ruby')
    end
    def test_php09
#<?php if ((((! ($a || $b)) && $c) || $d)) { ?>
	expected = <<END
<?php if ((! ($a || $b) && $c) || $d) { ?>
true
<?php } ?>
END
	_test_translate(@@input09, expected, 'php')

    end
    def test_jsp09
#<c:when test="${(((not (a or b)) and c) or d)}">
	expected = <<END
<c:choose>
<c:when test="${(not (a or b) and c) or d}">
true
</c:when>
</c:choose>
END
	_test_translate(@@input09, expected, 'jsp')
    end
    def test_velocity09
#if ((((! ($a || $b)) && $c) || $d))
	expected = <<END
#if ((! ($a || $b) && $c) || $d)
true
#end
END
	_test_translate(@@input09, expected, 'velocity')
    end


    
    
    ## --------------------
    ## :print string
    ## --------------------
    @@input11 = ":print('foo', '.txt')"
    def test_ruby11
	expected = 'print "foo", ".txt"' + "\n"
	_test_translate(@@input11, expected, 'ruby')
    end
    def test_php11
	expected = 'foo.txt'
	_test_translate(@@input11, expected, 'php')
    end
    def test_jsp11
        expected = 'foo.txt'
        _test_translate(@@input11, expected, 'jsp')
    end
    def test_velocity11
        expected = 'foo.txt'
        _test_translate(@@input11, expected, 'velocity')
    end

    ## --------------------
    ## :print with several arguments
    ## --------------------
    @@input12 = ":print('foo', h[k1]{'k2'}.k3[:k4], 'bar')"
    def test_ruby12
	expected = 'print "foo", h[k1]["k2"].k3[:k4], "bar"' + "\n"
	_test_translate(@@input12, expected, 'ruby')
    end
    def test_php12
	expected = "foo<?php echo $h[$k1]['k2']->k3['k4']; ?>bar"
	_test_translate(@@input12, expected, 'php')
    end
    def test_jsp12
        expected = "foo<c:out value=\"${h[k1]['k2'].k3['k4']}\" escapeXml=\"false\"/>bar"
        _test_translate(@@input12, expected, 'jsp')
    end
    def test_velocity12
        expected = "foo$!{h.get($k1).get('k2').k3.k4}bar"
        _test_translate(@@input12, expected, 'velocity')
    end

    ## --------------------
    ## :print complex expression
    ## --------------------
    @@input13 = ":print(a .+ b)"
    def test_ruby13
	expected = "print a + b\n"
	_test_translate(@@input13, expected, 'ruby')
    end
    def test_php13
	expected = '<?php echo $a . $b; ?>'
	_test_translate(@@input13, expected, 'php')
    end
    def test_jsp13
        expected = '<c:out value="${a}${b}" escapeXml="false"/>'
        _test_translate(@@input13, expected, 'jsp')
    end
    def test_velocity13
        expected = nil
	assert_raises(Kwartz::TranslationError) do
	    _test_translate(@@input13, expected, 'velocity')
	end
    end

    ## --------------------
    ## :print number
    ## --------------------
    @@input14 = ":print(100)"
    def test_ruby14
	expected = "print 100\n"
	_test_translate(@@input14, expected, 'ruby')
    end
    def test_php14
	expected = '100'
	_test_translate(@@input14, expected, 'php')
    end
    def test_jsp14
        expected = '100'
        _test_translate(@@input14, expected, 'jsp')
    end
    def test_velocity14
        expected = '100'
	_test_translate(@@input14, expected, 'velocity')
    end

    ## --------------------
    ## :set
    ## --------------------
    @@input21 = ":set(var = a + 1)"
    def test_ruby21
	expected = "var = a + 1\n"
	_test_translate(@@input21, expected, 'ruby')
    end
    def test_php21
	expected = "<?php $var = $a + 1; ?>\n"
	_test_translate(@@input21, expected, 'php')
    end
    def test_jsp21
        expected = "<c:set var=\"var\" value=\"${a + 1}\"/>\n"
	_test_translate(@@input21, expected, 'jsp')
    end
    def test_velocity21
	expected = "#set ($var = $a + 1)\n"
	_test_translate(@@input21, expected, 'velocity')
    end

    ## --------------------
    ## :set (operator is '.+=')
    ## --------------------
    @@input22 = ":set(str .+= 'x')"
    def test_ruby22
	expected = 'str << "x"' + "\n"
	_test_translate(@@input22, expected, 'ruby')
    end
    def test_php22
	expected = "<?php $str .= 'x'; ?>\n"
	_test_translate(@@input22, expected, 'php')
    end
    def test_jsp22
        expected = "<c:set var=\"str\" value=\"${str}${'x'}\"/>\n"
	_test_translate(@@input22, expected, 'jsp')
    end
    def test_velocity22
	expected = "#set ($str = \"${str}x\")\n"
	_test_translate(@@input22, expected, 'velocity')
    end

    ## --------------------
    ## :while
    ## --------------------
    @@input23 = <<END
:while(a = dbh.fetch)
  :set(b .+= 'x')
:end	
END
    def test_ruby23
	expected = <<END
while a = dbh.fetch do
  b << "x"
end
END
	_test_translate(@@input23, expected, 'ruby')
    end
    def test_php23
	expected = <<END
<?php while ($a = $dbh->fetch) { ?>
  <?php $b .= 'x'; ?>
<?php } ?>
END
	_test_translate(@@input23, expected, 'php')
    end
    def test_jsp23
        expected = ''
        assert_raises(Kwartz::TranslationError) do
            _test_translate(@@input23, expected, 'jsp')
        end
    end
    def test_velocity23
        expected = ''
        assert_raises(Kwartz::TranslationError) do
            _test_translate(@@input23, expected, 'velocity')
        end
    end

    ## --------------------
    ## :foreach
    ## --------------------
    @@input24 = <<END
:foreach(user=user_list)
  :print(user)
:end
END
    def test_ruby24
	expected = <<END
for user in user_list do
  print user
end
END
	_test_translate(@@input24, expected, 'ruby')
    end
    def test_php24
	expected = <<END
<?php foreach ($user_list as $user) { ?>
<?php echo $user; ?><?php } ?>
END
	_test_translate(@@input24, expected, 'php')
    end
    def test_jsp24
        expected = <<END
<c:forEach var="user" items="${user_list}">
<c:out value="${user}" escapeXml="false"/></c:forEach>
END
	_test_translate(@@input24, expected, 'jsp')
    end
    def test_velocity24
        expected = <<END
#foreach ($user in $user_list)
$!{user}#end
END
	_test_translate(@@input24, expected, 'velocity')
    end

    ## --------------------
    ## :::
    ## --------------------
    def test_ruby25
        input = "::: for(i=0; i>a.length; i++) {"
	expected = " for(i=0; i>a.length; i++) {\n"
	_test_translate(input, expected, 'ruby')
    end
    def test_php25
        input  = "::: for($i=0; $i>$a.length; $i++) {"
        expected = " for($i=0; $i>$a.length; $i++) {\n"
	_test_translate(input, expected, 'php')
    end
    def test_jsp25
        input  = '::: <c:forEach var="i" begin="0" end="3">'
        expected = ' <c:forEach var="i" begin="0" end="3">' + "\n"
        _test_translate(input, expected, 'jsp')
    end
    def test_velocity25
        input  = '::: #foreach ($i = [0..3])'
        expected = " #foreach ($i = [0..3])\n"
        _test_translate(input, expected, 'velocity')
    end

    ## --------------------
    ## :rawcode
    ## --------------------
    def test_ruby26
	input = ":rawcode('for(i=0; i>a.length; i++) {')"
	expected = "for(i=0; i>a.length; i++) {\n"
	_test_translate(input, expected, 'ruby')
    end
    def test_php26
	input = ":rawcode('for($i=0; $i>$a.length; $i++) {')"
	expected = "for($i=0; $i>$a.length; $i++) {\n"
	_test_translate(input, expected, 'php')
    end
    def test_jsp26
        input  = ':rawcode(\'<c:forEach var="i" begin="0" end="3">\')'
        expected = '<c:forEach var="i" begin="0" end="3">' + "\n"
        _test_translate(input, expected, 'jsp')
    end
    def test_velocity26
        input  = ':rawcode(\'#foreach ($i = [0..3])\')'
        expected = '#foreach ($i = [0..3])' + "\n"
        _test_translate(input, expected, 'velocity')
    end

    
    ## --------------------
    ## :if
    ## --------------------
    @@input31 = <<END
:if(a>0)
  :print(a)
:end
END
    def test_ruby31
	expected = <<END
if a > 0 then
  print a
end
END
	_test_translate(@@input31, expected, 'ruby')
    end
    def test_php31
	expected = <<END
<?php if ($a > 0) { ?>
<?php echo $a; ?><?php } ?>
END
	_test_translate(@@input31, expected, 'php')
    end
    def test_jsp31
        expected = <<END
<c:choose>
<c:when test="${a > 0}">
<c:out value="${a}" escapeXml="false"/></c:when>
</c:choose>
END
	_test_translate(@@input31, expected, 'jsp')
    end
    def test_velocity31
        expected = <<END
#if ($a > 0)
$!{a}#end
END
	_test_translate(@@input31, expected, 'velocity')
    end

    ## --------------------
    ## :if, :else
    ## --------------------
    @@input32 = <<'END'
:if(a>0)
  :set(s='true')
:else
  :set(s='false')
:end
END
    def test_ruby32
	expected = <<'END'
if a > 0 then
  s = "true"
else
  s = "false"
end
END
	_test_translate(@@input32, expected, 'ruby')
    end
    def test_php32
	expected = <<'END'
<?php if ($a > 0) { ?>
  <?php $s = 'true'; ?>
<?php } else { ?>
  <?php $s = 'false'; ?>
<?php } ?>
END
	_test_translate(@@input32, expected, 'php')
    end
    def test_jsp32
        expected = <<'END'
<c:choose>
<c:when test="${a > 0}">
 <c:set var="s" value="true"/>
</c:when>
<c:otherwise>
 <c:set var="s" value="false"/>
</c:otherwise>
</c:choose>
END
	_test_translate(@@input32, expected, 'jsp')
    end
    def test_velocity32
        expected = <<'END'
#if ($a > 0)
  #set ($s = 'true')
#else
  #set ($s = 'false')
#end
END
	_test_translate(@@input32, expected, 'velocity')
    end

    ## --------------------
    ## :if, :elsif
    ## --------------------
    @@input33 = <<END
:if(a>100)
  :print(100)
:elsif(a>10)
  :print(10)
:elsif(a>0)
  :print(0)
:end
END
    def test_ruby33
	expected = <<END
if a > 100 then
  print 100
elsif a > 10 then
  print 10
elsif a > 0 then
  print 0
end
END
	_test_translate(@@input33, expected, 'ruby')
    end
    def test_php33
	expected = <<END
<?php if ($a > 100) { ?>
100<?php } elseif ($a > 10) { ?>
10<?php } elseif ($a > 0) { ?>
0<?php } ?>
END
	_test_translate(@@input33, expected, 'php')
    end
    def test_jsp33
        expected = <<'END'
<c:choose>
<c:when test="${a > 100}">
100</c:when>
<c:when test="${a > 10}">
10</c:when>
<c:when test="${a > 0}">
0</c:when>
</c:choose>
END
        _test_translate(@@input33, expected, 'jsp')
    end
    def test_velocity33
        expected = <<'END'
#if ($a > 100)
100#elseif ($a > 10)
10#elseif ($a > 0)
0#end
END
        _test_translate(@@input33, expected, 'velocity')
    end

    ## --------------------
    ## :if, :elsif, :else
    ## --------------------
    @@input34 = <<END
:if(a>100)
  :print(100)
:elsif(a>10)
  :print(10)
:elsif(a>0)
  :print(0)
:else
  :print('error')
:end
END
    def test_ruby34
	expected = <<END
if a > 100 then
  print 100
elsif a > 10 then
  print 10
elsif a > 0 then
  print 0
else
  print "error"
end
END
	_test_translate(@@input34, expected, 'ruby')
    end
    def test_php34
	expected = <<END
<?php if ($a > 100) { ?>
100<?php } elseif ($a > 10) { ?>
10<?php } elseif ($a > 0) { ?>
0<?php } else { ?>
error<?php } ?>
END
	_test_translate(@@input34, expected, 'php')
    end
    def test_jsp34
        expected = <<END
<c:choose>
<c:when test="${a > 100}">
100</c:when>
<c:when test="${a > 10}">
10</c:when>
<c:when test="${a > 0}">
0</c:when>
<c:otherwise>
error</c:otherwise>
</c:choose>
END
	_test_translate(@@input34, expected, 'jsp')
    end
    def test_velocity34
        expected = <<'END'
#if ($a > 100)
100#elseif ($a > 10)
10#elseif ($a > 0)
0#else
error#end
END
        _test_translate(@@input34, expected, 'velocity')
    end

    ## --------------------
    ## nested :if
    ## --------------------
    @@input35 = <<'END'
:if(x>0)
  :if(y>0)
    :if(z>0)
      :print("x*y*z > 0\n")
    :elsif(z==0)
      :print("x*y*z == 0\n")
    :else
      :print("x*y*z < 0\n")
    :end
  :end
:end
END
    def test_ruby35
	expected = <<'END'
if x > 0 then
  if y > 0 then
    if z > 0 then
      print "x*y*z > 0\n"
    elsif z == 0 then
      print "x*y*z == 0\n"
    else
      print "x*y*z < 0\n"
    end
  end
end
END
	_test_translate(@@input35, expected, 'ruby')
    end
    def test_php35
	expected = <<'END'
<?php if ($x > 0) { ?>
  <?php if ($y > 0) { ?>
    <?php if ($z > 0) { ?>
x*y*z > 0
    <?php } elseif ($z == 0) { ?>
x*y*z == 0
    <?php } else { ?>
x*y*z < 0
    <?php } ?>
  <?php } ?>
<?php } ?>
END
	_test_translate(@@input35, expected, 'php')
    end
    def test_jsp35
        expected = <<END
<c:choose>
<c:when test="${x > 0}">
 <c:choose>
<c:when test="${y > 0}">
  <c:choose>
<c:when test="${z > 0}">
x*y*z > 0
  </c:when>
  <c:when test="${z == 0}">
x*y*z == 0
  </c:when>
  <c:otherwise>
x*y*z < 0
  </c:otherwise>
  </c:choose>
 </c:when>
 </c:choose>
</c:when>
</c:choose>
END
	_test_translate(@@input35, expected, 'jsp')
    end
    def test_velocity35
        expected = <<END
#if ($x > 0)
  #if ($y > 0)
    #if ($z > 0)
x*y*z > 0
    #elseif ($z == 0)
x*y*z == 0
    #else
x*y*z < 0
    #end
  #end
#end
END
	_test_translate(@@input35, expected, 'velocity')
    end

    ## --------------------
    ## multiple conditional operator ('? :')
    ## --------------------
    @@input36 = <<'END'
:print(v1 ? a1 : a2, v2.=='x' ? 'b1' : 'b2')
END
    def test_ruby36
        expected = <<'END'
print (v1 ? a1 : a2), (v2 == "x" ? "b1" : "b2")
END
	_test_translate(@@input36, expected, 'ruby')
    end
    def test_php36
        expected = "<?php echo ($v1 ? $a1 : $a2); ?><?php echo ($v2 == 'x' ? 'b1' : 'b2'); ?>"
	_test_translate(@@input36, expected, 'php')
    end
    def test_jsp36
        expected = <<'END'
<c:choose>
<c:when test="${v1}">
 <c:choose>
<c:when test="${v2 == 'x'}">
<c:out value="${a1}" escapeXml="false"/>b1</c:when>
 <c:otherwise>
<c:out value="${a1}" escapeXml="false"/>b2</c:otherwise>
 </c:choose>
</c:when>
<c:otherwise>
 <c:choose>
<c:when test="${v2 == 'x'}">
<c:out value="${a2}" escapeXml="false"/>b1</c:when>
 <c:otherwise>
<c:out value="${a2}" escapeXml="false"/>b2</c:otherwise>
 </c:choose>
</c:otherwise>
</c:choose>
END
	_test_translate(@@input36, expected, 'jsp')
    end
    def test_velocity36
        expected = <<'END'
#if ($v1)
  #if ($v2 == 'x')
$!{a1}b1#else
$!{a1}b2#end
#else
  #if ($v2 == 'x')
$!{a2}b1#else
$!{a2}b2#end
#end
END
	_test_translate(@@input36, expected, 'velocity')
    end


    ## --------------------
    ## :macro, :expand
    ## --------------------
    @@input41 = <<END
:macro(foo)
  :print(100)
:end
:expand(foo)
END
    def test_ruby41
	expected = "print 100\n"
	_test_translate(@@input41, expected, 'ruby')
    end
    def test_php41
	expected = '100'
	_test_translate(@@input41, expected, 'php')
    end
    def test_jsp41
	expected = '100'
        _test_translate(@@input41, expected, 'jsp')
    end
    def test_velocity41
	expected = '100'
	_test_translate(@@input41, expected, 'velocity')
    end

    ## --------------------
    ## :macro, :expand (nested)
    ## --------------------
    @@input42 = <<'END'
:macro(child)
  :set(v=100)
:end
:macro(parent)
  :expand(child)
:end
:expand(parent)
END
    def test_ruby42
	expected = "v = 100\n"
	_test_translate(@@input42, expected, 'ruby')
    end
    def test_php42
	expected = "<?php $v = 100; ?>\n"
	_test_translate(@@input42, expected, 'php')
    end
    def test_jsp42
	expected = '<c:set var="v" value="100"/>' + "\n"
	_test_translate(@@input42, expected, 'jsp')
    end
    def test_velocity42
	expected = "#set ($v = 100)\n"
	_test_translate(@@input42, expected, 'velocity')
    end

    ## --------------------
    ## overwrite :macro
    ## --------------------
    @@input43 = <<'END'
:macro(foo)
  :print(100)
:end
:expand(foo)
:macro(foo)
  :set(v=100)
:end
END
    def test_ruby43
	expected = "v = 100\n"
	_test_translate(@@input43, expected, 'ruby')
    end
    def test_php43
	expected = "<?php $v = 100; ?>\n"
	_test_translate(@@input43, expected, 'php')
    end
    def test_jsp43
	expected = '<c:set var="v" value="100"/>' + "\n"
	_test_translate(@@input43, expected, 'jsp')
    end
    def test_velocity43
	expected = "#set ($v = 100)\n"
	_test_translate(@@input43, expected, 'velocity')
    end


    ## --------------------
    ## @foo  (:expand2)
    ## --------------------
    @@input44 = <<'END'
:macro(foo)
  :print(100)
:end
@foo
:macro(foo)
  :set(v=100)
:end
END
    def test_ruby44
	expected = "v = 100\n"
	_test_translate(@@input44, expected, 'ruby')
    end
    def test_php44
	expected = "<?php $v = 100; ?>\n"
	_test_translate(@@input44, expected, 'php')
    end
    def test_jsp44
	expected = '<c:set var="v" value="100"/>' + "\n"
	_test_translate(@@input44, expected, 'jsp')
    end
    def test_velocity44
	expected = "#set ($v = 100)\n"
	_test_translate(@@input44, expected, 'velocity')
    end


    ## --------------------
    ## raise error if unknown macro found
    ## --------------------
    @@input45 = <<'END'
:macro(foo)
  :print(100)
:end
@foo2
END
    def test_ruby45
	expected = nil
	assert_raises(Kwartz::TranslationError) do
	    _test_translate(@@input45, expected, 'ruby')
	end
    end
    def test_php45
	expected = nil
	assert_raises(Kwartz::TranslationError) do
	    _test_translate(@@input45, expected, 'php')
	end
    end
    def test_jsp45
	expected = nil
	assert_raises(Kwartz::TranslationError) do
	    _test_translate(@@input45, expected, 'jsp')
	end
    end
    def test_velocity45
	expected = nil
	assert_raises(Kwartz::TranslationError) do
	    _test_translate(@@input45, expected, 'velocity')
	end
    end




#    ## --------------------
#    ## sanitizing
#    ## --------------------
#    @@input51 = <<'END'
#:print('<span ', flag ? aaa : bbb, '>\n')
#:print('<span ', flag ? 's' : bbb, '>\n')
#:print('<span ', flag ? aaa : 's', '>\n')
#:print('<span ', flag ? 's' : 'S', '>\n')
#:print('\n')
#:print('<span ', flag ? aaa : bbb, '>\n')
#:print('<span ', flag ? 100 : bbb, '>\n')
#:print('<span ', flag ? aaa : 100, '>\n')
#:print('<span ', flag ? 100 : 200, '>\n')
#:print('\n')
#:print('<span ', flag1 ? 'x' : (flag2 ? 'y' : 'z'), '>\n')
#:print('<span ', flag1 ? 'x' : (flag2 ? aaa : 'z'), '>\n')
#:print('<span ', flag1 ? (flag2 ? 'x' : 'y') : 'z', '>\n')
#:print('<span ', flag1 ? (flag2 ? 'x' : aaa) : 'z', '>\n')
#END
#
#    def test_rubys51
#	expected = <<'END'
#print "<span ", CGI.escapeHTML((flag ? aaa : bbb)), ">\n"
#print "<span ", CGI.escapeHTML((flag ? "s" : bbb)), ">\n"
#print "<span ", CGI.escapeHTML((flag ? aaa : "s")), ">\n"
#print "<span ", (flag ? "s" : "S"), ">\n"
#print "\n"
#print "<span ", CGI.escapeHTML((flag ? aaa : bbb)), ">\n"
#print "<span ", CGI.escapeHTML((flag ? 100 : bbb)), ">\n"
#print "<span ", CGI.escapeHTML((flag ? aaa : 100)), ">\n"
#print "<span ", (flag ? 100 : 200), ">\n"
#print "\n"
#print "<span ", (flag1 ? "x" : (flag2 ? "y" : "z")), ">\n"
#print "<span ", CGI.escapeHTML((flag1 ? "x" : (flag2 ? aaa : "z"))), ">\n"
#print "<span ", (flag1 ? (flag2 ? "x" : "y") : "z"), ">\n"
#print "<span ", CGI.escapeHTML((flag1 ? (flag2 ? "x" : aaa) : "z")), ">\n"
#END
#	_test_translate(@@input51, expected, 'ruby.s')
#    end
#
#    def test_erubys51
#	expected = <<'END'
#<span <%= CGI.escapeHTML((flag ? aaa : bbb)) %>>
#<span <%= CGI.escapeHTML((flag ? 's' : bbb)) %>>
#<span <%= CGI.escapeHTML((flag ? aaa : 's')) %>>
#<span <%= (flag ? 's' : 'S') %>>
#
#<span <%= CGI.escapeHTML((flag ? aaa : bbb)) %>>
#<span <%= CGI.escapeHTML((flag ? 100 : bbb)) %>>
#<span <%= CGI.escapeHTML((flag ? aaa : 100)) %>>
#<span <%= (flag ? 100 : 200) %>>
#
#<span <%= (flag1 ? 'x' : (flag2 ? 'y' : 'z')) %>>
#<span <%= CGI.escapeHTML((flag1 ? 'x' : (flag2 ? aaa : 'z'))) %>>
#<span <%= (flag1 ? (flag2 ? 'x' : 'y') : 'z') %>>
#<span <%= CGI.escapeHTML((flag1 ? (flag2 ? 'x' : aaa) : 'z')) %>>
#END
#	_test_translate(@@input51, expected, 'eruby.s')
#    end
#
#    def test_erbs51
#	expected = <<'END'
#<span <%= ERB::Util.h((flag ? aaa : bbb)) %>>
#<span <%= ERB::Util.h((flag ? 's' : bbb)) %>>
#<span <%= ERB::Util.h((flag ? aaa : 's')) %>>
#<span <%= (flag ? 's' : 'S') %>>
#
#<span <%= ERB::Util.h((flag ? aaa : bbb)) %>>
#<span <%= ERB::Util.h((flag ? 100 : bbb)) %>>
#<span <%= ERB::Util.h((flag ? aaa : 100)) %>>
#<span <%= (flag ? 100 : 200) %>>
#
#<span <%= (flag1 ? 'x' : (flag2 ? 'y' : 'z')) %>>
#<span <%= ERB::Util.h((flag1 ? 'x' : (flag2 ? aaa : 'z'))) %>>
#<span <%= (flag1 ? (flag2 ? 'x' : 'y') : 'z') %>>
#<span <%= ERB::Util.h((flag1 ? (flag2 ? 'x' : aaa) : 'z')) %>>
#END
#	_test_translate(@@input51, expected, 'erb.s')
#    end
#
#    def test_phps51
#	expected = <<'END'
#<span <?php echo htmlspecialchars(($flag ? $aaa : $bbb)); ?>>
#<span <?php echo htmlspecialchars(($flag ? 's' : $bbb)); ?>>
#<span <?php echo htmlspecialchars(($flag ? $aaa : 's')); ?>>
#<span <?php echo ($flag ? 's' : 'S'); ?>>
#
#<span <?php echo htmlspecialchars(($flag ? $aaa : $bbb)); ?>>
#<span <?php echo htmlspecialchars(($flag ? 100 : $bbb)); ?>>
#<span <?php echo htmlspecialchars(($flag ? $aaa : 100)); ?>>
#<span <?php echo ($flag ? 100 : 200); ?>>
#
#<span <?php echo ($flag1 ? 'x' : ($flag2 ? 'y' : 'z')); ?>>
#<span <?php echo htmlspecialchars(($flag1 ? 'x' : ($flag2 ? $aaa : 'z'))); ?>>
#<span <?php echo ($flag1 ? ($flag2 ? 'x' : 'y') : 'z'); ?>>
#<span <?php echo htmlspecialchars(($flag1 ? ($flag2 ? 'x' : $aaa) : 'z')); ?>>
#END
#	_test_translate(@@input51, expected, 'php.s')
#    end


    ## --------------------
    ## empty
    ## --------------------
    @@input61 = <<'END'
:if(s1 == empty && s2 != empty)
  :print("OK\n")
:end
END
    def test_ruby61
	expected = <<'END'
if (s1 == nil || s1 == "") && (s2 != nil && s2 != "") then
  print "OK\n"
end
END
	_test_translate(@@input61, expected, 'ruby')
    end
    def test_php61
	expected = <<'END'
<?php if (! $s1 && $s2) { ?>
OK
<?php } ?>
END
	_test_translate(@@input61, expected, 'php')
    end
    def test_jsp61
	expected = <<'END'
<c:choose>
<c:when test="${(empty s1) and !(empty s2)}">
OK
</c:when>
</c:choose>
END
	_test_translate(@@input61, expected, 'jsp')
    end
    def test_velocity61
	expected = <<'END'
#if ((! $s1 || $s1 == '') && ($s2 && $s2 != ''))
OK
#end
END
	_test_translate(@@input61, expected, 'velocity')
    end



    ## --------------------
    ## BEGIN / END macro
    ## --------------------
    @@input62 = <<'END'
:print("foo\n")
:macro(BEGIN)
  :print("--- begin ---\n")
:end
:macro(END)
  :print("--- end ---\n")
:end
END
    def test_ruby62
	expected = <<'END'
print "--- begin ---\n"
print "foo\n"
print "--- end ---\n"
END
	_test_translate(@@input62, expected, 'ruby')
    end
    def test_php62
	expected = <<'END'
--- begin ---
foo
--- end ---
END
	_test_translate(@@input62, expected, 'php')
    end
    def test_jsp62
	expected = <<'END'
--- begin ---
foo
--- end ---
END
	_test_translate(@@input62, expected, 'jsp')
    end
    def test_velocity62
	expected = <<'END'
--- begin ---
foo
--- end ---
END
	_test_translate(@@input62, expected, 'velocity')
    end


    
    ## --------------------
    ## complex input test
    ## --------------------
    @@input91 = <<'END'
:set(user_ctr=0)
:foreach(user=user_list)
  :set(user_ctr += 1)
  :set(klass = user_ctr % 2 == 0 ? 'even' : 'odd')
  :print("<tr class=\"", klass, "\">\n")
  :print("  <td>", user, "</td>\n")
  :print("</tr>\n")
:end
END
    @@input91crlf = @@input91.gsub(/\n/, "\r\n").gsub(/\\n/, '\r\n')

    def test_ruby91
	expected = <<'END'
user_ctr = 0
for user in user_list do
  user_ctr += 1
  klass = (user_ctr % 2 == 0 ? "even" : "odd")
  print "<tr class=\"", klass, "\">\n"
  print "  <td>", user, "</td>\n"
  print "</tr>\n"
end
END
	_test_translate(@@input91, expected, 'ruby')
	expected.gsub!(/\\n/, '\r\n')
	_test_translate(@@input91crlf, expected.gsub(/\n/,"\r\n"), 'ruby')
    end

    def test_php91
	expected = <<'END'
<?php $user_ctr = 0; ?>
<?php foreach ($user_list as $user) { ?>
  <?php $user_ctr += 1; ?>
  <?php $klass = ($user_ctr % 2 == 0 ? 'even' : 'odd'); ?>
<tr class="<?php echo $klass; ?>">
  <td><?php echo $user; ?></td>
</tr>
<?php } ?>
END
	_test_translate(@@input91, expected, 'php')
	_test_translate(@@input91crlf, expected.gsub(/\n/,"\r\n"), 'php')
    end

    def test_jsp91
        expected = <<'END'
<c:set var="user_ctr" value="0"/>
<c:forEach var="user" items="${user_list}">
 <c:set var="user_ctr" value="${user_ctr + 1}"/>
 <c:choose>
<c:when test="${user_ctr % 2 == 0}">
  <c:set var="klass" value="even"/>
 </c:when>
 <c:otherwise>
  <c:set var="klass" value="odd"/>
 </c:otherwise>
 </c:choose>
<tr class="<c:out value="${klass}" escapeXml="false"/>">
  <td><c:out value="${user}" escapeXml="false"/></td>
</tr>
</c:forEach>
END
	_test_translate(@@input91, expected, 'jsp')
	_test_translate(@@input91crlf, expected.gsub(/\n/,"\r\n"), 'jsp')
    end
    
    def test_velocity91
        expected = <<'END'
#set ($user_ctr = 0)
#foreach ($user in $user_list)
  #set ($user_ctr = $user_ctr + 1)
  #if ($user_ctr % 2 == 0)
    #set ($klass = 'even')
  #else
    #set ($klass = 'odd')
  #end
<tr class="$!{klass}">
  <td>$!{user}</td>
</tr>
#end
END
	_test_translate(@@input91, expected, 'velocity')
	_test_translate(@@input91crlf, expected.gsub(/\n/,"\r\n"), 'velocity')
    end

    ## --------------------
    ## complex :macro, :expand
    ## --------------------
    @@input92 = <<'END'
:macro(stag_user)
  :print("<tr>\n")
:end
:macro(cont_user)
  :print(" <td>", user, "</td>\n")
:end
:macro(etag_user)
  :print("</tr>\n")
:end
:macro(elem_user)
  :foreach(user=user_list)
    :expand(stag_user)
    :expand(cont_user)
    :expand(etag_user)
  :end
:end
:expand(elem_user)
END
    @@input92crlf = @@input92.gsub(/\n/, "\r\n").gsub(/\\n/, '\r\n')

    def test_ruby92
	expected = <<'END'
for user in user_list do
  print "<tr>\n"
  print " <td>", user, "</td>\n"
  print "</tr>\n"
end
END
	_test_translate(@@input92, expected, 'ruby')
	expected.gsub!(/\\n/, '\r\n')
	_test_translate(@@input92crlf, expected.gsub(/\n/,"\r\n"), 'ruby')
    end

    def test_php92
	expected = <<'END'
<?php foreach ($user_list as $user) { ?>
<tr>
 <td><?php echo $user; ?></td>
</tr>
<?php } ?>
END
	_test_translate(@@input92, expected, 'php')
	_test_translate(@@input92crlf, expected.gsub(/\n/,"\r\n"), 'php')
    end

    def test_jsp92
        expected = <<'END'
<c:forEach var="user" items="${user_list}">
<tr>
 <td><c:out value="${user}" escapeXml="false"/></td>
</tr>
</c:forEach>
END
	_test_translate(@@input92, expected, 'jsp')
	_test_translate(@@input92crlf, expected.gsub(/\n/,"\r\n"), 'jsp')
    end

    def test_velocity92
        expected = <<'END'
#foreach ($user in $user_list)
<tr>
 <td>$!{user}</td>
</tr>
#end
END
	_test_translate(@@input92, expected, 'velocity')
	_test_translate(@@input92crlf, expected.gsub(/\n/,"\r\n"), 'velocity')
    end

end


## ====================
## Converter test
## ====================
class ConverterTest < Test::Unit::TestCase
    def _test_convert(input, expected, toppings={}, flag_verbose=true)
    	#toppings[:include_path] =  ['.', 'test.d']
	converter = Kwartz::DefaultConverter.new(input, toppings)
	output = converter.convert()
	if flag_verbose then
	    assert_equal(expected, output)
	else
	    assert(expected == output)
	end
    end

    ## --------------------
    ## simple text
    ## --------------------
    def test_convert01	##  simple text
	input = 'hoge\r\n'
	expected = ':print("hoge\r\n")' + "\n"
	_test_convert(input, expected)
    end

    ## --------------------
    ## html without id-command
    ## --------------------
    def test_convert02	##  html without id-command
	input = <<'END'
  <tr class=odd>
    <td>user</td>
  </tr>
END
	expected = <<'END'
:print("  <tr class=odd>\n")
:print("    <td>user</td>\n")
:print("  </tr>\n")
END
	_test_convert(input, expected)
    end

    ## --------------------
    ## #{expr}#
    ## --------------------
    def test_convert03	##  #{expr}#
	input = <<'END'
<tr class=#{class}#>
  <td>#{user}#</td>
</tr>
END
	expected = <<'END'
:print("<tr class=", class, ">\n")
:print("  <td>", user, "</td>\n")
:print("</tr>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## tag width extra space
    ## --------------------
    def test_convert04	##  tag width extra space
	input = <<'END'
<tr >
  <td>user</td>
</tr >
END
	expected = <<'END'
:print("<tr>\n")
:print("  <td>user</td>\n")
:print("</tr>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## single tag
    ## --------------------
    def test_convert05	##  single tag
	input = <<'END'
<hr width="#{width}#%"/>
END
	expected = <<'END'
:print("<hr width=\"", width, "%\"/>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## #{ cond ? true-value : false-value }#
    ## --------------------
    def test_convert06	##  #{ cond ? true-value : false-value }#
	input = <<'END'
foo#{cond?'aaa':'bbb'}#bar
END
	expected = <<'END'
:print("foo", cond?'aaa':'bbb', "bar\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## foo'bar'baz
    ## --------------------
    def test_convert07	##  foo'bar'baz
	input = "foo'bar\"baz"
	expected = ':print("foo\'bar\"baz")' + "\n"
	_test_convert(input, expected)
    end


    ## --------------------
    ## #{@CHECK(flag)}#, #{@SELECT(flag)}#, #{@DIABLE(flag)}#, 
    ## --------------------
    def test_convert08	##  #{@CHECK(flag)}#, #{@SELECT(flag)}#, #{@DIABLE(flag)}#, 
	input = <<'END'
<input #{@CHECK(param[:gender]=='M')}#>
<option #{@SELECT(amount>100)}#>
<input #{@DISABLE(age<18)}#>
END
	expected = <<'END'
:print("<input ", X(param[:gender]=='M' ? ' checked="checked"' : ''), ">\n")
:print("<option ", X(amount>100 ? ' selected="selected"' : ''), ">\n")
:print("<input ", X(age<18 ? ' disabled="disabled"' : ''), ">\n")
END
	_test_convert(input, expected)
    end

    ## --------------------
    ## bug reported by Peter Palfrader
    ## --------------------
    def test_convert09	##  bug reported by Peter Palfrader
	input = <<'END'
<option value="foo" #{flag?'selected':''}#>
<foo kd="value:foo">foo</foo>
END
	expected = <<'END'
:print("<option value=\"foo\" ", flag?'selected':'', ">\n")
:print("<foo>", foo, "</foo>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## bug reported by Andreas Schwartz
    ## --------------------
    def test_convert10	##  bug reported by Andreas Schwartz
	input = <<'END'
<foo id="hoge">
  <foo>hogehoge</foo>
</foo>
<foo id="geji">
  <foo>
    <foo>
      geji<foo/>geji
    </foo>
  </foo>
</foo>
END
	expected = <<'END'
:macro(stag_hoge)
  :print("<foo id=\"hoge\">\n")
:end

:macro(cont_hoge)
  :print("  <foo>hogehoge</foo>\n")
:end

:macro(etag_hoge)
  :print("</foo>\n")
:end

:macro(elem_hoge)
  :expand(stag_hoge)
  :expand(cont_hoge)
  :expand(etag_hoge)
:end

:macro(stag_geji)
  :print("<foo id=\"geji\">\n")
:end

:macro(cont_geji)
  :print("  <foo>\n")
  :print("    <foo>\n")
  :print("      geji<foo/>geji\n")
  :print("    </foo>\n")
  :print("  </foo>\n")
:end

:macro(etag_geji)
  :print("</foo>\n")
:end

:macro(elem_geji)
  :expand(stag_geji)
  :expand(cont_geji)
  :expand(etag_geji)
:end

:expand(elem_hoge)
:expand(elem_geji)
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="attr:name=value"
    ## --------------------
    def test_convert11	##  kd="attr:name=value"
	input = <<'END'
  <tr class=odd style=color:#000000
      kd="attr:class=klass">
  </tr>
END
	expected = <<'END'
:print("  <tr class=\"", klass, "\" style=color:#000000>\n")
:print("  </tr>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="attr:a1=v1;attr:a2=v2"
    ## --------------------
    def test_convert12	##  kd="attr:a1=v1;attr:a2=v2"
	input = <<'END'
  <tr class=odd style=color:#000000
      kd="attr:class=klass;attr:bgcolor=a[:bg]">
  </tr>
END
#	expected = <<'END'
#:print('  <tr class="#{klass}#" style=color:#000000 bgcolor="#{a[:bg]}#">\n')
#:print('  </tr>\n')
#END
	expected = <<'END'
:print("  <tr class=\"", klass, "\" style=color:#000000 bgcolor=\"", a[:bg], "\">\n")
:print("  </tr>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="attr:name=value" with single-tag
    ## --------------------
    def test_convert13	##  kd="attr:name=value" with single-tag
	input = <<'END'
  <hr width="100" size="2" kd="attr:width=width;attr:size=size"/>
END
	expected = <<'END'
:print("  <hr width=\"", width, "\" size=\"", size, "\"/>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="attr:name=cond?v1:v2"
    ## --------------------
    def test_convert14	##  kd="attr:name=cond?v1:v2"
	input = <<'END'
<foo kd="attr:class=cond==0?'aaa':'bbb'"/>
END
	expected = <<'END'
:print("<foo class=\"", cond==0?'aaa':'bbb', "\"/>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="attr:namespace:name=expr"
    ## --------------------
    def test_convert15	##  kd="attr:namespace:name=expr"
	input = <<'END'
<c:hoge kd="if:cond">
 <a ccc:href="foo" hhhref="foo" c:href="bar" href="baz" kd="attr:c:href=var">aaa</a>
</c:hoge>
END
	expected = <<'END'
:if(cond)
  :print("<c:hoge>\n")
  :print(" <a ccc:href=\"foo\" hhhref=\"foo\" c:href=\"", var, "\" href=\"baz\">aaa</a>\n")
  :print("</c:hoge>\n")
:end
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="value:expr"
    ## --------------------
    def test_convert21	##  kd="value:expr"
	input = <<'END'
<tr>
 <td kd="value:user">hoo</td>
</tr>
END
	expected = <<'END'
:print("<tr>\n")
:print(" <td>", user, "</td>\n")
:print("</tr>\n")
END
	_test_convert(input, expected)
    end

    ## --------------------
    ## kd="value:expr;attr:name=expr"
    ## --------------------
    def test_convert22	##  kd="value:expr;attr:name=expr"
	input = <<'END'
<tr>
  <td kd="value:user;attr:bgcolor=bgkolor">
   hoo
  </td>
</tr>
END
	expected = <<'END'
:print("<tr>\n")
:print("  <td bgcolor=\"", bgkolor, "\">\n", user, "  </td>\n")
:print("</tr>\n")
END
	_test_convert(input, expected)
    end

    ## --------------------
    ## kd="value:expr" with single-tag
    ## --------------------
    def test_convert23	##  kd="value:expr" with single-tag
	input = <<'END'
<tr>
  <td kd="value:user;attr:bgcolor=bgkolor"/>
</tr>
END
	expected = <<'END'
:print("<tr>\n")
:print("  <td bgcolor=\"", bgkolor, "\">", user, "</td>\n")
:print("</tr>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="value:cond?val1:val2"
    ## --------------------
    def test_convert24	##  kd="value:cond?val1:val2"
	input = <<'END'
<tr>
  <td kd="value:ctr%2==0:'odd':'even'">foo</td>
</tr>
END
	expected = <<'END'
:print("<tr>\n")
:print("  <td>", ctr%2==0:'odd':'even', "</td>\n")
:print("</tr>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="mark:name"
    ## --------------------
    def test_convert31	##  kd="mark:name"
	input = <<'END'
<tr kd="mark:user">
  <td>foo</td>
</tr>
END
	expected = <<'END'
:macro(stag_user)
  :print("<tr>\n")
:end

:macro(cont_user)
  :print("  <td>foo</td>\n")
:end

:macro(etag_user)
  :print("</tr>\n")
:end

:macro(elem_user)
  :expand(stag_user)
  :expand(cont_user)
  :expand(etag_user)
:end

:expand(elem_user)
END
	_test_convert(input, expected)
    end

    ## --------------------
    ## nested kd="mark:name"
    ## --------------------
    def test_convert32	##  nested kd="mark:name"
	input = <<'END'
<ul kd="mark:parent">
  <li kd="mark:child">foo</li>
</ul>
END
	expected = <<'END'
:macro(stag_child)
  :print("  <li>")
:end

:macro(cont_child)
  :print("foo")
:end

:macro(etag_child)
  :print("</li>\n")
:end

:macro(elem_child)
  :expand(stag_child)
  :expand(cont_child)
  :expand(etag_child)
:end

:macro(stag_parent)
  :print("<ul>\n")
:end

:macro(cont_parent)
  :expand(elem_child)
:end

:macro(etag_parent)
  :print("</ul>\n")
:end

:macro(elem_parent)
  :expand(stag_parent)
  :expand(cont_parent)
  :expand(etag_parent)
:end

:expand(elem_parent)
END
	_test_convert(input, expected)
    end

    ## --------------------
    ## kd="mark:name" and kd="value:expr"
    ## --------------------
    def test_convert33	##  kd="mark:name" and kd="value:expr"
	input = <<'END'
<tr kd="mark:user">
  <td kd="value:user">foo</td>
</tr>
END
	expected = <<'END'
:macro(stag_user)
  :print("<tr>\n")
:end

:macro(cont_user)
  :print("  <td>", user, "</td>\n")
:end

:macro(etag_user)
  :print("</tr>\n")
:end

:macro(elem_user)
  :expand(stag_user)
  :expand(cont_user)
  :expand(etag_user)
:end

:expand(elem_user)
END
	_test_convert(input, expected)
    end

    ## --------------------
    ## kd="mark:name;attr:name=expr" and kd="value:expr"
    ## --------------------
    def test_convert34	##  kd="mark:name;attr:name=expr" and kd="value:expr"
	input = <<'END'
<table>
 <tr class="odd" kd="mark:user;attr:class=klass">
  <td kd="value:user">foo</td>
 </tr>
</table>
END
	expected = <<'END'
:macro(stag_user)
  :print(" <tr class=\"", klass, "\">\n")
:end

:macro(cont_user)
  :print("  <td>", user, "</td>\n")
:end

:macro(etag_user)
  :print(" </tr>\n")
:end

:macro(elem_user)
  :expand(stag_user)
  :expand(cont_user)
  :expand(etag_user)
:end

:print("<table>\n")
:expand(elem_user)
:print("</table>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="mark:name" with single-tag
    ## --------------------
    def test_convert35	##  kd="mark:name" with single-tag
	input = <<'END'
 <tr>
  <td kd="mark:user"/>
 </tr>
END
	expected = <<'END'
:macro(stag_user)
  :print("  <td/>\n")
:end

:macro(cont_user)
  ## empty
:end

:macro(etag_user)
  ## empty
:end

:macro(elem_user)
  :expand(stag_user)
  :expand(cont_user)
  :expand(etag_user)
:end

:print(" <tr>\n")
:expand(elem_user)
:print(" </tr>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## id="..."
    ## --------------------
    def test_convert36	##  id="..."
	input = <<'END'
<ul id="foo">
  <li>foo</li>
</ul>
END
	expected = <<'END'
:macro(stag_foo)
  :print("<ul id=\"foo\">\n")
:end

:macro(cont_foo)
  :print("  <li>foo</li>\n")
:end

:macro(etag_foo)
  :print("</ul>\n")
:end

:macro(elem_foo)
  :expand(stag_foo)
  :expand(cont_foo)
  :expand(etag_foo)
:end

:expand(elem_foo)
END
	_test_convert(input, expected)
    end
    

    ## --------------------
    ## id="..." and kd="..."
    ## --------------------
    def test_convert37	##  id="..." and kd="..."
	input = <<'END'
<ul>
  <li id="foo" kd="attr:class=klass;set:klass='odd'">foo</li>
</ul>
END
	expected = <<'END'
:print("<ul>\n")
:set(klass='odd')
:print("  <li id=\"foo\" class=\"", klass, "\">")
:print("foo")
:print("</li>\n")
:print("</ul>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## id="mark:xxx" and id="value:xxx"
    ## --------------------
    def test_convert38	##  id="mark:xxx" and id="value:xxx"
	input = <<'END'
<ul id="mark:list">
  <li id="value:item[0]">foo</li>
</ul>
END
	expected = <<'END'
:macro(stag_list)
  :print("<ul>\n")
:end

:macro(cont_list)
  :print("  <li>", item[0], "</li>\n")
:end

:macro(etag_list)
  :print("</ul>\n")
:end

:macro(elem_list)
  :expand(stag_list)
  :expand(cont_list)
  :expand(etag_list)
:end

:expand(elem_list)
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="dummy:name"
    ## --------------------
    def test_convert41	##  kd="dummy:name"
	input = <<'END'
<tr>
  <td>foo</td>
  <td kd="dummy:d1">bar</td>
</tr>
END
	expected = <<'END'
:print("<tr>\n")
:print("  <td>foo</td>\n")
:print("</tr>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="dummy:name" width single-tag
    ## --------------------
    def test_convert42	##  kd="dummy:name" width single-tag
	input = <<'END'
foo
<hr width="100%" kd="dummy:d1"/>
bar
END
	expected = <<'END'
:print("foo\n")
:print("bar\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="if:expr"
    ## --------------------
    def test_convert43	##  kd="if:expr"
	input = <<'END'
<font color="red" kd="if:error">
  Error happened.
</font>
END
	expected = <<'END'
:if(error)
  :print("<font color=\"red\">\n")
  :print("  Error happened.\n")
  :print("</font>\n")
:end
END
	_test_convert(input, expected)
    end

    ## --------------------
    ## kd="if:expr" width single-tag
    ## --------------------
    def test_convert44	##  kd="if:expr" width single-tag
	input = <<'END'
<hr color="red" kd="if:error"/>
END
	expected = <<'END'
:if(error)
  :print("<hr color=\"red\"/>\n")
:end
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="while:expr"
    ## --------------------
    def test_convert45	##  kd="while:expr"
	input = <<'END'
<tr kd="while:row=dbh.fetch">
  <td>#{row[0]}#</td>
</tr>
END
	expected = <<'END'
:while(row=dbh.fetch)
  :print("<tr>\n")
  :print("  <td>", row[0], "</td>\n")
  :print("</tr>\n")
:end
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="while:expr" width single-tag
    ## --------------------
    def test_convert46	##  kd="while:expr" width single-tag
	input = <<'END'
<hr color="red" kd="while:i<length"/>
END
	expected = <<'END'
:while(i<length)
  :print("<hr color=\"red\"/>\n")
:end
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## id="replace:name"
    ## --------------------
    def test_convert47	##  id="replace:name"
	input = <<'END'
<foobar id="replace:new">may be new</foobar>
<foobar id="replace:old"/>
END
	expected = <<END
:expand(elem_new)
:expand(elem_old)
END
	_test_convert(input, expected)

	## id="replace:hoge'ge"  cause error
	input = '<foobar id="replace:new!">may be new</foobar>' + "\n"
	assert_raises(Kwartz::ConvertionError) do
	    _test_convert(input, '')
	end
    end


    ## --------------------
    ## id="include:'filename'"
    ## --------------------
    def test_convert48	##  id="include:'filename'"
	include1 = <<'END'
<td id="include:'include2.html'">foo</td>
END
	include2 = <<'END'
<b>include2</b>
END
	Dir.mkdir('test.d') if !test(?d, 'test.d')
	File.open('test.d/include1.html', 'w') { |f| f.write(include1) }
	File.open('test.d/include2.html', 'w') { |f| f.write(include2) }

	input = <<'END'
<tr>
 <span id="include:'include1.html'"/>
</tr>
END
	File.open('test.d/include.html', 'w') { |f| f.write(input) }
	expected = <<'END'
:print("<tr>\n")
## :print(" <span>")
:print("<td>")
:print("<b>include2</b>\n")
:print("</td>\n")
## :print("</span>\n")
:print("</tr>\n")
END
	File.open('test.d/include.expected', 'w') { |f| f.write(expected) }
	toppings = { :include_path => ['.', 'test.d'] }
	_test_convert(input, expected, toppings)
	
	input2 = "<span id=\"include:include1.html\"/>\n"
	assert_raises(Kwartz::ConvertionError) do
	    _test_convert(input2, expected, toppings)
	end

	input3 = "<span id=\"include:'foo.html'\"/>\n"
	assert_raises(Kwartz::ConvertionError) do
	    _test_convert(input3, expected, toppings)
	end
    end


    ## --------------------
    ## id="load:'filename'"
    ## --------------------
    def test_convert49	##  id="load:'filename'"
	input = "<span id=\"load:'file.plogic'\"/>\n"
	expected = <<'END'
## :print("<span>")
:load('file.plogic')
## :print("</span>\n")
END
	_test_convert(input, expected)
	
	input2 = "<span id=\"load:dload1.html\"/>\n"
	assert_raises(Kwartz::ConvertionError) do
	    _test_convert(input2, expected)
	end
    end


    ## --------------------
    ## kd="foreach:var=expr"
    ## --------------------
    def test_convert51	##  kd="foreach:var=expr"
	input = <<'END'
<tr kd="foreach:user=user_list">
  <td>#{user}#</td>
</tr>
END
	expected = <<'END'
:foreach(user = user_list)
  :print("<tr>\n")
  :print("  <td>", user, "</td>\n")
  :print("</tr>\n")
:end
END
	_test_convert(input, expected)
    end

    ## --------------------
    ## kd="Foreach:name=expr"
    ## --------------------
    def test_convert52	##  kd="Foreach:name=expr"
	input = <<'END'
<tr kd="Foreach:user=user_list;attr:class=user_tgl">
  <td>#{user}#</td>
</tr>
END
	expected = <<'END'
:set(user_ctr = 0)
:foreach(user = user_list)
  :set(user_ctr += 1)
  :print("<tr class=\"", user_tgl, "\">\n")
  :print("  <td>", user, "</td>\n")
  :print("</tr>\n")
:end
END
	_test_convert(input, expected)
    end

    ## --------------------
    ## kd="list:var=expr"
    ## --------------------
    def test_convert53	##  kd="list:var=expr"
	input = <<'END'
<ul kd="list:user=user_list">
  <li>#{user}#</li>
</ul>
END
	expected = <<'END'
:print("<ul>\n")
:foreach(user = user_list)
  :print("  <li>", user, "</li>\n")
:end
:print("</ul>\n")
END
	_test_convert(input, expected)
    end

    ## --------------------
    ## kd="List:name=expr"
    ## --------------------
    def test_convert54	##  kd="List:name=expr"
	input = <<'END'
<ul kd="List:user=user_list">
  <li kd="attr:class=user_tgl">#{user}#</li>
</ul>
END
	expected = <<'END'
:print("<ul>\n")
:set(user_ctr = 0)
:foreach(user = user_list)
  :set(user_ctr += 1)
  :print("  <li class=\"", user_tgl, "\">", user, "</li>\n")
:end
:print("</ul>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="foreach:name=list" with single-tag
    ## --------------------
    def test_convert55	##  kd="foreach:name=list" with single-tag
	input = <<'END'
<hr width="100%" kd="foreach:width=width_list;attr:width=width"/>
END
	expected = <<'END'
:foreach(width = width_list)
  :print("<hr width=\"", width, "\"/>\n")
  ## empty
  ## empty
:end
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="LIST:..."
    ## --------------------
    def test_convert56	##  kd="LIST:..."
	input = <<'END'
<table>
  <tbody kd="LIST:user=user_list">
    <tr class="odd" kd="attr:class=user_tgl">
      <td kd="value:user{'name'}">foo</td>
      <td kd="value:user{'email'}">foo@mail.com</td>
    </tr>
  </tbody>
</table>
END
	expected = <<'END'
:print("<table>\n")
:print("  <tbody>\n")
:set(user_ctr = 0)
:foreach(user = user_list)
  :set(user_ctr += 1)
  :set(user_tgl = user_ctr % 2 == 0 ? 'even' : 'odd')
  :print("    <tr class=\"", user_tgl, "\">\n")
  :print("      <td>", user{'name'}, "</td>\n")
  :print("      <td>", user{'email'}, "</td>\n")
  :print("    </tr>\n")
:end
:print("  </tbody>\n")
:print("</table>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="FOREACH:..."
    ## --------------------
    def test_convert57	##  kd="FOREACH:..."
	input = <<'END'
<table>
  <tbody>
    <tr class="odd" kd="FOREACH:user=user_list;attr:class=user_tgl">
      <td kd="value:user{'name'}">foo</td>
      <td kd="value:user{'email'}">foo@mail.com</td>
    </tr>
  </tbody>
</table>
END
	expected = <<'END'
:print("<table>\n")
:print("  <tbody>\n")
:set(user_ctr = 0)
:foreach(user = user_list)
  :set(user_ctr += 1)
  :set(user_tgl = user_ctr % 2 == 0 ? 'even' : 'odd')
  :print("    <tr class=\"", user_tgl, "\">\n")
  :print("      <td>", user{'name'}, "</td>\n")
  :print("      <td>", user{'email'}, "</td>\n")
  :print("    </tr>\n")
:end
:print("  </tbody>\n")
:print("</table>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="foreach:var" -- ConvertionError
    ## --------------------
    def test_convert58	##  kd="foreach:var" -- ConvertionError
	input = <<'END'
<ul kd="foreach:foo[0]=bar">
 <li>foo</li>
</ul>
END
	expected = ""
	assert_raises(Kwartz::ConvertionError) do
	    _test_convert(input, expected)
	end
    end


    ## --------------------
    ## kd="set:var=value"
    ## --------------------
    def test_convert61	##  kd="set:var=value"
	input = <<'END'
<li kd="set:ctr+=1">foo</li>
END
	expected = <<'END'
:set(ctr+=1)
:print("<li>")
:print("foo")
:print("</li>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="set:var=value" with <span></span>
    ## --------------------
    def test_convert62	##  kd="set:var=value" with <span></span>
	input = <<'END'
<span kd="set:ctr+=1"/>
END
	expected = <<'END'
:set(ctr+=1)
## :print("<span/>\n")
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## kd="set:var=value" with table
    ## --------------------
    def test_convert63	##  kd="set:var=value" with table
	input = <<'END'
<span kd="set:ctr=0"/>
<span kd="set:klass='odd'"/>
<tr kd="foreach:item=list">
  <td kd="value:ctr">1</td>
  <td kd="value:item">hoge</td>
  <span kd="set:ctr+=1"/>
  <span kd="set:klass=ctr%2==0?'even':'odd'"/>
</tr>
END
	expected = <<'END'
:set(ctr=0)
## :print("<span/>\n")
:set(klass='odd')
## :print("<span/>\n")
:foreach(item = list)
  :print("<tr>\n")
  :print("  <td>", ctr, "</td>\n")
  :print("  <td>", item, "</td>\n")
  :set(ctr+=1)
  ## :print("  <span/>\n")
  :set(klass=ctr%2==0?'even':'odd')
  ## :print("  <span/>\n")
  :print("</tr>\n")
:end
END
	_test_convert(input, expected)
    end



    ## --------------------
    ## kd="set:var" -- ConvertionError
    ## --------------------
    def test_convert64	##  kd="set:var" -- ConvertionError
	input = <<'END'
<li kd="set:var">foo</li>
END
	expected=''
	assert_raises(Kwartz::ConvertionError) do
	    _test_convert(input, expected)
	end
    end


    ## --------------------
    ## delete <span></span>
    ## --------------------
    def test_convert71	##  delete <span></span>
	input = <<'END'
<span kd="if:true">foo</span>
END
	expected = <<'END'
:if(true)
  ## :print("<span>")
  :print("foo")
  ## :print("</span>\n")
:end
END
	_test_convert(input, expected)
    end


#    ## --------------------
#    ## delete <div></div>
#    ## --------------------
#    def test_convert72	##  delete <div></div>
#	input = <<'END'
#<div kd="if:true">
#foo
#</div>
#END
#	expected = <<'END'
#:if(true)
#  ## :print("<div>\n")
#  :print("foo\n")
#  ## :print("</div>\n")
#:end
#END
#	_test_convert(input, expected)
#    end


    ## --------------------
    ## leave <span></span> if attribtes exist
    ## --------------------
    def test_convert73	##  leave <span></span> if attribtes exist
	input = <<'END'
<span kd="if:true" class="foo">foo</span>
END
	expected = <<'END'
:if(true)
  :print("<span class=\"foo\">")
  :print("foo")
  :print("</span>\n")
:end
END
	_test_convert(input, expected)
    end

    ## --------------------
    ## leave <span></span> if attr:name:value exists
    ## --------------------
    def test_convert74	## leave <span></span> if attr:name:value exists
	input = <<'END'
<span kd="if:true;attr:name:value">foo</span>
END
	expected = <<'END'
:if(true)
  :print("<span name=\"", value, "\">")
  :print("foo")
  :print("</span>\n")
:end
END
	_test_convert(input, expected)
    end


#    ## --------------------
#    ## leave <div></div> if attributes exist
#    ## --------------------
#    def test_convert74	##  leave <div></div> if attributes exist
#	input = <<'END'
#<div kd="if:true;attr:class=foo">
#foo
#</div>
#END
#	expected = <<'END'
#:if(true)
#  :print("<div class=\"", foo, "\">\n")
#  :print("foo\n")
#  :print("</div>\n")
#:end
#END
#	_test_convert(input, expected)
#    end


#    ## --------------------
#    ## delete <div></div>
#    ## --------------------
#    def test_convert75	##  delete <div></div>
#	input = <<'END'
#<div kd="foreach:item=list">
#foo
#</div>
#END
#	expected = <<'END'
#:foreach(item = list)
#  ## :print("<div>\n")
#  :print("foo\n")
#  ## :print("</div>\n")
#:end
#END
#	_test_convert(input, expected)
#    end


    ## --------------------
    ## single tag
    ## --------------------
    def test_convert76	## single tag	##  single tag
	input = <<'END'
<span kd="mark:foo"/>
END
	expected = <<'END'
:macro(stag_foo)
  ## :print("<span/>\n")
:end

:macro(cont_foo)
  ## empty
:end

:macro(etag_foo)
  ## empty
:end

:macro(elem_foo)
  :expand(stag_foo)
  :expand(cont_foo)
  :expand(etag_foo)
:end

:expand(elem_foo)
END
	_test_convert(input, expected)
    end



    ## --------------------
    ## --delete_idattr=true
    ## --------------------
    def test_convert77	## --delete_idattr=true	##  --delete_idattr=true
	input = <<'END'
<ul id="user_list">
 <li><span id="user">foo</span></li>
</ul>
END
	expected = <<'END'
:macro(stag_user)
  ## :print("<span>")
:end

:macro(cont_user)
  :print("foo")
:end

:macro(etag_user)
  ## :print("</span>")
:end

:macro(elem_user)
  :expand(stag_user)
  :expand(cont_user)
  :expand(etag_user)
:end

:macro(stag_user_list)
  :print("<ul>\n")
:end

:macro(cont_user_list)
  :print(" <li>")
  :expand(elem_user)
  :print("</li>\n")
:end

:macro(etag_user_list)
  :print("</ul>\n")
:end

:macro(elem_user_list)
  :expand(stag_user_list)
  :expand(cont_user_list)
  :expand(etag_user_list)
:end

:expand(elem_user_list)
END
	toppings = { :delete_idattr => true }
	_test_convert(input, expected, toppings)
    end



    ## --------------------
    ## \r\n (crlf)
    ## --------------------
    def test_convert78	## \r\n (crlf)
	input = <<END
<ul id="list">\r
  <li id="value:item">foo</li>\r
</ul>\r
END
	expected = <<'END'
:macro(stag_list)
  :print("<ul>\r\n")
:end

:macro(cont_list)
  :print("  <li>", item, "</li>\r\n")
:end

:macro(etag_list)
  :print("</ul>\r\n")
:end

:macro(elem_list)
  :expand(stag_list)
  :expand(cont_list)
  :expand(etag_list)
:end

:expand(elem_list)
END
	expected.gsub!(/\n/, "\r\n")
	toppings = { :delete_idattr => true }
	_test_convert(input, expected, toppings)
    end

    
    ## --------------------
    ## IllegalDirectiveError
    ## --------------------
    def test_convert81	##  IllegalDirectiveError
	input = <<'END'
<ul kd="foeach:user=user_list">
  <li kd="value:user">foo</li>
</ul>
END
	expected = <<'END'
END
	#assert_raises(Kwartz::ConvertionError) do
	assert_raises(Kwartz::IllegalDirectiveError) do
	    _test_convert(input, expected)
	end
    end


    ## --------------------
    ## UnclosedTagError
    ## --------------------
    def test_convert82	##  UnclosedTagError
	input = <<'END'
<body>
<ul kd="foreach:user=user_list">
<li kd="value:user">foo</li>
</body>
END
	expected = <<'END'
END
	assert_raises(Kwartz::UnclosedTagError) do
	    _test_convert(input, expected)
	end
    end


    ## --------------------
    ## id="attr:name:value", id="foreach:item:list", ...
    ## --------------------
    def test_convert91	##  id="attr:name:value", id="foreach:item:list", ...
	input = <<'END'
<ul id="foreach:item:list;attr:class:flag?odd:even">
  <li id="set:var:value">foo</li>
  <li id="attr:ns:class:klass" ns:class=dummy>foo</li>
</ul>
END
	expected = <<'END'
:foreach(item = list)
  :print("<ul class=\"", flag?odd:even, "\">\n")
  :set(var=value)
  :print("  <li>")
  :print("foo")
  :print("</li>\n")
  :print("  <li ns:class=\"", klass, "\">foo</li>\n")
  :print("</ul>\n")
:end
END
	_test_convert(input, expected)
    end


    ## --------------------
    ## practical sample
    ## --------------------
    def test_convert92	##  practical sample
	input = <<'END'
<table>
  <thead>
    <tr>
      <th>uid</th>
      <th>name</th>
      <th>email</th>
    </tr>
  </thead>
  <tbody kd="List:user=user_list">
    <span kd="set:user_tgl=user_ctr%2==0?'even':'odd'"/>
    <tr class="odd" kd="attr:class=user_tgl">
      <td kd="value:user.uid">101</td>
      <td kd="value:user.name">Foo</td>
      <td kd="value:user.email">foo@foo.com</td>
    </tr>
    <tr class="even" kd="dummy:d1">
      <td>102</td>
      <td>Bar</td>
      <td>bar@bar.org</td>
    </tr>  
    <tr class="odd" kd="dummy:d2">
      <td>103</td>
      <td>Baz</td>
      <td>baz@baz.net</td>
    </tr>  
  </tbody>
</table>
END
	expected = <<'END'
:print("<table>\n")
:print("  <thead>\n")
:print("    <tr>\n")
:print("      <th>uid</th>\n")
:print("      <th>name</th>\n")
:print("      <th>email</th>\n")
:print("    </tr>\n")
:print("  </thead>\n")
:print("  <tbody>\n")
:set(user_ctr = 0)
:foreach(user = user_list)
  :set(user_ctr += 1)
  :set(user_tgl=user_ctr%2==0?'even':'odd')
  ## :print("    <span/>\n")
  :print("    <tr class=\"", user_tgl, "\">\n")
  :print("      <td>", user.uid, "</td>\n")
  :print("      <td>", user.name, "</td>\n")
  :print("      <td>", user.email, "</td>\n")
  :print("    </tr>\n")
:end
:print("  </tbody>\n")
:print("</table>\n")
END
	_test_convert(input, expected)
    end

end



## ====================
## CompilerTest
## ====================

class CompilerTest < Test::Unit::TestCase
    @@inputs  = {}
    @@plogics = {}

    ## --------------------
    ## generate HTML table, without plogic (only with directive)
    ## --------------------
    @@inputs['01'] = <<END
<table>
  <thead>
    <tr>
      <th>uid</th>
      <th>name</th>
      <th>email</th>
    </tr>
  </thead>
  <tbody kd="List:user=user_list">
    <tr class="odd"
        kd="attr:class=klass;set:klass=user_ctr%2==0?'even':'odd'">
      <td kd="value:user['uid']">101</td>
      <td kd="Value:user['name']">Foo</td>
      <td kd="VALUE:user['email']">foo@foo.com</td>
    </tr>
    <tr class="even" kd="dummy:d1">
      <td>102</td>
      <td>Bar</td>
      <td>bar@bar.org</td>
    </tr>  
    <tr class="odd" kd="dummy:d2">
      <td>103</td>
      <td>Baz</td>
      <td>baz@baz.net</td>
    </tr>  
  </tbody>
</table>
END

    @@plogics['01'] = <<END
END


    ## --------------------
    ## generate HTML table, with plogic
    ## --------------------
    @@inputs['02'] = <<END
<table>
  <tr>
    <th>uid</th>
    <th>name</th>
    <th>email</th>
  </tr>
  <tr class="odd" id="mark:user" kd="Attr:class=klass">
    <td id="mark:uid">101</td>
    <td id="mark:name">Foo</td>
    <td id="mark:email">foo@foo.com</td>
  </tr>
  <tr class="even" kd="dummy:d1">
    <td>102</td>
    <td>Bar</td>
    <td>bar@bar.org</td>
  </tr>  
  <tr class="odd" kd="dummy:d2">
    <td>103</td>
    <td>Baz</td>
    <td>baz@baz.net</td>
  </tr>  
</table>
END

    @@plogics['02'] = <<END
:elem(user)
  :set(is_odd = 0)
  :foreach(user = user_list)
    :set(is_odd = 1 - is_odd)
    :set(klass = is_odd == 1 ? 'odd' : 'even')
    @stag
    @cont
    @etag
  :end
:end
:value(uid=user.uid)
:value(name=E(user.name))
:value(email=X(user.email))
END


    ## --------------------
    ## generate CSV file
    ## --------------------
    @@inputs['03'] = <<END
<span id="list:items=items_list">
  <span id="mark:item">
<span id="value:item">foo</span>
  </span>
</span>
END

    @@plogics['03'] = <<END
:macro(elem_item)
  :set(item_ctr = 0)
  :foreach(item = items)
    :set(item_ctr += 1)
    :if(item_ctr > 1)
      :print(', ')
    :end    
    :expand(stag_item)
    :expand(cont_item)
    :expand(etag_item)
  :end
:end
END


    ## --------------------
    ## #{cond?val1:val2}#
    ## --------------------
    @@inputs['04'] = <<'END'
<input type="radio" name="gender" #{param['gender']=='M'?'checked':''}#>Man
<input type="radio" name="gender" #{E(param['gender']=='W'?'checked':'')}#>Woman
<input type="radio" name="gender" #{X(param['gender']==empty?'checked':'')}#>Unknown

<option name="win"#{@CHECK(param['os']=='win')}#>Windows
<option name="mac"#{@SELECT(param['os']=='mac')}#>MacOS X
<option name="unix"#{@DISABLE(param['os']=='unix')}#>UNIX
END

    @@plogics['04'] = <<END
END


    ## --------------------
    ## indent
    ## --------------------
    @@inputs['05'] = <<END
<span><i kd="list:user=user_list">
<b><span kd="set:ctr+=1"/><s kd="if:ctr%2==0">even</s></b></i>
</span>
END

    @@plogics['05'] = <<END
END

    ## --------------------
    ## special macro expantion
    ## --------------------
    @@inputs['06'] = <<END
<table>
 <tbody id="mark:userlist">
  <tr bgcolor="#FFCCCC" id="mark:odd">
   <td kd="value:user[:name]">foo</td>
   <td kd="value:user[:mail]">foo@mail.com</td>
  </tr>
  <tr bgcolor="#CCCCFF" id="mark:even">
   <td kd="value:user[:name]">bar</td>
   <td kd="value:user[:mail]">bar@mail.com</td>
  </tr>
 </tbody>
</table>
END
    @@plogics['06'] = <<END
## dummy macro
:elem(foobar)
    @etag
    @cont
    @stag
:end

:elem(userlist)
  :set(user_ctr=0)
  :foreach(user=userlist)
    :set(user_ctr+=1)
    @stag
    @cont
    @etag
  :end
:end

:macro(cont_userlist)
  :if(user_ctr%2==0)
    @elem_even
  :else
    @elem_odd
  :end
:end
END


    ## --------------------
    ## "\r\n" (CRLF)
    ## --------------------
    @@inputs['07']  = @@inputs['06'].gsub(/\n/, "\r\n")
    @@plogics['07'] = @@plogics['06'].gsub(/\n/, "\r\n")

    
    ## ---------------------------------------
    

    ## presentation data and presentation logic
    Dir.mkdir(Dirname) unless test(?d, Dirname)
    @@inputs.each do |num,str|
	filename = "#{Dirname}/compile#{num}.in"
	File.open(filename, "w") { |f| f.write(str) }
    end
    @@plogics.each do |num,str|
	filename = "#{Dirname}/compile#{num}.plogic"
	File.open(filename, "w") { |f| f.write(str) }
    end

    ## expected data
    if test(?d, 'test.expected')
	#require 'fileutils'
	#Dir.glob("test.expected/*.exp.*").each do |filename|
	#    FileUtils.cp(filename, Dirname)
	#end
	Dir.glob("test.expected/*.exp.*").each do |source|
	    dest = source.sub(/^test.expected/, Dirname)
	    File.open(source) do |f_in|
		File.open(dest, 'w') do |f_out|
		    f_out.write(f_in.read())
		end
	    end
	end
    else
	$stderr.print "\n*** directory 'test.expected' not found.\n"
	exit 1
    end

    
    ## ---------------------------------------

    ## utility method for test
    def _test_compile(input, plogic, lang, n, toppings, flag_verbose=true)
	compiler = Kwartz::Compiler.new(lang, toppings)
	output   = compiler.compile(input, plogic)
	output_filename = "compile#{n}.out.#{lang}#{toppings[:escape] ? '.s':''}"
	expect_filename = "compile#{n}.exp.#{lang}#{toppings[:escape] ? '.s':''}"
	File.open("#{Dirname}/#{output_filename}", "w") { |f| f.print output }
	expected = File.open("#{Dirname}/#{expect_filename}") { |f| f.read }
	if flag_verbose then
	    assert_equal(expected, output)
	else
	    assert(expected == output)
	end
    end

    ## test method definition
    langs = %w(ruby php jsp eruby erb erbscan velocity ruby2)
    langs.concat %w(ruby.s php.s jsp.s eruby.s erb.s erbscan.s ruby2.s)
    @@inputs.keys.sort.each do |n|
	langs.each do |lang|
	    opt = lang =~ /\.s$/ ? ':escape=>true,' : ''
	    opt << ':load_path=>[",","test.d"],' if n == '07'
	    testname = "#{lang.sub(/\./, '_')}#{n}"
	    code = <<-END
	    def test_compile_#{testname}
		_test_compile(@@inputs['#{n}'], @@plogics['#{n}'], '#{lang.sub(/\.s$/, '')}', '#{n}', {#{opt}})
	    end
	    END
	    eval code
	end
    end

end


## ====================
## :load(...) and id="load:..."
## ====================
class LoadTest < Test::Unit::TestCase
    pdata = <<END
  <table>
   <tr id="item_list">
    <td id="value:item">foo</td>
   </tr>
  </table>
  <span id="load:'dload1.plogic'"/>
END

    plogic1 = <<'END'
:elem(item_list)
  :foreach(item=list)
    @stag
    @cont
    @etag
  :end
:end
:load('dload2.plogic')
END

    plogic2 = <<'END'
:macro(BEGIN)
  :print("<html>\n")
  :print(" <body>\n")
:end
:macro(END)
  :print(" </body>\n")
  :print("</html>\n")
:end
END

    expected = <<'END'
<html>
 <body>
  <table>
#foreach ($item in $list)
   <tr id="item_list">
    <td>$!{item}</td>
   </tr>
#end
  </table>
 </body>
</html>
END

    Dir.mkdir('test.d') if !test(?d, 'test.d')
    File.open('test.d/dload.html',    'w') { |f| f.write(pdata) }
    File.open('test.d/dload1.plogic', 'w') { |f| f.write(plogic1) }
    File.open('test.d/dload2.plogic', 'w') { |f| f.write(plogic2) }
    File.open('test.d/dload.expected','w') { |f| f.write(expected) }
    @@pdata    = pdata
    @@expected = expected

    def test_load
	toppings = { :load_path => ['.', 'test.d'] }
	output = Kwartz::Helper.compile(@@pdata, '', 'velocity', toppings)
	expect = @@expected
	assert_equal(output, expect)
    end

end


## ====================
## test for Kwaser::compile() utility function
## ====================
class HelperTest < Test::Unit::TestCase
    
    def _touch(filename)
	File.open(filename, 'r+') do |file|
	    file.flock(File::LOCK_EX)
	    file.seek(0, IO::SEEK_SET)
	    ch = file.getc
	    file.seek(0, IO::SEEK_SET)
	    file.putc(ch)
	end
    end


    InputData = <<END
<b kd="mark:foo">bar</b>
END

    PlogicData = <<END
:macro(cont_foo)
  :print(foo)
:end
END

    ExpectData1 = <<END
<b>bar</b>
END

    ExpectData2 = <<END
<b><?php echo $foo; ?></b>
END

    DummyData = "abc\n"

    Basename = "test.d/util"
    File.open(Basename + '.in',     'w') { |f| f.write(InputData)  }
    File.open(Basename + '.plogic', 'w') { |f| f.write(PlogicData) }

    def _test_util(flag_output, flag_plogic, flag_touch_input, flag_touch_plogic)
	base = Basename
	input_fname  = "#{base}.in"
	output_fname = "#{base}.out"
	plogic_fname = "#{base}.plogic"
	expect_fname = "#{base}.exp"
	plogic_fname = nil if !flag_plogic
	dummy        = DummyData
	if flag_output then
	    File.open(output_fname, 'w') { |f| f.write(dummy) }
	else
	    File.unlink(output_fname) if test(?f, output_fname)
	    assert(!test(?f, output_fname))
	end
	if flag_touch_input then
	    sleep(1)
	    _touch(input_fname)
	end
	if flag_plogic && flag_touch_plogic then
	    sleep(1)
	    _touch(plogic_fname) 
	end
	Kwartz::compile(input_fname, output_fname, plogic_fname, 'php')
	assert(test(?f, output_fname))
	output = File.open(output_fname) { |f| f.read }
	expect = flag_plogic ? ExpectData2 : ExpectData1
	if !flag_output || flag_touch_input || (flag_plogic && flag_touch_plogic) then
	    assert_equal(expect, output)	## compiled
	else
	    assert_equal(dummy, output)		## not compiled
	end
	return output
    end

    ## ---- without plogic file -----

    ## there is no output file
    def test_util11
	output = _test_util(false, false, nil, nil)
	assert_equal(ExpectData1, output)
    end
    
    ## output file is older than input file
    def test_util12
	output = _test_util(true, false, true, nil)
	#assert_equal(ExpectData1, output)
    end
    
    ## output file is newer than input file
    def test_util13
	output = _test_util(true, false, false, nil)
	assert_equal(DummyData, output)
    end

    
    ## ---- with plogic file -----

    ## there is no output file
    def test_util15
	output = _test_util(false, true, nil, false)
	assert_equal(ExpectData2, output)
    end

    ## output file is older than input file and plogic file
    def test_util16
	output = _test_util(true, true, true, true)
	assert_equal(ExpectData2, output)
    end
    
    ## output file is older than input file, and newer than plogic file
    def test_util17
	output = _test_util(true, true, true, false)
	assert_equal(ExpectData2, output)
    end

    ## output file is newer than input file, and older than plogic file
    def test_util18
	output = _test_util(true, true, false, true)
	assert_equal(ExpectData2, output)
    end

    ## output file is newer than input file and plogic file
    def test_util19
	output = _test_util(true, true, false, false)
	assert_equal(DummyData, output)
    end

    ## ----
    def test_util21
	input = <<'END'
<table kd="Foreach:user=list">
  <tr bgcolor="#FFCCCC"
      kd="attr:bgcolor=color;set:color=user_ctr%2==0?'#CCCCFF':'#FFCCCC'">
    <td kd=value:user>foo</td>
  </tr>
</table>
END
	expected = <<'END'
<table >
  <tr bgcolor="#FFCCCC"
      >
    <td >foo</td>
  </tr>
</table>
END
	str = Kwartz::Helper.delete_directives(input)
	assert_equal(expected, str)
    end

end



## ====================
## Analyzer test
## ====================
class AnalyzerTest < Test::Unit::TestCase

    def _test(pdata, plogic, expected, flag_verbose=true)
	result = Kwartz::Helper.analyze(pdata, plogic, expected)
	if flag_verbose then
	    assert_equal(expected, result)
	else
	    assert(expected == result)
	end
    end


    ## --------------------
    ## :set stmt
    ## --------------------
    def test_analyze01	##  :set stmt
    pdata = ''
    plogic = <<'END'
:set(var = 100)
END
    expected = <<'END'
global variables:
local variables:  var
END
	_test(pdata, plogic, expected)
    end


    ## --------------------
    ## :print stmt
    ## --------------------
    def test_analyze02	##  :print stmt
    pdata = ''
    plogic = <<'END'
:print(var)
END
    expected = <<'END'
global variables: var
local variables: 
END
	_test(pdata, plogic, expected)
    end


    ## --------------------
    ## :foreach stmt
    ## --------------------
    def test_analyze03	##  :foreach stmt
    pdata = ''
    plogic = <<'END'
:foreach(item = list)
  :print(item)
  :print(item2)
:end
END
    expected = <<'END'
global variables: list item2
local variables:  item
END
	_test(pdata, plogic, expected)
    end


    ## --------------------
    ## :if stmt
    ## --------------------
    def test_analyze04	##  :if stmt
    pdata = ''
    plogic = <<'END'
:set(i=0)
:if(i==0)
  :print(yes)
:end
END
    expected = <<'END'
global variables: yes
local variables:  i
END
	_test(pdata, plogic, expected)
    end


    ## --------------------
    ## :if :else
    ## --------------------
    def test_analyze05	##  :if :else
    pdata = ''
    plogic = <<'END'
:set(i=0)
:if(i>0)
  :print(i)
:else
  :print(yes)
:end
END
    expected = <<'END'
global variables: yes
local variables:  i
END
	_test(pdata, plogic, expected)
    end


    ## --------------------
    ## :if :elsif :else
    ## --------------------
    def test_analyze06	##  :if :elsif :else
    pdata = ''
    plogic = <<'END'
:set(i=0)
:if(i>0)
  :print(i+1)
:elsif(i<0)
  :print(i-1)
:else
  :print(yes .+ 'foo')
:end
END
    expected = <<'END'
global variables: yes
local variables:  i
END
	_test(pdata, plogic, expected)
    end


    ## --------------------
    ## :while stmt
    ## --------------------
    def test_analyze07	##  :while stmt
    pdata = ''
    plogic = <<'END'
:set(i=1+2)
:while(i>0 && j>0)
  :print(j)
:end
END
    expected = <<'END'
global variables: j
local variables:  i
END
	_test(pdata, plogic, expected)
    end


    ## --------------------
    ## array
    ## --------------------
    def test_analyze08	##  array
    pdata = ''
    plogic = <<'END'
:print(params[:key])
:set(key='name')
:print('foo' .+ params[key])
END
    expected = <<'END'
global variables: params
local variables:  key
END
	_test(pdata, plogic, expected)
    end


    ## --------------------
    ## negative number (reported by Andreas)
    ## --------------------
    def test_analyze09	##  negative number (reported by Andreas)
	pdata = ''
    plogic = <<'END'
:print(array[-1])
:print(array[-n])
:print(array[-n*(-1)])
END
	expected = <<'END'
global variables: n array
local variables: 
END
	_test(pdata, plogic, expected)
    end


    ## --------------------
    ## bug reported by Shu-yu Guo 
    ## --------------------
    def test_analyze10	##  bug reported by Shu-yu Guo 
    	pdata = <<'END'
<span id="set:test:global_one"/>
<span id="set:test:global_two"/>
END
	plogic = ''
    	expected = <<'END'
global variables: global_one global_two
local variables:  test
END
	_test(pdata, plogic, expected)
    end

    ## --------------------
    ## bug reported by Andreas
    ## --------------------
    def test_analyze11	##  bug reported by Andreas
    	pdata = <<'END'
<tr id="foreach:local=global1">
 <td id="value:local">foo</td>
</tr>
<tr id="foreach:local=global2">
 <td id="value:local">bar</td>
</tr>
END
	plogic = ''
    	expected = <<'END'
global variables: global1 global2
local variables:  local
END
	_test(pdata, plogic, expected)
    end


    ## --------------------
    ## only pdata
    ## --------------------
    def test_analyze21	##  only pdata
    pdata = <<'END'
<ul id="user_list">
  <li>#{user}#</li>
</ul>
END
    plogic = ''
    expected = <<'END'
global variables: user
local variables: 
END
	_test(pdata, plogic, expected)
    end


    ## --------------------
    ## pdata + plogic
    ## --------------------
    def test_analyze22	##  pdata + plogic
    pdata = <<'END'
<ul id="user_list">
  <li>#{user}#</li>
</ul>
END
    plogic = <<'END'
:elem(user_list)
  @stag
  :foreach(user=user_list)
    @cont
  :end
  @etag
:end
END
    expected = <<'END'
global variables: user_list
local variables:  user
END
	_test(pdata, plogic, expected)
    end


    ## --------------------
    ## errors - :set(var = var+1)
    ## --------------------
    def test_analyze31	##  errors - :set(var = var+1)
	pdata = ''
	plogic = <<'END'
:set(var = var + 1)
END
	expected = <<'END'
*** warning: var: uninitialized local var is used, or tryng to assign into global var.
global variables:
local variables:  var
END
	_test(pdata, plogic, expected)
	#expected = ''
	#assert_raises(Kwartz::AnalysisError) do
	#    _test(pdata, plogic, expected)
	#end
    end

    
    ## --------------------
    ## errors - :set(var += 1)
    ## --------------------
    def test_analyze32	##  errors - :set(var += 1)
	pdata = ''
	plogic = <<'END'
:set(var += 1)
END
	expected = <<'END'
*** warning: var: uninitialized local var is used, or tryng to assign into global var.
global variables:
local variables:  var
END
	_test(pdata, plogic, expected)
	#expected = ''
	#assert_raises(Kwartz::AnalysisError) do
	#    _test(pdata, plogic, expected)
	#end
    end


    ## --------------------
    ## errors - assign into global var
    ## --------------------
    def test_analyze33	##  errors - assign into global var
	pdata = ''
	plogic = <<'END'
:print(var)
:set(var = 0)
END
	expected = <<'END'
*** warning: var: trying to assign into global var.
global variables: var
local variables: 
END
	_test(pdata, plogic, expected)
	#expected = ''
	#assert_raises(Kwartz::AnalysisError) do
	#    _test(pdata, plogic, expected)
	#end
    end


    ## --------------------
    ## errors - use global var as loopvar
    ## --------------------
    def test_analyze34	##  errors - use global var as loopvar
	pdata = ''
	plogic = <<'END'
:print(var)
:foreach(var=list)
  :print(var)
:end
END
	expected = <<'END'
*** warning: var: global var is used as loop var.
global variables: list var
local variables: 
END
	_test(pdata, plogic, expected)
	#expected = ''
	#assert_raises(Kwartz::AnalysisError) do
	#    _test(pdata, plogic, expected)
	#end
    end


#    ## --------------------
#    ## errors - complex left-hand expr of assignment
#    ## --------------------
#    def test_analyze35	##  errors - complex left-hand expr of assignment
#	pdata = ''
#	plogic = <<'END'
#:set(var[0]=0)
#END
#	expected = ''
#	#assert_raises(Kwartz::AnalysisError) do
#	assert_raises(Kwartz::SemanticError) do
#	    _test(pdata, plogic, expected)
#	end
#    end


#    ## --------------------
#    ## errors - complex expr as loopvar
#    ## --------------------
#    def test_analyze36	##  errors - complex expr as loopvar
#	pdata = ''
#	plogic = <<'END'
#:foreach(loop[:var]=list)
#  :print(foo)
#:end
#END
#	expected = ''
#	#assert_raises(Kwartz::AnalysisError) do
#	assert_raises(Kwartz::SemanticError) do
#	    _test(pdata, plogic, expected)
#	end
#    end


    ## --------------------
    ## practice
    ## --------------------
    def test_analyze91	##  practice
    pdata = <<'END'
     <table cellpadding="2" summary="Calendar of #{month}# #{year}#">
       <caption>
	 <i id="value:month">Jan</i>&nbsp;<i id="value:year">20XX</i>
       </caption>
       <thead>
	 <tr bgcolor="#CCCCCC">
	   <th class="holiday">S</th>
	   <th>M</th>
	   <th>T</th>
	   <th>W</th>
	   <th>T</th>
	   <th>F</th>
	   <th class="saturday">S</th>
	 </tr>
       </thead>
       <tbody>
	 <tr id="mark:week">
	   <td id="mark:holiday"  class="holiday">&nbsp;</td>
	   <td id="mark:normal"   class="normal">&nbsp;</td>
	   <td id="dummy:d1">&nbsp;</td>
	   <td id="dummy:d2">1</td>
	   <td id="dummy:d3">2</td>
	   <td id="dummy:d4">3</td>
	   <td id="dummy:d5">4</td>
	   <td id="mark:saturday" class="saturday">5</td>
	 </tr>
	 <tr id="dummy:w1">
	   <td class="holiday">6</td>
	   <td>7</td><td>8</td><td>9</td><td>10</td><td>11</td>
	   <td class="saturday">12</td>
	 </tr>
	 <tr id="dummy:w2">
	   <td class="holiday">13</td>
	   <td>14</td><td>15</td><td>16</td><td>17</td><td>18</td>
	   <td class="saturday">19</td>
	 </tr>
	 <tr id="dummy:w3">
	   <td class="holiday">20</td>
	   <td>21</td><td>22</td><td>23</td><td>24</td><td>25</td>
	   <td class="saturday">26</td>
	 </tr>
	 <tr id="dummy:w4">
	   <td class="holiday">27</td>
	   <td>28</td><td>29</td><td>30</td><td>31</td><td>&nbsp;</td>
	   <td class="saturday">&nbsp;</td>
	 </tr>
       </tbody>
     </table>
     &nbsp;
END
    plogic = <<'END'
:elem(week)

  :set(day = '&nbsp')
  :set(wday = 1)
  :while(wday < first_weekday)
    :if(wday == 1)
      @stag
    :end
    @cont
    :set(wday += 1)
  :end

  :set(day = 0)
  :set(wday -= 1)
  :while(day < num_days)
    :set(day += 1)
    :set(wday = wday % 7 + 1)
    :if(wday == 1)
      @stag
    :end
    @cont
    :if(wday == 7)
      @etag
    :end
  :end

  :if(wday != 7)
    :set(day = '&nbsp;')
    :while(wday != 6)
      @cont
      :set(wday += 1)
    :end
    @etag
  :end

:end

:macro(cont_week)
  :if(wday == 1)
    @elem_holiday
  :elsif(wday == 7)
    @elem_saturday
  :else
    @elem_normal
  :end
:end

:elem(holiday)
  @stag
  :print(day)
  @etag
:end

:elem(saturday)
  @stag
  :print(day)
  @etag
:end

:elem(normal)
  @stag
  :print(day)
  @etag
:end
END
    expected = <<'END'
global variables: month first_weekday year num_days
local variables:  day wday
END
	_test(pdata, plogic, expected)
    end

end



## ====================
## test execution
## ====================

opt = ARGV.shift
flag_all = (!opt || opt != '-s')

suite = Test::Unit::TestSuite.new
suite << ScannerTest.suite
suite << ParserTest.suite
suite << TranslatorTest.suite
suite << ConverterTest.suite
suite << CompilerTest.suite
suite << LoadTest.suite
suite << AnalyzerTest.suite
suite << HelperTest.suite  if flag_all
Test::Unit::UI::Console::TestRunner.run(suite)

