#!/usr/bin/ruby

###
### kwartz.rb -- a library for kwartz template system
###
### Copyright (C) 2004 kuwata-lab
### All rights reserved.
### author::  kwa(at)kuwata-lab.com
### id::      $Id: kwartz.rb,v 0.103 2004/05/09 07:46:40 kwatch Exp kwatch $
###

### This project is subsidized by Exploratory Software Project of IPA
### (Information-Technology Promotion Agency Japan).
### See http://www.ipa.go.jp/about/english/index.html for IPA.

### This program is free software; you can redistribute it and/or
### modify it under the terms of the GNU General Public License
### as published by the Free Software Foundation; either version 2
### of the License, or (at your option) any later version.
### 
### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.
### 
### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

module Kwartz

    AUTHOR	= 'kwa <kwa(at)kuwata-lab.com>'
    VERSION	= ('$Revision: 0.103 $' =~ /\d+(?:\.\d+)+/ && $&)
    LAST_UPDATE = ('$Date: 2004/05/09 07:46:40 $' =~ /\d+[-\/]\d+[-\/]\d+/ && $&)

    ##
    ## ex.
    ##	Kwartz::assert() unless arg == nil
    ##
    def self.assert(message=nil)
	raise AssertionError.new(caller()[0], message)
    end

    ## ============================================================

    ##
    ## Error class for Kwartz
    ##
    class KwartzError < StandardError
	def initialize(message=nil)
	    super('*** ' + message)
	end
    end


    ##
    ## abstract class for SyntaxError and SemanticError
    ##
    class ParseError < KwartzError
	def initialize(message=nil, scanner=nil)
	    if scanner
		submsg = nil
		case token = scanner.token
		when '"'
		    submsg = "use '...' instead of \"...\" for string"
		when :name
		    submsg = "maybe you forgot ':set(...)' for assignment statement"
		end
		message << "\n  Hint: #{submsg}" if submsg
		#message << "\n  lienum:#{linenum}, line: #{line}"
		message << "\n  line: #{scanner.line}"
	    end
	    super(message)
	end
    end

    class SyntaxError < ParseError
    end

    class SemanticError < ParseError
    end

    class ScanError < KwartzError
    end

    class ConvertionError < KwartzError
    end

    class IllegalDirectiveError < ConvertionError
    end

    class UnclosedTagError < ConvertionError
    end

    class TranslationError < KwartzError
    end

    class CompileError < KwartzError
    end
    
    class InvalidLanguageError < CompileError
	def initialize(lang)
	    super("#{lang}: no such language.\n")
	end
    end

    class AssertionError < KwartzError
	def initialize(location, msg=nil)
	    super("assertion error: #{location}" + (msg ? ": msg=#{msg};" : ""))
	end
    end

    class AbstractMethodInvokationError < KwartzError
	def initialize(class_and_method_name)
	    super("abstract method invoked: #{class_and_method_name}\n")
	end
    end
    
    ## ============================================================

    ##
    ## sample:
    ##	## presentation data
    ##	pdata = <<'END'
    ##	  <tr id="mark:user;attr:class=klass">
    ##	    <td id="value:user">foo</td>
    ##	  </tr>
    ##	END
    ##	
    ##	## presentation logic
    ##	plogic = <<'END'
    ##	  :macro(elem_user)
    ##	    :set(ctr = 0)
    ##	    :foreach(user = user_list)
    ##	      :set(ctr += 1)
    ##	      :set(klass = ctr%2 == 0 ? 'even' : 'odd')
    ##	      :expand(head_user)
    ##	      :expand(body_user)
    ##	      :expand(foot_user)
    ##	    :end
    ##	  :end
    ##	END
    ##	
    ##	## compile
    ##	compiler = Kwartz::Compiler.new('ruby')
    ##	ruby_code = compiler.compile(pdata, plogic)
    ##	print ruby_code
    ##
    class Compiler
	def initialize(lang='ruby', toppings={})
	    @lang  = lang
	    @toppings = toppings
	end

	def compile(input, plogic=nil, lang=@lang)
	    converter = Kwartz::DefaultConverter.new(input, @toppings)
	    kpl_code = converter.convert
	    kpl_code << plogic if plogic
	    scanner    = Kwartz::Scanner.new(kpl_code, @toppings)
	    parser     = Kwartz::Parser.new(scanner, @toppings)
	    translator = Kwartz::Translator.instance(lang, @toppings, scanner.newline)
	    unless translator
		raise InvalidLanguageError.new(lang)
	    end
	    nodelist   = parser.parse
	    output     = translator.translate_all(nodelist)
	    return output
	end
    end


    ##
    ## read input-file and plogic-file, compile it, and write output
    ## into output-file. 
    ##
    ## if output-file is newer than input-file and plogic-file,
    ## compile is not done, and return nil.
    ## Else compile and write into output-file, and return output string.
    ##
    ## arguments:
    ## * in_filename   -- mandatory. filename of input.
    ## * out_filename  -- mandatory. filename of output.
    ## * plogic_filename -- optional(defalt is nil). filename of
    ##   presentation logic file.
    ##   use nil when plogic-file is not required.
    ## * lang -- optional(default is "ruby")
    ##
    ## ex. for eRuby:
    ##	require 'kwartz'
    ##	Kwartz::compile('file.html', 'file.eruby', 'file.plogic', 'eruby')
    ##	ERuby::import('file.eruby')
    ##
    def self.compile(input_filename, output_filename, plogic_filename=nil, lang='ruby', toppings={})
	## check argument
	if !input_filename || input_filename.empty?
	    raise ArgumentError.new("input_filename required.")
	end
	if !output_filename || output_filename.empty?
	    raise ArgumentError.new("output_filename required.")
	end

	## check timestamp;
	## do nothing if output file is newer than input file and plogic file
	if test(?f, output_filename) &&
	   (out_mtime = File.mtime(output_filename)) >= File.mtime(input_filename)  &&
	   ( !plogic_filename || !test(?f, plogic_filename) || (out_mtime >= File.mtime(plogic_filename)) )
	    return nil
	end
	return compile!(input_filename, output_filename, plogic_filename, lang, toppings)
	####
	#dummy_loop = true
	#while dummy_loop
	#    break if !test(?f, output_filename)
	#    break if File.mtime(input_filename) > File.mtime(output_filename)
	#    break if test(?f, plogic_filename) && File.mtime(input_filename) > File.mtime(plogic_filename)
	#    return nil
	#end
	#return compile!(input_filename, output_filename, plogic_filename, lang)
	####
    end

    ##
    ## compile input-file and plogic-file, and write output into out-file.
    ## always compile even if out-file is newer than input-file or plogic-file.
    ##
    def self.compile!(input_filename, output_filename, plogic_filename=nil, lang='ruby', toppings={})
	input = File.open(input_filename) { |f| f.flock(File::LOCK_SH); f.read() }
	plogic = nil
	if plogic_filename && FileTest.exist?(plogic_filename)
	    plogic = File.open(plogic_filename) { |f| f.flock(File::LOCK_SH); f.read() }
	end
	compiler = Kwartz::Compiler.new(lang, toppings)
	output = compiler.compile(input, plogic)
	File.open(output_filename, "w") { |f| f.flock(File::LOCK_EX); f.write(output) }
	return output
    end

    ## ============================================================


    ## abstract class
    ## 
    ## ex.
    ##  class MyConverter < Converter
    ##    def initialize(input_str='', toppings={})
    ##      super(input_str, toppings)
    ##      ...
    ##    end
    ##    def convert
    ##      ...
    ##    end
    ##  end
    ##  
    ##  str = File.open('file.html') { |f| f.read }
    ##  converter = MyConverter.new(str)
    ##  print converter.convert
    ##
    class Converter
	def initialize(input_str='', toppings={})
	    #raise AbstractMethodInvokationError.new('Converter#initialize()')
	    @input = input_str
	    @toppings = toppings
	    @newline = "\n"
	    if (pos = input_str.index(?\n)) != nil
		@newline = "\r\n" if input_str[pos - 1] == ?\r
	    end
	end
	attr_accessor :input, :toppings, :newline
	
	def convert
	    raise AbstractMethodInvokationError.new('Converter#convert()')
	end
    end

    ##
    ## convert HTML template to intermediate-language program.
    ## 
    ## usage.
    ##	input = "<table>\n<tr kd="mark:foo">\n...</tr>\n</table>\n"
    ##	converter = Kwartz::DefaultConverter.new(input)
    ##	output = converter.convert
    ##	print output
    ##	
    class DefaultConverter < Converter

	##
	## utility class for DefaultConverter#_fetch(). don't use this.
	##
	class FetchValue
	    def initialize(tag_name, tag_type, attr_str, head_space, tail_space, \
			   head_text, tail_text, directive_name, directive_arg)
		@tag_name   = tag_name
		@tag_type   = tag_type
		@attr_str   = attr_str
		@head_space = head_space
		@tail_space = tail_space
		@head_text  = head_text
		@tail_text  = tail_text
		@directive_name = directive_name
		@directive_arg  = directive_arg
	    end
	    
	    attr_accessor :tag_name, :tag_type, :attr_str
	    attr_accessor :head_space, :tail_space, :head_text, :tail_text
	    attr_accessor :directive_name, :directive_arg

	    ##
	    ## re-struct tag string and return it
	    ##
	    def tag_str
		s = ''
		s << @head_space if @head_space
		s << (@tag_type == :end_tag ? '</' : '<')
		s << @tag_name
		s << @attr_str
		s << (@tag_type == :empty_tag ? '/>' : '>')
		s << @tail_space
		return s
	    end

	    ##
	    ## convert empty-tag to a pair of begin-tag and end-tag
	    ##
	    def empty2pair
		Kwartz::assert() unless self.tag_type == :empty_tag
		stag_fv = self
		etag_fv = stag_fv.dup
		#
		etag_fv.tag_type   = :end_tag
		etag_fv.attr_str   = ''
		etag_fv.head_space = ''
		etag_fv.head_text  = ''
		#
		stag_fv.tag_type   = :start_tag
		stag_fv.tail_space = ''
		stag_fv.tail_text  = ''
		# 
		return stag_fv, etag_fv
	    end

	end  ## end of class FetchValue
	

	##
	## * toppings key:
	##   attr_name     :: attribute name of marking (default "kd")
	##   header_text   :: header text (ex: '<%@ taglib ...%>' in JSP)
	##   footer_text   :: footer text
	##   header_plcode :: Presentation Language code for header
	##   footer_plcode :: Presentation Language code for footer
	##   odd_value     :: odd value for FOREACH or LIST (default "'odd'")
	##   even_value    :: even value for FOREACH or LIST (defalt "'even'")
	##
	def initialize(input_str='', toppings={})
	    #@input       = input_str
	    #@toppings     = toppings
	    super(input_str, toppings)
	    @tag_stack   = []
	    @macro_table = {}
	    @macro_list  = []
	    @attr_name   = (@toppings[:attr_name] || 'kd')	# 'id'? 'kwartz'?
	    @delete_idattr = @toppings[:delete_idattr]		# true/false
	end

	##
	def _add_macro(name, body)
	    @macro_table[name] = body
	    @macro_list << name
	end
	private :_add_macro

	##
	## usage.
	##  _each_macro do |name, body|
	##    ...
	##  end
	##
	def _each_macro(&block)
	    @macro_list.each do |name|
		body = @macro_table[name]
		yield(name, body)
	    end
	end
	private :_each_macro


	##
	## pattern for parsing tag
	##
	TAG_PATTERN = /(^[ \t]*)?<(\/?)([-.:\w]+)((?:\s+[-.:\w]+=(?:"[^"]*"|[^>\s]+))*)\s*(\/)?>/m     #"


	##
	## 
	##
	def _fetch
	    return nil unless @input =~ TAG_PATTERN
	    head_space    = $1
	    flag_endtag   = ($2 && !$2.empty?)
	    tag_name      = $3
	    attr_str      = $4
	    flag_emptytag = ($5 && !$5.empty?)
	    head_text	= $`
	    tail_text	= $'
	    tail_text	=~ /^([ \t]*\r?\n?)/
	    tail_space	= $1
	    tail_text	= $'

	    if flag_endtag
		tag_type = :end_tag
	    elsif flag_emptytag
		tag_type = :empty_tag
	    else
		tag_type = :start_tag
	    end

	    directive_name = directive_arg = nil
	    if attr_str =~ /\bid="(.*?)"/
		kd_str = $1
		directive_name, directive_arg, attr_str = _parse_directives(kd_str, attr_str, 'id')
	    end
	    if attr_str =~ /\b#{@attr_name}="(.*?)"/
		kd_str = $1
		d_name, d_arg, attr_str = _parse_directives(kd_str, attr_str, @attr_name)
		if d_name
		    directive_name = d_name
		    directive_arg  = d_arg
		end
	    end

	    @input = tail_text

	    return FetchValue.new(tag_name, tag_type, attr_str, head_space, \
		tail_space, head_text, tail_text, directive_name, directive_arg)
	end
	private :_fetch


	##
	## parse kwartz-attr string (like 'kwartz="mark:user;attr:class=klass"'),
	## and change other attributes if needed.
	## only _fetch() calls this method.
	##
	def _parse_directives(kwartz_str, attr_str, attr_name)
	    directive_name = directive_arg = nil
	    replace_list = nil
	    while kwartz_str.sub!(/(\w+):([^;]*);?/, '')
		d_name, d_arg = $1, $2
		case d_name
		when 'attr', 'Attr', 'ATTR'
		    case d_arg
		    when /\A([-.\w]+:[-.\w]+)[=:](.+)\z/
			a_name, a_expr = $1, $2
		    when /\A([-.\w]+)[=:](.+)\z/
			a_name, a_expr = $1, $2
		    else
			msg = "'#{d_name}:' must be '#{d_name}:name=value' or '#{d_name}:name:value'."
			raise ConvertionError.new(msg)
		    end
		    case d_name
		    when 'Attr' ;  a_expr = "E(#{a_expr})"
		    when 'ATTR' ;  a_expr = "X(#{a_expr})"
		    end
		    replace_pat = /\b#{a_name}=(".*?"|\S+)/
		    replace_str = a_name + '="#{' + a_expr + '}#"'		## *refpat*
		    replace_list ||= []
		    replace_list << [replace_pat, replace_str]
		else
		    directive_name = d_name;  directive_arg = d_arg
		    case directive_name
		    when 'mark', 'replace'
			unless directive_arg =~ /^\w+$/
			    msg = "'mark:' contains a invalid char."
			    raise ConvertionError.new(msg)
			end
		    when 'value', 'while', 'if', 'dummy', 'Value', 'VALUE'   #, 'unless'
			# do nothing
		    when 'set'
			unless directive_arg =~ /\A\w+(=|:|[-+*\/%]=)/
			    msg = "#{attr_name}=\"#{directive_name}:#{directive_arg}\": invalid assign expression."
			    raise ConvertionError.new(msg)
			end
			directive_arg.sub!(/\A(\w+):/, '\1=')
		    when 'foreach', 'list', 'Foreach', 'List', 'FOREACH', 'LIST'
			unless directive_arg =~ /\A\w+[=:]/
			    msg = "#{attr_name}=\"#{directive_name}:#{directive_arg}\": must be 'loopvar=expression'."
			    raise ConvertionError.new(msg)
			end
			directive_arg.sub!(/\A(\w+):/, '\1=')
		    when 'include', 'load'
			unless d_arg =~ /^'(.*)'$/
			    msg = "'#{d_name}' directive should be \"#{d_name}:'filename'\"."
			    raise ConvertionError.new(msg)
			end
			filename = $1
			directive_arg = filename
		    #else
		    #	msg = "#{attr_name}=\"#{directive_name}:#{directive_arg}\": invalid directive."
		    #	#raise ConvertionError.new(msg)
		    #	raise IllegalDirectiveError.new(msg)
		    end
		end
	    end

	    ## change attr_str
	    rest = kwartz_str
	    if rest.empty?
		attr_str.sub!(/\s+#{attr_name}=".*?"/, "")
	    else
		if !directive_name && rest =~ /^\w+$/
		    directive_name = 'mark'
		    directive_arg  = rest
		end
		if @delete_idattr
		    attr_str.sub!(/\s+#{attr_name}=".*?"/, "")
		else
		    attr_str.sub!(/(\s+)#{attr_name}=".*?"/, "\\1#{attr_name}=\"#{rest}\"")
		end
	    end
	    if replace_list
		replace_list.each do |pair|
		    replace_pat, replace_str = pair
		    if ! attr_str.sub!(replace_pat, replace_str)
			attr_str << ' ' << replace_str
		    end
		end
	    end

	    return directive_name, directive_arg, attr_str
	end
	private :_parse_directives


	##
	## escape "\n" and "'"
	##
	def _escape_text(str)
	    str.gsub!(/\n/, '\n')
	    str.gsub!(/\r/, '\r')
	    str.gsub!(/\t/, '\t')
	    #str.gsub!(/'/, '\\\\\'')	# "'" => "\'"		#"	## *str*
	    str.gsub!(/"/, '\\"')	# '"' => '\"'		#"	## *str*
	    return str
	end
	private :_escape_text


	##
	## parse embedded expression in text
	##
	## ex.
	##  "<tr class=#{klass}#>"  => "'<tr class=', klass, '>'"
	##
	def _embedded_expr(text)
	    output = ''
	    return output if text.empty?
	    rest = nil
	    text.scan(/(.*?)?\#\{(.*?)\}\#/) do |str, expr|		## *refpat*
		output << ", " if rest
		output << "\"#{_escape_text(str)}\", " if str && !str.empty?	## *str*
		rest = $'
		if expr[0] == ?@
		    case expr 
		    when /\A\@C(?:HECK)\((.*)\)\z/
			expr = "X(#{$1} ? ' checked=\"checked\"' : '')"
		    when /\A\@S(?:ELECT)\((.*)\)\z/
			expr = "X(#{$1} ? ' selected=\"selected\"' : '')"
		    when /\A\@D(?:ISABLE)\((.*)\)\z/
			expr = "X(#{$1} ? ' disabled=\"disabled\"' : '')"
		    end
		end
		output << expr
	    end
	    if rest
		output << ", \"#{_escape_text(rest)}\"" if !rest.empty?		## *str*
	    else
		output << "\"#{_escape_text(text)}\""				## *str*
	    end
	    return output
	end
	private :_embedded_expr


	##
	## convert text to :print() statement
	## * split each line
	## * expand '#{}#' notation
	## * escape "\n" and "'"
	##
	## ex.
	##     <td class="#{klass}">\nI'm file.\n</td>
	##     ==>  :print('<td class="', klass, '>\n')
	##          :print('I\'m fine.\n')
	##          :print('<td>')
	##
	def _print_text(text)
	    output = ''
	    return output if text.empty?
	    text.scan(/^.*\r?\n?/) do |line|		# /^.*\$/
		output << ":print("
		output << _embedded_expr(line)
		output << ")" << @newline
	    end
	    return output
	end
	private :_print_text



	##
	## convert a template file string into PL code and return it.
	##
	def convert
	    plcode = ''
	    if @toppings[:header_text]
		@toppings[:header_text].each_line do |line|
		    plcode << ":print(\"#{_escape(line)}\")#{@newline}"
		end
	    end
	    if @toppings[:header_plcode]
		plcode << @toppings[:header_plcode]
	    end

	    text = ''
	    while fv = _fetch() do
		text << fv.head_text
		if fv.directive_name
		    plcode << _print_text(text)
		    text = ''
		    plcode << _convert_element(fv)
		else
		    text << fv.tag_str()
		end
	    end
	    plcode << _print_text(text)

	    rest = @input
	    plcode << _print_text(rest) if ! rest.empty?

	    s = ''
	    _each_macro do |name, body|
		s << ":macro(#{name})" << @newline
		s << body.gsub(/^/, '  ')
		s << ":end" << @newline
		s << @newline
	    end
	    plcode = s + plcode

	    if @toppings[:footer_text]
		@toppings[:footer_text].each_line do |line|
		    plcode << ":print('#{_escape(line)}')#{@newline}"
		end
	    end
	    if @toppings[:footer_plcode]
		plcode << @toppings[:footer_plcode]
	    end

	    return plcode
	end


	##
	## convert '<tag id="...">....</tag>' into PL code and return it.
	##
	## this is called from convert() and _convert_content().
	##
	def _convert_element(stag_fv)
	    Kwartz::assert() unless stag_fv.tag_type != :end_tag
	    Kwartz::assert() unless stag_fv.directive_name
	    case stag_fv.tag_type
	    when :start_tag
		cont_plcode, etag_fv = _convert_content(stag_fv.tag_name)
		unless etag_fv
		    msg = "'<#{stag_fv.tag_name}>': start tag not closed."
		    raise UnclosedTagError.new(msg)
		end
	    when :empty_tag
		cont_plcode = "## empty#{@newline}"
		etag_fv = nil
	    when :end_tag
		Kwartz::assert()
	    end

	    if !etag_fv
		## ex. '<li id="value:user"/>' => '<li>' and '</li>'
		case stag_fv.directive_name
		when 'value', 'Value', 'VALUE', 'include', 'load'
		    stag_fv, etag_fv = stag_fv.empty2pair()
		end
	    end

	    stag_str    = stag_fv.tag_str
	    stag_plcode = _print_text(stag_str)
	    if etag_fv
		etag_str    = etag_fv.tag_str
		etag_plcode = _print_text(etag_str)
	    else
		etag_str    = ''
		etag_plcode = "## empty#{@newline}"
	    end
	    if stag_str =~ /\A\s*<span\s*\/?>\s*\z/
		stag_str = etag_str = ''
		stag_plcode = '## ' + stag_plcode
		etag_plcode = '## ' + etag_plcode if etag_fv
	    end

	    plcode = ''
	    directive_name = stag_fv.directive_name
	    directive_arg  = stag_fv.directive_arg
	    case directive_name
	    when 'value', 'Value', 'VALUE'
		case directive_name
		when 'value' ;  expr = directive_arg
		when 'Value' ;  expr = "E(#{directive_arg})"
		when 'VALUE' ;  expr = "X(#{directive_arg})"
		end
		if stag_str.empty?
		    plcode << ":print(#{expr})#{@newline}"
		else
		    plcode << ":print(#{_embedded_expr(stag_str)}, "
		    plcode << "#{expr}, "
		    plcode << "#{_embedded_expr(etag_str)})#{@newline}"
		end

	    when 'dummy'
		# do nothing; ignore output

	    when 'set'
		assign_expr = directive_arg
		plcode << ":set(#{assign_expr})#{@newline}"
		plcode << stag_plcode
		plcode << cont_plcode if etag_fv
		plcode << etag_plcode if etag_fv

	    when 'if', 'while'
		expr = directive_arg
		stag_plcode.gsub!(/^/, '  ')
		cont_plcode.gsub!(/^/,  '  ')
		etag_plcode.gsub!(/^/,   '  ')
		plcode << ":#{directive_name}(#{expr})#{@newline}"
		plcode << stag_plcode
		plcode << cont_plcode if etag_fv
		plcode << etag_plcode if etag_fv
		plcode << ":end#{@newline}"

	    when 'mark'
		name = directive_arg
		plcode << ":expand(elem_#{name})#{@newline}"
		#
		_add_macro("stag_#{name}", stag_plcode)
		_add_macro("cont_#{name}", cont_plcode)
		_add_macro("etag_#{name}", etag_plcode)
		c = ''
		c <<  ":expand(stag_#{name})#{@newline}"
		c <<  ":expand(cont_#{name})#{@newline}"
		c <<  ":expand(etag_#{name})#{@newline}"
		_add_macro("elem_#{name}", c)

	    when 'foreach', 'list', 'Foreach', 'List', 'FOREACH', 'LIST'
		directive_arg =~ /(\w+)=(.*)/
		name, expr = $1, $2
		case directive_name
		when 'Foreach', 'List'
		    counter_name = name + '_ctr'
		when 'FOREACH', 'LIST'
		    counter_name = name + '_ctr'
		    toggle_name	 = name + '_tgl'
		    odd_value	 = @toppings[:odd_value]  || "'odd'"
		    even_value	 = @toppings[:even_value] || "'even'"
		else
		    counter_name = toggle_name = nil
		end
		if @toppings[:create_foreach_macro]
		    _add_macro("stag_#{name}", stag_plcode)
		    _add_macro("cont_#{name}", cont_plcode)
		    _add_macro("etag_#{name}", etag_plcode)
		    stag_plcode = ":expand(stag_#{name})#{@newline}"
		    cont_plcode = ":expand(cont_#{name})#{@newline}"
		    etag_plcode = ":expand(etag_#{name})#{@newline}"
		end
		c = ''
		case directive_name
		when 'foreach', 'Foreach', 'FOREACH'
		    stag_plcode.gsub!(/^/, '  ')
		    cont_plcode.gsub!(/^/, '  ')
		    etag_plcode.gsub!(/^/, '  ')
		    c <<  ":set(#{counter_name} = 0)#{@newline}"    if counter_name
		    c <<  ":foreach(#{name} = #{expr})#{@newline}"
		    c <<  "  :set(#{counter_name} += 1)#{@newline}" if counter_name
		    c <<  "  :set(#{toggle_name} = #{counter_name} % 2 == 0 ? #{even_value} : #{odd_value})#{@newline}" if toggle_name
		    c <<  stag_plcode
		    c <<  cont_plcode
		    c <<  etag_plcode
		    c <<  ":end#{@newline}"
		when 'list', 'List', 'LIST'
		    cont_plcode.gsub!(/^/, '  ')
		    c <<  stag_plcode
		    c <<  ":set(#{counter_name} = 0)#{@newline}"     if counter_name
		    c <<  ":foreach(#{name} = #{expr})#{@newline}"
		    c <<  "  :set(#{counter_name} += 1)#{@newline}" if counter_name
		    c <<  "  :set(#{toggle_name} = #{counter_name} % 2 == 0 ? #{even_value} : #{odd_value})#{@newline}" if toggle_name
		    c <<  cont_plcode
		    c <<  ":end#{@newline}"
		    c <<  etag_plcode
		end
		if @toppings[:create_foreach_macro]
		    _add_macro("elem_#{name}", c)
		    plcode << ":expand(elem_#{name})#{@newline}"
		else
		    plcode << c
		end

	    when 'replace'
		plcode << ":expand(elem_" << directive_arg << ")#{@newline}"

	    when 'include'
		filename = directive_arg
		if @toppings[:include_path]
		    @toppings[:include_path].each do |dir|
			if test(?d, dir) && test(?f, "#{dir}/#{filename}")
			    filename = "#{dir}/#{filename}"
			    break
			end
		    end
		end
		unless test(?f, filename)
		    msg = "include directive: file '#{filename}' not found."
		    raise ConvertionError.new(msg)
		end
		pdata = File.open(filename) { |f| f.read }
		plcode << stag_plcode
		plcode << Kwartz::Helper.convert(pdata, @toppings)
		plcode << etag_plcode

	    when 'load'
		filename = directive_arg
		plcode << stag_plcode
		plcode << ":load('#{filename}')#{@newline}"
		plcode << etag_plcode

	    else
		msg = "#{directive_name}: illegal directive."
		raise IllegalDirectiveError.new(msg)
	    end
	    return plcode
	end
	private :_convert_element

	
	##
	## convert body (between '<tag id="...">' and '</tag>')
	## into PL code.
	##
	## this is called from _convert_element().
	##
	def _convert_content(stag_tagname)
	    plcode = ''
	    tag_level = 1
	    text = ''
	    while fv = _fetch() do
		text << fv.head_text
		if fv.directive_name
		    plcode << _print_text(text)
		    text = ''
		    plcode << _convert_element(fv)
		elsif fv.tag_name == stag_tagname
		    case fv.tag_type
		    when :start_tag
			tag_level += 1
		    when :end_tag
			tag_level -= 1
			break if tag_level == 0		# break!
		    end
		    text << fv.tag_str()
		else
		    text << fv.tag_str()
		end
	    end
	    plcode << _print_text(text)
	    return plcode, fv
	end
	private :_convert_content


	private
	
	def _escape(line)
	    return _escape!(line.dup)
	end
	
	def _escape!(line)
	    line.gsub!(/\n/, '\n')
	    line.gsub!(/\r/, '\r')
	    line.gsub!(/\t/, '\t')
	    #line.gsub!(/'/, '\\\\\'')
	    line.gsub!(/"/, '\\"')	#'
	    return line
	end
	
    end # end of class DefaultConverter


    
    ## returns Node of intermediate code instead of string of it.
    ## *it is under construction*.
    class NodeConverter < DefaultConverter
	
	def initialize(input_str='', toppings={})
	    super(input_str, toppings)
	end
	
	def convert
	    ## under construction
	    return ndelist
	end
	
    end

    
    ## ============================================================


    ##
    ## vistor pattern for Node and it's subclass
    ##
    module Action
	## don't override in subclass
	def exec(node, arg1=nil, arg2=nil)
	    node.execute(self, arg1, arg2) if node
	end

	## abstract method
	def exec_expr(exprnode, arg1=nil, arg2=nil)
	    raise AbstractMethodInvocationError("#{self.class.name}#exec_expr()")
	end

	## abstract method
	def exec_stmt(stmtnode, arg1=nil, arg2=nil)
	    raise AbstractMethodInvocationError("#{self.class.name}#exec_stmt()")
	end

	## abstract method
	def exec_nodelist(nodelist, arg1=nil, arg2=nil)
	    #raise AbstractMethodInvocationError("#{self.class.name}#exec_nodelist()")
	    nodelist.each do |node|
		node.execute(self, arg1, arg2)
	    end
	end
    end


    ##
    ## abstract class for ExprNode, StmtNode, NodeList
    ##
    class Node
	def initialize(token, left=nil, right=nil)
	    @token = token
	    @left  = left
	    @right = right
	end
	attr_accessor :token, :left, :right

	## for visitor pattern
	def execute(action, arg1=nil, arg2=nil)
	    raise AbstractMethodInvocationError("#{self.class.name}#execute()")
	end
	#alias act execute

	## for translator
	def translate(translator, code='', level=0)
	    raise AbstractMethodInvocationError("#{self.class.name}#translate()")
	end

	##
	## ex.
	##  expression_node.traverse { |node| p node.token }
	##
	def traverse(&block)
	    yield(self)
	    @left.traverse(&block)	if @left      && @left.is_a?(Node)
	    @right.traverse(&block)	if @right     && @right.is_a?(Node)
	    @condition.traverse(&block) if @condition && @condition.is_a?(Node)
	end

	##
	## ex.
	##  expression_node.traverse_with_depth { |node, depth|
	##    indent = ' ' * depth
	##    print indent
	##    p node.token
	##  }
	##
	def traverse_with_depth(depth=0, &block)
	    yield(self, depth)
	    [@left, @right, @condition].each do |node|
		if node && node.is_a?(Node)
		    node.traverse_with_depth(depth+1, &block)
		end
	    end
	end

	##
	## ex.
	##  condop_node = nil
	##  parent_node = nil
	##  expression_node.traverse_with_parent { |node, depth, parent|
	##    if node.token == '?'
	##	condop_node = node
	##	parent_ndoe = parent
	##	break
	##    end
	##  }
	##
	def traverse_with_parent(depth=0, parent=nil, &block)
	    yield(self, depth, parent)
	    [@left, @right, @condition].each do |node|
		if node && node.is_a?(Node)
		    node.traverse_with_parent(depth+1, self, &block)
		end
	    end
	end

	##
	## ex.
	##   ## find a expression node which token is '?'
	##   condop_node = expr_node.find { |node| node.token == '?' }
	##
	def find(&block)
	    return self if yield(self)
	    [@condition, @left, @right].each do |child|
		if child && child.is_a?(Node)
		    ret = child.find(&block)
		    return ret if ret
		end
	    end
	    return nil
	end
	
	##
	## ex.
	##   ## get expression node which token is '?', and it's parent
	##   condop_node, parent_node = expr_node.find_with_parent {
	##	 |node| node.token == '?'
	##   }
	##
	def find_with_parent(parent=nil, &block)
	    return self, parent if yield(self)
	    [@condition, @left, @right].each do |child|
		if child && child.is_a?(Node)
		    ret = child.find_with_parent(self, &block)
		    return ret if ret
		end
	    end
	    return nil
	end

	##
	def deepcopy
	    node = self.dup
	    (node.left  = @left.is_a?(Node)  ? @left.deepcopy  : @left.dup)  if @left
	    (node.right = @right.is_a?(Node) ? @right.deepcopy : @right.dup) if @right
	    (node.condition = @condition.is_a?(Node) ? @condition.deepcopy : @condition.dup) if @condition
	    return node
	end

	## for debug
	def print_all(newline="\n")
	    str = ''
	    NodePrinter.new(newline).exec(self, str, 0)
	    return str
	end
    end


    ##
    ## for IfStmtNode, WhileStmtNode, ExprNode(conditional operator)
    ##
    module Conditional
	attr_accessor :condition
    end

    
    ##
    ## Node for Expression
    ##
    class ExprNode < Node
	include Conditional
	
	## for visitor pattern
	def execute(action, arg1=nil, arg2=nil)
	    action.exec_expr(self, arg1, arg2)
	end

	## for translator
	def translate(translator, code='', level=0)
	    translator.translate_expr(self, code, level)
	    return code
	end
    end


    ##
    ## Node for Statement
    ##
    ## class StmtNode is an abstract class.
    ## you should use subclasses instead of StmtNode.
    ##
    class StmtNode < Node
	## for visitor pattern
	def execute(action, arg1=nil, arg2=nil)
	    action.exec_stmt(self, arg1, arg2)
	end

	## abstract method. subclass should override this method.
	def translate(translator, code='', level=0)
	    #translator.translate_stmt(self, code, level)
	    #return code
	    name = "StmtNode#translate()"
	    raise AbstractMethodInvokationError.new(name)
	end
    end

    class SetStmtNode < StmtNode
	def translate(translator, code='', level=0)
	    translator.translate_set_stmt(self, code, level)
	    return code
	end
    end

    class PrintStmtNode < StmtNode
	def translate(translator, code='', level=0)
	    translator.translate_print_stmt(self, code, level)
	    return code
	end
    end

    class Print2StmtNode < StmtNode
	def translate(translator, code='', level=0)
	    translator.translate_print2_stmt(self, code, level)
	    return code
	end
    end

    class IfStmtNode < StmtNode
	include Conditional
	def translate(translator, code='', level=0)
	    translator.translate_if_stmt(self, code, level)
	    return code
	end
	def initialize(token, left, right, cond_expr=nil)
	    super(token, left, right)
	    self.condition = cond_expr
	end
    end

    class WhileStmtNode < StmtNode
	include Conditional
	def translate(translator, code='', level=0)
	    translator.translate_while_stmt(self, code, level)
	    return code
	end
	def initialize(token, left, right, cond_expr=nil)
	    super(token, left, right)
	    self.condition = cond_expr
	end
    end

    class ForeachStmtNode < StmtNode
	def translate(translator, code='', level=0)
	    translator.translate_foreach_stmt(self, code, level)
	    return code
	end
    end

    class MacroStmtNode < StmtNode
	def translate(translator, code='', level=0)
	    translator.translate_macro_stmt(self, code, level)
	    return code
	end
    end

    class ExpandStmtNode < StmtNode
	def translate(translator, code='', level=0)
	    translator.translate_expand_stmt(self, code, level)
	    return code
	end
    end

    class RawcodeStmtNode < StmtNode
	def translate(translator, code='', level=0)
	    translator.translate_rawcode_stmt(self, code, level)
	    return code
	end
    end


    ##
    ## List of StmtNode
    ##
    class NodeList < Node
	def initialize(list=[])
	    @list = list
	end
	attr_reader :list

	def <<(node)
	    @list << node
	end

	def each(&block)
	    @list.each(&block)
	end

	def each_with_index(&block)
	    @list.each_with_index(&block)
	end

	def [](idx)
	    @list.idx
	end

	def []=(idx,value)
	    @list[idx] = value
	end

	## for visitor pattern
	def execute(action, arg1=nil, arg2=nil)
	    action.exec_nodelist(self, arg1, arg2)
	end

	## for translator
	def translate(translator, code='', level=0)
	    translator.translate_nodelist(self, code, level)
	    return code
	end
	
	def deepcopy
	    newlist = []
	    @list.each do |node|
		newlist << node.deepcopy
	    end
	    node = super()
	    node.instance_eval("@list = newlist")
	    return node
	end
    end


    ##
    class NodePrinter
	include Action
	
	def initialize(newline="\n")
	    @newline = newline
	end

	##
	## action for expr_node
	##
	def exec_expr(expr_node, str, level)
	    token = expr_node.token
	    left  = expr_node.left
	    right = expr_node.right
	    cond  = expr_node.condition
	    #
	    str << '  ' * level
	    case token
	    when :string
		str << left.inspect << @newline
	    when :variable, :number
		str << left << @newline
	    when '[:]', '.'
		str << token << @newline
		exec(left, str, level+1)
		str << '  ' * (level+1) << right << @newline
	    when :true, :false, :null
		str << token.id2name << @newline
	    when :function
		str << left << "()" << @newline
		exec(right, str, level+1) if right
	    else
		str << (token.is_a?(Symbol) ? token.inspect : token)
		str << @newline
		Kwartz::assert() unless !cond  || cond.is_a?(ExprNode)
		Kwartz::assert() unless !left  || left.is_a?(ExprNode)
		Kwartz::assert() unless !right || right.is_a?(ExprNode)
		exec(cond,  str, level+1) if cond
		exec(left,  str, level+1) if left
		exec(right, str, level+1) if right
	    end
	    return str
	end

	##
	## action for StmtNode
	##
	def exec_stmt(stmt_node, str, level)
	    token = stmt_node.token
	    left  = stmt_node.left
	    right = stmt_node.right
	    #
	    str << '  ' * level
	    str << (token.is_a?(Symbol) ? token.inspect : token)
	    str << @newline
	    if stmt_node.is_a?(Conditional) && (cond = stmt_node.condition)
		exec(cond, str, level+1) 
	    end
	    case token
	    when :expand, :macro, :rawcode, ':::'
		Kwartz::assert() unless left.is_a?(String)
		str << '  ' * (level+1) << left << @newline
	    else
		exec(left, str, level+1) if left
	    end
	    exec(right, str, level+1) if right
	    return str
	end

	##
	## action for NodeList
	##
	def exec_nodelist(nodelist, str, level)
	    list = nodelist.list
	    #
	    str << '  ' * level << "<nodelist>" << @newline
	    list.each do |node|
		exec(node, str, level+1)
		#node.execute(self, str, level+1)
	    end
	    return str
	end
    end


    ## ============================================================


    ##
    ## abstract class for RubyTranslator, PHPTranslator, ...
    ##
    ## usage.
    ##	scanner = Kwartz::Scanner.new(input_str)
    ##	parser	= Kwartz::Parser.new(scanner)
    ##	nodelist = parser.parse()
    ##	translator = Kwartz::Translator.instance("ruby", scanner.newline)
    ##	output_str = translator.translate(nodelist)
    ##	print output_str
    ##
    ## If you are defining a subclass of Translator, you should do the
    ## following task:
    ## * override a method translate_expr(node, code='', level=0),
    ##	 which translate ExprNode to program code
    ## * override a method translate_stmt(node, code='', level=0),
    ##	 which translate StmtNode to program code
    ## * set @codedect appropriately, which is a hash to
    ##	 convert :if, :then, :end, ... to "if (", ") {\n", "}\n", ... ,
    ##	 for example.
    ##	 notice that @codedect is not an instance variable, but a
    ##	 class-instance variable.
    ## * register the subclass to Translater by Translator::register().
    ## 
    ## See the definition of RubyTranslator for detail.
    ##
    class Translator
	##
	## if you define a subclass of Translator, you should call
	## this method with self and a language name.
	##
	def self.register(class_obj, lang_str)
	    @@class_list[lang_str] = class_obj
	end
	@@class_list = {}

	
	##
	## return true if class-object which is associated with lang_str is registered
	##
	def self.registered?(lang_str)
	    return @@class_list[lang_str] != nil
	end
	
	##
	## get a instance of appropriate Translator subclass.
	##
	## usage.
	##  translator = Translator.instance("ruby")
	##
	def self.instance(lang_str, toppings={}, newline="\n")
	    class_obj = @@class_list[lang_str]
	    instance = class_obj ? class_obj.new(toppings, newline) : nil
	    return instance
	end


	##
	##
	##
	def encode(key)
	    return @codedict[key]
	    #return @word[key]
	    #return self.class.codedict[key]
	end
	#private :encode

	
	##
	## base of codedict hashtable
	##
	def self.base_codedict
	    return @@base_codedict
	end
	@@base_codedict = {
	    :indent	=> '	',
	    
	    :set	=> "",
	    :endset	=> ";",
	    :if		=> "if (",
	    :then	=> ") {\n",
	    :endthen	=> nil,
	    :elsif	=> "} else if (",
	    :else	=> "} else {\n",
	    :endelse	=> nil,
	    :endif	=> "}\n",
	    :while	=> "while (",
	    :do		=> ") {\n",
	    :endwhile	=> "}\n",
	    :foreach	=> "foreach (",
	    :in		=> " in ",
	    :endforeach	=> "}\n",
	    :print	=> "print(",
	    :endprint	=> ")\n",
	    :print2	=> "print(",
	    :endprint2	=> ")\n",
	    :rawcode	=> "",
	    :endrawcode	=> "\n",

	    :true       => 'true',
	    :false      => 'false',
	    :null       => 'null',

	    '['		=> '[',
	    ']'		=> ']',
	    '{'		=> '{',
	    '}'		=> '}',
	    '[:'	=> "{'",
	    ':]'	=> "'}",

	    '+'		=> ' + ',
	    '-'		=> ' - ',
	    '*'		=> ' * ',
	    '/'		=> ' / ',
	    '%'		=> ' % ',
	    '^'		=> ' ^ ',

	    '.+'	=> ' + ',

	    '=='	=> ' == ',
	    '!='	=> ' != ',
	    '>'		=> ' > ',
	    '>='	=> ' >= ',
	    '<'		=> ' < ',
	    '<='	=> ' <= ',

	    '.=='	=> ' == ',
	    '.!='	=> ' != ',
	    '.>'	=> ' > ',
	    '.>='	=> ' >= ',
	    '.<'	=> ' < ',
	    '.<='	=> ' <= ',

	    '='		=> ' = ',
	    '+='	=> ' += ',
	    '-='	=> ' -= ',
	    '*='	=> ' *= ',
	    '/='	=> ' /= ',
	    '%='	=> ' %= ',
	    '^='	=> ' ^= ',

	    '.+='	=> ' .= ',

	    '('		=> '(',
	    ')'		=> ')',
	    '?'		=> ' ? ',
	    ':'		=> ' : ',
	    '.'		=> '.',

	    '!'		=> '! ',
	    '&&'        => ' && ',
	    '||'        => ' || ',

	    ','         => ', ',

	    'E' 	=> ['', ''],
	    'X' 	=> ['', ''],
	}

	def self.init_codedict(codedict_diff={})
	    @codedict = base_codedict().dup
	    @codedict.update(codedict_diff)
	    
	    @codedict_crlf = {}
	    @codedict.each do |key, value|
		@codedict_crlf[key] = value.is_a?(String) ? value.gsub(/\n/, "\r\n") : value
	    end
	end


	## return a codedict hashtable
	def self.codedict
	    return @codedict
	end
	
	def self.codedict_crlf
	    return @codedict_crlf
	end


	##
	def initialize(toppings={}, newline="\n")
	    @toppings = toppings
	    @macros = {}
	    @newline = newline
	    #@word = self.class.codedict
	    @codedict = newline == "\r\n" ? self.class.codedict_crlf : self.class.codedict
	end


	##
	def _add_macro(name, nodelist)
	    @macros[name] = nodelist
	end
	private :_add_macro

	## return value: NodeList
	def _get_macro(name)
	    return @macros[name]
	end
	private :_get_macro

	##
	def translate_all(nodelist, code='', level=0)
	    Kwartz::assert() unless nodelist.is_a?(NodeList)
	    code << @toppings[:header] if @toppings[:header]
	    translate(nodelist, code, level)
	    begin_nodelist = _get_macro('BEGIN')  # || _get_macro('elem_BEGIN')
	    end_nodelist   = _get_macro('END')    # || _get_macro('elem_END')
	    if begin_nodelist
		begin_code = ''
		translate(begin_nodelist, begin_code, 0)
		code = begin_code + code
	    end
	    if end_nodelist
	    	end_code = ''
	    	translate(end_nodelist, end_code, 0)
	    	code << end_code
	    end
	    code << @toppings[:footer] if @toppings[:footer]
	    return code
	end

	## node may be ExprNode, StmtNode or NodeList.
	def translate(node, code='', level=0)
	    return code unless node
	    node.translate(self, code, level)
	    return code
	end

	## translate expr node, with comparing priority of operators
	def _translate_child_expr(code, operator, child_node)
	    if Parser.prior?(operator, child_node.token)
		code << encode('(')
		translate(child_node, code)
		code << encode(')')
	    else
		translate(child_node, code)
	    end
	    return code
	end
	private :_translate_child_expr


	## tranlsate ExprNode
	def translate_expr(node, code='', level=0)
	    case tkn = node.token
	    when :string
		value_str = node.left
		code << "'"
		code << value_str.gsub(/'/, "\\\\'")		#"
		code << "'"

	    when :variable, :number
		value_str = node.left
		code << value_str

	    when :true, :false, :null	#, :empty
		code << encode(tkn)

	    when :function
		funcname = node.left
		obj = encode(funcname)
		if !obj
		    translate(node.right, code)
		elsif obj.is_a?(Array)
		    code << obj[0]
		    translate(node.right, code)
		    code << obj[1]
		elsif obj.is_a?(Proc)
		    obj.call(self, node.right, code)
		else
		    translate(node.right, code)
		end
		
	    when '[]'
		translate(node.left, code)
		code << encode('[')
		translate(node.right, code)
		code << encode(']')

	    when '{}'						## obsolete
		translate(node.left, code)
		code << encode('{')
		translate(node.right, code)
		code << encode('}')

	    when '[:]'
		translate(node.left, code)
		code << encode('[:')
		code << node.right
		code << encode(':]')

	    when '.'
		translate(node.left, code)
		code << encode('.')
		code << node.right

	    when '+', '-', '*', '/', '%', '^', '.+'
		if node.right
		    #translate(node.left, code)
		    #code << encode(tkn)
		    #translate(node.right, code)
		    _translate_child_expr(code, tkn, node.left)
		    code << encode(tkn)
		    _translate_child_expr(code, tkn, node.right)
		else			## unary operator '+' or '-'
		    code << encode('(')
		    code << encode(tkn)
		    _translate_child_expr(code, tkn, node.left)
		    code << encode(')')
		end

	    when :empty, :notempty
		## s==empty  => (s==null || s==''),  s!=empty => (s!=null && s!='')
		op1, op2 = '==', '||'
		op1, op2 = '!=', '&&' if tkn == :notempty
		str_node  = ExprNode.new(:string, '', nil)
		null_node = ExprNode.new(:null, nil, nil)
		l_node    = ExprNode.new(op1, node.left, null_node)
		r_node    = ExprNode.new(op1, node.left, str_node)
		node2     = ExprNode.new(op2, l_node, r_node)
		code << encode('(')
		translate_expr(node2, code, level)
		code << encode(')')

	    when '==', '!=', '<', '<=', '>', '>='
		#translate(node.left, code)
		#code << encode(tkn)
		#translate(node.right, code)
		_translate_child_expr(code, tkn, node.left)
		code << encode(tkn)
		_translate_child_expr(code, tkn, node.right)

	    when '.==', '.!=', '.<', '.<=', '.>', '.>='			## obsolete
		#translate(node.left, code)
		#code << encode(tkn)
		#translate(node.right, code)
		_translate_child_expr(code, tkn, node.left)
		code << encode(tkn)
		_translate_child_expr(code, tkn, node.right)

	    when '=', '+=', '-=', '*=', '/=', '%=', '^=', '.+='
		translate(node.left, code)
		code << encode(tkn)
		translate(node.right, code)

	    when '!'
		#code << encode('(')
		code << encode(tkn)
		_translate_child_expr(code, tkn, node.left)
		#code << encode(')')

	    when '?'
		code << encode('(')
		translate(node.condition, code)
		code << encode('?')
		translate(node.left, code)
		code << encode(':')
		translate(node.right, code)
		code << encode(')')

	    when ','
		#translate(node.left, code)
		#code << encode(',')
		#translate(node.right, code)
		_translate_child_expr(code, tkn, node.left)
		code << encode(tkn)
		_translate_child_expr(code, tkn, node.right)

	    when '&&', '||'
		flag_l = tkn == '||' && node.left.token  == '&&'
		flag_r = tkn == '||' && node.right.token == '&&'
		code << encode('(') if flag_l
		_translate_child_expr(code, tkn, node.left)
		code << encode(')') if flag_l
		code << encode(tkn)
		code << encode('(') if flag_r
		_translate_child_expr(code, tkn, node.right)
		code << encode(')') if flag_r

	    else
		Kwartz::assert("node.token=#{node.token.inspect}")
	    end

	    return code
	end

	def indent(code, level)
	    prefix = encode(:prefix)
	    indent = encode(:indent)
	    if prefix
		code << @newline if code[-1] != ?\n
		code << prefix
		code << indent * level if level > 0
	    elsif level > 0
		code << indent * level if code[-1] == ?\n
	    end
	end

	## tranlsate StmtNode
	def translate_stmt(stmt_node, code='', level=0)
	    stmt_node.translate_stmt(self, code, level)
	    return code
	end

	def translate_set_stmt(stmt_node, code='', level=0)
	    assign_op = stmt_node.left.token
	    lhs_expr  = stmt_node.left.left
	    rhs_expr  = stmt_node.left.right
	    #
	    indent(code, level)
	    code << encode(:set)
	    translate(lhs_expr, code)
	    code << encode(assign_op)
	    translate(rhs_expr, code)
	    code << encode(:endset)
	    return code
	end

	#--
	#def translate_print_stmt(node, code='', level=0)
	#    arg_expr = node.left
	#    #
	#    indent(code, level)
	#    code << encode(:print)
	#    translate(arg_expr, code)
	#    code << encode(:endprint)
	#    return code
	#end
	#++
	def translate_print_stmt(stmt_node, code='', level=0)
	    case @toppings[:escape]
	    when true
		startkey = :print2 ; endkey = :endprint2
	    when false
	    	startkey = :print  ; endkey = :endprint
	    else
		startkey = :print  ; endkey = :endprint		# default: no-sanitize
		#startkey = :print2 ; endkey = :endprint2	# default: sanitize
	    end
	    return _translate_print_args(stmt_node.left, code, level, startkey, endkey)
	end

	def _translate_print_args(expr_node, code, level, startkey, endkey)
	    case expr_node.token
	    when :string, :number
		code << expr_node.left.to_s
		return code
	    when ','
		_translate_print_args(expr_node.left,  code, level, startkey, endkey)
		_translate_print_args(expr_node.right, code, level, startkey, endkey)
		return code
	    else
		if expr_node.token == :function
		    case expr_node.left
		    when 'X'
			startkey = :print  ; endkey = :endprint
			expr_node = expr_node.right
			case expr_node.token
			when :string, :number
			    code << expr_node.left.to_s
			    return code
			end
		    when 'E'
			startkey = :print2 ; endkey = :endprint2
			expr_node = expr_node.right
		    end
		end
		code << encode(startkey)
		translate_expr(expr_node, code, level)
		code << encode(endkey)
	    end
	    return code
	end
	private :_translate_print_args

	def translate_if_stmt(stmt_node, code='', level=0)
	    node = stmt_node
	    flag_first = true
	    while node.token == :if do
		cond_expr     = node.condition
		then_nodelist = node.left
		else_node     = node.right		# StmtNode or NodeList
		#
		indent(code, level)
		code << (flag_first ? encode(:if) : encode(:elsif))
		translate(cond_expr, code)
		code << encode(:then)
		translate(then_nodelist, code, level+1)
		if encode(:endthen)
		    indent(code, level)
		    code << encode(:endthen)
		end
		#
		node = else_node
		break if node == nil
		flag_first = false
	    end
	    if node
		Kwartz::assert() unless node.is_a?(NodeList)
		indent(code, level)
		code << encode(:else)
		translate(node, code, level+1)
		if encode(:endelse)
		    indent(code, level)
		    code << encode(:endelse)
		end
	    end
	    indent(code, level)
	    code << encode(:endif)
	    return code
	end

	def translate_while_stmt(stmt_node, code='', level=0)
	    cond_expr = stmt_node.condition
	    nodelist  = stmt_node.left
	    #
	    indent(code, level)
	    code << encode(:while)
	    translate(cond_expr, code)
	    code << encode(:do)
	    translate(nodelist, code, level+1)
	    indent(code, level)
	    code << encode(:endwhile)
	    return code
	end

	def translate_foreach_stmt(stmt_node, code='', level=0)
	    assign_node = stmt_node.left
	    nodelist    = stmt_node.right
	    Kwartz::assert() unless assign_node.token == '='
	    var_expr    = assign_node.left
	    list_expr   = assign_node.right
	    #
	    indent(code, level)
	    code << encode(:foreach)
	    translate(var_expr, code)
	    code << encode(:in)
	    translate(list_expr, code)
	    code << encode(:do)
	    translate(nodelist, code, level+1)
	    indent(code, level)
	    code << encode(:endforeach)
	    return code
	end

	def translate_macro_stmt(stmt_node, code='', level=0)
	    name = stmt_node.left
	    nodelist = stmt_node.right
	    _add_macro(name, nodelist)
	    return code
	end

	def translate_expand_stmt(stmt_node, code='', level=0)
	    name = stmt_node.left
	    nodelist = _get_macro(name)
	    unless nodelist
		msg = "macro '#{name}' not found."
		raise Kwartz::TranslationError.new(msg)
	    end
	    translate(nodelist, code, level)
	    return code
	end

	def translate_rawcode_stmt(stmt_node, code='', level=0)
	    rawcode = stmt_node.left
	    code << encode(:rawcode)
	    code << rawcode
	    code << encode(:endrawcode)
	end

	def translate_nodelist(nodelist, code='', level=0)
	    nodelist.each do |stmt_node|
		Kwartz::assert() unless stmt_node && stmt_node.is_a?(StmtNode)
		translate(stmt_node, code, level)
	    end
	    return code
	end


	##
	## Some language such as JSP+JSTL or Velocity doesn't support
	## 'v += 1', 'v -= 1', 'v *= 1', ...
	## normalize_assignment() convert a assign-statement node
	## to normal assignment, such as 'v = v+1', 'v = v-1', 'v = v*1', ...
	##
	## ex.
	##  :set(v += x)  ==> :set(v = v+x)
	##  :set(v -= x)  ==> :set(v = v-x)
	##  :set(v *= x)  ==> :set(v = v*x)
	##  :set(v /= x)  ==> :set(v = v/x)
	##  :set(v %= x)  ==> :set(v = v%x)
	##  :set(v .+= x)  ==> :set(v = v.+x)
	##
	def normalize_assignment(setstmt_node)
	    Kwartz::assert() unless setstmt_node.is_a?(SetStmtNode)
	    assign_node = setstmt_node.left
	    assign_op   = assign_node.token
	    if assign_op != '='
		left  = assign_node.left
		right = assign_node.right
		op = assign_op.sub(/=$/, '')
		assign_node.token = '='
		assign_node.right = ExprNode.new(op, left, right)
	    end
	end
	
	##
	## Some language such as JSP+JSTL or Velocity doesn't have conditional
	## operator ('? :').
	## condop_to_ifstmt() convert a statement node which have conditioanl
	## operator to if-statement node which is equal to the origianl node.
	##
	## ex.
	##  :set(var = cond ? val1 : val2)
	##   ==> :if(cond) :set(var = val1) :else :set(var = val2) :end
	##
	def condop_to_ifstmt(stmt_node)
	    ## conditional operator node
	    left = stmt_node.left
	    ifstmt_node = nil
	    if left && left.is_a?(ExprNode)
		condop_expr, parent_expr = left.find_with_parent do |expr_node|
		    expr_node.token == '?'
		end
		if condop_expr
		    ## expression node when true/false
		    cond_expr  = condop_expr.condition
		    true_expr  = condop_expr.left
		    false_expr = condop_expr.right

		    ## statement node when true/false
		    parent_node = parent_expr || stmt_node
		    l_or_r = parent_node.left == condop_expr ? 'left' : 'right'
		    eval "parent_node.#{l_or_r} = true_expr"
		    true_stmt  = stmt_node.deepcopy
		    eval "parent_node.#{l_or_r} = false_expr"
		    false_stmt = stmt_node.deepcopy

		    ## create if-statement node
		    true_nodelist  = NodeList.new([true_stmt])
		    false_nodelist = NodeList.new([false_stmt])
		    ifstmt_node = IfStmtNode.new(:if, true_nodelist, false_nodelist, cond_expr)
		end
	    end
	    return ifstmt_node
	end

    end


    ##
    ## translate Node to Ruby code
    ##
    class RubyTranslator < Translator
	self.register(self, "ruby")

	codedict_diff = {
	    :indent	=> '  ',
	    
	    :set	=> "",
	    :endset	=> "\n",
	    :if		=> "if ",
	    :then	=> " then\n",
	    :elsif	=> "elsif ",
	    :else	=> "else\n",
	    :endelse	=> nil,
	    :endif	=> "end\n",
	    :while	=> "while ",
	    :do		=> " do\n",
	    :endwhile	=> "end\n",
	    :foreach	=> "for ",
	    :in		=> " in ",
	    :endforeach => "end\n",
	    :print	=> "print ",
	    :endprint	=> "\n",
	    :print2	=> "print CGI.escapeHTML((",
	    :endprint2	=> ").to_s)\n",

	    :true       => 'true',
	    :false      => 'false',
	    :null       => 'nil',

	    '{'		=> '[',
	    '}'		=> ']',
	    '[:'	=> '[:',
	    ':]'	=> ']',
	    '.+'	=> ' + ',
	    '.+='	=> ' << ',

	    'E'		=> ['CGI.escapeHTML(', ')'],
	}
	self.init_codedict(codedict_diff)

	##
	def translate_expr(node, code='', level=0)
	    case tkn = node.token
	    when :string
		value_str = node.left
		code << value_str.dump

	#    when :empty, :notempty
	#	## s==empty  => (!s || s==''),  s!=empty => (s && s!='')
	#	flag = (tkn == :empty)
	#	str_node  = ExprNode.new(:string, '', nil)
	#	l_node    = flag ? ExprNode.new('!', node.left, nil) : node.left
	#	r_node    = ExprNode.new(flag ? '==' : '!=' , node.left, str_node)
	#	node2     = ExprNode.new(flag ? '||' : '&&', l_node, r_node)
	#	translate_expr(node2, code, level)

	    else
		super(node, code, level)
	    end
	    return code
	end

	def _translate_expr(expr_node, code, flag_to_s=nil)
	    code << '('      if flag_to_s
	    translate_expr(expr_node, code)
	    code << ').to_s' if flag_to_s
	    return code
	end

	def translate_print_stmt(node, code='', level=0)
	    arg_expr = node.left
	    #
	    indent(code, level)
	    code << encode(:print)
	    #translate(arg_expr, code)
	    _translate_print_args(node.left, code)
	    code << encode(:endprint)
	    return code
	end
	
	def _translate_print_args(expr_node, code='')
	    case expr_node.token
	    when :string
		_translate_expr(expr_node, code, false)
	    when :string, :number, :true, :false, :null
		_translate_expr(expr_node, code)
	    when ','
		_translate_print_args(expr_node.left, code)
		code << encode(',')
		_translate_print_args(expr_node.right, code)
	    else
		case flag_escape = @toppings[:escape]
		when true, false
		    # do nothing
		else
		    flag_escape = false		# default no-sanitize
		    #flag_escape = true		# default sanitize
		end
		if expr_node.token == :function
		    funcname = expr_node.left
		    case funcname
		    when 'X'
			flag_escape = false
			expr_node = expr_node.right
		    when 'E'
			flag_escape = true
			expr_node = expr_node.right
		    end
		end
		if flag_escape
		    keys = encode('E')
		    code << keys[0]
		    _translate_expr(expr_node, code, true)
		    code << keys[1]
		else
		    _translate_expr(expr_node, code)
		end
	    end
	    return code
	end
    end

    ##
    ## translate Node to eRuby code
    ##
    class Ruby2Translator < RubyTranslator
	self.register(self, "ruby2")
	@codedict = RubyTranslator.codedict.dup
	@codedict[:print]    = '_s << '
	@codedict[:endprint] = "\n"
	@codedict[',']       = ' << '
	@codedict_crlf = RubyTranslator.codedict_crlf.dup
	@codedict_crlf[:print]    = '_s << '
	@codedict_crlf[:endprint] = "\r\n"
	@codedict_crlf[',']       = ' << '

	def _translate_expr(expr_node, code, flag_to_s=nil)
	    flag_to_s = true if flag_to_s == nil
	    super(expr_node, code, flag_to_s)
	end

    end


    ##
    ## translate Node to eRuby code
    ##
    class ErubyTranslator < Translator
	self.register(self, "eruby")

	codedict_diff = {
	    :indent	=> '',			# no indent
	    
	    :set	=> "<% ",
	    :endset	=> "\n %>",
	    :if		=> "<% if ",
	    :then	=> " then\n %>",
	    :elsif	=> "<% elsif ",
	    :else	=> "<% else\n %>",
	    :endelse	=> nil,
	    :endif	=> "<% end\n %>",
	    :while	=> "<% while ",
	    :do		=> " do\n %>",
	    :endwhile	=> "<% end\n %>",
	    :foreach	=> "<% for ",
	    :in		=> " in ",
	    :endforeach => "<% end\n %>",
	    :print	=> "<%= ",
	    :endprint	=> " %>",
	    :print2	=> "<%= CGI.escapeHTML((",
	    :endprint2	=> ").to_s) %>",

	    :true       => 'true',
	    :false      => 'false',
	    :null       => 'nil',

	    '{'		=> '[',
	    '}'		=> ']',
	    '[:'	=> '[:',
	    ':]'	=> ']',
	    '.+'	=> ' + ',
	    '.+='	=> ' << ',

	    'E'		=> ['CGI.escapeHTML((', ').to_s)'],
	}
	self.init_codedict(codedict_diff)

    end


    ##
    ## translate Node to eRuby code, with '%'
    ##
    class ErbTranslator < Translator
	self.register(self, "erb")

	codedict_diff = {
	    :indent	=> '  ',
	    :prefix	=> '% ',
	    
	    :set	=> "",
	    :endset	=> "\n",
	    :if		=> "if ",
	    :then	=> " then\n",
	    :elsif	=> "elsif ",
	    :else	=> "else\n",
	    :endelse	=> nil,
	    :endif	=> "end\n",
	    :while	=> "while ",
	    :do		=> " do\n",
	    :endwhile	=> "end\n",
	    :foreach	=> "for ",
	    :in		=> " in ",
	    :endforeach => "end\n",
	    :print	=> "<%= ",
	    :endprint	=> " %>",
	    #:print2	=> "<%= CGI.escapeHTML((",
	    #:endprint2	=> ").to_s) %>",
	    :print2	=> "<%= html_escape(",
	    :endprint2	=> ") %>",

	    :true       => 'true',
	    :false      => 'false',
	    :null       => 'nil',

	    '{'		=> '[',
	    '}'		=> ']',
	    '[:'	=> '[:',
	    ':]'	=> ']',
	    '.+'	=> ' + ',
	    '.+='	=> ' << ',

	    #'E' 	=> ['CGI.escapeHTML((', ').to_s)'],
	    'E'		=> ['html_escape(', ')'],
	}
	self.init_codedict(codedict_diff)

    end


    ##
    ## translate Node to erbscan code
    ##
    class ErbscanTranslator < Translator
	self.register(self, "erbscan")

	codedict_diff = {
	    :indent	=> '  ',
	    
	    :set	=> "<%- ",
	    :endset	=> " -%>\n",
	    :if		=> "<%- if ",
	    :then	=> " then -%>\n",
	    :elsif	=> "<%- elsif ",
	    :else	=> "<%- else -%>\n",
	    :endelse	=> nil,
	    :endif	=> "<%- end -%>\n",
	    :while	=> "<%- while ",
	    :do		=> " do -%>\n",
	    :endwhile	=> "<%- end -%>\n",
	    :foreach	=> "<%- for ",
	    :in		=> " in ",
	    :endforeach => "<%- end -%>\n",
	    :print	=> "<%= ",
	    :endprint	=> " %>",
	    #:print2	=> "<%= ERB::Util.h(",
	    #:endprint2	=> ").to_s) %>",
	    :print2	=> "<%= html_escape(",
	    :endprint2	=> ") %>",

	    :true       => 'true',
	    :false      => 'false',
	    :null       => 'nil',

	    '{'		=> '[',
	    '}'		=> ']',
	    '[:'	=> '[:',
	    ':]'	=> ']',
	    '.+'	=> ' + ',
	    '.+='	=> ' << ',

	    #'E' 	=> ['CGI.escapeHTML((', ').to_s)'],
	    'E' 	=> ['html_escape(', ')'],
	}
	self.init_codedict(codedict_diff)
    end


    ##
    ## translate Node to PHP code
    ##
    class PhpTranslator < Translator
	self.register(self, "php")

	codedict_diff = {
	    :indent	=> '  ',
	    
	    :set	=> "<?php ",
	    :endset	=> "; ?>\n",
	    :if		=> "<?php if (",
	    :then	=> ") { ?>\n",
	    :elsif	=> "<?php } elseif (",
	    :else	=> "<?php } else { ?>\n",
	    :endelse	=> nil,
	    :endif	=> "<?php } ?>\n",
	    :while	=> "<?php while (",
	    :do		=> ") { ?>\n",
	    :endwhile	=> "<?php } ?>\n",
	    :foreach	=> "<?php foreach (",
	    :in		=> " as ",
	    :endforeach => "<?php } ?>\n",
	    :print	=> "<?php echo ",
	    :endprint	=> "; ?>",
	    :print2	=> "<?php echo htmlspecialchars(",
	    :endprint2	=> "); ?>",

	    :true       => 'TRUE',
	    :false      => 'FALSE',
	    :null       => 'NULL',

	    '['		=> '[',
	    '{'		=> "[",
	    '}'		=> "]",
	    '[:'	=> "['",
	    ':]'	=> "']",
	    '.+'	=> ' . ',
	    '.+='	=> ' .= ',
	    '.'		=> '->',

	    'E'		=> ['htmlspecialchars(', ')'],
	}
	self.init_codedict(codedict_diff)

	##
	def translate_expr(node, code='', level=0)
	    case tkn = node.token
	    when :variable
		code << '$'
		super(node, code, level)
	    when :empty, :notempty
		## s==empty => '!s', s!=empty => 's'
		left = node.left
		node2 = tkn == :empty ? ExprNode.new('!', left, nil) : left
		translate_expr(node2, code, level)
	    else
		super(node, code, level)
	    end
	    return code
	end

	##
	def translate_foreach_stmt(node, code='', level=0)
	    assign_node = node.left
	    tmp = assign_node.left
	    assign_node.left = assign_node.right
	    assign_node.right = tmp
	    super(node, code, level)
	    return code
	end

    end

    ##
    ## translate Node to PHP code, with output buffering
    ##
    class Php2Translator < PhpTranslator
	self.register(self, "php2")
	@codedict = PhpTranslator.codedict.dup
	
	def translate_all(nodelist, code='', level=0)
	    code << "<?php  ob_start(); ?>" << @newline
	    super(nodelist, code, level)
	    code << "<?php  $_s = ob_get_contents();  ob_end_clean(); ?>" << @newline
	    return code
	end
    end

    ##
    ## translate Node to JSP(+JSTL) code
    ##
    class JspTranslator < Translator
	self.register(self, "jsp")

	codedict_diff = {
	    :indent	=> ' ',
	    :set	=> "<c:set var=\"",
	    :endset	=> "}\"/>\n",
	    :if		=> "<c:choose>\n<c:when test=\"${",
	    :then	=> "}\">\n",
	    :endthen	=> "</c:when>\n",
	    :elsif	=> "<c:when test=\"${",
	    :else	=> "<c:otherwise>\n",
	    :endelse	=> "</c:otherwise>\n",
	    :endif	=> "</c:choose>\n",
	    :while	=> "<WHILE test=\"${",
	    :do		=> "}\">\n",
	    :endwhile	=> "</WHILE>\n",
	    :foreach	=> "<c:forEach var=\"",
	    :in		=> "\" items=\"${",
	    :endforeach => "</c:forEach>\n",
	    :print	=> "<c:out value=\"${",
	    :endprint	=> "}\" escapeXml=\"false\"/>",
	    :print2	=> "<c:out value=\"${",
	    :endprint2	=> "}\"/>",

	    :true       => 'true',
	    :false      => 'false',
	    :null       => 'null',

	    '['		=> '[',
	    '{'		=> "[",
	    '}'		=> "]",
	    '[:'	=> "['",
	    ':]'	=> "']",
	    '='		=> "\" value=\"${",
	    '.+'	=> '',
	    '.'		=> '.',
	    #'!'	=> 'empty '
	    '!'         => 'not ',
	    '&&'        => ' and ',
	    '||'        => ' or ',

	    #'E' 	=> ['', ''],
	}
	self.init_codedict(codedict_diff)

	def translate_expr(node, code='', level=0)
	    case tkn = node.token
	    when '.+'
		translate_expr(node.left, code, level)
		code << '}${'
		translate_expr(node.right, code, level)
		#left  = node.left
		#right = node.right
		#node.left  = ExprNode.new('()', node.left)
		#node.right = ExprNode.new('()', node.right)

	    when :empty, :notempty
		## s==empty => '(empty s)',  s!=empty => '!(empty s)'
		code << (tkn == :empty ? '(empty ' : '!(empty ')
		translate_expr(node.left, code, level)
		code << ')'

	    else
		super(node, code, level)
	    end
	end


	def translate_set_stmt(node, code='', level=0)
	    normalize_assignment(node)	# ex. :set(v+=1) -> :set(v=v+1)
	    assign_node = node.left
	    assign_op = assign_node.token
	    lhs_expr  = assign_node.left
	    rhs_expr  = assign_node.right
	    case rhs_expr.token
	    when :string, :number
		indent(code, level)
		code << encode(:set)
		translate(lhs_expr, code)
		code << '" value="'
		value = rhs_expr.left
		code << value.to_s.gsub(/"/, '\\"')    #'
		code << "\"/>" << @newline
	    else
		super(node, code, level)
	    end
	    return code
	end

	def translate_while_stmt(node, code='', level=0)
	    msg = "Jsp cannot support 'while' statment."
	    raise TranslationError.new(msg)
	end

	def translate_nodelist(nodelist, code='', level=0)
	    nodelist.each_with_index do |stmt_node, idx|
		new_node = condop_to_ifstmt(stmt_node)
		nodelist[idx] = new_node if new_node
	    end
	    return super(nodelist, code, level)
	end

    end


    ##
    ## translate Node to Velocity code
    ##
    class VelocityTranslator < Translator
	self.register(self, "velocity")

	codedict_diff = {
	    :indent	=> '  ',
	    
	    :set	=> "#set (",
	    :endset	=> ")\n",
	    :if		=> "#if (",
	    :then	=> ")\n",
	    :elsif	=> "#elseif (",
	    :else	=> "#else\n",
	    :endelse	=> nil,
	    :endif	=> "#end\n",
	    :while	=> nil,
	    :do		=> ")\n",
	    :endwhile	=> nil,
	    :foreach	=> "#foreach (",
	    :in		=> " in ",
	    :endforeach => "#end\n",
	    :print	=> '$!{',
	    :endprint	=> '}',
	    :print2	=> '$!{',
	    :endprint2	=> '}',

	    :true       => 'true',
	    :false      => 'false',
	    :null       => 'null',

	    '['		=> '.get(',
	    ']'		=> ')',
	    '{'		=> '.get(',
	    '}'		=> ')',
	    '[:'	=> '.',
	    ':]'	=> '',
	    '.+'	=> '',
	    '.'		=> '.',

	    #'E' 	=> ['', ''],
	}
	self.init_codedict(codedict_diff)

	##
	## ex. a .+ b ==> $a$b,  'a' .+ b ==> a${b}
	##
	def _translate_strcat(expr_node)
	    node = expr_node
	    case token = node.token
	    when :string
		str_value = node.left
		return str_value
	    when :variable, '[]', '{}', '[:]', '.'
		code = translate_expr(node)
		code.sub!(/^\$/, '')
		return "${#{code}}"
	    when '.+'
		left_str  = _translate_strcat(node.left)
		right_str = _translate_strcat(node.right)
		return "#{left_str}#{right_str}"
	    end
	end
	private :_translate_strcat
	
	##
	def translate_expr(expr_node, code='', level=0)
	    node = expr_node
	    case tkn = node.token
	    when :variable
		code << '$'
		super(node, code, level)
	    #when '[]', '{}'
	    #	right = node.right
	    #	if right.token == :string && right.left =~ /\A\w+\z/
	    #	    translate_expr(node.left, code, level)
	    #	    code << '.'
	    #	    code << right.left
	    #	else
	    #	    super(node, code, level)
	    #	end
	    when '.+'
		code << '"'
		code << _translate_strcat(node)
		code << '"'
	    when '!='
		if node.left.token == :null
		    translate_expr(node.right, code, level)
		elsif node.right.token == :null
		    translate_expr(node.left, code, level)
		else
		    super(node, code, level)
		end
	    when '=='
		if node.left.token == :null
		    node.token = '!'
		    node.left = node.right
		    node.right = nil
		elsif node.right.token == :null
		    node.token = '!'
		    node.right = nil
		end
		super(node, code, level)
	    when :empty, :notempty
		## s==empty  => (!s || s==''),  s!=empty => (s && s!='')
		flag = (tkn == :empty)
		str_node = ExprNode.new(:string, '', nil)
		l_node   = flag ? ExprNode.new('!', node.left, nil) : node.left
		r_node   = ExprNode.new(flag ? '==' : '!=', node.left, str_node)
		node2    = ExprNode.new(flag ? '||' : '&&', l_node, r_node)
		code << encode('(')
		translate_expr(node2, code, level)
		code << encode(')')
	    when :null
		msg = "velocity cannot support keyword 'null'."
		raise TranslationError.new(msg)
	    else
		super(node, code, level)
	    end
	    return code
	end

	def translate_set_stmt(stmt_node, code='', level=0)
	    # ex. :set(v+=1) ==> :set(v=v+1)
	    normalize_assignment(stmt_node)
	    # ex. :set(hash['key'] = value) ==> :set(hash.key = value)
	    lvalue = stmt_node.left.left
	    case lvalue.token
	    when '[]', '{}'
		index = lvalue.right
		if index.token == :string && index.left =~ /\A\w+\z/
		    lvalue.token = '.'
		    lvalue.right = index.left
		end
	    end
	    super(stmt_node, code, level)
	    return code
	end

	#def translate_print_stmt(stmt_node, code='', level=0)
	#    _translate_print_args(stmt_node.left, code, level, startkey, endkey)
	#end

	def _translate_print_args(expr_node, code, level, startkey, endkey)
	    if expr_node.token == :function
		funcname = expr_node.left
		case funcname
		when 'X', 'E'
		    expr_node = expr_node.right
		end
	    end
	    case expr_node.token
	    when :string, :number, ','
		super(expr_node, code, level, startkey, endkey)
	    else
		code << encode(startkey)
		#_translate_print_args(expr_node, code, level, startkey, endkey)
		_translate_expr_for_print(expr_node, code)
		code << encode(endkey)
	    end
	    return code
	end
	private :_translate_print_args

	def _translate_expr_for_print(expr_node, code)
	    node = expr_node
	    case node.token
	    when :variable
		code << node.left
	    when '.'
		_translate_expr_for_print(node.left, code)
		code << '.'
		code << node.right
	    when '[]'
		_translate_expr_for_print(node.left, code)
		#if node.right.token == :string && node.right.left =~ /\A\w+\z/
		#    code << encode('.')
		#    code << node.right.left
		#else
		    code << encode('[')
		    translate_expr(node.right, code)
		    code << encode(']')
		#end
	    when '{}'
		_translate_expr_for_print(node.left, code)
		#if node.right.token == :string && node.right.left =~ /\A\w+\z/
		#    code << encode('.')
		#    code << node.right.left
		#else
		    code << encode('{')
		    translate_expr(node.right, code)
		    code << encode('}')
		#end
	    when '[:]'
		_translate_expr_for_print(node.left, code)
		code << encode('[:')
		code << node.right
		code << encode(':]')
	    else
		msg = 'Velocity cannot print complex expression.'
		raise TranslationError.new(msg)
	    end
	    return code
	end
	private :_translate_expr_for_print

	def translate_while_stmt(stmt_node, code='', level=0)
	    msg = "Velocity cannot support 'while' statment."
	    raise TranslationError.new(msg)
	end

	def translate_nodelist(nodelist, code='', level=0)
	    nodelist.each_with_index do |stmt_node, idx|
		new_node = condop_to_ifstmt(stmt_node)
		nodelist[idx] = new_node if new_node
	    end
	    return super(nodelist, code, level)
	end
    end


    ## ============================================================

    class AnalysisError < KwartzError
    end
    
    
    ## 
    ## analyze local/global variables
    ## 
    class Analyzer
	include Action
	
	def initialize(node, toppings={})
	    @node      = node
	    @toppings   = toppings
	    @macros    = {}
	    @variables = {}
	    @warnings  = []
	    @globals   = nil
	    @locals    = nil
	end
	attr_reader :variables, :warnings, :globals, :locals

	private
	
	def _registered?(varname)
	    return @variables.key?(varname)
	end
	
	def _debut?(varname)
	    return ! @variables.key?(varname)
	end
	
	def _register(varname, scope=:global, initialized=true)
	    Kwartz::assert() if _registered?(varname)
	    data = { :name => varname, :scope => scope, :initialized => initialized, }
	    @variables[varname] = data
	end
	
	def _initialized?(varname)
	    Kwartz::assert() if _debut?(varname)
	    return @variables[varname][:initialized]
	end
	
	def _set_initialized(varname)
	    Kwartz::assert() if _scope(varname) == :global
	    @variables[varname][:initialized] = true
	end

	def _scope(varname)
	    Kwartz::assert() if _debut?(varname)
	    Kwartz::assert() if ! @variables[varname]
	    if @variables[varname]
		return @variables[varname][:scope]
	    else
		Kwartz::assert()
	    end
	end

	public
	
	def analyze
	    return nil if @node == nil
	    @node.execute(self)
	    @globals = []
	    @locals  = []
	    @variables.each do |varname, data|
		case data[:scope]
		when :global
		    @globals << varname
		when :local
		    @locals  << varname
		end
	    end
	end
	
	def exec_expr(expr_node, arg1=nil, arg2=nil)
	    case tkn = expr_node.token
	    when :variable
		varname = expr_node.left
		if _debut?(varname)
		    _register(varname)
		elsif _scope(varname) == :local && ! _initialized?(varname)
		    msg = "#{varname}: uninitialized local var is used, or tryng to assign into global var."
		    @warnings << msg
		end

	    when '[]', '{}'
		exec_expr(expr_node.right)
		exec_expr(expr_node.left)
		
	    when '[:]', '.'
		exec_expr(expr_node.left)
		
	    when :string, :number, :true, :false, :null
		# do nothing
		
	    when :function
		exec_expr(expr_node.right) if expr_node.right
		
	    when '+', '-'
		exec_expr(expr_node.left, arg1, arg2)
		exec_expr(expr_node.right, arg1, arg2) if expr_node.right
		
	    when '*', '/', '%'   #, '^'
		exec_expr(expr_node.left, arg1, arg2)
		exec_expr(expr_node.right, arg1, arg2)
		
	    when '!'
		exec_expr(expr_node.left)

	    when '==', '!=', '<', '<=', '>', '>=', '&&', '||'
		exec_expr(expr_node.left)
		exec_expr(expr_node.right)

	    when '.+', ','
		exec_expr(expr_node.left)
		exec_expr(expr_node.right)

	    when '.==', '.!=', '.<', '.<=', '.>', '.>='
		exec_expr(expr_node.left)
		exec_expr(expr_node.right)

	    when :empty, :notempty
		exec_expr(expr_node.left)

	    when '?'
		exec_expr(expr_node.condition)
		exec_expr(expr_node.left)
		exec_expr(expr_node.right)

	    else
		Kwartz::assert("tknn=#{tkn.inspect}")
	    end
	end
	
	def exec_stmt(stmt_node, arg1=nil, arg2=nil)
	    case stmt_node.token
	    when :set
		node = stmt_node.left
		op       = node.token		# = += -= *= /= %=
		lhs_node = node.left
		rhs_node = node.right
		#Kwartz::assert() unless lhs_node.token == :variable
		if lhs_node.token == :variable
		    varname = lhs_node.left
		    if _debut?(varname)
			unless op == '='
			    msg = "#{varname}: uninitialized local var is used, or tryng to assign into global var."
			    @warnings << msg
			end
			_register(varname, :local, false)
			exec_expr(rhs_node, arg1, arg2)
			_set_initialized(varname)
		    else
			if _scope(varname) == :global
			    msg = "#{varname}: trying to assign into global var."
			    @warnings << msg
			end
			exec_expr(rhs_node, arg1, arg2)
		    end
		else
		    #msg = "left-hand of assignment should be a normal variable."
		    #@warnings << msg
		    exec_expr(lhs_node, arg1, arg2)
		    exec_expr(rhs_node, arg1, arg2)
		end
		
	    when :foreach
		Kwartz::assert() unless stmt_node.left.token == '='
		loopvar_node = stmt_node.left.left
		listvar_node = stmt_node.left.right
		nodelist     = stmt_node.right
		#Kwartz::assert() unless loopvar_node.token == :variable
		if loopvar_node.token == :variable
		    varname = loopvar_node.left
		    if _debut?(varname)
			_register(varname, :local, false)
			exec_expr(listvar_node, arg1, arg2)
			_set_initialized(varname)
		    else
			if _scope(varname) == :global
			    msg = "#{varname}: global var is used as loop var."
			    @warnings << msg
			end
			exec_expr(listvar_node, arg1, arg2)
		    end
		else
		    #msg = "loop var should be a normal variable."
		    #@warnings << msg
		    exec_expr(loopvar_node, arg1, arg2)
		    exec_expr(listvar_node, arg1, arg2)
		end
		exec_nodelist(nodelist, arg1, arg2)

	    when :print
		exec_expr(stmt_node.left, arg1, arg2)

	    when :while
		#cond_expr = stmt_node.left
		#nodelist  = stmt_node.right
		cond_expr = stmt_node.condition
		nodelist  = stmt_node.left
		exec_expr(cond_expr, arg1, arg2)
		exec_nodelist(nodelist, arg1, arg2)
		
	    when :if
		cond_expr     = stmt_node.condition
		then_nodelist = stmt_node.left
		else_node     = stmt_node.right
		exec_expr(cond_expr, arg1, arg2)
		exec_nodelist(then_nodelist, arg1, arg2)
	        else_node.execute(self, arg1, arg2) if else_node

	    when :rawcode

	    when :macro
		macro_name = stmt_node.left
		nodelist = stmt_node.right
		@macros[macro_name] = nodelist
		
	    when :expand
		macro_name = stmt_node.left
		nodelist = @macros[macro_name]
		unless nodelist
		    msg = "'#{macro_name}': no such macro."
		    raise AnalysisError.new(msg)
		end
		exec_nodelist(nodelist, arg1, arg2)
		
	    else
		Kwartz::assert()
	    end
	end
	
	#def exec_nodelist(nodelist, arg1=nil, arg2=nil)
	#    nodelist.list.each do |stmt_node|
	#	stmt_node.execute(self, arg1, arg2)
	#    end
	#end
    end
    

    ## ============================================================


    ##
    ## usage.
    ##	 input = <<END
    ##	   :if(ctr % 2 == 0)
    ##	     :print('even')
    ##	   :else
    ##	     :print('odd')
    ##	   :end
    ##	 END
    ##	 scanner = Kwartz::Scanner.new(input)
    ##	 parser	 = Kwartz::Parser.new(scanner)
    ##	 nodelist = parser.parse()
    ##
    class Parser
	def initialize(scanner, toppings={})
	    @scanner = scanner
	    @scanner.scan
	    @toppings = toppings
	    @name_stack = []
	    @newline = scanner.newline
	end

	def scan
	    return @scanner.scan
	end

	def token
	    return @scanner.token
	end

	def token_str
	    return @scanner.token_str
	end
	
	def newline
	    return @scanner.newline
	end
	
	def _push_element_name(name)
	    @name_stack.push(name)
	    return name
	end
	private :_push_element_name
	
	def _pop_element_name()
	    return @name_stack.pop
	end
	private :_pop_element_name
	
	def _element_name()
	    return @name_stack.last
	end
	private :_element_name

	def _syntaxerr(error_message)
	    raise SyntaxError.new(error_message, @scanner)
	end
	private :_syntaxerr
	
	
	
	##
	## compare priority of operators. this is called from Translator#translate_expr().
	##
	## return true if op1 > op2, nil if op1 == op2, false if op1 < op2.
	##
	def self.prior?(op1, op2)
	    pr1 = @@priority[op1]
	    pr2 = @@priority[op2]
	    Kwartz::assert() unless pr1
	    #Kwartz::assert() unless pr2
	    pr2 = 0 if !pr2
	    return nil if pr1 == pr2
	    return pr1 < pr2
	end
	@@priority = {
	    '[]'	=>   1,
	    '{}'	=>   1,
	    '[:]'	=>   1,
	    '.'		=>   1,
	    '*'		=>   3,
	    '/'		=>   3,
	    '%'		=>   3,
	    #'^' 	=>   3,
	    '+'		=>   4,
	    '-'		=>   4,
	    '.+'	=>   4,
	    '=='	=>   5,
	    '!='	=>   5,
	    '<'		=>   5,
	    '<='	=>   5,
	    '>'		=>   5,
	    '>='	=>   5,
	    '.=='	=>   5,
	    '.!='	=>   5,
	    '.<'	=>   5,
	    '.<='	=>   5,
	    '.>'	=>   5,
	    '.>='	=>   5,
	    :empty	=>   5,
	    :notempty	=>   5,
	    '!'		=>   5,		## ??
	    '&&'	=>   6,
	    '||'	=>   7,
	    '?'		=>   8,
	    ','		=>   9,
	}


	
	## --------------------
	
	##
	## * BNF:
	##    item ::= variable | function '(' [expr {',' expr}] ')' | '(' expression ')'
	##
	def parse_item
	    case token()
	    when :name
		name = token_str()
		scan()
		if token() != '('		# variable
		    expr = ExprNode.new(:variable, name, nil)
		else				# function
		    scan()
		    if token() == ')'
			arg_expr = nil
		    else
			arg_expr = parse_expression()
			while token() == ',' do
			    scan()
			    arg2_expr = parse_expression()
			    arg_expr = ExprNode.new(',', arg_expr, arg2_expr)
			end
		    end
		    _syntaxerr("function '#{name}(' is not closed.") unless token() == ')'
		    scan()
		    expr = ExprNode.new(:function, name, arg_expr)
		end
	    when '('
		scan()
		expr = parse_expression()
		_syntaxerr("')' expected.") unless token() == ')'
		scan()
	    else
		Kwartz::assert()
	    end
	    return expr
	end

	##
	## * BNF:
	##    array ::= item | item '[' expr ']' | item '{' expr '}' 
	##                   | item '[:' name ']' | item '.' property
	##	    ::= item { '[' expr ']' | '{' expr '}' | '[:' name ']' | '.' propperty }
	##
	def parse_array
	    expr = parse_item()
	    while true do
		case token()
		when '['
		    scan()
		    index_expr = parse_expression()
		    _syntaxerr("array '[' is not closed.") unless token() == ']'
		    scan()
		    expr = ExprNode.new('[]', expr, index_expr)
		when '{'
		    scan()
		    index_expr = parse_expression()
		    _syntaxerr("hash '{' is not closed.") unless token() == '}'
		    scan()
		    expr = ExprNode.new('{}', expr, index_expr)
		when '[:'
		    scan()
		    _syntaxerr("'[:' require a word.") unless token() == :name
		    word = token_str
		    scan()
		    _syntaxerr("']' required.") unless token() == ']'
		    scan()
		    expr = ExprNode.new('[:]', expr, word)
		when '.'
		    scan()
		    _syntaxerr("property name is invalid.") unless token() == :name
		    property = token_str
		    scan()
		    expr = ExprNode.new('.', expr, property)
		else
		    break
		end
	    end
	    return expr
	end

	##
	## * BNF:
	##    factor ::= array | number | string | 'true' | 'false' | 'nil'
	##
	def parse_factor
	    expr = nil
	    case token()
	    when :name, '('
		expr = parse_array()
	    when :number
		value = token_str
		scan()
		expr = ExprNode.new(:number, value, nil)
	    when :string
		value = token_str
		scan()
		expr = ExprNode.new(:string, value, nil)
	    when :true, :false, :null
		value = token_str
		expr = ExprNode.new(token, value, nil)
		scan()
	    when :empty
		_syntaxerr("'empty' is allowed only in right-side of '==' or '!='.")
	    else
	        #assert()
		_syntaxerr("'#{token()}': invalid expression syntax.")
	    end
	    return expr
	end

	##
	## * BNF:
	##    term ::= factor | factor '*' term | factor '/' term | factor '%' term | factor
	##	   ::= factor { ('*' | '/' | '%') factor }
	##
	def parse_term
	    expr = parse_factor()
	    while (t = token()) == '*' || t == '/' || t == '%'
		scan()
		expr2 = parse_factor()
		expr = ExprNode.new(t, expr, expr2)
	    end
	    return expr
	end

	##
	## * BNF:
	##    unary ::= [ '+' | '-' | '!' ] term
	##
	def parse_unary
	    case t = token()
	    when '+', '-', '!'
		scan()
		expr2 = parse_term()
		expr = ExprNode.new(t, expr2, nil)
	    else
		expr = parse_term()
	    end
	    return expr
	end
	
	##
	## * BNF:
	##    arith2 ::= term | term '+' arith | term '-' arith | term '.+' arith
	##    arith  ::= unary | unary '+' arith2 | unary '-' arith2 | unary '.+' arith2
	##           ::= unary { ('+' | '-' | '.+') term }
	##
	def parse_arith
	    expr = parse_unary()
	    while (t = token()) == '+' || t == '-' || t == '.+'
		scan()
		expr2 = parse_term()
		expr = ExprNode.new(t, expr, expr2)
	    end
	    return expr
	end

	##
	## * BNF:
	##    compare-op ::=   '==' |  '!=' |  '>' |  '>=' |  '<' |  '<='
	##		    | '.==' | '.!=' | '.>' | '.>=' | '.<' | '.<='
	##    compare	 ::= arith compare-op arith | arith ['==' | '!='] 'empty'
	##
	def parse_compare
	    expr = parse_arith()
	    case t = token()
	    when '==', '!=', '>', '>=', '<', '<=', '.>', '.>=', '.<', '.<=', '.==', '.!='
		scan()
		if token() == :empty && (t == '==' || t == '!=')
		    scan()
		    t2 = t == '==' ? :empty : :notempty
		    expr = ExprNode.new(t2, expr, nil)
		else
		    expr2 = parse_arith()
		    expr = ExprNode.new(t, expr, expr2)
		end
	    end
	    return expr
	end

	##
	## * BNF:
	##    logical-and ::= compare '&&' logical-and
	##		  ::= compare { '&&' compare }
	##
	def parse_logical_and
	    expr = parse_compare()
	    while token() == '&&' do
		t = token()
		scan()
		expr2 = parse_compare()
		expr = ExprNode.new(t, expr, expr2)
	    end
	    return expr
	end

	##
	## * BNF:
	##    logical-or ::= logical-and '||' logical-or
	##		 ::= logical-and { '||' logical-and }
	##
	def parse_logical_or
	    expr = parse_logical_and()
	    while token() == '||' do
		t = token()
		scan()
		expr2 = parse_logical_and()
		expr = ExprNode.new(t, expr, expr2)
	    end
	    return expr
	end

	##
	## * BNF:
	##    expression ::= logical-or | logical-or '?' logical-or ':' logical-or
	##		 ::= logical-or { '?' logical-or ':' logical-or }
	##
	def parse_expression
	    expr = parse_logical_or()
	    while token() == '?' do
		scan()
		left_expr = parse_logical_or()
		_syntaxerr("':' expected.") unless token() == ':'
		scan()
		right_expr = parse_logical_or()
		condition_expr = expr
		expr = ExprNode.new('?', left_expr, right_expr)
		expr.condition = condition_expr
	    end
	    return expr
	end

	##
	## * BNF:
	##    assign-op ::= '=' | '.+=' | '+=' | '-=' | '*=' | '/=' | '%=' | '^='
	##
	def _assign_op?(tkn)
	    case tkn
	    when '=', '.+=', '+=', '-=', '*=', '/=', '%=', '^='
		return true
	    end
	    return false
	end
	private :_assign_op?

	##
	## * BNF:
	##    assignment ::= expression	 assign-op expression | expression
	##		 ::= expression [ assign-op expression ]
	##
	def parse_assignment(flag_foreach_stmt=false)
	    left_expr = parse_expression()
	    if _assign_op?(tkn = token)
		scan()
		right_expr = parse_expression()
		unless left_expr.token == :variable
		    msg = nil
		    flag = flag_foreach_stmt
		    case left_expr.token
		    when '[]', '{}', '[:]'
			if @toppings[:inhibit_array_assign]
			    msg = "cannot use array/hash as loop var."    if     flag_foreach_stmt
			    msg = "cannot assign value into array/hash."  unless flag_foreach_stmt
			end
		    when '.'
			if @toppings[:inhibit_property_assign]
			    msg = "cannot use property as loop var."      if     flag_foreach_stmt
			    msg = "cannot assign value into property."    unless flag_foreach_stmt
			end
		    when :true, :false, :null
			msg = "cannot use true/false/null as loop var."   if     flag_foreach_stmt
			msg = "cannot assign value into true/false/null." unless flag_foreach_stmt
		    else
			msg = "cannot use expression as loop var."        if     flag_foreach_stmt
			msg = "cannot assign value into expression."      unless flag_foreach_stmt
		    end
		    raise SemanticError.new(msg) if msg
		end
		expr = ExprNode.new(tkn, left_expr, right_expr)
	    else
		expr = left_expr
	    end
	    return expr
	end

	## --------------------

	METHOD_DISPATHER = {
	    :set	=> :parse_stmt_set,
	    :if		=> :parse_stmt_if,
	    :while	=> :parse_stmt_while,
	    :foreach	=> :parse_stmt_foreach,
	    :print	=> :parse_stmt_print,
	    :macro	=> :parse_stmt_macro,
	    :elem	=> :parse_stmt_elem,
	    :expand	=> :parse_stmt_expand,
	    :expand2	=> :parse_stmt_expand2,
	    :stag	=> :parse_stmt_specialexpand,
	    :cont	=> :parse_stmt_specialexpand,
	    :etag	=> :parse_stmt_specialexpand,
	    :value	=> :parse_stmt_value,
	    :rawcode	=> :parse_stmt_rawcode,
	    ':::'	=> :parse_stmt_rawcode,
	    :load	=> :parse_stmt_load,
	}

	##
	## * BNF:
	##    stmt	::= set-stmt | if-stmt | while-stmt | foreach-stmt
	##		   | print-stmt | macro-stmt | expand-stmt | rawcode-stmt
	##		   | print2-stmt
	##    stmt-list ::= stmt*
	##
	## * return node: NodeList
	##
	def parse_stmt_list
	    list = []
	    while (method_symbol = METHOD_DISPATHER[token()]) != nil do
		node = self.__send__(method_symbol)
		if method_symbol == :parse_stmt_load
		    nodelist = node
		    list.concat(nodelist.list)
		else
		    stmt_node = node
		    list << stmt_node
		end
	    end
	    return NodeList.new(list)
	end


	##
	## * BNF:
	##    set-stmt ::= ':set' '(' assignment ')'
	##
	## * return node:
	##   token::  :set
	##   left::   assignment ExprNode
	##   right::  nil
	##
	def parse_stmt_set
	    Kwartz::assert() unless token() == :set
	    scan()
	    _syntaxerr("':set' requires '('.") unless token() == '('
	    scan()
	    expr = parse_assignment()
	    _syntaxerr("':set(' is not closed by ')'.") unless token() == ')'
	    scan()
	    _syntaxerr("':set' requires assignment.") unless _assign_op?(expr.token)
	    return SetStmtNode.new(:set, expr, nil)
	end


	##
	## * BNF:
	##    if-stmt ::= ':if' '(' expression ')' stmt-list 
	##		 { ':elsif' '(' expression ')' stmt-list }
	##		 [ ':else' stmt-list ] ':end'
	##
	## * return node
	##   token::	  :if
	##   left::	  NodeList
	##   right::	  if StmtNode (when :elsif) / NodeList (when no :elsif)
	##   condition::  ExprNode
	##
	def parse_stmt_if
	    Kwartz::assert() unless token() == :if || token() == :elsif
	    t = token()
	    scan()
	    _syntaxerr("':#{t.id2name}' requires '('.") unless token() == '('
	    scan()
	    cond_expr = parse_expression()
	    _syntaxerr("':#{t.id2name}(' is not closed by ')'.") unless token() == ')'
	    scan()
	    then_node = parse_stmt_list()	   ## nodelist
	    else_node = nil
            if token() == :elsif
	        else_node = parse_stmt_if()	   ## stmtnode
	    else
		if token() == :else
		    scan()
		    else_node = parse_stmt_list()  ## nodelist
		end
		_syntaxerr("':else' is not closed by ':end'.") unless token() == :end
		scan()
	    end
	    return IfStmtNode.new(:if, then_node, else_node, cond_expr)
	end


	##
	## * BNF:
	##    while-stmt ::= ':while' '(' assignment ')' stmt-list ':end'
	##
	## * return node:
	##   token::	  :while
	##   left::	  NodeList
	##   right::	  nil
	##   condition::  expression ExprNode
	##
	def parse_stmt_while
	    Kwartz::assert() unless token() == :while
	    scan()
	    _syntaxerr("':while' requires '('.") unless token() == '('
	    scan()
	    cond_expr = parse_assignment()
	    _syntaxerr("':while(' is not closed.") unless token() == ')'
	    scan()
	    nodelist = parse_stmt_list()
	    _syntaxerr("':while' is not closed by ':end'.") unless token() == :end
	    scan()
	    return WhileStmtNode.new(:while, nodelist, nil, cond_expr)
	end


	##
	## * BNF:
	##    foreach-stmt ::= ':foreach' '(' expression '=' expression ')'
	##		       stmt-list ':end'
	##
	## * return node:
	##   token::  :foreach
	##   left::   assignment ExprNode
	##   right::  NodeList
	##
	def parse_stmt_foreach
	    Kwartz::assert() unless token() == :foreach
	    scan()
	    _syntaxerr("':foreach' requires '('.") unless token() == '('
	    scan()
	    assign_expr = parse_assignment(true)
	    _syntaxerr("':foreach(' is not closed.") unless token() == ')'
	    scan()
	    _syntaxerr("':foreach' requires assignment.") unless assign_expr.token == '='
	    nodelist = parse_stmt_list()
	    _syntaxerr("':foreach' is not closed by ':end'.") unless token() == :end
	    scan()
	    return ForeachStmtNode.new(:foreach, assign_expr, nodelist)
	end


	##
	## * BNF:
	##    print-stmt ::= ':print'  '(' expression { ',' expression } ')'
	##
	## * return node:
	##   token::  :print
	##   left::   expression ExprNode
	##   right::  nil
	##
	def parse_stmt_print
	    Kwartz::assert() unless token() == :print
	    scan()
	    _syntaxerr("':print' requires '('.") unless token() == '('
	    scan()
	    arg_expr = parse_expression()
	    while token() == ',' do
		scan()
		arg_expr2 = parse_expression()
		arg_expr  = ExprNode.new(',', arg_expr, arg_expr2)
	    end
	    _syntaxerr("':print(' is not closed.") unless token() == ')'
	    scan()
	    return PrintStmtNode.new(:print, arg_expr, nil)
	end


	##
	## * BNF:
	##    macro-stmt ::= ':macro' '(' name ')' stmt-list ':end'
	##
	## * return node:
	##   token::  :macro
	##   left::   name String
	##   right::  NodeList
	##
	def parse_stmt_macro
	    Kwartz::assert() unless token() == :macro || token() == :elem
	    scan()
	    _syntaxerr("':macro' requires '('.") unless token() == '('
	    scan()
	    name_expr = parse_expression()
	    _syntaxerr("':macro(' is not closed.") unless token() == ')'
	    scan()
	    _syntaxerr("':macro' requires a name.") unless name_expr.token == :variable
	    name = name_expr.left
	    nodelist = parse_stmt_list()
	    _syntaxerr("':macro' is not closed by ':end'.") unless token() == :end
	    scan()
	    return MacroStmtNode.new(:macro, name, nodelist)
	end


	##
	## * BNF:
	##    elem-stmt  ::= ':elem'  '(' name ')' stmt-list ':end'
	##
	## * return node:
	##   token::  :macro
	##   left::   name String
	##   right::  NodeList
	##
	def parse_stmt_elem
	    Kwartz::assert() unless token() == :elem
	    scan()
	    _syntaxerr("':elem' requires '('.") unless token() == '('
	    scan()
	    name_expr = parse_expression()
	    _syntaxerr("':elem(' is not closed.") unless token() == ')'
	    scan()
	    _syntaxerr("':elem' requires an element name.") unless name_expr.token == :variable
	    name = name_expr.left
	    _push_element_name(name)
	    nodelist = parse_stmt_list()
	    _syntaxerr("':elem' is not closed by ':end'.") unless token() == :end
	    _pop_element_name()
	    scan()
	    return MacroStmtNode.new(:macro, 'elem_' + name, nodelist)
	end


	##
	## * BNF:
	##    expand-stmt ::= ':expand' '(' name ')'
	##
	## * return node:
	##   token::  :expand
	##   left::   name String
	##   right::  nil
	##
	def parse_stmt_expand
	    Kwartz::assert() unless token() == :expand
	    scan()
	    _syntaxerr("':expand' requires '('.") unless token() == '('
	    scan()
	    name_expr = parse_expression()
	    _syntaxerr("':expand(' is not closed.") unless token() == ')'
	    scan()
	    _syntaxerr("':macro' requires a name.") unless name_expr.token == :variable
	    name = name_expr.left
	    return ExpandStmtNode.new(:expand, name, nil)
	end


	##
	## * BNF:
	##     expand2-stmt ::= '@' macro-name
	##
	## * return node:
	##   token::   :expand
	##   left::    name String
	##   right::   nil
	##
	def parse_stmt_expand2
	    Kwartz::assert() unless token() == :expand2
	    macro_name = token_str()
	    scan()
	    return ExpandStmtNode.new(:expand, macro_name, nil)
	end


	##
	## * BNF:
	##     special-expand-stmt ::= '@stag' | '@cont' | '@etag' | ':stag' | ':cont' | ':etag'
	##
	## * return node:
	##   token::   :expand
	##   left::    name String
	##   right::   nil
	##
	def parse_stmt_specialexpand
	    Kwartz::assert() unless token() == :stag || token() == :cont || token() == :etag
	    elem_name = _element_name()
	    unless elem_name
		msg = "@stag, @cont or @etag must be in :elem() statement."
		raise SemanticError.new(msg)
	    end
	    macro_name = "#{token_str()}_#{elem_name}"
	    scan()
	    return ExpandStmtNode.new(:expand, macro_name, nil)
	end


	##
	## * BNF:
	##     value-stmt ::= ':value' '(' name '=' expression ')'
	##
	## * return node:
	##   token::   :value
	##   left::    name String ('cont_' + name)
	##   right::   NodeList (of PrintStmtNode)
	##
	def parse_stmt_value
	    Kwartz::assert() unless token() == :value
	    scan()
	    _syntaxerr("':value' requires '('.") unless token() == '('
	    scan()
	    assign_expr = parse_assignment()
	    _syntaxerr("':value(' is not closed.") unless token() == ')'
	    scan()
	    _syntaxerr("':value' requires assignment.") unless assign_expr.token == '='
	    name_expr = assign_expr.left
	    _syntaxerr("':value' requires a name.") unless name_expr.token == :variable
	    name = name_expr.left
	    expr = assign_expr.right
	    print_stmt = PrintStmtNode.new(:print, expr, nil)
	    nodelist = NodeList.new([print_stmt])
	    return  MacroStmtNode.new(:macro, 'cont_' + name, nodelist)
	end


	##
	## * BNF:
	##    rawcode ::= ':::' raw-string | ':rawcode' '(' string ')'
	##
	## * return node:
	##   token::  :rawcode
	##   left::   rawcode String
	##   right::  nil
	##
	def parse_stmt_rawcode
	    t = token()
	    if t == ':::'
		str = token_str()
	    elsif t == :rawcode
		scan()
		_syntaxerr("':rawcode' needs '('.") unless token() == '('
		scan()
		expr = parse_expression()
		_syntaxerr("':rawcode(' is not closed.") unless token() == ')'
		_syntaxerr("':rawcode()' needs a string.") unless expr.token == :string
		str = expr.left
	    end
	    scan()
	    return RawcodeStmtNode.new(:rawcode, str, nil)
	end


	##
	## * BNF:
	##    load-stmt ::= ':load' '(' string ')'
	##
	## * return:  nodelist
	##
	def parse_stmt_load
	    Kwartz::assert() unless token() == :load
	    scan()
	    _syntaxerr("':load' requires '('.") unless token() == '('
	    scan()
	    filename_expr = parse_expression()
	    _syntaxerr("':load(' is not closed.") unless token() == ')'
	    scan()
	    _syntaxerr("':load' requires filename as a string.") unless filename_expr.token == :string
	    filename = filename_expr.left
	    @toppings[:load_path].each do |dir|
		if test(?d, dir) && test(?f, "#{dir}/#{filename}")
		    filename = "#{dir}/#{filename}"
		    break
		end
	    end if @toppings[:load_path]
	    unless test(?f, filename)
		msg = "'#{filename}': not found."
		raise ParseError.new(msg)
	    end
	    plcode = File.open(filename) { |f| f.read }
	    nodelist = Kwartz::Helper.parse(plcode, @toppings)
	    return nodelist
	end

	##
	## main parse method
	##
	def parse
	    nodelist = parse_stmt_list()
	    unless token() == nil
		if token.is_a?(String)
		    s = "'#{token()}'"
		else
		    case token()
		    when :name, :number
			s = "'#{token_str()}'"
		    when :string
			s = "\"'#{token_str()}'\""
		    else
			s = token()
		    end
		end
		_syntaxerr("#{s}: invalid statement.")
	    end
	    #
	    macro_list = []
	    stmt_list = []
	    nodelist.each do |stmtnode|
		if stmtnode.token == :macro
		    macro_list << stmtnode
		else
		    stmt_list << stmtnode
		end
	    end
	    return NodeList.new(macro_list + stmt_list)
	end

    end


    ## ============================================================


    ##
    ## usage.
    ##	 str = File.open('file.txt') { |f| f.read }
    ##	 scanner = Scanner.new(str)
    ##	 while (token = scanner.scan()) != nil do
    ##	   print token, ':', scanner.token_str, "\n"
    ##	 end
    ##
    class Scanner
	def initialize(plcode_str, toppings={})
	    @content = plcode_str
	    @pos = -1
	    @token = nil
	    @token_str = nil
	    @linenum = 1
	    @toppings = toppings
	    if toppings[:enable_eruby]
		unless toppings[:lang]
		    raise ScanError.new("option '--enable_eruby' requires language name.")
		end
		@content = _exec_eruby(@content, toppings[:lang])
	    end
	    @newline = "\n"
	    if (idx = plcode_str.index(?\n)) != nil
		@newline = "\r\n" if plcode_str[idx - 1] == ?\r
	    end
	end
	attr_reader :token, :token_str, :linenum, :newline

	## ----- private methods -----

	private

	def _exec_eruby(_plogic, __lang__)
	    require 'erb'
	    erb = ERB.new(_plogic, $SAFE, '%')
	    return erb.result(binding())
	end

	def getchar
	    @pos += 1
	    @linenum += 1 if (ch = @content[@pos]) == ?\n
	    return ch
	end
	
	def nextchar
	    return @content[@pos+1]
	end

	def cutstr(ch)
	    s = ''
	    while (c = getchar()) != nil
		s << c
		break if c == ch
	    end
	    return s
	end

	def alpha?(ch)
	    return nil if ch == nil
	    return ?a <= ch && ch <= ?z || ?A <= ch && ch <= ?Z
	end

	def num?(ch)
	    return nil if ch == nil
	    return ?0 <= ch && ch <= ?9
	end

	def word?(ch)
	    return nil if ch == nil
	    return alpha?(ch) || num?(ch) || ch == ?_
	end

	def space?(ch)
	    return nil if ch == nil
	    return ch == ?\  || ch == ?\t || ch == ?\n || ch == ?\r
	end

	## ----- public methods -----

	public

	## return a line string
	def line(linenum=@linenum)
	    lines = @content.split(/\n/)
	    return lines[linenum-1]
	end

	Keywords = {
	    ":if"      => :if,
	    ":elsif"   => :elsif,
	    ":else"    => :else,
	    ":while"   => :while,
	    ":loop"    => :loop,
	    ":foreach" => :foreach,
	    #":for"     => :for,
	    ":macro"   => :macro,
	    ":expand"  => :expand,
	    ":set"     => :set,
	    ":print"   => :print,
	    ":end"     => :end,
	    ":rawcode" => :rawcode,
	    ":elem"    => :elem,
	    #":stag"    => :stag,
	    #":cont"    => :cont,
	    #":etag"    => :etag,
	    ":value"   => :value,
	    ":load"    => :load,
	}

	## return a token, and set @token_str.
	## see token() and token_str().
	def scan
	    @token = '*** debug ***'
	    @token_str = '*** debug ***'

	    ## ignore space and comment
	    while (ch = getchar()) != nil do
		if space?(ch)	# ignore space
		    next
		end
		if ch == ?\#	# ignore comment
		    while (ch = getchar()) != ?\n && ch != nil do; end
		    next
		end
		break			# break if else
	    end
		

	    ## check token

	    ## nil
	    if ch == nil
		@token = @token_str = nil
		return @token
	    end

	    ## name (variable, macro, ...), or true/false/nil
	    if alpha?(ch)
		s = ch.chr
		s << getchar() while word?(nextchar())
		@token_str = s
		case s
		when 'true'
		    @token = :true
		when 'false'
		    @token = :false
		when 'null', 'nil'
		    @token = :null
		when 'empty'
		    @token = :empty
		else
		    @token = :name
		end
		return @token
	    end

	    ## number
	    if num?(ch)
		s = ch.chr
		s << getchar() while num?(nextchar())
		if nextchar() == ?.
		    s << getchar()
		    s << getchar() while num?(nextchar())
		end
		@token_str = s
		@token = :number
		return @token
	    end

	    
	    ## keywords(:if, :while, ...), ':::' and ':'
	    if ch == ?\:
		s = ':'
		if nextchar() == ?\:
		    getchar()
		    if nextchar() == ?\:
			getchar()
			@token = ':::'
			@token_str = cutstr(?\n)
			@token_str.chomp!
		    else
			@token = '::'
			@token_str = cutstr(?\n)
			@token_str.chomp!
		    end
		elsif alpha?(nextchar())
		    s << getchar() while alpha?(nextchar())
		    @token_str = s
		    @token = Keywords[s]
		    unless @token != nil
			raise SyntaxError.new("'#{s}': invalid token.", self)
		    end
		else
		    @token = @token_str = ':'
		end
		return @token
	    end
	    
	    ## "string"
	    if ch == ?"					#"
		s = ""
		while (ch = getchar()) != ?"		#"
		    raise SyntaxError.new("string is not closed.", self) unless ch
		    if ch == ?\\
			case ch = getchar()
			when ?n  ; ch = ?\n
			when ?r  ; ch = ?\r
			when ?t  ; ch = ?\t
			when nil ; raise SyntaxError.new("string is not closed.", self)
			end
		    end
		    s << ch
		end
		@token_str = s
		@token = :string
		return @token
	    end

	    ## 'string'
	    if ch == ?'					#'
		s = ""
		while (ch = getchar()) != ?'		#'
		    raise SyntaxError.new("string is not closed.", self) unless ch
		    if ch == ?\\
			ch = getchar()
			raise SyntaxError.new("string is not closed.", self) unless ch
			if ch != ?\\ && ch != ?'	#'
			    s << '\\' 
			end
		    end
		    s << ch
		end
		@token_str = s
		@token = :string
		return @token
	    end

	    ## other symbol
	    case ch
		
	    when ?(, ?), ?{, ?}, ?]
		@token = @token_str = ch.chr
		
	    when ?[
		if nextchar() == ?:
		    getchar()
		    @token = @token_str = '[:'
		else
		    @token = @token_str = '['
		end

	    when ?.
		case nextchar()
		when ?+, ?<, ?>, ?=, ?!
		    c = getchar()
		    if nextchar() == ?=
			getchar()
			@token = @token_str = ".#{c.chr}="
		    else
			@token = @token_str = ".#{c.chr}"
		    end
		else
		    @token = @token_str = '.'
		end

	    when ?!
		if nextchar() == ?=
		    getchar()
		    @token = @token_str = '!='
		else
		    @token = @token_str = '!'
		end
		
	    when ?<, ?>, ?=
		case nextchar()
		when ?\=
		    getchar()
		    @token = @token_str = ch.chr + '='
		else
		    @token = @token_str = ch.chr
		end

	    when ?+, ?-, ?*, ?/, ?%, ?^
		if nextchar() == ?=
		    getchar()
		    @token = @token_str = ch.chr + '='
		else
		    @token = @token_str = ch.chr
		end

	    when ?\\, ??, ?\,
		@token = @token_str = ch.chr

	    when ?&, ?|
		if nextchar() == ch
		    getchar()
		    @token = @token_str = ch.chr + ch.chr
		else
		    raise SyntaxError.new("invalid char: #{ch.chr}", self)
		end

	    when ?@
		s = ''
		s << getchar() while word?(nextchar())
		case s
		when 'stag', 'cont', 'etag'
		    @token = s.intern
		    @token_str = s
		else
		    @token = :expand2
		    @token_str = s
		    if s.empty?
		    	msg = "'@': macro name required."
		    	raise SyntaxError.new(msg)
		    end
		    #raise SyntaxError.new("'@#{s}': invalid token.", self)
		end

	    else
		raise SyntaxError.new("invalid char: #{ch.chr}", self)
	    end

	    return @token
	end


	## for debug
	def scan_all
	    s = ''
	    while scan() != nil do
		if @token == :name || @token == :number
		    s << @token_str
		elsif @token == :true || @token == :false || @token == :null || @token == :empty
		    s << @token_str
		elsif @token == :string
		    s << @token_str.inspect
		#elsif @token == :stag || @token == :cont || @token == :etag
		#    s << '@' + @token.id2name
		elsif @token == :expand2
		    s << '@' + @token_str
		elsif @token.is_a?(Symbol)
		    s << @token.inspect
		elsif @token == ':::'
		    s << @token + @token_str
		elsif @token == '::'
		    s << @token + @token_str
		else
		    s << @token
		end
		s << @newline
	    end
	    return s
	end
    end

    ### ------------------------------------------------------------

    module Helper
	def self.compile(pdata_str, plogic_str, lang, toppings)
	    compiler = Kwartz::Compiler.new(lang, toppings)
	    output = compiler.compile(pdata_str, plogic_str)
	    #converter  = Kwartz::DefaultConverter.new(pdata_str)
	    #scanner    = Kwartz::Scanner.new(converter.convert() + plogic_str)
	    #parser     = Kwartz::Parser.new(scanner)
	    #translator = Kwartz::Translator.instance(lang, scanner.newline)
	    #nodelist   = parser.parse()
	    #output     = translator.translate_all(nodelist)
	    return output
	end
	
	def self.convert(pdata_str, toppings)
	    converter  = Kwartz::DefaultConverter.new(pdata_str, toppings)
	    output     = converter.convert()
	    return output
	end
	
	def self.translate(plogic_str, lang, toppings)
	    unless Kwartz::Translator.registered?(lang)
		raise InvalidLanguageError.new(lang)
	    end
	    scanner    = Kwartz::Scanner.new(plogic_str, toppings)
	    parser     = Kwartz::Parser.new(scanner, toppings)
	    translator = Kwartz::Translator.instance(lang, toppings, scanner.newline)
	    nodelist   = parser.parse()
	    output     = translator.translate_all(nodelist)
	    return output
	end
	
	def self.parse(plcode_str, toppings)
	    scanner    = Kwartz::Scanner.new(plcode_str, toppings)
	    parser     = Kwartz::Parser.new(scanner, toppings)
	    nodelist   = parser.parse()
	    return nodelist
	end

	def self.parse_stmt(plcode_str, toppings)
	    scanner    = Kwartz::Scanner.new(plcode_str, toppings)
	    parser     = Kwartz::Parser.new(scanner, toppings)
	    nodelist   = parser.parse()
	    output     = nodelist.print_all(scanner.newline)
	    return output
	end

	def self.parse_expr(expression_str, toppings)
	    scanner = Kwartz::Scanner.new(expression_str, toppings)
	    parser	= Kwartz::Parser.new(scanner, toppings)
	    #node = parser.parse_expression()
	    node = parser.parse_assignment()
	    if parser.token
		msg = ''
		raise Kwartz::SyntaxError.new(msg, scanner)
	    end
	    output = node.print_all
	    return output
	end
	
	def self.scan(pdata_str, toppings)
	    scanner    = Kwartz::Scanner.new(pdata_str, toppings)
	    output     = scanner.scan_all()
	    return output
	end
	
	def self.delete_directives(input_str, attr_name='kd')
	    s = input_str.gsub(/\b#{attr_name}=(\".*?\"|[\-\:\#\@\w]+)/, '')
	    return s
	end

	def self.fetch(input_str)
	    converter  = Kwartz::DefaultConverter.new(input_str, {})
	    converter.instance_eval {
		def self.fetch
		    return _fetch()
		end
	    }
	    s = ''
	    while fv = converter.fetch() do
		s << fv.head_text.inspect << " + " << fv.tag_str.inspect << "\n"
	    end
	    return s
	end
	
	def self.analyze(pdata_str, plogic_str, toppings)
	    converter  = Kwartz::DefaultConverter.new(pdata_str)
	    scanner    = Kwartz::Scanner.new(converter.convert() + plogic_str)
	    parser     = Kwartz::Parser.new(scanner)
	    nodelist   = parser.parse()
	    analyzer   = Kwartz::Analyzer.new(nodelist)
	    analyzer.analyze()
	    newline    = scanner.newline
	    s = ''
	    analyzer.warnings.each do |message|
		s << '*** warning: ' << message << newline
	    end
	    s << "global variables:"
	    analyzer.globals.each { |varname| s << ' ' << varname }
	    s << newline
	    s << "local variables: "
	    analyzer.locals.each { |varname| s << ' ' << varname }
	    s << newline
	    return s
	end

	def self.do_action(action, input, plogic, lang, toppings)
	    output = nil
	    case action
	    when 'compile'
		output = self.compile(input, plogic, lang, toppings)
	    when 'convert'
		output = self.convert(input, toppings)
	    when 'translate'
		output = self.translate(input + plogic, lang, toppings)
	    when 'parse'
		output = self.parse_stmt(input + plogic, toppings)
	    when 'expr', 'parse-expr'
		output = self.parse_expr(input + plogic, toppings)
	    when 'scan'
		output = self.scan(input + plogic, toppings)
	    when 'delete_directives'
		attr_name = toppings[:attr_name] || 'kd'
		output = self.delete_directives(input, attr_name)
	    when 'fetch'
		output = self.fetch(input)
	    when 'analyze'
		output = self.analyze(input, plogic, toppings)
	    end
	    return output
	end

    end

    module Util
	class CommandOptionError < StandardError
	    def initialize(msg)
		super(msg)
	    end
	end
	
	def self.parse_argv(argv, single_options=nil, argneed_options=nil)
	    options  = {}
	    toppings = {}
	    return options, toppings if !argv
	    
	    opttypes = {}
	    single_options.each_byte do |ch|
		opttypes[ch] = :single
	    end if single_options
	    argneed_options.each_byte do |ch|
		opttypes[ch] = :arg_need
	    end if argneed_options
	    opttypes[?-] = :topping
	    
	    while argv[0] && argv[0][0] == ?- do
		optstr = argv.shift		# argv.shift.sub(/^-/, '')
		optstr = optstr[1, optstr.length-1]
		
		while optstr && !optstr.empty? do
		    optchar = optstr[0]
		    optstr[0] = ''		# optstr = optstr[1, optstr.length-1]
		    optstr = nil if optstr && optstr.empty?
		    case opttypes[optchar]
		    when :single
			options[optchar] = true
		    when :arg_need
			arg = optstr || argv.shift
			raise CommandOptionError.new("-#{optchar.chr}: argument reguired.") unless arg
			options[optchar] = arg
			optstr = nil
		    when :topping
			if optstr == 'help'
			    options[?h] = true
			elsif optstr =~ /^(\w+)(=.*)?/
			    key   = $1.intern
			    value = $2
			    if !value || value.empty?
				value = true
			    else
				value.sub!(/^=/, '')
				value = str2value(value)
			    end
			    toppings[key] = value
			end
			optstr = nil
		    else
			raise CommandOptionError.new("-#{optchar.chr}: invalid option.")
		    end
		end
	    end
	    return options, toppings
	end

	def self.str2value(str)
	    case str
	    when /^\d+$/
		value = str.to_i
	    when /^true$/i, /^yes$/i
		value = true
	    when /^false$/i, /^no$/i
		value = false
	    when /^nil$/, /^null$/i
		value = nil
	    when /^\d+$/
		value = str.to_i
	    when /^(\d*\.\d+|\d+\.\d*)$/
		value = str.to_f
	    when /^\/(.*)\/$/
		value = Regexp.new($1)
	    when /^'(.*)'$/
		value = eval value
	    when /^"(.*)"$/
		value = eval value
	    else
		value = str
	    end
	    return value
	end

    end
end



##
## test program
##
if $0 == __FILE__
    command = File::basename($0)
    usage = <<END
kwartz.rb -- version #{Kwartz::VERSION} (#{Kwartz::LAST_UPDATE})

Usage:
    #{command} [options...] [file ...]

Options:
    -a action	    : compile,convert,translate,scan,parse,parse-expr
    -l lang	    : ruby,php,jsp,eruby,erb,erbscan,velocity,...
    -p plogic-file  : presentation logic file
    -s              : sanitize
    -h		    : help
END

    ## default value
    action = 'compile'
    lang = 'ruby'
    plogic = ''

    ## parse command line options
    begin
	options, toppings = Kwartz::Util.parse_argv(ARGV, "hs", "alp")
    rescue Kwartz::Util::CommandOptionError => ex
	$stderr.print ex.message, "\n"
	$stderr.print usage
	exit 1
    end

    ## set values
    action = options[?a] if options[?a]
    lang   = options[?l] if options[?l]
    plogic = options[?p] if options[?p]
    if options[?h]
	$stderr.print usage
	exit 0
    end
    if options[?s]
	toppings[:escape] = true
    end
    if lang == 'jsp'
	header = ''
	s = "<%@ page contentType=\"text/html; charset=#{toppings[:charset]}\" %>\n"
	header << s if toppings.key?(:charset)
	s = "<%@ taglib prefix=\"c\" uri=\"http://java.sun.com/jstl/core\" %>\n"
	header << (toppings[:header_text] || s)
	toppings[:header_text] = header
    end

    ## main
    input = ARGF.read()
    print Kwartz::Helper.do_action(action, input, plogic, lang, toppings)
end

#[EOF]
