 /**************************************************************************
 *   Copyright (C) 2005,2006 by Danny Kukawka                              *
 *                            <dkukawka@suse.de>, <danny.kukawka@web.de>   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

/*! 
 * \file 	dbusPowersave.cpp
 * \brief 	This file contains the functionality to connect to powersave
 *	        via D-BUS for kpowersave.
 * \author 	Danny Kukawka, <dkukawka@suse.de>, <danny.kukawka@web.de>
 * \date    	2005, 2006
 */

 // KDE Header
#include <klocale.h>

// DBUS - Header
#include "dbusPowersave.h"
#include <powersave_dbus.h>

#include <iostream>

static void* myInstance = 0;

/*! The default constructor of the class dbusPowersaveConnection. */
dbusPowersaveConnection::dbusPowersaveConnection(){
	myDebug ("dbusPowersaveConnection::dbusPowersaveConnection");
	is_connected = false;
	no_rights = false;
	
	// add pointer to this for filter_function()
	myInstance=this;
	// init connection to dbus
	if(!initDBUS()) {
		myDebug("ERROR: Can't connect to DBUS");
		m_dBusQtConnection = NULL;
	}
}

/*! This is the default destructor of class dbusPowersaveConnection. */
dbusPowersaveConnection::~dbusPowersaveConnection(){
	myDebug ("dbusPowersaveConnection::~dbusPowersaveConnection");
	myInstance = NULL;
}

/*!
 * Use this SLOT to emit a reviced messages to the pdaemon.
 * \param  type enum with the type of the message
 * \param  _string String with the message
 * \param  _int Integer with e.g. percentage values for the progressbar
 */
void dbusPowersaveConnection::emitMsgReceived( msg_type type, QString _string, int _int ) {
	
	emit msgReceived_withStringInteger( type, _string, _int );
}

/*! 
 * This function return information about connection status
 * to powersave over the DBUS daemon.
 * \return boolean with the state of the connection to D-BUS
 * \retval true if connected
 * \retval false if disconnected
 */
bool dbusPowersaveConnection::isConnected() {
	return is_connected;
}

/*! 
 * This function return information about rights to acced powersaved via DBUS.
 * \return boolean with information if user has not the needed rights
 * \retval true if not enough rights
 * \retval false if enough rights
 */
bool dbusPowersaveConnection::noRights() {
	return no_rights;
}

/*! 
 * This function try a reconnect to powersave over the DBUS daemon.
 * \return boolean with the result of the operation
 * \retval true if successful reconnected to D-BUS/powersave
 * \retval false if unsuccessful
 */
bool dbusPowersaveConnection::reconnect() {
	return initDBUS();
}

/*! 
 * This function close the connection to powersave over the DBUS daemon.
 * \return boolean with the result of the operation
 * \retval true if successful closed the connection
 * \retval false if any problems
 */
bool dbusPowersaveConnection::close() {
	if ( m_dBusQtConnection != NULL ) {
		m_dBusQtConnection->close();
		m_dBusQtConnection = NULL;
	}
	is_connected = false;

	return true;
}

/*! 
 * This function initialise the connection to powersave over the DBUS daemon.
 * \return boolean with the result of the operation
 * \retval true if successful initialised D-Bus connection
 * \retval false if unsuccessful
 */
bool dbusPowersaveConnection::initDBUS(){
	myDebug ("dbusPowersaveConnection::initDBUS");

        DBusError error;
        dbus_error_init(&error);
	
	DBusConnection *dbus_connection = dbus_connection_open_private( DBUS_SYSTEM_BUS_SOCKET, &error );

	if (dbus_connection == NULL){
		myDebug ( "Failed to open connection to system message bus: %s\n", error.message);
		dbus_error_free (&error);
		return false;
	}

	dbus_bus_register( dbus_connection, &error);

	if ( dbus_error_is_set( &error ) ) {
		myDebug ( "Failed to register connection with system message bus: %s\n", error.message);
		return false;
	}
	dbus_connection_set_exit_on_disconnect( dbus_connection, false );

        /* add the filter function which should be executed on events on the bus */
        if ( ! dbus_connection_add_filter( dbus_connection, filter_function, this, NULL) ) {
                myDebug ("Error: Not enough memory to add filter to dbus connection\n");
                exit(EXIT_FAILURE);
        }

	/* connect to the daemon ( from powersave_dbus )*/
	int capabilities = CAPABILITY_NOTIFICATIONS | CAPABILITY_SCREENLOCK | CAPABILITY_BRIGHTNESS;
	int reply = establishConnection( capabilities, dbus_connection );
	if ( reply != REPLY_SUCCESS ) {
		// send a ping to find out if the user has not the correct rights
		reply = dbusSendSimpleMessage(ACTION_MESSAGE, "Ping");
		if (reply == REPLY_NO_RIGHTS) {
			myDebug ( "User is not permitted to connect to daemon" );
			no_rights = true;
		}

		is_connected = false;
		return false;
        }

        /* add a match rule to catch all signals going through the bus with
         * powersave manager interface */
	dbus_bus_add_match( dbus_connection, "type='signal',"
			    "interface='com.novell.powersave.manager',"
			    "path='/com/novell/powersave',", NULL);

	dbus_bus_add_match( dbus_connection, "type='signal',"
			    "interface='org.freedesktop.DBus'," 
			    "member='NameOwnerChanged'", NULL);

	m_dBusQtConnection = new DBusQt::Connection(this);
        m_dBusQtConnection->dbus_connection_setup_with_qt_main(dbus_connection);
	
	is_connected = true;
	no_rights = false;
	return true;
}


/*! 
 * This function is needed filter function for the DBUS connection to filter
 * all needed messages from the bus which are needful for KPowersave. 
 * \param connection	existing connection to the D-BUS daemon
 * \param message 	the recieved message from the D-Bus daemon
 * \param data		void pointer (see dbus bindings for more information)
 * \return DBusHandlerResult
 */
DBusHandlerResult 
filter_function (DBusConnection *connection, DBusMessage *message, void *data) {

	if( connection == NULL || data == NULL ) ; // to prevent compiler warning

	char *value;
	DBusError error;
        dbus_error_init( &error );

	if (dbus_message_is_signal (message,
				    DBUS_INTERFACE_LOCAL,
				    "Disconnected")){
		((dbusPowersaveConnection*) myInstance)->emitMsgReceived( DBUS_EVENT, "dbus.terminate", 0 );
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

        if ( dbus_message_get_type( message ) != DBUS_MESSAGE_TYPE_SIGNAL ) {
                return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }

        /* get the name of the signal */
        const char *signal = dbus_message_get_member( message );
	
	/* get the first argument. This must be a string at the moment */
        dbus_message_get_args( message, &error, DBUS_TYPE_STRING, &value, DBUS_TYPE_INVALID );

        if ( dbus_error_is_set( &error ) ) {
                myDebug ("Warning: Received signal %s, but no string argument\n", error.message );
                dbus_error_free( &error );
                return DBUS_HANDLER_RESULT_HANDLED;
        }

	myDebug("filter_function::SIGNAL=%s|VALUE=%s", signal, value);

        /* our name is... */
        if ( ! strcmp( signal, "NameAcquired" ) ) {
                return DBUS_HANDLER_RESULT_HANDLED;
        }
	
	else if ( ! strcmp( signal, "NameOwnerChanged" )) {
		char *old_owner;
                char *new_owner;
		
		if (!dbusGetMessageString(message, &old_owner, 0) && !dbusGetMessageString(message, &new_owner, 2) &&
		    old_owner != NULL && new_owner != NULL) {
			if (!strcmp(new_owner, "")) {
				// test if hal broke away
				if (!strcmp(old_owner, "org.freedesktop.Hal")) {
					// Hal service stopped.
					((dbusPowersaveConnection*) myInstance)->emitMsgReceived( DBUS_EVENT, "hal.terminate", 0 );
				}
			}
			// okay this is ugly, but we reuse the code from above ;-)
			else if (!strcmp(old_owner, "org.freedesktop.Hal")) {
				// Hal service started.
				((dbusPowersaveConnection*) myInstance)->emitMsgReceived( DBUS_EVENT, "hal.started", 0 );
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;
	}

        /* acpi event received */
        else if ( ! strcmp( signal, "AcpiEvent" ) ) {
		// do nothing! uninteresting event!
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }
        /* powersave event received */
        else if ( ! strcmp( signal, "PowersaveEvent" ) ) {
		((dbusPowersaveConnection*) myInstance)->emitMsgReceived( POWERSAVE_EVENT, value, 0 );
		return DBUS_HANDLER_RESULT_HANDLED;
        }
        /* progress information received */
        else if ( ! strcmp( signal, "Progress" ) ) {
		/*
		int value2;
		
		dbus_message_get_args( message, &error, DBUS_TYPE_INT32, &value2, DBUS_TYPE_INVALID );
		if ( dbus_error_is_set( &error ) ) {
			qDebug ("Warning: Received signal %s, but no string argument\n", error.message );
			dbus_error_free( &error );
                	return DBUS_HANDLER_RESULT_HANDLED;
        	}
		
		((dbusPowersaveConnection*) myInstance)->emitMsgReceived( PROGRESS, value, value2 );
		*/
		((dbusPowersaveConnection*) myInstance)->emitMsgReceived( PROGRESS, value, 0 );
		return DBUS_HANDLER_RESULT_HANDLED;
        }
        /* notification received */
        else if ( ! strcmp( signal, "Notification" ) ) {
		((dbusPowersaveConnection*) myInstance)->emitMsgReceived( NOTIFICATION, value, 0 );
		return DBUS_HANDLER_RESULT_HANDLED;
        }
        /* screenlock request received */
        else if ( ! strcmp( signal, "Screenlock" ) ) {
		((dbusPowersaveConnection*) myInstance)->emitMsgReceived( SCREENLOCK, value, 0 );
		return DBUS_HANDLER_RESULT_HANDLED;
        }
        else {
                myDebug ("Received unknown signal %s. Value: %s\n", signal, value );
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }
}
