 /**************************************************************************
 *   Copyright (C) 2004 by  Thomas Renninger                               *
 *                            <trenn@suse.de> and                          *
 *                          Danny Kukawka                                  *
 *                            <dkukawka@suse.de>, <danny.kukawka@web.de>   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#ifndef _PDAEMON_H_
#define _PDAEMON_H_

// include powersave specific header
#include <powerlib.h>
#include <powersave_hal.h>

// include own header
#include "dbusPowersave.h"
#include "schemes.h"

#ifdef DISTRO_IS_SLACKWARE
#include <unistd.h>
#endif

 /*! 
 *  \file 	pdaemon.h
 *  \brief 	Headerfile for pdaemon.cpp and the class \ref pDaemon.
 */
 
 /*! 
 *  \class 	pDaemon
 *  \brief 	contains all functions for the connection to powersaved
 *  \author 	Thomas Renninger, <trenn@suse.de> (2004)
 *  \author 	Danny Kukawka, <dkukawka@suse.de>, <danny.kukawka@web.de> (2004,2005)
 *  \author	Holger Macht, <hmacht@suse.de> (2005)
 *  \version 	0.6.0
 *  \date    	2004 - 2005
 */
class pDaemon :public QObject{
	
	Q_OBJECT

private:

	//!
	/*! This is used to get notified about powersave events and requests */
	dbusPowersaveConnection *dbus_conn;

	//! Timer to poll battery state from /proc
	/*!
	* This timer is used if the powersave daemon isn't running and AC is plugged in.
	* We use this timer to control the polling of /proc/acpi/battery instead of get 
	* battery states from powersaved. The timerinterval is defined through 
	* \ref BAT_PdOffAcOn_timeout .
	*/
	QTimer *BAT_PdOffAcOn_Timer;
	
	//! basic timer for polling
	/*!
	* Base time interval timer. This is the lowest interval for any 
	* constant polling. The timerintervall is defined in \ref system_timer_interval .
	*/
	QTimer *check_system_timer;
	
	//! Timer to poll battery state from /proc
	/*!
	* This timer is used if the powersave daemon isn't running and AC is plugged out.
	* We use this timer to control the polling of /proc/acpi/battery instead of get 
	* battery states from powersaved. The timerinterval is defined through 
	* \ref BAT_PdOffAcOff_timeout .
	*/
	QTimer *BAT_PdOffAcOff_Timer;

	//! QTimer-interval to poll battery state from /proc
	/*!
	* This intervall used to poll battery state from /proc if the powersave 
	* daemon isn't running and the machine is on AC. The value is 60000 msec 
	* (60 sec/1 min) because: if AC on, user don't need so often updated infos.
	*/
	static const int BAT_PdOffAcOn_timeout = 60000; // 60sec
	//! QTimer-interval to poll battery state from /proc
	/*!
	* This interval used to poll battery state from /proc, if the powersave 
	* daemon isn't running and the AC is plugged out. The value is 20000 msec 
	* (20 sec). This value is a compromise between system load (reading
	* battery state from /proc/acpi cost a lot processor usage) and fresh 
	* infomation about battery.
	*/
	static const int BAT_PdOffAcOff_timeout = 20000; // 20sec
	//! QTimer-interval for poll the system
	/*! Smalles time slice to check something or poll the system. The value is 1000 msec. */
	static const int system_timer_interval = 1000; // 1sec
	
	//! Integer with the number of availabl brightness level
	/*! This contains the number of available brightness levels for internal usage. */
	int availableBrightnessLevels;	

	//! Integer with the numbers of CPUs in the system
	int numOfCPUs;

	//! read data from daemon
	int readDaemonData();
	//! Go back to daemon mode
	void switchToDaemonMode();
	//! Daemon broke away get data directly from /proc via polling (timers)
	void switchToNonDaemonMode();

	// -------------- PRIVATE HELPER FUNCTIONS ------------------ //
	//! handle the errors from batteryinfos get from powerlib
	void handleGetBatteryInfoError( int );
	//! simply get a integer value from powersave via DBUS
	int simpleGetIntegerValue ( QString );
	
private slots:
	
	//! for check_system_timer
	void checkSystemTimeout();
	//! recheck the powersave daemon if problems with connection to library
	void recheckDaemon();
        //! checks the battery state directly from the procfs
	int checkBatteryProcInfo();
	//! informs about powersave messages and events
	void processClientMessage( msg_type, QString, int );
	
signals:

	//! signal to check and update graphical elements
	void generalDataChanged();
        //! signal if the schemes changed
	void schemeDataChanged();
	//! signal if there is a notification message
	void forwardClientMessage( QString );
	//! signal if the 'lid button close' event triggered
	void lidclosed();
	//! signal if the 'lid button open' event triggered
	void lidopened();
	//! signal for a 'lock screen' event
	void lockscreen();
	//! signal for a progress dialog
	// void progressDialogRequest( QString, int );
	void progressDialogRequest( QString );
	//! signal if back from suspend
	void resumeFromSuspend( bool );
	//! signal with info from what we resumed
	void resumedFrom( QString );
	//! signal for the current type of suspend
	void setSuspend( QString );
	//! signal which say that we recieved a battery event from powersave
	void batteryInfoEvent();
	//! signal for battery warning states
	void batteryWARNState( QString );
	//! signal for battery warning states
	void ACStatus( bool );
	//! signal if powersave is not running
	void powersaveNotRunningDialog();
	//! signal for throttling
	void throttlingChanged();
	
public:
	//! default constructor
	pDaemon();
	//! default destructor
	virtual ~pDaemon();

	//! represent the current Throttling of the CPU
	/*!
	* This interge value represent the current throttling step of the CPU in percent
	* \li a value >= 0 in %
	*/
	QValueList <int> cpu_throttling;
		
	//! represent the current CPU speeds
	/*!
	* This integer value represent the current speed/frequency of the CPUs in Mhz
	* \li a value > 0 in Mhz
	*/
	QValueList <int> cpufreq_speed;

	//! represent the max. CPU speeds
	/*!
	* This integer value represent the max speed/frequency of the CPUs in Mhz
	* \li a value > 0 in Mhz
	*/
	QValueList <int> cpufreq_max_speed;

	//! indicates if the machine is on AC power
	/*!
	* This integer value inidcates if the machine is on AC power or rather if the
	* AC adapter is plugged in.
	* \li value == 1 or true if on AC power
	* \li value == 0 or FALSE if not on AC power
	* \li value == other for error as e.g. HAL_ERROR or UNKNOWN
	*/
	int on_AC_power;
	
	//! represent the percentage of battery power 
	/*!
	* This integer value indicates the actually percentage of the battery power state.
	* \li a value between 0 and 100
	* \li value == -1 if there is a problem / error / or if no battery present
	*/
	int perc;
	//! represent the left beeing battery time 
	/*!
	* This integer value represent the remaining battery time in minutes.
	* \li value >= 0 the remaining time
	* \li value == -1 if there is a problem / error / or if no battery present
	*/
	int left;
	//! represent the different battery states
	/*!
	* This value represent all the different battery states. The states are in some cases 
	* defined through the settings in the powersave daemon and the related config files.
	* \li BAT_NORM: if battery state is normal
	* \li BAT_WARN: if battery state is warning
	* \li BAT_LOW:  if battery state is low
	* \li BAT_CRIT: if battery state is critical
	* \li BAT_NONE: if no battery present
	*/
	int battery_state;
	//! represent the different charing states of the battery
	/*!
	* This integer indicates the different chargingstates of the battery.
	* \li CHARG_STATE_CHARGING:         if the battery is charging
	* \li CHARG_STATE_UNKNOWN: if the charging state is unknown
	*/
	int charge_state;

	//! represent the different CPUFregPolicies
	/*!
	* This integer value is related to this CPUFregPolicies:
	* \li CPU_HIGH   for performance 
	* \li CPU_AUTO   for dynamic CPUFreqency scaling
	* \li CPU_LOW    for powersave related CPUFreqency scaling
	* \li CPU_UNSUPP for CPUFreqency scaling not supported
	*/
	int cpufreq_policy;
	//! Indicates if the daemon is running. 
	/*!
	* This integer value represent the daemon stater and indicates if the powersave daemon 
	* is present and running.
	* \li value == 1 if the daemon is running
	* \li value == 0 if not
	*/
	int daemon_running;

	//! indicates if 'suspend to disk' is allowed
	/*!
	* This interger value indicates if 'suspend to disk' is allowed by root though YaST
	* or modified the configure files of the powersave daemon by other way.
	* \li value == 1 or true if 'suspend to disk' is allowed
	* \li value == 0 or FALSE if disallowed
	*/
	int suspend2disk_allowed;
	//! indicates if 'suspend to RAM' is allowed
	/*!
	* This interger value indicates if 'suspend to RAM' is allowed by root though YaST
	* or modified the configure files of the powersave daemon by other way.
	* \li value == 1 or true if 'suspend to RAM' is allowed
	* \li value == 0 or FALSE if disallowed
	*/
	int suspend2ram_allowed;
	//! indicates if 'stand-by' is allowed
	/*!
	* This interger value indicates if 'stand-by' is allowed by root though YaST
	* or modified the configure files of the powersave daemon by other way.
	* \li value == 1 or true if 'stand-by' is allowed
	* \li value == 0 or FALSE if disallowed
	*/
	int standby_allowed;

	//! indicates if 'suspend to disk' is supported
	/*!
	* This interger value indicates if 'suspend to disk' is supported by the machine.
	* \li value == 1 or true if 'suspend to disk' is supported
	* \li value == 0 or FALSE if unsupported
	*/
	int suspend2disk_supported;
	///! indicates if 'suspend to RAM' is supported
	/*!
	* This interger value indicates if 'suspend to RAM' is supported by the machine.
	* \li value == 1 or true if 'suspend to RAM' is supported
	* \li value == 0 or FALSE if unsupported
	*/
	int suspend2ram_supported;
	//! indicates if 'stand-by' is supported
	/*!
	* This interger value indicates if 'stand-by' is supported by the machine.
	* \li value == 1 or true if 'stand-by' is supported
	* \li value == 0 or FALSE if unsupported
	*/
	int standby_supported;

	//! tells if something on AC Power changed
	/*!
	* This boolean represent information about changes on AC power.
	* \li true: if something changed
	* \li false: if nothing changed (or this is reset to false if the message was consumed)
	*/
	bool update_info_ac_changed;
	//! tells if something at the sleeping states changed
	/*!
	* This boolean represent information about changes at the sleeping states.
	* \li true: if something changed
	* \li false: if nothing changed (or this is reset to false if the message was consumed)
	*/
	bool update_info_sleep_state_changed;
	//! tells if something on CPUFreq Policy changed
	/*!
	* This boolean represent information about changes at the CPUFreq Policy.
	* \li true: if something changed
	* \li false: if nothing changed (or this is reset to false if the message was consumed)
	*/
	bool update_info_cpufreq_policy_changed;
	//! tells if something changed within the battery states
	/*!
	* This boolean represent information about changes on the battery states.
	* \li true: if something changed
	* \li false: if nothing changed (or this is reset to false if the message was consumed)
	*/
	bool update_info_battery_state_changed;
	//! tells if something changed on the state of battery charging
	/*!
	* This boolean represent information about changes on the state of battery charging.
	* \li true: if something changed
	* \li false: if nothing changed (or this is reset to false if the message was consumed)
	*/
	bool update_info_battery_charge_changed;
	//! tells if the percentage of battery power changed 
	/*!
	* This boolean represent information about changes on the percentage of battery power.
	* \li true: if something changed
	* \li false: if nothing changed (or this is reset to false if the message was consumed)
	*/
	bool update_info_battery_perc_changed;
	//! tells if the  CPUFreq Speed changed
	/*! 
	* This boolean represent information about CPUFreq Speed changes.
	* \li true: if something changed
	* \li false: if nothing changed (or this is reset to false if the message was consumed)
	*/
	bool update_info_cpufreq_speed_changed;
	//! tells if the schemes changed
	/*!
	* This boolean represent information about scheme changes.
	* \li true: if something changed
	* \li false: if nothing changed (or this is reset to false if the message was consumed)
	*/
	bool update_info_scheme_info_changed;

	// this one is whether the battery state changed to a lower one
	//! used to tell the kicker applet to send a 'battery state changed' message
	/*!
	* This boolean is used to inform kpowersave about changes of the battery state and 
	* to popup a messagebox if the battery is in state BAT_WARN, BAT_LOW or BAT_CRIT.
	* \li true:  If the message should be send
	* \li false: If no message should be send or if the last message was consumed
	*/
	bool send_battery_state_change_message;
	//! used to tell the kicker applet to send a 'no daemon' message
	/*!
	* This boolean contains information about the message which displayed if no powersave daemon
	* is present or kpowersave is not able to connect to the daemon by other reasons. Through this
	* we guarantee that the message is displayed only one time.
	* \li true:  If the message should be send
	* \li false: If no message should be send or if the last message was consumed
	*/
	bool send_no_daemon_message;

	//! Including the sleeping states 
	/*! This 64bit integer contains information about the sleeping states supported by the system. */
	int64_t supported_sleeping_states;

	//! contains a list with the available schemes 
	/*!
	 * The list is constantly refreshed from the daemon through a DBus
	 * connection.
	 */
	Schemes schemes;

	//! checks the status of the AC adapter
	int checkACProcInfo();
	//! queries the powersave daemon for information
	int checkDaemon();
	//! checks if the machine is a laptop
	bool isLaptop();
	//! set the CPUFreqPolicy to the powersave daemon
	int setCPUFreqPolicy( int policy );
	//! Set the active scheme
	int setActiveScheme( QString schemename );
	//! Requests available schemes and changes from the powersave daemon
	int updateSchemeInfo();
	
	//! checks the current CPU Speed from sysfs
	int checkCPUSpeed();
	//! checks the Speed of throttling CPUs from /proc/cpuinfo
	int checkCPUSpeedThrottling();
	//! read the current throttling state of the CPUs from /proc/acpi/processor/CPUX/throttling
	bool getCPUThrottlingState();
	//! read the max speed of the CPUs
	void getCPUMaxSpeed();
	//! counts the total number of CPUs
	int getCPUNum();

	//! to get the current Brightness of the display
	int getBrightness( bool percentage );
	//! to get the max number of brightness levels supported by the display
	int getBrightnessLevels();
	//! to set the current Brightness of the display
	int setBrightness( int value, bool percentage, bool force_min = false );
};

// CPUFREQ is used for the global cpufreq_policy _and_ for initating actions!
enum CPUFREQ   {CPU_UNSUPP, CPU_LOW, CPU_AUTO, CPU_HIGH};
enum BAT_STATE {BAT_NONE, BAT_WARN, BAT_LOW, BAT_CRIT, BAT_NORM, BAT_HAL_ERROR};

#endif
