/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// renderer.cpp

#include "renderer.h"
#include "project.h"

#include <qpainter.h>
#include <qpaintdevice.h>
#include <qpaintdevicemetrics.h>
#include <qwmatrix.h>
#include <qfontmetrics.h>
#include <qapplication.h>

#include <kmessagebox.h>
#include <klocale.h>

#include <math.h>

#include <iostream>


namespace KoverArtist
{


Renderer::Renderer(Project* aProject)
:Inherited(aProject)
,mLandscape(false)
,mRenderFront(true)
,mRenderBack(true)
,mRendering(false)
,mAbort(false)
,mAbortable(false)
{
}


Renderer::~Renderer()
{
}


void Renderer::renderSurfaceBackground(int aX, int aY, const QPixmap& aPix)
{
   QPixmap pix(mWidth, mHeight);
   QPainter p(&pix);

   p.fillRect(0, 0, mWidth, mHeight, mProject->backgroundColor());
   p.drawPixmap(aX, aY, aPix);

   p.end();
   setSurfaceBackground(pix);
}


void Renderer::renderOutlines(int aLWidth, int aRWidth)
{
   int w=mWidth, h=mHeight-1;
   mPaint.setPen(mProject->outlineColor());
   mPaint.drawLine(0, 0, w, 0);
   mPaint.drawLine(0, h, w, h);
   mPaint.drawLine(0, 0, 0, h);
   if (aLWidth>0) mPaint.drawLine(aLWidth, 0, aLWidth, h);
   if (aRWidth>0 && aRWidth<mWidth)
   {
      int x = w-aRWidth;
      mPaint.drawLine(x, 0, x, h);
   }
   mPaint.drawLine(w, 0, w, h);
}


void Renderer::renderSideText(int /*aWidth*/, int aHeight, int aLWidth, int aRWidth,
                              bool aRotated)
{
   const TextStyle& style = mProject->sideStyle();
   int hfnt, w, th, cnt, xind=scaleX(50);
   QRect rect;

   QString sideText = mProject->title();
   if (!mProject->subTitle().isEmpty()) sideText += " - "+mProject->subTitle();
   sideText = sideText.simplifyWhiteSpace();

   const Effect& eff = mProject->sideEffect();
   QFont fnt(style.font());

   th = aHeight - xind;

   if (aLWidth>0)
   {
      for (w=99999,cnt=100,hfnt=int(aLWidth*0.9); w>th && cnt>0; --cnt,hfnt-=2)
      {
         fnt.setPixelSize(hfnt);
         QFontMetrics fm(fnt);
         w = fm.boundingRect(sideText).width();
      }

      if (aRotated) rect.setRect(xind, -aLWidth, th, aLWidth);
      else rect.setRect(-aHeight+xind, 0, aHeight-xind, aLWidth);

      beginEffect(eff);
      renderText(sideText, rect, style.alignment(), fnt, style.color(),
                 aRotated?90:270);
      endEffect();
   }

   if (aRWidth>0)
   {
      if (aRWidth!=aLWidth)
      {
         for (w=99999,cnt=100,hfnt=int(aRWidth*0.9); w>th && cnt>0; --cnt,hfnt-=2)
         {
            fnt.setPixelSize(hfnt);
            QFontMetrics fm(fnt);
            w = fm.boundingRect(sideText).width();
         }
      }
      if (aRotated) rect.setRect(-aHeight+xind, mWidth-aRWidth, th, aRWidth);
      else rect.setRect(xind, -mWidth, aHeight-xind, aRWidth);

      beginEffect(eff);
      renderText(sideText, rect, style.alignment(), fnt, style.color(),
                 aRotated?270:90);
      endEffect();
   }
}


void Renderer::renderFront(QPaintDevice* aDev, const QWMatrix& aMatrix)
{
   const Image& img = mProject->imgFront();
   const Case& c = mProject->box();
   int lWidth = scaleX(c.frontLeftSide());
   int rWidth = scaleX(c.frontRightSide());
   QSize front = scale(c.front());
   int y, h;

   beginSurface(aDev, aMatrix, front.width()+lWidth+rWidth, front.height());
   renderSurfaceBackground(img.onSides()?0:lWidth, 0, mPixFront);
   beginEffect(mProject->titleEffect());

   // Calculate position of title and sub-title
   int xind=scaleX(20), yind=scaleY(20);
   QRect outer(xind, yind, front.width()-(xind<<1), front.height()-(yind<<1));
   Qt::AlignmentFlags titleAlignHoriz = Qt::AlignmentFlags(mProject->titleStyle().alignment() & Qt::AlignHorizontal_Mask);
   Qt::AlignmentFlags titleAlignVert = Qt::AlignmentFlags(mProject->titleStyle().alignment() & Qt::AlignVertical_Mask);
   Qt::AlignmentFlags subTitleAlignHoriz =
      Qt::AlignmentFlags((mProject->subTitleStyle().alignment() &
                          Qt::AlignHorizontal_Mask) | Qt::DontClip);

   QFontMetrics titleMetrics(mProject->titleStyle().font());
   QRect titleRect = titleMetrics.boundingRect(outer.x(), outer.y(), outer.width(),
                           outer.height(), titleAlignHoriz, mProject->title());
   h = titleRect.height();

   QFontMetrics subTitleMetrics(mProject->subTitleStyle().font());
   QRect subTitleRect;
   if (mProject->subTitle() != "")
   {
      QRect outerSub(outer);
      outerSub.setY(outerSub.y() + h);
      outerSub.setHeight(outerSub.height() - h);
      subTitleRect = subTitleMetrics.boundingRect(outerSub.x(), outerSub.y(),
                  outerSub.width(), outerSub.height(), subTitleAlignHoriz,
                  mProject->subTitle());
   }

   int hh = titleRect.height() + subTitleRect.height();
   int ww = titleRect.width();
   if (subTitleRect.width()>ww) ww = subTitleRect.width();

   if (titleAlignVert==Qt::AlignTop) y = yind;
   else if (titleAlignVert==Qt::AlignVCenter) y = (mHeight-hh)>>1;
   else y = mHeight-hh-yind;
   y -= titleRect.top();

   // Title and sub-title text bounding rectangles
   QRect rtitle(lWidth+xind, y, front.width()-(xind<<1), hh);
   QRect rsubTitle(titleRect.left(), y+titleRect.height(), titleRect.width(),
                   subTitleRect.height());

   // Draw title and sub-title
   renderText(mProject->subTitle(), rsubTitle, subTitleAlignHoriz,
              mProject->subTitleStyle().font(), mProject->subTitleStyle().color());

   renderText(mProject->title(), rtitle, titleAlignHoriz,
              mProject->titleStyle().font(), mProject->titleStyle().color());

   endEffect();

   if (mAbortable) qApp->processEvents();
   if (!mAbort && (lWidth>0 || rWidth>0))
      renderSideText(front.width(), front.height(), lWidth, rWidth, true);

   endSurface();

   // Draw outlines
   renderOutlines(lWidth, rWidth);
}


void Renderer::renderContents(QRect bb)
{
   int numDiscs = mProject->count();
   QValueVector<QSize> contentsSize(numDiscs);
   QValueVector<QString> contentsText(numDiscs);
   QValueVector<QString> discTitleText(numDiscs);
   int contentsAlign = (mProject->contentsStyle().alignment() & Qt::AlignHorizontal_Mask) | Qt::AlignTop;
   int discTitleAlign = contentsAlign;
   int sumHeight, i, j, x, y, w, h, hc, hdt, num, wmax;
   const Effect& eff = mProject->contentsEffect();
   QRect r, rt;
   QSize sz, maxSize;
   const Disc* d;

   if (eff.type()!=Effect::NONE)
   {
      bb.setWidth(bb.width() - eff.size());
      bb.setHeight(bb.height() - eff.size());
   }

   int bbw=bb.width(), bbh=bb.height();

   QFont contentsFont = mProject->contentsStyle().font();
   mPaint.setFont(contentsFont);
   QFontMetrics contentsMetrics(mPaint.fontMetrics());

   QFont discTitleFont = mProject->discTitleFont();
   mPaint.setFont(discTitleFont);
   QFontMetrics discTitleMetrics(mPaint.fontMetrics());

   hdt = discTitleMetrics.boundingRect(0, 0, bbw, bbh, Qt::AlignLeft, "M").height();
   hc = contentsMetrics.boundingRect(0, 0, bbw, bbh, Qt::AlignLeft, "M").height();

   //
   // Collect disc contents and title strings. If the contents has too many
   // entries for the bounding box the contents is split into several entries.
   //
   for (i=0,j=0; i<numDiscs; ++i,++j)
   {
      d = &mProject->disc(i);
      discTitleText[j] = d->title();

      h = discTitleText[j].isEmpty() ? 0 : hdt;
      if (int(h + d->count()*hc) <= bbh)
      {
         // Contents fits into the bounding box.
         contentsText[j] = d->entries().join("\n");
      }
      else
      {
         // Contens does not fit into the bounding box. Split it into pieces.
         QStringList lst;
         QStringList::const_iterator it;
         const QStringList& ents = d->entries();

         for (it=ents.begin(); it!=ents.end(); ++it)
         {
            h += hc;
            if (h>=bbh-hc && !lst.isEmpty())
            {
               num = contentsText.count()+1;
               contentsText.resize(num);
               discTitleText.resize(num);
               contentsSize.resize(num);

               contentsText[j++] = lst.join("\n");
               lst.clear();
               h = hc;
            }
            lst.append(*it);
         }
         if (!lst.isEmpty())
            contentsText[j] = lst.join("\n");
      }
      contentsText[j].stripWhiteSpace();
   }

   //
   // Calculate geometry of all disc contents and title texts
   //
   num = contentsText.count();
   for (i=0,sumHeight=0; i<num; ++i)
   {
      r = contentsMetrics.boundingRect(0, 0, bbw, bbh, contentsAlign,
                                       contentsText[i]);

      if (!discTitleText[i].isEmpty())
      {
         rt = discTitleMetrics.boundingRect(0, 0, bbw, bbh,
                           discTitleAlign, discTitleText[i]);

	 if (r.width()>rt.width()) sz.setWidth(r.width());
	 else sz.setWidth(rt.width());
	 sz.setHeight(r.height() + rt.height());
      }
      else
      {
         sz = r.size();
      }

      contentsSize[i] = sz;
      sumHeight += sz.height();

      if (sz.width() > maxSize.width()) maxSize.setWidth(sz.width());
      if (sz.height() > maxSize.height()) maxSize.setHeight(sz.height());
   }

   //
   // Draw contents and title texts
   //
   num = contentsText.count();
   for (i=0,x=bb.x(),y=bb.y(),wmax=0; i<num; ++i)
   {
      beginEffect(mProject->contentsEffect());

      if (y+contentsSize[i].height() > bbh)
      {
         x += wmax + scaleX(30);
         y = bb.y();
         wmax = 0;
      }

      w = maxSize.width();

      r.setX(x);
      r.setY(y);
      r.setWidth(w);

      if (w>wmax) wmax = w;

      if (!discTitleText[i].isEmpty())
      {
         r.setHeight(hdt);

         renderText(discTitleText[i], r, discTitleAlign, mProject->discTitleFont(),
                    mProject->contentsStyle().color());
         y += hdt;
         r.setY(y);
         r.setHeight(contentsSize[i].height() - hdt);
      }
      else r.setHeight(contentsSize[i].height());

      if (!contentsText[i].isEmpty())
      {
         renderText(contentsText[i], r, contentsAlign,
                 mProject->contentsStyle().font(),
                 mProject->contentsStyle().color());
      }

      y += contentsSize[i].height();

      endEffect();

      if (mAbortable) qApp->processEvents();
      if (mAbort) return;
   }
}


void Renderer::renderBack(QPaintDevice* aDev, const QWMatrix& aMatrix)
{
   const Image& img = mProject->imgBack();
   const Case& c = mProject->box();
   int lWidth = scaleX(c.backLeftSide());
   int rWidth = scaleX(c.backRightSide());
   QSize back = scale(c.back());
   QString str;

   beginSurface(aDev, aMatrix, back.width()+lWidth+rWidth, back.height());
   renderSurfaceBackground(img.onSides()?0:lWidth, 0, mPixBack);

   // Draw contents
   int indentX = scaleX(40); //in 1/10mm
   int indentY = scaleY(40); //in 1/10mm
   QRect cbb(lWidth+indentX, indentY, back.width()-indentX-indentX,
             back.height()-indentY-indentY);
   renderContents(cbb);
   if (mAbort) return;

   // Draw left and right side texts
   if (!mAbort && (lWidth>0 || rWidth>0) && (!c.hasFrontRightSide() || !c.hasFrontLeftSide()))
      renderSideText(back.width(), back.height(), lWidth, rWidth, false);

   endSurface();

   // Draw outlines
   renderOutlines(lWidth, rWidth);
}


void Renderer::render(QPaintDevice* aDev, bool aAutoFit)
{
   if (mRendering) return;

   mRendering = true;
   mAbort = false;

   doRender(aDev, aAutoFit);

   if (mPaint.isActive()) finish();
   mRendering = false;
}


void Renderer::doRender(QPaintDevice* aDev, bool aAutoFit)
{
   QPaintDeviceMetrics pdm(aDev);
   const Case& c = mProject->box();
   Case::ConnectSide con = c.frontBackConnected();
   int areaWidth, areaHeight;
   int frontWidth = c.front().width() + c.frontLeftSide() + c.frontRightSide();
   int backWidth = c.back().width() + c.backLeftSide() + c.backRightSide();
   int frontHeight = c.front().height();
   int backHeight = c.back().height();
   QPoint frontPos, backPos;
   double zoom = 1.0;

   // Usually we render front and back above each other. For some
   // cases, e.g. dvd boxes, we need to rotate the whole scene to fit on
   // a standard A4 paper
   if (frontHeight+backHeight < frontWidth+backWidth && con==Case::NotConnected)
   {
      mLandscape = false;
      int wmax = frontWidth>backWidth ? frontWidth : backWidth;
      frontPos = QPoint((wmax-frontWidth)>>1, 0);
      backPos = QPoint((wmax-backWidth)>>1, frontHeight+50);
      areaWidth = wmax;
      areaHeight = frontHeight + backHeight + 50;
   }
   else
   {
      // The rendering will be rotated. We continue as if the output
      // medium (paper) is in landscape. The painter's matrix transformation
      // will do the coordinate roation for us.
      mLandscape = true;
      int hmax = frontHeight>backHeight ? frontHeight : backHeight;
      int sep = 0;
      if (con==Case::ConnectLeft)
      {
         backPos = QPoint(0, (hmax-backHeight)>>1);
         frontPos = QPoint(backWidth, (hmax-frontHeight)>>1);
      }
      else if (con==Case::ConnectRight)
      {
         backPos = QPoint(frontWidth, (hmax-backHeight)>>1);
         frontPos = QPoint(0, (hmax-frontHeight)>>1);
      }
      else
      {
         sep = 50;
         backPos = QPoint(0, (hmax-backHeight)>>1);
         frontPos = QPoint(backWidth+sep, (hmax-frontHeight)>>1);
      }
      areaWidth = frontWidth + backWidth + sep;
      areaHeight = hmax;
   }

//    std::cout<<"Resolution in dpi: "<<pdm.logicalDpiX()<<"x"<<pdm.logicalDpiY()<<std::endl;
//    std::cout<<"Drawing area in 1/10mm: "<<areaWidth<<"x"<<areaHeight<<std::endl;

   int drawAreaWidth = pdm.width()-20;
   int drawAreaHeight = pdm.height()-20;

   scaleFactorX = (pdm.logicalDpiX() / 254.0);
   scaleFactorY = (pdm.logicalDpiY() / 254.0);
//    std::cout<<"Scale x="<<scaleFactorX<<" y="<<scaleFactorY<<std::endl;

   if (aAutoFit)
   {
      if (mLandscape)
         zoom = fmin(double(drawAreaWidth)/scaleX(areaHeight), double(drawAreaHeight)/scaleY(areaWidth));
      else zoom = fmin(double(drawAreaWidth)/scaleX(areaWidth), double(drawAreaHeight)/scaleY(areaHeight));
//       std::cout<<"Autofit zoom factor "<<zoom<<std::endl;
   }

   preparePixmaps();

   frontPos = scale(frontPos);
   backPos = scale(backPos);

   if (con==Case::ConnectLeft) frontPos.setX(frontPos.x()-1);
   else if (con==Case::ConnectRight) backPos.setX(backPos.x()-1);

   QWMatrix wm;

   if (mLandscape)
   {
      wm.translate(0, drawAreaHeight);
      wm.rotate(-90);
   }

   if (aAutoFit && fabs(zoom-1.0)>0.001)
      wm.scale(zoom, zoom);

   // Render front
   if (mAbortable) qApp->processEvents();
   if (c.hasFront() && mRenderFront && !mAbort)
   {
      QWMatrix wmf(wm);
      wmf.translate(frontPos.x()+10, frontPos.y()+10);
      renderFront(aDev, wmf);
   }

   // Render back
   if (mAbortable) qApp->processEvents();
   if (c.hasBack() && mRenderBack && !mAbort)
   {
      QWMatrix wmb(wm);
      wmb.translate(backPos.x()+10, backPos.y()+10);
      renderBack(aDev, wmb);
   }

   finish();
}


} //namespace KoverArtist
