/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* disc.h
 */
#ifndef disc_h
#define disc_h

#include <qstring.h>
#include <qstringlist.h>

class QDomDocument;
class QDomElement;


namespace KoverArtist
{

/** A disc description. A @ref KoverArtist::Project can have multiple disc
 * descriptions.
 */
class Disc
{
public:
   Disc();
   virtual ~Disc();

   /** Clear the object. */
   virtual void clear();

   /** Test if the object is empty. */
   virtual bool isEmpty() const;

   /** Returns the title of the disc. */
   QString title() const { return mTitle; }

   /** Set the disc's title to t. */
   void setTitle(const QString& t) { mTitle = t; }

   /** Returns the number of entries the disc has. */
   int count() const { return mEntries.count(); }

   /** Returns the idx-th entry. */
   QString entry(int idx) const { return mEntries[idx]; }

   /** Set the idx-th entry to str. */
   virtual void setEntry(int idx, const QString& str);

   /** Access to the list of entries. */
   const QStringList& entries() const { return mEntries; }
   QStringList& entries() { return mEntries; }

   /** Load the disc from the file fname. */
   virtual bool loadFile(const QString& fname);

   /** Save the disc to the file fname. */
   virtual bool saveFile(const QString& fname) const;

   /** Fill the DOM element de with the object's contents. */
   virtual void toDom(QDomDocument& doc, QDomElement& de) const;

   /** Set the object from the DOM element de. */
   virtual bool fromDom(const QDomElement& de);

protected:
   friend class Project;

   /** Load a KMediaFactory project. */
   virtual bool loadFileKmf(const QString& fname, QDomDocument& doc);

   /** Load a KoverArtist disc file. */
   virtual bool loadFileKoad(const QString& fname, QDomDocument& doc);

   /** Load a plaintext file. */
   virtual bool loadFileText(const QString& fname);

   /** Save a KoverArtist disc file. */
   virtual bool saveFileKoad(const QString& fname) const;

   /** Save a plaintext file. */
   virtual bool saveFileText(const QString& fname) const;

private:
   QString mTitle;
   QStringList mEntries;
};


} //namespace

#endif /*disc_h*/
