/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include "disc.h"
#include "project.h"

#include <qdom.h>
#include <qfile.h>
#include <qtextstream.h>

#include <kmessagebox.h>
#include <klocale.h>

#include "../config.h"



namespace KoverArtist
{

Disc::Disc()
:mTitle()
,mEntries()
{
}


Disc::~Disc()
{
}


void Disc::setEntry(int idx, const QString& str)
{
   mEntries[idx] = str;
}


void Disc::clear()
{
   mTitle = "";
   mEntries.clear();
}


bool Disc::isEmpty() const
{
   return mTitle.isEmpty() && mEntries.isEmpty();
}


void Disc::toDom(QDomDocument& aDoc, QDomElement& aElem) const
{
   QStringList::const_iterator it;
   QDomElement ent;

   QDomElement title = aDoc.createElement("Title");
   title.appendChild(aDoc.createTextNode(mTitle));
   aElem.appendChild(title);

   for (it=mEntries.begin(); it!=mEntries.end(); ++it)
   {
      ent = aDoc.createElement("Entry");
      ent.appendChild(aDoc.createTextNode(*it));
      aElem.appendChild(ent);
   }
}


bool Disc::fromDom(const QDomElement& aDisc)
{
   QDomElement elem;
   QString nm;

   clear();

   for (elem=aDisc.firstChild().toElement(); !elem.isNull();
        elem=elem.nextSibling().toElement())
   {
      nm = elem.tagName();
      if (nm=="Title")
      {
         setTitle(elem.text());
      }
      else if (nm=="Entry")
      {
         mEntries.append(elem.text());
      }
      else
      {
         int rc = KMessageBox::warningContinueCancel(0,
                  i18n("Unknown XML element: %1").arg("Disc/"+nm));
         if (rc!=KMessageBox::Continue) return false;
      }
   }

   return true;
}


bool Disc::loadFile(const QString& aFileName)
{
   QString ext(Project::fileExt(aFileName));

   if (ext=="txt") return loadFileText(aFileName);

   QDomDocument doc;
   QFile in(aFileName);

   if (!in.open(IO_ReadOnly)) return false;
   if (!doc.setContent(&in)) return false;

   QDomDocumentType dt = doc.doctype();
   QString type = dt.name();

   if (type=="kmf_project") return loadFileKmf(aFileName, doc);
   else if (type=="koverartist_disc") return loadFileKoad(aFileName, doc);

   KMessageBox::error(0, i18n("File %1 has unsupported document type %2")
      .arg(aFileName).arg(dt.name()));

   return true;
}


bool Disc::loadFileText(const QString& aFileName)
{
   QFile fin(aFileName);
   QTextStream in(&fin);
   QString line;

   if (!fin.open(IO_ReadOnly)) return false;

   clear();

   while (!in.atEnd())
   {
      line = in.readLine().stripWhiteSpace();
      if (!line.isEmpty()) mEntries.append(line);
   }

   return true;
}


bool Disc::loadFileKoad(const QString& aFileName, QDomDocument& aDoc)
{
   QDomElement root = aDoc.documentElement();
   QString nm, str, type;

   QDomElement elem=root.firstChild().toElement();
   if (elem.tagName()!="Disc")
   {
      KMessageBox::error(0, i18n("%1: expected XML element %2")
         .arg(aFileName).arg("Disc"));
      return false;
   }

   clear();
   return fromDom(elem);
}


bool Disc::loadFileKmf(const QString& aFileName, QDomDocument& aDoc)
{
   QDomElement root = aDoc.documentElement();
   QDomElement elem, elem2;
   QString nm, str, type;

   if (root.tagName()!="project")
   {
      KMessageBox::error(0, i18n("%1: expected XML element %2")
         .arg(aFileName).arg(root.tagName()));
      return false;
   }

   clear();
   mTitle = root.attribute("title");

   for (elem=root.firstChild().toElement(); !elem.isNull();
        elem=elem.nextSibling().toElement())
   {
      nm = elem.tagName();
      if (nm=="media")
      {
         type = elem.attribute("object");
         if (type=="video")
         {
            for (elem2=elem.firstChild().toElement(); !elem2.isNull();
                 elem2=elem2.nextSibling().toElement())
	    {
	       nm = elem2.tagName();
               if (nm=="video")
               {
                  str = elem2.attribute("title");
                  if (!str.isEmpty()) mEntries.append(str);
	       }
	    }
	 }
      }
   }

   return true;
}


bool Disc::saveFile(const QString& aFileName) const
{
   QString ext(Project::fileExt(aFileName));

   if (ext=="txt") return saveFileText(aFileName);
   if (ext=="koad") return saveFileKoad(aFileName);

   KMessageBox::error(0, i18n("Unsupported file format %1").arg(ext));
   return false;
}


bool Disc::saveFileKoad(const QString& aFileName) const
{
   QDomDocument doc("koverartist_disc");
   QDomElement elem;

   // Root
   QDomElement root = doc.createElement("KoverArtistDisc");
   root.setAttribute("creator", PACKAGE);
   root.setAttribute("version", VERSION);
   doc.appendChild(root);

   // The disc
   elem = doc.createElement("Disc");
   toDom(doc, elem);
   root.appendChild(elem);

   // Save the DOM
   QFile out(aFileName);
   if (!out.open(IO_WriteOnly)) return false;

   QCString hdr("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n");
   out.writeBlock(hdr, strlen(hdr));

   QCString domStr = doc.toString().utf8();
   int domStrLen = strlen(domStr);
   return (out.writeBlock(domStr, domStrLen)==domStrLen);
}


bool Disc::saveFileText(const QString& aFileName) const
{
   QStringList::const_iterator it;
   QDomElement ent;

   QFile fout(aFileName);
   if (!fout.open(IO_WriteOnly)) return false;
   QTextStream out(&fout);

   for (it=mEntries.begin(); it!=mEntries.end(); ++it)
      out<<*it<<endl;

   return true;
}



} //namespace
