/*
   Adapted for KoverArtist by Stefan Taferner <taferner@kde.org>
   Original version Copyright (C) 2001-2003 by Adrian Reber <adrian@lisas.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "cdinfo.h"

#include <klocale.h>

#include <errno.h>
#include <fcntl.h>
#include <iostream>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef __FreeBSD__
# include <sys/cdio.h>
# define CDROM_LEADOUT 0xAA
#else
# include <linux/cdrom.h>
#endif



CdInfo::CdInfo(const char *aDevice)
:mFd(-1)
,mErrorCode(0)
,mLength(0)
,mDevice(aDevice)
,mTitle()
,mPerformer()
,mDiscId()
,mUpc()
,mCddbId(0)
,mTracks()
,mErrorText()
{
}


CdInfo::~CdInfo()
{
   close();
}


bool CdInfo::readDisc()
{
   mCddbId = 0;
   mLength = 0;
   mTitle = QString::null;

   if (!isOpen() && !open()) return false;
   if (!readToc()) return false;
   return true;
}


bool CdInfo::open()
{
   mErrorCode = 0;
   if (mFd>=0) return true;

   mFd = ::open(mDevice, O_RDONLY|O_NONBLOCK);
   if (mFd<0)
   {
      mErrorCode = errno;
      if (mErrorCode==EACCES)
         mErrorText = i18n("You do not have permission to read from %1").arg(mDevice);
      else if (mErrorCode==ENOMEDIUM)
         mErrorText = i18n("Please insert a disc into %1 and try again").arg(mDevice);
      else if (mErrorCode==EBUSY)
         mErrorText = i18n("%1 is busy").arg(mDevice);
      else mErrorText = i18n("Cannot open %1: %2").arg(mDevice).arg(strerror(mErrorCode));

      return false;
   }

#ifndef __FreeBSD__
   int st = ioctl(mFd, CDROM_DISC_STATUS);
   if (st!=CDS_AUDIO && st!=CDS_MIXED && st!=CDS_NO_INFO)
   {
      mErrorCode = -100;
      mErrorText = i18n("There is no audio-cd in %1").arg(mDevice);
      return false;
   }
#endif

   return true;
}


void CdInfo::close()
{
   mErrorCode = 0;
   if (mFd>=0)
   {
      ::close(mFd);
      mErrorText = errno;
      mFd = -1;
   }
}


const QString& CdInfo::errorText() const
{
   if (mErrorText.isEmpty())
   {
      if (mErrorCode>=0) mErrorText = strerror(mErrorCode);
      else mErrorText.sprintf("Unknown error %d", mErrorCode);
   }
   return mErrorText;
}


bool CdInfo::readToc()
{
   int track, i, t, ntracks, pos;

   mErrorText = QString::null;
   mErrorCode = 0;

   ntracks = readTocHeader();
   if (ntracks<0)
   {
      mErrorCode = -101;
      mErrorText = i18n("Cannot read audio-cd table of contents: %1").arg(strerror(errno));
      return false;
   }

   mTracks.resize(ntracks+1);

   for (i=0,pos=0; i<=ntracks; ++i)
   {
      if (i==ntracks) track = CDROM_LEADOUT;
      else track = i + 1;

      if (!readTocEntry(track, mTracks[i]))
      {
         mErrorCode = -102;
         mErrorText = i18n("Cannot read audio-cd table of contents: %1").arg(strerror(errno));
         return false;
      }

      if (i<ntracks)
         pos += cddbSum(mTracks[i].min*60 + mTracks[i].sec);
   }

   t = (mTracks[ntracks].min - mTracks[0].min)*60 +
        mTracks[ntracks].sec - mTracks[0].sec;

   mCddbId = (pos % 0xff) << 24 | t << 8 | ntracks;
   mLength = mTracks[ntracks].min*60 + mTracks[ntracks].sec;

   return true;
}


int CdInfo::cddbSum(int n)
{
   int ret = 0;
   while (n>0)
   {
      ret = ret + (n % 10);
      n = n / 10;
   }
   return ret;
}


int CdInfo::readTocHeader()
{
#ifdef __FreeBSD__
   ioc_toc_header hdr;
   if (ioctl(mFd, CDIOREADTOCHEADER, &hdr)<0) return -1;
   return hdr.ending_track;

#else /* not __FreeBSD__*/

   cdrom_tochdr hdr;
   if (ioctl(mFd, CDROMREADTOCHDR, &hdr)<0) return -1;
   return hdr.cdth_trk1;
#endif /*__FreeBSD__*/
}


bool CdInfo::readTocEntry(int aEntryTrack, Track& aTrack)
{
   int frame = 0;

#ifdef __FreeBSD__
   ioc_read_toc_single_entry entry;
   entry.track = aEntryTrack;
   entry.format = CD_MSF_FORMAT;
   if (ioctl(mFd, CDIOREADTOCENTRY, &entry)) return false;
   aTrack.min = entry.entry.addr.msf.minute;
   aTrack.sec = entry.entry.addr.msf.second;
   frame = entry.entry.addr.msf.frame;

#else /* not __FreeBSD__*/

   cdrom_tocentry entry;
   entry.cdte_track = aEntryTrack;
   entry.cdte_format = CDROM_MSF;
   if (ioctl(mFd, CDROMREADTOCENTRY, &entry)) return false;
   aTrack.min = entry.cdte_addr.msf.minute;
   aTrack.sec = entry.cdte_addr.msf.second;
   frame = entry.cdte_addr.msf.frame;
#endif /*__FreeBSD__*/

   aTrack.length = aTrack.min*60 + aTrack.sec;
   aTrack.start = aTrack.length*75 + frame;

   return true;
}


QString CdInfo::cddbQueryString() const
{
   QString result, str;
   int i, num=count();

   result.sprintf("%08lx %d", mCddbId, num);

   for (i=0; i<num; ++i)
   {
      str.sprintf(" %d", mTracks[i].start);
      result += str;
   }

   str.sprintf(" %d", mLength);
   result += str;

   return result;
}




#ifdef BROKEN

//from k3b
bool CdInfo::readCdInfo()
{
   static const int headerMaxLen = 2048;
   unsigned char header[headerMaxLen];
   struct cdrom_generic_command ccmd;
   int dataLen, format=5, track=0;
   bool time = false;

   mErrorText = QString::null;
   mErrorCode = 0;

   if (!isOpen() && !open())
      return false;

   memset(&ccmd, 0, sizeof(ccmd));
   ccmd.cmd[0] = 0x43;
   ccmd.cmd[1] = (time ? 0x2 : 0x0);
   ccmd.cmd[2] = format & 0x0F;
   ccmd.cmd[6] = track;
   ccmd.cmd[8] = 2;       // we only read the length first

   ccmd.buffer = (unsigned char *) header;
   ccmd.buflen = 2;
   ccmd.data_direction = CGC_DATA_READ;

   if (ioctl(mFd, CDROM_SEND_PACKET, &ccmd))
   {
      mErrorCode = errno;
      mErrorText.sprintf("Error accessing cdrom drive: %s", strerror(errno));
      return false;
   }

   dataLen = from2Byte(header) + 2;
   ccmd.cmd[7] = 2048 >> 8;
   ccmd.cmd[8] = 2048;
   ccmd.buflen = 2048;
   ::ioctl(mFd, CDROM_SEND_PACKET, &ccmd);
   dataLen = from2Byte(header) + 2;

   ::memset(header, 0, dataLen);

   ccmd.cmd[7] = dataLen >> 8;
   ccmd.cmd[8] = dataLen;
   ccmd.buffer = (unsigned char *) header;
   ccmd.buflen = dataLen;
   ::ioctl(mFd, CDROM_SEND_PACKET, &ccmd);

   return parseCdInfo(header);
}


#define SIZE 61
//from k3b and xsadp (which has it from cdda2wav)
bool CdInfo::parseCdInfo(unsigned char *buffer)
{
   short posBuffer22;
   char block_no, old_block_no, dbcc, track, code, c;
   int length, j, bufferSize;
   char buffer2[SIZE];
   unsigned char *bufptr, *txtstr;

   bufptr = buffer;
   bufferSize = (buffer[0] << 8) | buffer[1];
   bufferSize -= 2;

   posBuffer22 = 0;
   old_block_no = 0xff;

   for (bufptr=buffer+4; bufferSize>=18; bufptr+=18, bufferSize-=18)
   {
      code = *bufptr;
      if ((code & 0x80) != 0x80) continue;

      block_no = *(bufptr + 3);
      dbcc = block_no & 0x80;
      block_no &= 0x70;

      if (block_no != old_block_no)
      {
         posBuffer22 = 0;
         old_block_no = block_no;
      }

      if (dbcc)
      {
         mErrorCode = -202;
         mErrorText = "Error reading cd-text: Double byte code not supported";
         return false;
      }

      track = *(bufptr + 1);
      if (track & 0x80) continue;

      txtstr = bufptr + 4;

      for (length = 11; length >= 0 && *(txtstr + length) == '\0'; length--)
         ;

      ++length;
      if (length < 12) ++length;

      for (j=0; j<length; ++j)
      {
         c = *(txtstr + j);

         if (c=='\0')
         {
            buffer2[posBuffer22] = c;
            saveCdInfo(code, track, (const char*)buffer2);

            posBuffer22 = 0;
            track++;
         }
         else if (posBuffer22 < (SIZE - 1))
            buffer2[posBuffer22++] = c;
      }
   }

   return true;
}


//inspired by k3b
void CdInfo::saveCdInfo(char code, char track_no, const char *data)
{
   if (track_no == 0)
   {
      if (code == (char)0xFFFFFF80) setTitle(data);
      if (code == (char)0xFFFFFF81) setPerformer(data);
      if (code == (char)0xFFFFFF86) setDiscId(data);
      if (code == (char)0xFFFFFF8E) setUpc(data);
   }
   else
   {
      if (code == (char)0xFFFFFF80) mTracks[track_no].title = data;
      if (code == (char) 0xFFFFFF81) mTracks[track_no].performer = data;
   }
}


//from k3b
unsigned short CdInfo::from2Byte(const unsigned char *d)
{
   return (d[0] << 8 & 0xFF00 | d[1] & 0xFF);
}


#endif /*BROKEN*/
