/***************************************************************************
                                qsworkbook.h
                             -------------------                                         
    begin                :
    copyright            : (C) 2001 by Kamil Dobkowski
    email                : kamildobk@poczta.onet.pl                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#ifndef QSWORKBOOK_H
#define QSWORKBOOK_H

#include <qobject.h>
#include <qstring.h>
#include "qscobject.h"

//---------------------------------------------------------------------------------------------------//

template<class CHILD_OBJECT> class QSChildList;

class QSDrv;
class QPainter;
class QWidget;
/**
  * \brief Single page of QSWorkbook.
  *
  * Contains a list of QSCObjects.
  * Most of its functionality is available through objects() collection. See also QSWorkbook.
  * @author Kamil Dobkowski
  */
class QSPage : public QObject
 {
	Q_OBJECT
  public:
   	/**
	  * Constructor.
	  */
	QSPage( QObject *parent=NULL );
	/**
     	  * Destructor. Deletes all objects on the page.
	  */
	virtual ~QSPage();
	/**
	  * Sets a title of this page. It appears on a tab.
	  * Does not emit sigPageChanged - emits sigTitleChanged instead.
	  */
	void setTitle( const QString& title );
	/**
	  * Returns a title of this page.
	  */
	QString title() const { return m_title; }
	/**
	  * Collection of objcects on this page.
	  */
	QSCObjectCollection *objects() const { return m_objects; }
	/**
	  * Prints the page on the painter. If drawing in background it duplicates painter for each object
	  * using QSDrvQt::copyPainter and each object is drawing itself in the background.
	  */
	void paint( QPainter *p, double dpi, bool blocking=false );

  signals:
	/**
 	  * Page title has changed
	  */
	void sigTitleChanged( const QString& newTitle );
	/**
	  * Page has changed ( the same as collection()::sigChanged ). Does not include sigTitleChanged()
	  * It informs when page needs redrawnig.
	  */
	void sigPageChanged();

  private:
	QSCObjectCollection *m_objects;
	QString m_title;

  private slots:
	void slot_collection_changed();

  };

//---------------------------------------------------------------------------------------------------//

class QPrinter;

/**
  * \brief  Main document class
  *
  * Workbook holds a list of pages. Use QSPlotView to display documents on the screen.
  * @author Kamil Dobkowski
  */
class QSWorkbook : public QObject
 {
  Q_OBJECT
  public:
   /**
     * Constructor.
     */
   QSWorkbook( QObject *parent  );
   /**
     * Destructor. Deletes all pages. Doesn't delete printer()
     */
   virtual ~QSWorkbook();
   /**
     * Returns console for messages, errors, etc.
     */
   //QSConsole *console();
   /**
     * Returns a number of pages in this workbook
     */
   int pageCount() const;
   /**
     * Clears a workbook to the initial state
     */
   virtual void clear();
   /**
     * Print the whole workbook on the given printer.QPrinter - QPainter
     */
   void print( QPainter *painter, QPrinter *printer, double dpi );
   /**
     * Adds a new page.
     */
   void pageAdd( QSPage *object );
   /**
     * Insert page at position 'beforePos'.
     */
   void pageInsert( int position, QSPage *page );
   /**
     * Removes a page but doesn't delete it.
     */
   void pageRemove( int index );
   /**
     * Removes a page ad deletes it
     */
   void pageDelete( int index );
   /**
     * Raises a page.
     */
   void pageRaise( int index );
   /**
     * Lowers a page
     */
   void pageLower( int index );
   /**
     * Brings a page to the front.
     */
   void pageToFront( int index );
   /**
     * Sends a page to the back
     */
   void pageToBack( int index );
   /**
     * Move page 'index' to position 'position'.
     */
   void pageReorder( int position, int index );
   /**
     * Returns an index of a page or -1 if it wasn't found.
     */
   int pageFind( QSPage *page ) const;
   /**
     * Find page
     */
   QSPage *pageFind( QSCObject *object ) const;
   /**
     * Returns a page.
     */
   QSPage *page( int index ) const;
       /**
         * Set a printer. Page size and margins are used to
         * display full page preview. The printer must be
         * set each time its settings changes. Emits sigWorkbookChanged.
	 * printer is not deleted - the pointer still belongs to caller.
         */
   void setPrinter( QPrinter *printer );
       /**
         * Returns a current printer
         */
   QPrinter *printer() const { return m_printer; }

  signals:
    /**
      * Page added
      */
   void sigPageAdded( QSPage *o );
    /**
      * Page removed
      */
   void sigPageRemoved( QSPage *o );
    /**
      * Order of pages changed.
      */
   void sigPageOrder();
    /**
      * Object added ( to any page )
      */
   void sigObjectAdded( QSCObject *o );
    /**
      * Object removed  ( from any page )
      */
   void sigObjectRemoved( QSCObject *o );
    /**
      * Order of objects changed. ( on any page )
      */
   void sigObjectOrder();
   /**
     * Page added/removed/reordered, page title has changed.
     * Page tab bar needs refreshing.
     */
   void sigPageListChanged();
   /**
     * Printer has changed.
     */
   void sigPrinterChanged();
   /**
     * Page added/removed/reordered, page changed, page title changed, a new printer is set.
     * Workbook is 'dirty' and needs to be saved when exiting.
     */
   void sigWorkbookChanged();

  private slots:
   void slot_object_added(QSCObject*);
   void slot_object_removed(QSCObject*);
   void slot_object_order();
   void slot_page_changed();
   void slot_page_title_changed( const QString& );

  private:
   QPrinter *m_printer;
   QSChildList<QSPage> *m_page_list;
 };

/**
  * \mainpage
  *
  * \section intro Introduction
  *
  * The main object in this library is QSWorkbook, which represents the document. Workbook can be displayed
  * on the screen using a provided view named QSPlotView. Workbook holds a list of QSPage's and each page contains a list
  * of QSCObject's which are placed on it, QSCGroup is an object which can contain other objects, but behaves as a single
  * one. You can display graphs using QSCAxesShadow object. It is a simple wrapper around QSAxes2D and QSAxes3D - they don't
  * inherit QSCObject directly because of problems with multiple inheritance. These objects don't draw much by themselves.
  * They serve as containers for datasets and axes. Datasets inherit QSPlot class. These are QSCurve, QSImage, QSContour, QSSurface, QSFigure.
  * QSAxis class represents the single X, Y, V or Z axis. Axes and datasets inherit QSData interface. This means that they contain
  * data organised as a list of matrices. For example QSCurve accepts four matrices: XVector, YVector, DXVector, DYVector and
  * draws successive points ( x, y, dx, dy ) as a curve with dx and dy used for error bars. Your data is not assumed to be in some
  * peculiar format - you will only have to implement an abstract QSMatrix interface. It is very simple, you may look at providen examples.
  * You can repaint objects not only using a Qt's painter but also using a driver of your own, see QSDrv. This library uses its
  * own graphic attributes such as QSGFill, QSGFont, QSGLine, QSGPoint, QSGArrow, QSGGradient, but you can easily convert them to/from
  * coresponding Qt attributes using funtions from QSDrvQt class. See the inheritace tree under QSSerializable, it is a good point to
  * start.
  */
#endif

