/***************************************************************************
                                 qscontour.cpp
                             -------------------                                         
    begin                : 01-January-2000
    copyright            : (C) 2000 by Kamil Dobkowski                         
    email                : kamildobk@poczta.onet.pl                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include "qscontour.h"
#include "qsconsole.h"
#include <math.h>
#include <list.h>
#include <algo.h>
#include <vector.h>

const double QSContour::EPS_VALUE = 1e-100;

/**
  * Possible problems
  */
struct QSContour::contour_levels_data {
     int number;
     QSGFill *fill;
     QSGLine *line;
     QSGFont *font;
     int     *angle;
     double  *value;
     double  *zvalue;
     QString *label;
     list<QSPt2f> *labels_pos;
    };

#define MAX_BUFF_SIZE	10

struct QSContour::contour_runtime_data {
     enum Stage { Prepare = 0,
		  DrawingFills  = 1,
     		  DrawingGrid	= 2,
     		  DrawingLines  = 3,
     		  DrawingLabels = 4,
     		  DrawingPoints = 5,
     		  DrawingEnds   = 6 } stage;
     int  pi, pj;
     int  curr_level;
     int  triangles;
     QSAxis *xaxis;
     QSAxis *yaxis;
     QSAxis *vaxis;
     bool rectangle_grid;
     contour_levels_data level;
     int stack_ptr;
     int stack_triangle[3];
     int stack_edge[3];
     bool stack_first[3];
     bool label_drawn;

     QSPt3f buff1[MAX_BUFF_SIZE];
     QSPt3f buff2[MAX_BUFF_SIZE];
     QSPt3f buff3[MAX_BUFF_SIZE];
     QSPt2f cbuff1[MAX_BUFF_SIZE];

     QSPt2f rbuff[4];
     bool rbuff_is_empty;
     int rbuff_curr_level;

     list<QSPt2f> curr_line;
     double curr_line_len;
     QSPt2f curr_pos;

     char *processed_triangles;
     inline void set_triangle_processed( int triangle_number );
     inline bool triangle_processed( int triangle_number );
     inline void stack_push( int triangle, int edge, bool is_first );
     inline void stack_pop( int *triangle, int *edge, bool *is_first );
     };



//-------------------------------------------------------------//

bool QSContour::contour_runtime_data::triangle_processed( int triangle_number )
// 0x01 & ( d->processed_triangles[triangle_number/8] >> (triangle_number%8) )
 {
  return 0x01 & ( processed_triangles[triangle_number>>3] >> (triangle_number&0x07) );
 }

//-------------------------------------------------------------//

void QSContour::contour_runtime_data::set_triangle_processed( int triangle_number )
 {
  processed_triangles[triangle_number>>3] |= 1U<<(triangle_number&0x07);
 }

//-------------------------------------------------------------//

void QSContour::contour_runtime_data::stack_push( int triangle, int edge, bool first )
 {
  assert( stack_ptr < 2 && stack_ptr >= 0 );
  stack_triangle[stack_ptr] = triangle;
  stack_edge[stack_ptr] = edge;			
  stack_first[stack_ptr++] = first;	 	
 }

//-------------------------------------------------------------//

void QSContour::contour_runtime_data::stack_pop( int *triangle, int *edge, bool *first )
 {
  assert( stack_ptr < 3 && stack_ptr > 0 );
  *triangle = stack_triangle[--stack_ptr];
  *edge = stack_edge[stack_ptr];
  *first = stack_first[stack_ptr];
 }
	 	
//-------------------------------------------------------------//
//-------------------------------------------------------------//

QSContour::QSContour(QSAxes* parent, const char * name)
:QSPlot2D(parent,name)
 {
  assert( parent );
  m_label_spacing = 0.2;
  m_labels  = true;
  m_fills   = true;
  m_lines   = true;
  m_title_str = tr("Untitled contour");

  #define FILLS_NUM	0
  #define FONTS_NUM	0
  #define LINES_NUM	1
  #define POINTS_NUM	1

  initAttributeTables( FONTS_NUM, FILLS_NUM, LINES_NUM, POINTS_NUM );
  m_settings.lines[Grid] = QSGLine::invisibleLine;
 }

//-------------------------------------------------------------//

QSContour::~QSContour()
 {
 }

//-------------------------------------------------------------//

void QSContour::setContourFills( bool visible )
 {
  SET_PROPERTY( m_fills, visible );
 }

//-------------------------------------------------------------//

void QSContour::setContourLines( bool visible )
 {
  SET_PROPERTY( m_lines, visible );
 }

//-------------------------------------------------------------//

void QSContour::setContourLabels( bool visible )
 {
  SET_PROPERTY( m_labels, visible );
 }

//-------------------------------------------------------------//

void QSContour::setLabelSpacing( double value )
 {
  if ( value > 0.0 )
  SET_PROPERTY( m_label_spacing, value );
 }

//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//

void QSContour::allocRuntimeData()
 {
  QSPlot2D::allocRuntimeData();
  d = new contour_runtime_data();

  d->pi    = 0;
  d->curr_level = 0;
  d->xaxis = defaultAxis(QSAxis::XAxisType);
  d->yaxis = defaultAxis(QSAxis::YAxisType);
  d->vaxis = defaultAxis(QSAxis::VAxisType);
  d->rectangle_grid = rectangleGrid();

  d->level.number = 0;
  d->level.fill   = NULL;
  d->level.font   = NULL;
  d->level.angle  = NULL;
  d->level.line   = NULL;
  d->level.value  = NULL;
  d->level.zvalue = NULL;
  d->level.label  = NULL;
  d->level.labels_pos = NULL;

  d->stack_ptr = 0;
  d->processed_triangles = NULL;

  d->rbuff_is_empty = true;
  d->rbuff_curr_level = -1;
 }



//-------------------------------------------------------------//

void QSContour::freeRuntimeData()
 {
  delete[] d->processed_triangles;
  free_levels_data( &d->level );
  delete d; d = NULL;
  QSPlot2D::freeRuntimeData();
 }



//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//

void QSContour::calculate_levels_data( contour_levels_data *level )
// each level contains also a fill style used to fill areas below this level.
// przerobic legendItemSize
 {
  QSAxis *axis = defaultAxis( QSAxis::VAxisType );
  const list<QSAxisTic> *tics = axis->tics();

  level->number = tics->size();
  level->value  = new  double[level->number+2];
  level->zvalue = new  double[level->number+2];
  level->fill   = new QSGFill[level->number+2];
  level->line   = new QSGLine[level->number+2];
  level->font   = new QSGFont[level->number+2];
  level->angle  = new int[level->number+2];
  level->label  = new QString[level->number+2];
  level->labels_pos = new list<QSPt2f>[level->number+2];

  // first level at 0, all data below this level is drawn as transparent
  level->line[0]   = QSGLine::invisibleLine;
  level->fill[0]   = QSGFill::Transparent;
  level->value[0]  = 0.0;
  level->label[0]  = QString::null;
  level->zvalue[0] = axis->rangeMin();
  level->number = level->number+1;	

  // all tics from axis
  list<QSAxisTic>::const_iterator curr_tic = tics->begin();
  for( int i=1; i<level->number; i++ ) {	
        level->line[i]   = curr_tic->m_line;
        level->font[i]   = curr_tic->m_font;
        level->angle[i]  = curr_tic->m_angle;	
  	level->value[i]  = curr_tic->m_pos;
  	level->label[i]  = curr_tic->m_label;
  	level->zvalue[i] = curr_tic->m_value;
	level->fill[i]   = curr_tic->m_fill;	
	// put gradient to axis
        if ( !curr_tic->m_is_fill_defined ) m_gradient.fill( level->value[i], level->fill[i] );
  	curr_tic++;
  	}

  // the last level
  if ( level->value[level->number-1] < 1.0 ) {
	level->line[level->number]   = QSGLine::invisibleLine;
  	level->value[level->number]  = 1.0;
  	level->label[level->number]  = QString::null;;
  	level->zvalue[level->number] = axis->rangeMax(); 	
  	if ( !axis->reversed() && axis->lastTic().m_is_fill_defined )
		level->fill[level->number] = axis->lastTic().m_fill;
		else m_gradient.fill( 1.0, level->fill[level->number] );
	level->number = level->number+1;
	}
 }

//-------------------------------------------------------------//

void QSContour::free_levels_data( contour_levels_data *level )
 {
  delete[] level->fill;
  delete[] level->font;
  delete[] level->angle;
  delete[] level->line;
  delete[] level->value;
  delete[] level->zvalue;
  delete[] level->label;
  delete[] level->labels_pos;
  level->fill       = NULL;
  level->font       = NULL;
  level->angle      = NULL;
  level->line       = NULL;
  level->value      = NULL;
  level->zvalue     = NULL;
  level->label      = NULL;
  level->labels_pos = NULL;
 }

//-------------------------------------------------------------//

bool QSContour::start()
 {
  QSPlot2D::start();
  d->stage = contour_runtime_data::Prepare;
  return true;
 }

//-------------------------------------------------------------//

bool QSContour::step()
 {
  switch( d->stage ) {
   	case contour_runtime_data::Prepare:		call_prepare();  break;
  	case contour_runtime_data::DrawingFills:	drawing_fills();  break;
  	case contour_runtime_data::DrawingGrid: 	drawing_grid();   break;
  	case contour_runtime_data::DrawingLines:	drawing_lines();  break;
  	case contour_runtime_data::DrawingLabels:	drawing_labels(); break;
  	case contour_runtime_data::DrawingPoints:	drawing_points(); break;
  	default: return false;
  	}
  return true;
 }

//-------------------------------------------------------------//

void QSContour::call_prepare()
  {
   if ( prepare() ) {
	d->pi = 0;
	d->triangles = triangles();
	calculate_levels_data( &d->level );
	start_drawing_fills();
	}
  }

//-------------------------------------------------------------//

void QSContour::start_drawing_fills()
 {
  m_drv->setFill( QSGFill::transparentFill );
  m_drv->setLine( QSGLine::invisibleLine );
  d->stage = contour_runtime_data::DrawingFills;
  if ( !m_fills ) start_drawing_grid();
 }

//-------------------------------------------------------------//

void QSContour::drawing_fills()
 {
  QSPt3f min;  // bounding cube
  QSPt3f max;

  QSPt3f *inpts; // rotating buffers  - input points
  QSPt3f *abpts; // above points
  QSPt3f *unpts; // under points
  int ninpts;
  int nabpts;
  int nunpts;

  //QSGLine line;
  QSPt2f buff[MAX_BUFF_SIZE];

  int curr_step = 0;
  while( d->pi < d->triangles ) {
	// try to process the whole rectangle ( for faster redrawing )
	if ( (d->pi%4)==0 && d->rectangle_grid ) {
		getRectangle( d->pi, d->buff1 );
		if ( draw_rectangle( d->buff1 ) ) { d->pi = d->pi + 4; continue; }
		}
	
	flush_rectangle_buffer();
		
	// get triangle
	getTriangle( d->pi++, d->buff1 );
		
	// bounding cube
	min.x = d->buff1[0].x; min.x = QMIN(min.x,d->buff1[1].x); min.x = QMIN(min.x,d->buff1[2].x);
	max.x = d->buff1[0].x; max.x = QMAX(max.x,d->buff1[1].x); max.x = QMAX(max.x,d->buff1[2].x);
	min.y = d->buff1[0].y; min.y = QMIN(min.y,d->buff1[1].y); min.y = QMIN(min.y,d->buff1[2].y);
	max.y = d->buff1[0].y; max.y = QMAX(max.y,d->buff1[1].y); max.y = QMAX(max.y,d->buff1[2].y);
	min.z = d->buff1[0].z; min.z = QMIN(min.z,d->buff1[1].z); min.z = QMIN(min.z,d->buff1[2].z);
	max.z = d->buff1[0].z; max.z = QMAX(max.z,d->buff1[1].z); max.z = QMAX(max.z,d->buff1[2].z);
	
	// check if is out of xy area
	if ( !(min.x>1.0 || max.x<0.0 || min.y>1.0 || max.y<0.0) ) {
	   inpts  = d->buff1;
	   ninpts = 3;
	   abpts  = d->buff2;
           unpts  = d->buff3;
           // draw levels
	   int level_nr = find_level_greater_than( min.z );
	   while( level_nr<d->level.number && ninpts>0 ) {
		// divide polygon to: under part ( under contour level ) and upper part ( over contour level )
		cut_polygon( d->level.value[level_nr], inpts, ninpts, abpts, &nabpts, unpts, &nunpts );
		// draw polygon under
		if ( d->level.fill[level_nr].style != QSGFill::Transparent && nunpts > 1 ) {
			for( int j=0; j<nunpts; j++ ) { buff[j].set( unpts[j].x, unpts[j].y ); }
			m_drv->setFill( d->level.fill[level_nr] );
			m_drv->drawPoly2( buff, nunpts );
			}
		// rotate buffers
		QSPt3f *temp = inpts; inpts = abpts; ninpts = nabpts; abpts = temp;
		// next level
		level_nr++;
		}
	    }

  	// end with this little part ...
        if ( curr_step++ > work_steps && m_bkg_handler ) return;
  	}

  // end drawing
  flush_rectangle_buffer();
  start_drawing_grid();
 }

//-------------------------------------------------------------//

inline int QSContour::find_level_greater_than( double value )
// find first level > value or return level.number if all levels are lower then value
 {
  double *first_level      = &d->level.value[0];
  double *after_last_level = &d->level.value[d->level.number];
  double *ptr = lower_bound( first_level, after_last_level, value );
  int result = ptr < after_last_level ? ptr-first_level : d->level.number;
  return result;
 }

//-------------------------------------------------------------//
/*
pts, rbuff layout

0 _________ 1
 |\      / |
 |  \  /   |
 |    \    |     -> drawing direction
 |  /   \  |
 |/_______\|
3           2

*/
bool QSContour::draw_rectangle( const QSPt3f pts[4] )
// tries to draw rectangle if it is fully contained in some level ( single color )
// or returns false otherwise.
// This function does not draw rectangles immediately on the screen but rather
// tries to collect them into the longer horizontal strips.
 {
  double min_z = pts[0].z; min_z = QMIN(min_z,pts[1].z); min_z = QMIN(min_z,pts[2].z); min_z = QMIN(min_z,pts[3].z);
  double max_z = pts[0].z; max_z = QMAX(max_z,pts[1].z); max_z = QMAX(max_z,pts[2].z); max_z = QMAX(max_z,pts[3].z);
  int level_nr = find_level_greater_than( max_z );
  // check if it is fully contained in some level ( drawn in single color )
  if ( level_nr == 0 || d->level.value[level_nr-1] <= min_z ) {
	// try to join the new rectangle to the longer strip
	if ( !d->rbuff_is_empty &&
	      d->rbuff_curr_level == level_nr &&
	      d->rbuff[1].y == pts[1].y &&
	      d->rbuff[2].y == pts[2].y ) {
		d->rbuff[1].x = pts[1].x;
		d->rbuff[2].x = pts[2].x;
		} else {
	// flush rectangle buffer to the screen, and set it to the new rectangle
		flush_rectangle_buffer();
		d->rbuff[0].x = pts[0].x; d->rbuff[0].y = pts[0].y;
		d->rbuff[1].x = pts[1].x; d->rbuff[1].y = pts[1].y;
		d->rbuff[2].x = pts[2].x; d->rbuff[2].y = pts[2].y;
		d->rbuff[3].x = pts[3].x; d->rbuff[3].y = pts[3].y;
		d->rbuff_is_empty = false;
		d->rbuff_curr_level = level_nr;
		}
	return true;
	}
  return false;
 }

 //-------------------------------------------------------------//

void QSContour::flush_rectangle_buffer()
// draws the strip of rectangles collected by draw rectangle
 {
  if ( !d->rbuff_is_empty && d->rbuff_curr_level < d->level.number ) {
	m_drv->setFill( d->level.fill[d->rbuff_curr_level] );
	m_drv->drawPoly2( d->rbuff, 4 );
	d->rbuff_is_empty = true;	
	}
 }

//-------------------------------------------------------------//

void QSContour::start_drawing_grid()
 {
  d->stage = contour_runtime_data::DrawingGrid;
 }

//-------------------------------------------------------------//

void QSContour::drawing_grid()
 {
  drawGrid();  	
  start_drawing_lines();
 }

//-------------------------------------------------------------//

void QSContour::start_drawing_lines()
 {
  d->pi = 0;
  d->stack_ptr = 0;	
  d->curr_level = 0;	
  if ( m_lines ) {
  	 init_triangle_buffer();  	
  	 d->stage = contour_runtime_data::DrawingLines;
  	} else {
  	 start_drawing_points();
  	}
 }

//-------------------------------------------------------------//

void QSContour::drawing_lines()
 {
  int curr_step = 0;
  while( d->curr_level < d->level.number ) {
     double level = d->level.value[d->curr_level];

     if (  d->level.line[d->curr_level].style == QSGLine::Invisible &&
	   d->level.label[d->curr_level] == QString::null ) {
	d->curr_level++;
     	continue;
        }

     while( d->pi < d->triangles ) {

	// tracing contour line when stack isn't empty
  	while( d->stack_ptr > 0 ) {
  		int t, e; bool is_first; d->stack_pop( &t, &e, &is_first );
  		
  		// if this triangle was processed make its neighboutr the current triangle
  		// if there is no neighbour or the neighbour is already processed - end with this
  		// tracing ( Wow ! Look at those operators ! )
  		if ( d->triangle_processed(t) )
  			if ( !getNeighbouringTriangle(&t,&e) || d->triangle_processed(t) )  { end_contour(d->curr_level); continue; }
  		
  		getTriangle( t, d->buff1, level );
    		if ( is_first ) start_contour( d->buff1[e], d->buff1[(e+1)%3], d->curr_level );
  			
   		// find cross point on other edges
  		if ( e != 0 && (d->buff1[0].z<level) != (d->buff1[1].z<level) ) { d->stack_push(t,0,false ); draw_contour(d->buff1[0],d->buff1[1],d->curr_level); }
  		else
  		if ( e != 1 && (d->buff1[1].z<level) != (d->buff1[2].z<level) ) { d->stack_push(t,1,false ); draw_contour(d->buff1[1],d->buff1[2],d->curr_level);  }
  		else
  		if ( e != 2 && (d->buff1[2].z<level) != (d->buff1[0].z<level) ) { d->stack_push(t,2,false ); draw_contour(d->buff1[2],d->buff1[0],d->curr_level);  }
  	
  		d->set_triangle_processed(t);
  		
  		// put assert( d->stack_ptr != 0 ) rather
  		if ( d->stack_ptr == 0 ) end_contour(d->curr_level);
  		
		// end with this little part ...
        	if ( curr_step++ > work_steps && m_bkg_handler ) return;
  		}
  		 	
	// search for triangles where we can start a isocurve tracing algorithm
  	if ( !d->triangle_processed(d->pi) ) {

		// try to process the whole rectangle ( for faster redrawing )
		if ( (d->pi%4)==0 && d->rectangle_grid ) {
			getRectangle( d->pi, d->buff1 );
			double min_z = d->buff1[0].z; min_z = QMIN(min_z,d->buff1[1].z); min_z = QMIN(min_z,d->buff1[2].z); min_z = QMIN(min_z,d->buff1[3].z);
			double max_z = d->buff1[0].z; max_z = QMAX(max_z,d->buff1[1].z); max_z = QMAX(max_z,d->buff1[2].z); max_z = QMAX(max_z,d->buff1[3].z);
			if ( (min_z<level) == (max_z<level) ) { d->pi = d->pi + 4; continue; }
			}		  	

  		// check if the current triangle crosses the current level
  		// put on the stack the first found point ( its triangle and edge )
  		// put it two times ( it will be traced in both directions )
  		getTriangle( d->pi, d->buff1, level );
  		if ( (d->buff1[0].z<level) != (d->buff1[1].z<level) ) { d->stack_push(d->pi,0,true); d->stack_push(d->pi,0,true); }
  		else
  		if ( (d->buff1[1].z<level) != (d->buff1[2].z<level) ) { d->stack_push(d->pi,1,true); d->stack_push(d->pi,1,true); }
  		else
  		if ( (d->buff1[2].z<level) != (d->buff1[0].z<level) ) { d->stack_push(d->pi,2,true); d->stack_push(d->pi,2,true); }	
  		}
  		 		
  	d->pi++;
  	// end with this little part ...
        if ( curr_step++ > work_steps && m_bkg_handler ) return;
  	}
		
     init_triangle_buffer();
     d->curr_level++; d->pi= 0;
    }

  start_drawing_labels();
 }

//-------------------------------------------------------------//

void QSContour::start_drawing_labels()
 {
  d->curr_level = 0;
  d->stage = contour_runtime_data::DrawingLabels;
 }

//-------------------------------------------------------------//

void QSContour::drawing_labels()
 {
  while( d->curr_level < d->level.number ) {
	 m_drv->setFont( d->level.font[d->curr_level] );
  	 list<QSPt2f>::iterator first = d->level.labels_pos[d->curr_level].begin();
  	 list<QSPt2f>::iterator last  = d->level.labels_pos[d->curr_level].end();
  	 list<QSPt2f>::iterator curr  = first;
	
	 while( curr != last ) m_drv->drawRText2( *curr++, d->level.angle[d->curr_level], d->level.label[d->curr_level], AlignVCenter | AlignHCenter );
	 	
         d->level.labels_pos[d->curr_level].erase( first, last );
         d->curr_level++;
  	}
  	
  start_drawing_points();
 }

//-------------------------------------------------------------//

void QSContour::start_drawing_points()
 {
  d->stage = contour_runtime_data::DrawingPoints;
 }

//-------------------------------------------------------------//

void QSContour::drawing_points()
 {
  drawPoints();
  d->stage = contour_runtime_data::DrawingEnds;
 }

//-------------------------------------------------------------//

void QSContour::end()
 {
  // all clean-up work is done in freeRuntimeData()
  QSPlot2D::end();
 }

//-------------------------------------------------------------//

void QSContour::init_triangle_buffer()
 {
  int size = d->triangles/8+7;
  if ( !d->processed_triangles ) d->processed_triangles = new char[size];
  char *ptr = d->processed_triangles;
  for( int i=0; i<size; i++ ) *ptr++ = 0;
 }

//-------------------------------------------------------------//

void QSContour::start_contour( const QSPt3f& p1, const QSPt3f& p2, int level )
// start contour 'level' one the edge from p1 to p2
 {
  // calculate the position where the contour plane crosses the edge
  double t = (p2.z-p1.z) ? (d->level.value[level]-p1.z)/(p2.z-p1.z) : 0.0;
  QSPt2f w = QSPt2f(p1.x+t*(p2.x-p1.x),p1.y+t*(p2.y-p1.y));

  d->label_drawn = false;
  // clean the buffer
  d->curr_pos = w;
  d->curr_line_len = 0.0;
  d->curr_line.erase( d->curr_line.begin(), d->curr_line.end() );

  m_drv->setFont( d->level.font[level] );
  m_drv->setLine( d->level.line[level] );
  m_drv->beginPolyline2( w );
 }

//-------------------------------------------------------------//

#define GRID_SPACING 0.8

void QSContour::draw_contour( const QSPt3f& p1, const QSPt3f& p2, int level )
// draw conotur line to the edge (p1,p1)
// accumulate lines in buffer instead of direct drawing. When line in
// buffer has a apriopriate length, flush the buffer using draw_line_buffer
 {
  // calculate position where the level plane crosses the edge
  double t = (p2.z-p1.z) ? (d->level.value[level]-p1.z)/(p2.z-p1.z) : 0.0;
  QSPt2f new_pos = QSPt2f( p1.x + t*(p2.x-p1.x), p1.y + t*(p2.y-p1.y) );

  // don't draw, write the point to the buffer
  d->curr_line.push_back( new_pos );

  // calculate the length of a line in the buffer
  d->curr_line_len += distance( new_pos, d->curr_pos );
  d->curr_pos = new_pos;

  // if the line has an apriopriate length draw it on the screen
  if ( d->curr_line_len > m_label_spacing*2.0 ) { draw_line_buffer( level ); d->curr_line_len = 0.0; }
 }

//-------------------------------------------------------------//

void QSContour::end_contour( int level )
 {
  draw_line_buffer( level );
  m_drv->endPolyline2();
 }

//-------------------------------------------------------------//

void QSContour::draw_line_buffer( int level )
 {
  list<QSPt2f>::iterator first = d->curr_line.begin();
  list<QSPt2f>::iterator last  = d->curr_line.end();
  list<QSPt2f>::iterator curr  = first;

  // calculate the position of the label - in the middle of this segment
  if ( m_labels && !d->label_drawn && first != last && !d->level.label[level].isEmpty() && d->curr_line_len > m_label_spacing ) {
    // search for the middle
    QSPt2f prev = *curr;
    double curr_len = 0;
    while( curr != last ) {
  	curr_len += distance( *curr, prev );
  	if ( curr_len > m_label_spacing*(0.5*(level%3)+0.5) ) {
		// calculate the position of the curve label
    		QSPt2f lpos = QSPt2f( prev.x+(curr->x-prev.x)/2, prev.y+(curr->y-prev.y)/2 );
    		m_drv->setPolylineLabelPlace2( d->level.label[level], lpos, d->level.angle[level] );
    		d->level.labels_pos[level].push_back( lpos );
		d->label_drawn = true;
		break;
		}
  	prev = *curr; curr++;
  	}	
    } else {
	// turns off labels
     	m_drv->setPolylineLabelPlace2( QString::null, QSPt2f(), 0 );
    }

  // draw contour line ( leave the place for the label )
  curr = first;
  while( curr != last ) m_drv->drawPolylineTo2( *curr++ );

  d->curr_line.erase( first, last );	
 }

//-------------------------------------------------------------//


void QSContour::cut_polygon( double level, const QSPt3f *pts, int npts,
					   QSPt3f *above_pts, int *above_npts,
					   QSPt3f *under_pts, int *under_npts,
					   QSPt3f *cross_line, int *cross_npts )
 {
  bool p_under;
  bool s_under;
  *above_npts = 0;
  *under_npts = 0;
  if ( cross_npts ) *cross_npts = 0;

  const QSPt3f *p;
  const QSPt3f *s = &pts[npts-1]; s_under = ( s->z <= level );
  for( int i=0; i<npts; i++ ) {
  	p = &pts[i]; p_under = ( p->z <= level );
  	
  	if (  p_under &&  s_under ) under_pts[(*under_npts)++] = *p;
  	else
  	if ( !p_under && !s_under ) above_pts[(*above_npts)++] = *p;
  	else {
  	      double t = (level-s->z)/(p->z-s->z);
  	      QSPt3f c( s->x + t*(p->x-s->x),
  	      		s->y + t*(p->y-s->y),
  	      		level );
  	      under_pts[(*under_npts)++] = c;
  	      above_pts[(*above_npts)++] = c;
    	      if ( cross_line && cross_npts ) cross_line[(*cross_npts)++]=c;
  	
  	      if (  p_under ) under_pts[(*under_npts)++] = *p;
  	      if ( !p_under ) above_pts[(*above_npts)++] = *p;
             }

        s = p; s_under = p_under;
        }
 }


//-------------------------------------------------------------//

QString QSContour::posInfo( QSPt2f& pos )
 {
  QString result = QString::null;
  if ( m_busy ) return result;

  //QSPt2f p = m_axes->fcanvasToWorld( QSPt2f(pos.x,pos.y) );
  QSRectf clip_area( pos.x-2.0, pos.y-2.0, 5.0, 5.0 );

  allocRuntimeData();
  calculate_levels_data( &d->level );
  d->triangles = triangles();

  int triangle = triangleAtPos( pos );

  if ( triangle >= 0 ) {
  	 QSPt3f pts[3];
 	 QSPt2f ppts[5];
 	 for( int i=0; i<d->level.number; i++ ) {
		getTriangle( triangle, pts, d->level.value[i] );
		int nabpts = 0;
		int nunpts = 0;
		int ncpts  = 0;
		
		cut_polygon( d->level.value[i], pts, 3, d->buff1, &nabpts, d->buff2, &nunpts, d->buff3, &ncpts );
		
		// hit at contour filled area
		if ( result.isEmpty() && m_fills && d->level.fill[i].style != QSGFill::Transparent && nunpts > 0 ) {
			 for( int j=0; j<nunpts; j++ ) ppts[j] = m_proj->world2DToCanvas( QSPt2f(d->buff2[j].x,d->buff2[j].y) );
			 if ( QSProjection::pointInPoly(pos,ppts,nunpts) ) {
			 	 result  = QString(tr(" Level "))+QString::number(i)+"\n";
			 	 result += QString(tr(" Level max z-value = "))+QString::number(d->level.zvalue[i])+" \n";
			 	 if ( i>0 ) result += QString(tr(" Level min z-value = "))+QString::number(d->level.zvalue[i-1])+" \n";
			 	 result += "\n";
			 	}
			}
			
		// hit exactly at contour line	
		if ( m_lines && d->level.line[i].style != QSGLine::Invisible && ncpts>1 ) {
			 d->cbuff1[0] = m_proj->world2DToCanvas( QSPt2f(d->buff3[0].x,d->buff3[0].y) );
			 d->cbuff1[1] = m_proj->world2DToCanvas( QSPt2f(d->buff3[1].x,d->buff3[1].y) );
			 if ( m_proj->clipLine(&d->cbuff1[0],&d->cbuff1[1],clip_area.pos,clip_area.size) ) {
			 	result  = QString(tr(" Level "))+QString::number(i)+"\n";
			 	result += QString(tr(" Level z-value = "))+QString::number(d->level.zvalue[i])+" \n";
			 	result += "\n";
			 	}
			}
		
		// hit exacty at triangle vertex
		if ( i == 0 )	
		if ( point(PointMark).style != QSGPoint::Invisible || line(Grid).style != QSGLine::Invisible  )
		for( int j=0; j<3; j++ )
		   if ( clip_area.contains( m_proj->world2DToCanvas( QSPt2f(pts[j].x,pts[j].y) ) ) )
			result += vertexInfo(triangle,j,pos);
		}	
  	}
  	
  freeRuntimeData();
  return result;
 }



//-------------------------------------------------------------//

bool QSContour::isClicked( const QSPt2f& pos )
 {
  QSPt2f p = pos;
  if ( posInfo(p) != QString::null ) return true;
  return false;
 }

//-------------------------------------------------------------//

#define BOX_SPACE  2.0
#define BOX_WIDTH  20.0
#define BOX_HEIGHT 100.0

QSPt2f QSContour::legendItemSize( QSDrv *drv )
 {
  double boxSpace = drv->toPixels(BOX_SPACE);
  QSPt2f boxSize( drv->toPixels(BOX_WIDTH),
  		  drv->toPixels(BOX_HEIGHT) );
  QSPt2f tsize = drv->rTextSize( 270, title() );
  QSPt2f lsize;
  QString label;
  int nr_labels = 0;
//  int axis = defaultAxis(QSAxes::VAxisType);
  contour_levels_data level;
  calculate_levels_data( &level );
  for ( int i=0; i<level.number; i++ ) {
	 if ( !level.label[i].isEmpty() ) {
	 	QSPt2f size = drv->textSize( level.label[i] );
	 	lsize.x = QMAX(lsize.x, size.x);
	 	lsize.y = QMAX(lsize.y, size.y);
         	nr_labels++;
         	}
         }	  	
         	
  free_levels_data( &level );
  boxSize.y = QMAX(boxSize.y,nr_labels*lsize.y);       	
  return QSPt2f( tsize.x+boxSpace+boxSize.x+boxSpace+boxSpace+boxSpace+lsize.x, QMAX(tsize.y,boxSize.y) );
 }

//-------------------------------------------------------------//

void QSContour::drawLegendItem( const QSPt2f& pos, QSDrv *drv )
 {
  double boxSpace = drv->toPixels(BOX_SPACE);
  QSPt2f boxSize( drv->toPixels(BOX_WIDTH),
  		  drv->toPixels(BOX_HEIGHT) );
  		
  contour_levels_data level;
  calculate_levels_data( &level );		
  	 		
  // title
  QSPt2f tsize = drv->rTextSize( 270, title() );
  double height = boxSize.y = legendItemSize(drv).y;
  drv->drawRText( QSPt2f(pos.x,pos.y+height/2.0), 270, title(), AlignHCenter | AlignTop );

  double y0 = pos.y+(height-boxSize.y)/2.0;
  double x1 = pos.x+tsize.x+boxSpace;
  double x2 = x1+boxSize.x-1;

  // gradient
  for( int e=0; e<3; e++ ) {
    double prev_pos = 0.0;
    double curr_pos = 0.0;

    for ( int i=0; i<level.number; i++ ) {
  	 prev_pos = curr_pos;
  	 curr_pos = level.value[i];
  	 if  ( prev_pos <= 1.0 && curr_pos >= 0.0 ) {
  	 	 double y1 =  y0 + (1.0-QMAX(0.0,prev_pos)) * (boxSize.y-1) ;
  	 	 double y2 =  y0 + (1.0-QMIN(1.0,curr_pos)) * (boxSize.y-1) ;
    	 	 if ( e == 1 && m_fills ) {
  	 	 	drv->setFill( level.fill[i] );
  	 	 	drv->setLine( QSGLine::invisibleLine );
  	 	 	QSPt2f pts[4];
  	 	 	pts[0].set(x1,y1);
  	 	 	pts[1].set(x2,y1);
  	 	 	pts[2].set(x2,y2);
  	 	 	pts[3].set(x1,y2);
  		 	drv->drawPoly( pts, 4 );
  	 		}
  	 	 if ( curr_pos>=0.0 && curr_pos<=1.0 ) {
  	 	  	if ( e == 2 && m_lines ) {
  	 	  		drv->setLine( level.line[i] );
  	 	 		drv->drawLine( QSPt2f(x1+1,y2), QSPt2f(x2-2,y2) );
  	 	 		}
  	 	 	if ( e == 0 && !level.label[i].isEmpty() ) {
  	 	 		drv->drawText( QSPt2f(x2+boxSpace+boxSpace+boxSpace,y2), level.label[i], AlignLeft | AlignVCenter );
  	 			}
  	 		}
  	 	}	
  	}
    }
  	
  free_levels_data( &level );
 }


//-------------------------------------------------------------//

void QSContour::loadStateFromStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSPlot2D::loadStateFromStream( stream, factory );
 }

//-------------------------------------------------------------//

void QSContour::saveStateToStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSPlot2D::saveStateToStream( stream, factory );
 }




//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//

struct QSGriddedContour::gridded_contour_runtime_data {
	QSAxis *xaxis;
	QSAxis *yaxis;
	QSAxis *vaxis;
	QSMatrix *xvector;
	QSMatrix *yvector;
	QSMatrix *data;
	int w;
	int h;
	int dw;		// working area width
	int dh;		// working area height
	bool is_x_vector;
	bool is_y_vector;
     	QSPt2 min;
     	QSPt2 max;
	int curr_x_pos;
	int curr_y_pos;
	int curr_grid_number;
	QSPt3f curr_grid[5];
	};

QSGriddedContour::QSGriddedContour(QSAxes* parent, const char * name)
:QSContour( parent, name )
 {
  d = NULL;
  m_dmin  = 0.0;
  m_dmax  = 0.0;
  m_evalid  = false;
  static const int CHANNELS_NUM	= 3;
  initChannelTable( CHANNELS_NUM );
 }

 //-------------------------------------------------------------//

QSGriddedContour::~QSGriddedContour()
 {
 }

//-------------------------------------------------------------//

int QSGriddedContour::triangles()
// each grid mesh is divided into four triangles
 {
  return QMAX(0,d->dw-1)*QMAX(0,d->dh-1)*4;
 }

//-------------------------------------------------------------//



/*

 Triangle numbers

  ___________________
 |\   1  / |\   5  /
 |  \  /   |  \   /
 | 0  \  2 | 4  \   6   ....
 |  / 3 \  |  / 7 \
 |/_______\|/_______\


 Number of vertices in buffer

0 _________ 1
 |\      / |
 |  \  /   |
 |    \    |     4 - in the middle
 |  /   \  |
 |/_______\|
3           2

 Triangle vertices
1			   0
 |\			  /|	  2
 | \			2/ |	 / \
 | / 2   0 --- 1	 \ |	1---0
 |/        \ /		  \|
0           2		   1

 edge 0 is between 0 and 1
 edge 1 is between 1 and 2
 edge 3 is between 2 and 0
*/
void QSGriddedContour::getTriangle( int number, QSPt3f pts[3], double level )
 {
  load_mesh_into_buffer( number );
  	
  int tr_number = number % 4; 	

  switch( tr_number ) {
	 case 0: pts[0] = d->curr_grid[3]; pts[1] = d->curr_grid[0]; pts[2] = d->curr_grid[4]; break;
	 case 1: pts[0] = d->curr_grid[0]; pts[1] = d->curr_grid[1]; pts[2] = d->curr_grid[4]; break;
	 case 2: pts[0] = d->curr_grid[1]; pts[1] = d->curr_grid[2]; pts[2] = d->curr_grid[4]; break;
	 case 3: pts[0] = d->curr_grid[2]; pts[1] = d->curr_grid[3]; pts[2] = d->curr_grid[4]; break;
	 default: break;
	}

  // we don't want our triangle to cross a level plane exactly at an vertex	
  if ( pts[0].z == level ) pts[0].z += EPS_VALUE;
  if ( pts[1].z == level ) pts[1].z += EPS_VALUE;
  if ( pts[2].z == level ) pts[2].z += EPS_VALUE;
 }

//-------------------------------------------------------------//

void QSGriddedContour::getRectangle( int triangleNumber, QSPt3f pts[4] )
 {
  load_mesh_into_buffer( triangleNumber );
  pts[0] = d->curr_grid[0];
  pts[1] = d->curr_grid[1];
  pts[2] = d->curr_grid[2];
  pts[3] = d->curr_grid[3];
 }

//-------------------------------------------------------------//

void QSGriddedContour::load_mesh_into_buffer( int triangleNumber )
// loads mesh ( rectangle ) to the d->curr_grid buffer
 {
  int gr_number = triangleNumber >> 2 ; // divide by 4

  // check if is in buffer
  if ( gr_number != d->curr_grid_number ) {
  	
  	int x_pos = d->min.x + gr_number % (d->dw-1);
  	int y_pos = d->min.y + gr_number / (d->dw-1);
 		
  	// most requests will be with the increased x_pos
  	if ( x_pos == d->curr_x_pos+1 && y_pos == d->curr_y_pos ) {
  		 d->curr_grid[0] = d->curr_grid[1];
  		 d->curr_grid[3] = d->curr_grid[2];
  		} else {
  		 d->curr_grid[0].x = d->curr_grid[3].x = xvector(x_pos);
  		 d->curr_grid[0].y = yvector(y_pos+0);
  		 d->curr_grid[3].y = yvector(y_pos+1);
  		 d->curr_grid[0].z = d->data->value(y_pos+0,x_pos);
  		 d->curr_grid[3].z = d->data->value(y_pos+1,x_pos);
  		 d->curr_grid[0] = dataToWorldV( d->curr_grid[0] );
  		 d->curr_grid[3] = dataToWorldV( d->curr_grid[3] );
  		}
	
	d->curr_grid[1].x = d->curr_grid[2].x = xvector(x_pos+1);
	d->curr_grid[1].y = yvector(y_pos+0);
	d->curr_grid[2].y = yvector(y_pos+1);
	d->curr_grid[1].z = d->data->value(y_pos+0,x_pos+1);
	d->curr_grid[2].z = d->data->value(y_pos+1,x_pos+1);
	d->curr_grid[1] = dataToWorldV( d->curr_grid[1] );
	d->curr_grid[2] = dataToWorldV( d->curr_grid[2] );
	
	d->curr_grid[4] = d->curr_grid[0] +
			  d->curr_grid[1] +
			  d->curr_grid[2] +
			  d->curr_grid[3];
	d->curr_grid[4].x /= 4.0;
	d->curr_grid[4].y /= 4.0;
	d->curr_grid[4].z /= 4.0;
	
	d->curr_grid_number = gr_number;
  	d->curr_x_pos = x_pos;
  	d->curr_y_pos = y_pos;
  	}
 }

//-------------------------------------------------------------//

bool QSGriddedContour::getNeighbouringTriangle( int *number, int *edge )
 {
  //int grid_x;
  //int grid_y;
  int tr_number = *number & 0x03;
  switch( *edge ) {
  	case 1: *number = tr_number==3 ? *number-3 : *number+1; *edge = 2; return true;
  	case 2: *number = tr_number==0 ? *number+3 : *number-1; *edge = 1; return true;
  	case 0: {
  		  int grid_number = *number>>2;
  		  int grid_x = grid_number % (d->dw-1);
  		  int grid_y = grid_number / (d->dw-1);
  		  switch( tr_number ) {
  			case 0: if ( grid_x > 0       ) { *number -= 2;                  return true; } else return false;
  			case 1: if ( grid_y > 0       ) { *number -= (d->dw-1 << 2) - 2; return true; } else return false;
  			case 2: if ( grid_x < d->dw-2 ) { *number += 2;                  return true; } else return false;
  			case 3: if ( grid_y < d->dh-2 ) { *number += (d->dw-1 << 2) - 2; return true; } else return false;
  			}
  		}
  	}
  return false;
 }

//-------------------------------------------------------------//

bool QSGriddedContour::getAxisRange( QSAxis *axis, double& min, double& max )
 {
  allocRuntimeData();
  if ( d->w == 0 || d->h == 0 ) { freeRuntimeData(); return false; }

  if ( !m_evalid ) {
    for( int j=0; j<d->h; j++ ) {		
      for( int i=0; i<d->w; i++ ) {		 	
		double vmax;
		double vmin;
		vmax = vmin = value(j,i,Data);	 	
        	if ( i == 0 && j == 0 ) {
        		 m_dmax = vmax;
        		 m_dmin = vmin;
                       } else {
		         m_dmin = QMIN( m_dmin, vmin );
		 	 m_dmax = QMAX( m_dmax, vmax );
		 	}
		} // for ( i= ..
	} // for ( j= ...
      m_evalid = true;
     }

   double xmin = xvector(0);
   double xmax = xvector(d->w-1);
   double ymin = yvector(0);
   double ymax = yvector(d->h-1);
   if ( xmin > xmax ) { double xtemp = xmin; xmin = xmax; xmax = xtemp; }
   if ( ymin > ymax ) { double ytemp = ymin; ymin = ymax; ymax = ytemp; }

   if ( axis == d->xaxis ) { min = xmin; max = xmax; }
   else if ( axis == d->yaxis ) { min = ymin; max = ymax; }
   else if ( axis == d->vaxis ) { min = m_dmin; max = m_dmax; }
   else { freeRuntimeData(); return false; }

  freeRuntimeData();		
  return true;
 }

//-------------------------------------------------------------//

QString QSGriddedContour::vertexInfo( int t, int v, QSPt2f& pos )
 {
  QString result = QString::null;

  QSPt3f buff[MAX_BUFF_SIZE];
  getTriangle( t, buff ); 	

  int row = 0;
  int col = 0;
  if ( v!=2 ) {
  	 switch( ((t%4)+v)%4 ) {
  	 	case 0: col = d->curr_x_pos;   row = d->curr_y_pos+1; break;
  	 	case 1: col = d->curr_x_pos;   row = d->curr_y_pos  ; break;
  	 	case 2: col = d->curr_x_pos+1; row = d->curr_y_pos  ; break;
  	 	case 3: col = d->curr_x_pos+1; row = d->curr_y_pos+1; break;
  	 	}
  	 double x = xvector(col);
  	 double y = yvector(row);
  	 double z = d->data->value(row,col);
  	 pos = m_axes->dataToCanvas( QSPt2f(x,y), d->xaxis, d->yaxis );
  	 result = tr(" Data point: \n");
  	 result += tr(" row = ")+QString::number(row)+tr(", col = ")+QString::number(col)+" \n";
  	 result += tr(" X = ")+QString::number(x) + "\n";
  	 result += tr(" Y = ")+QString::number(y) + "\n";
  	 result += tr(" Z = ")+QString::number(z) + "\n";
  	 result += "\n";
  	}
  	
  return result;
 }

//-------------------------------------------------------------//

int QSGriddedContour::triangleAtPos( const QSPt2f& pos )
// the very ineffective
 {
  QSPt2f pts[3];
  //QSPt2f p = m_proj->canvasToWorld2D( QSPt2f(pos.x,pos.y) );
  QSPt3f buff[MAX_BUFF_SIZE];
  int triangles_count = triangles();
  for ( int i=0; i<triangles_count; i++ ) {
		getTriangle( i, buff );
		pts[0] = m_proj->world2DToCanvas( QSPt2f(buff[0].x,buff[0].y) ); //.set( buff[0].x, buff[0].y );
		pts[1] = m_proj->world2DToCanvas( QSPt2f(buff[1].x,buff[1].y) ); //.set( buff[1].x, buff[1].y );
		pts[2] = m_proj->world2DToCanvas( QSPt2f(buff[2].x,buff[2].y) ); //.set( buff[2].x, buff[2].y );
		if ( QSProjection::pointInPoly(pos,pts,3) ) return i;
		}
  return -1;
 }

//-------------------------------------------------------------//

void QSGriddedContour::dataChanged( int channel )
// Ouu. We need to calculate new extremes in data
// Refresh data on screen also.
  {
   if ( channel == Data || channel == -1 ) m_evalid = false;
   QSContour::dataChanged( channel );
  }

//-------------------------------------------------------------//

void QSGriddedContour::allocRuntimeData()
 {
  QSContour::allocRuntimeData();
  d = new gridded_contour_runtime_data();
  d->xaxis = defaultAxis(QSAxis::XAxisType);
  d->yaxis = defaultAxis(QSAxis::YAxisType);
  d->vaxis = defaultAxis(QSAxis::VAxisType);
  d->curr_grid_number = -999;
  d->curr_x_pos       = -999;
  d->curr_y_pos       = -999;
  d->is_x_vector = false;
  d->is_y_vector = false;
  d->w     = matrixCols( Data );
  d->h     = matrixRows( Data );
  d->data    = matrix( Data    );
  d->xvector = matrix( XVector );
  d->yvector = matrix( YVector );
  if ( matrixCols( XVector ) == d->w &&
       matrixRows( XVector ) > 0    ) d->is_x_vector = true;
                                 else d->is_x_vector = false;

  if ( matrixRows( YVector ) == d->h &&
       matrixCols( YVector ) > 0    ) d->is_y_vector = true;
                                 else d->is_y_vector = false;
  calculate_data_range();
 }

//-------------------------------------------------------------//

void QSGriddedContour::freeRuntimeData()
 {
  QSContour::freeRuntimeData();
  delete d; d = NULL;
 }

//-------------------------------------------------------------//

void QSGriddedContour::drawGrid()
 {
  if ( line(Grid).style == QSGLine::Invisible ) return;

  m_drv->setLine( line(Grid) );
  int pi = d->min.x;
  int pj = d->min.y;

  while( pi >= d->min.x && pi <= d->max.x ) {
  	 QSPt2f p1 = dataToWorld(QSPt2f(xvector(pi),yvector(0)     ) );
  	 QSPt2f p2 = dataToWorld(QSPt2f(xvector(pi),yvector(d->h-1)) );
  	 m_drv->drawLine2( p1, p2 );
  	 pi++;
  	}
  	
  while( pj >= d->min.y && pj <= d->max.y ) {
 	 QSPt2f p1 = dataToWorld(QSPt2f(xvector(0),     yvector(pj)) );
  	 QSPt2f p2 = dataToWorld(QSPt2f(xvector(d->w-1),yvector(pj)) );
  	 m_drv->drawLine2( p1, p2 );  	
  	 pj++;
  	}	
 }

//-------------------------------------------------------------//

void QSGriddedContour::drawPoints()
 {
  if ( point(PointMark).style == QSGPoint::Invisible ) return;

  int pi = d->min.x;
  int pj = d->min.y;

  QString label;
  QSGPoint point_style = point(PointMark);
  while( pj >= d->min.y && pj <= d->max.y ) {
     while( pi >= d->min.x && pi <= d->max.x ) {
   	  m_drv->drawPoint2( dataToWorld(QSPt2f(xvector(pi),yvector(pj))), point_style );
   	  pi++;
   	 }
     pj++; pi = d->min.x;
     }
 }

//-------------------------------------------------------------//

double QSGriddedContour::yvector( int row )
 {
   return d->is_y_vector ? d->yvector->value(row,0) : row;
 }

//-------------------------------------------------------------//

double QSGriddedContour::xvector( int col )
 {
   return d->is_x_vector ? d->xvector->value(0,col) : col;
 }

//-------------------------------------------------------------//

void QSGriddedContour::calculate_data_range()
// which grid rectangles are to be drawn
// ( including those partialy overlapping a drawing area )
 {
  int i;
  bool is_p_in;
  bool is_c_in;
  double cvalue = 0.0;
  double pvalue = 0.0;

  d->min.x = d->w; // empty area
  d->max.x =   -1;
  for( i=0;i<d->w;i++ ) {
        pvalue = cvalue;
  	cvalue = d->xaxis->dataToWorld(xvector(i));
        is_p_in = ( pvalue >= 0.0 && pvalue <= 1.0 && i>0 );
        is_c_in = ( cvalue >= 0.0 && cvalue <= 1.0 && i>0 );
        if ( is_c_in || is_p_in ) {
        	d->min.x = QMIN( d->min.x, i-1 );
        	d->max.x = QMAX( d->max.x, i );
        	}
        if ( i>0 && pvalue < 0.0 && cvalue > 1.0 ||
             i>0 && cvalue < 0.0 && pvalue > 1.0 ) {
             	d->min.x = i-1; d->max.x = i;
             	}
        }

  d->min.y = d->h; // empty area
  d->max.y =   -1;
  for( i=0;i<d->h;i++ ) {
        pvalue = cvalue;
  	cvalue = d->yaxis->dataToWorld(yvector(i));
        is_p_in = ( pvalue >= 0.0 && pvalue <= 1.0 && i>0 );
        is_c_in = ( cvalue >= 0.0 && cvalue <= 1.0 && i>0 );
        if ( is_c_in || is_p_in ) {
        	d->min.y = QMIN( d->min.y, i-1 );
        	d->max.y = QMAX( d->max.y, i );
        	}
	if ( i>0 && pvalue < 0.0 && cvalue > 1.0 ||
             i>0 && cvalue < 0.0 && pvalue > 1.0 ) {
             	d->min.y = i-1;
             	d->max.y = i;
             	}
        }

  d->dw = QMAX( 0, d->max.x-d->min.x+1 );
  d->dh = QMAX( 0, d->max.y-d->min.y+1 );
  }

//-------------------------------------------------------------//

void QSGriddedContour::loadStateFromStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSContour::loadStateFromStream( stream, factory );
 }

//-------------------------------------------------------------//

void QSGriddedContour::saveStateToStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSContour::saveStateToStream( stream, factory );
 }

//-------------------------------------------------------------//

QString QSGriddedContour::channelVariable( int channel ) const
 {
  switch( channel ) {
	case Data:	return "v";
	case XVector:	return "x";
	case YVector:	return "y";
	}
  return QString::null;
 }

//-------------------------------------------------------------//

QSGriddedContour::ColumnType QSGriddedContour::columnType( int, int ) const
 {
  return ColumnUnknown;
 }




//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//

struct QSNonGriddedContour::non_gridded_contour_runtime_data {
	QSAxis *xaxis;
	QSAxis *yaxis;
	QSAxis *vaxis;
	int m_npoints;
	int m_ntriangles;
	QSMatrix *m_points_x;
	QSMatrix *m_points_y;
	QSMatrix *m_points_z;
	QSMatrix *m_triangles;
	int *m_index_to_visible_list;
        vector<int> m_visible_triangles;
        bool m_use_visible_list;
        int curr_triangle;

	inline int triangles() {
		return m_use_visible_list ? m_visible_triangles.size() : m_ntriangles;
		}
	inline int visible_to_real( int triangle_index ) {
		return m_use_visible_list ? m_visible_triangles[triangle_index] : triangle_index;
		}
	inline int real_to_visible( int triangle_index ) {
		return m_use_visible_list ? m_index_to_visible_list[triangle_index] : triangle_index;
		}
	inline int edge( int triangle, int triangle_edge ) {
		return triangle*3+triangle_edge;
		}
	inline int triangle( int edge ) {
		return edge/3;
		}
	inline int triangle_edge( int edge ) {
		return edge%3;
		}
        inline int point_1_index( int edge ) {
		return (int )m_triangles->value( triangle(edge), triangle_edge(edge) );
		}
	inline int point_2_index( int edge ) {
		return (int )m_triangles->value( triangle(edge), (triangle_edge(edge)+1)%3 );
		}

	inline bool edges_equal( int edge1, int edge2 ) {
		int edge1_p1 = point_1_index( edge1 );
		int edge1_p2 = point_2_index( edge1 );
		int edge2_p1 = point_1_index( edge2 );
		int edge2_p2 = point_2_index( edge2 );
		if ( QMIN(edge1_p1,edge1_p2) == QMIN(edge2_p1,edge2_p2) &&
		     QMAX(edge1_p1,edge1_p2) == QMAX(edge2_p1,edge2_p2) ) return true;
		return false;		
		}

        // compare edges
	struct edge_comp_t : public binary_function<int, int, bool> {
		// if we sort edge buffer using this compare functiona all
		// shared edges will be on neighbouring positions
		bool operator()( int edge1, int edge2 ) {
			// compare
			int edge1_p1 = d->point_1_index( edge1 );
			int edge1_p2 = d->point_2_index( edge1 );
                        int edge2_p1 = d->point_1_index( edge2 );
			int edge2_p2 = d->point_2_index( edge2 );

			// remember that the same edge is when:
			// edge1_p1 == edge2_p2 && edge1_p2 == edge2_p1 or
                        // edge1_p1 == edge2_p1 && edge1_p2 == edge2_p2
			// this is the same as:
			//  QMIN(edge1_p1,edge1_p2) == QMIN(edge1_p1,edge1_p2) &&
			//  QMAX(edge1_p1,edge1_p2) == QMAX(edge1_p1,edge1_p2) &&
			int edge1_min = QMIN(edge1_p1,edge1_p2);
                        int edge2_min = QMIN(edge2_p1,edge2_p2);
			if ( edge1_min < edge2_min ) {
				return true;
				}
			else if ( edge1_min == edge2_min ) {
				int edge1_max = QMAX(edge1_p1,edge1_p2);
                        	int edge2_max = QMAX(edge2_p1,edge2_p2);
				return edge1_max < edge2_max;
				}	
			else return false;				
			}
                 non_gridded_contour_runtime_data *d;
		} edge_comp;

	};

 //-------------------------------------------------------------//

QSNonGriddedContour::QSNonGriddedContour(QSAxes* parent, const char * name)
:QSContour( parent, name )
 {
  d = NULL;
  m_evalid = false;
  m_neighbours[0] = NULL;
  m_neighbours[1] = NULL;
  m_neighbours[2] = NULL;
  static const int CHANNELS_NUM	= 4;
  initChannelTable( CHANNELS_NUM );
 }

 //-------------------------------------------------------------//

QSNonGriddedContour::~QSNonGriddedContour()
 {
  delete[] m_neighbours[0];
  delete[] m_neighbours[1];
  delete[] m_neighbours[2];
 }

//-------------------------------------------------------------//

int QSNonGriddedContour::triangleAtPos( const QSPt2f& pos )
 {
  QSPt2f pts[3];
  QSPt3f buff[MAX_BUFF_SIZE];
  int triangles_count = triangles();
  for ( int i=0; i<triangles_count; i++ ) {
               	getTriangle( i, buff );
		pts[0] = m_proj->world2DToCanvas( QSPt2f(buff[0].x,buff[0].y) );
		pts[1] = m_proj->world2DToCanvas( QSPt2f(buff[1].x,buff[1].y) );
		pts[2] = m_proj->world2DToCanvas( QSPt2f(buff[2].x,buff[2].y) );
		if ( QSProjection::pointInPoly(pos,pts,3) ) return i;
		}
  return -1;
 }

//-------------------------------------------------------------//

QString QSNonGriddedContour::vertexInfo( int triangle, int vertex, QSPt2f& )
 {
  // map a visible index to a "real" index
  triangle = d->visible_to_real(triangle);
  int point_index = (int )d->m_triangles->value(triangle,vertex);
  QSPt3f p = point( point_index );
  QString result = tr(" Data point: \n");
  result += tr(" Vertex table: row = ")+QString::number(point_index)+" \n";
  result += tr(" Triangles: row = ")+QString::number(triangle)+tr(", col = ")+QString::number(vertex)+" \n";
  result += tr(" X = ")+QString::number(p.x) + "\n";
  result += tr(" Y = ")+QString::number(p.y) + "\n";
  result += tr(" Z = ")+QString::number(p.z) + "\n";
  return result;
 }

//-------------------------------------------------------------//

QSPt3f QSNonGriddedContour::point( int index )
 {
  if ( index < 0 || index >= d->m_points_x->rows() ) {
	QSConsole::write(tr(" Contour %1 : Bad index detected. Index value = %2. ").arg(title()).arg(index) );
	return QSPt3f( sqrt(-1.0), sqrt(-1.0), sqrt(-1.0) );
	}
  return QSPt3f( d->m_points_x->value(index,0),
		 d->m_points_y->value(index,0),
		 d->m_points_z->value(index,0) );
 }

//-------------------------------------------------------------//

bool QSNonGriddedContour::getAxisRange( QSAxis *axis, double& min, double& max )
 {
   allocRuntimeData();
   if ( d->m_npoints == 0 ) {
	freeRuntimeData();
	return false;
	}
   if ( !m_evalid ) {
	QSPt3f p = point( 0 );	
	m_xmin = p.x;
	m_xmax = p.x;
	m_ymin = p.y;
	m_ymax = p.y;
	m_zmin = p.z;
	m_zmax = p.z;
	for( int i=1; i<d->m_npoints; i++ ) {	
		QSPt3f p = point( i );
		m_xmin = QMIN( m_xmin, p.x );
		m_xmax = QMAX( m_xmax, p.x );
		m_ymin = QMIN( m_ymin, p.y );
		m_ymax = QMAX( m_ymax, p.y );
		m_zmin = QMIN( m_zmin, p.z );
		m_zmax = QMAX( m_zmax, p.z );		
		}
	m_evalid = true;
	}

   if ( axis == d->xaxis ) { min = m_xmin; max = m_xmax; }
   else if ( axis == d->yaxis ) { min = m_ymin; max = m_ymax; }
   else if ( axis == d->vaxis ) { min = m_zmin; max = m_zmax; }
   else { freeRuntimeData(); return false; }

  freeRuntimeData();	
  return true;
 }

//-------------------------------------------------------------//

void QSNonGriddedContour::dataChanged( int channel )
  {
   if ( channel == -1 || channel == VXCoord || channel == VYCoord || channel == VZCoord ) m_evalid = false;
   if ( channel == -1 || channel == VXCoord || channel == VYCoord || channel == VZCoord || channel == Triangles ) {
	delete[] m_neighbours[0]; m_neighbours[0] = NULL;
	delete[] m_neighbours[1]; m_neighbours[1] = NULL;
 	delete[] m_neighbours[2]; m_neighbours[2] = NULL;
	}
   QSContour::dataChanged( channel );
  }

//-------------------------------------------------------------//

void QSNonGriddedContour::allocRuntimeData()
 {
  QSContour::allocRuntimeData();
  d = new non_gridded_contour_runtime_data();
  d->xaxis = defaultAxis(QSAxis::XAxisType);
  d->yaxis = defaultAxis(QSAxis::YAxisType);
  d->vaxis = defaultAxis(QSAxis::VAxisType);

  d->m_ntriangles = matrixCols( Triangles );
  d->m_triangles = matrix( Triangles );

  d->m_npoints = 0;
  if ( matrixRows( VXCoord ) == matrixRows( VYCoord ) &&
       matrixRows( VXCoord ) == matrixRows( VZCoord ) &&
       matrixCols( VXCoord ) > 0 &&
       matrixCols( VYCoord ) > 0 &&
       matrixCols( VZCoord ) > 0  ) {
	d->m_npoints = matrixRows( VXCoord );
	}

  d->m_ntriangles = 0;
  if ( matrixCols( Triangles ) == 3 ) {
	d->m_ntriangles = matrixRows( Triangles );
	}

  d->m_points_x = matrix( VXCoord );
  d->m_points_y = matrix( VYCoord );
  d->m_points_z = matrix( VZCoord );
  d->edge_comp.d = d;

  if ( m_neighbours[0] == NULL ) search_for_neighbours();
  d->m_index_to_visible_list = new int[d->m_ntriangles];	
  d->m_use_visible_list = false;
  d->curr_triangle = 0;
 }

//-------------------------------------------------------------//

void QSNonGriddedContour::freeRuntimeData()
 {
  delete[] d->m_index_to_visible_list;
  delete d; d = NULL;
  QSContour::freeRuntimeData();
 }

//-------------------------------------------------------------//

int QSNonGriddedContour::triangles()
 {
  // there is a little trick. This object can provide two list of
  // triangles - full or visible ones, depending if a list of visible
  // triangles is calculated. The list of triangles should be available
  // immediately after allocRuntimeData(), but for optimized drawing
  // we provide a list of visible triangles after prepare(). We cannot
  // provide a list of visible triangles immediately after allorRuntime data
  // because it takes much time to make such list and this must be done in the
  // background.
  return d->triangles();
 }

//-------------------------------------------------------------//

bool QSNonGriddedContour::getNeighbouringTriangle( int *number, int *edge )
 {
  // map visible triangle index to a "real" triangle index
  int triangle = d->visible_to_real(*number);
  int neighbouring_edge = m_neighbours[*edge][triangle];
  if ( neighbouring_edge >= 0 ) {
	 // map a "real" index to a visible index
	 *number = d->real_to_visible( d->triangle(neighbouring_edge) );
	 *edge = d->triangle_edge(neighbouring_edge);
	 if ( *number >= 0 ) return true;
	}
  return false;
 }

//-------------------------------------------------------------//

void QSNonGriddedContour::getTriangle( int number, QSPt3f pts[3], double level )
 {
  // map visible triangle index to a "real" triangle index
  int triangle = d->visible_to_real(number);
  pts[0] = dataToWorldV( point( (int )d->m_triangles->value(triangle,0) ) );
  pts[1] = dataToWorldV( point( (int )d->m_triangles->value(triangle,1) ) );
  pts[2] = dataToWorldV( point( (int )d->m_triangles->value(triangle,2) ) );
  if ( pts[0].z == level ) pts[0].z += EPS_VALUE;
  if ( pts[1].z == level ) pts[1].z += EPS_VALUE;
  if ( pts[2].z == level ) pts[2].z += EPS_VALUE;
 }

//-------------------------------------------------------------//

void QSNonGriddedContour::drawGrid()
 {
  if ( line(Grid).style == QSGLine::Invisible ) return;
  QSPt3f pts[3];
  m_drv->setLine( line(Grid) );
  for( int i=0; i<triangles(); i++ ) {
	getTriangle( i, pts );
	for( int j=0; j<3; j++ ) {
		QSPt3f *p1 = &pts[j];
		QSPt3f *p2 = &pts[(j+1)%3];
		m_drv->drawLine2( QSPt2f(p1->x,p1->y), QSPt2f(p2->x,p2->y) );
		}	
	}
 }

//-------------------------------------------------------------//

void QSNonGriddedContour::drawPoints()
 {
  if ( QSGraphicalData::point(PointMark).style == QSGPoint::Invisible ) return;

  QSGPoint point_style = QSGraphicalData::point(PointMark);
  for( int i=0; i<d->m_npoints; i++ ) {
	  QSPt3f p = dataToWorldV( point( i ) );
	  if ( p.x >= 0.0 && p.x <= 1.0 &&
	       p.y >= 0.0 && p.y <= 1.0 )
   	  	m_drv->drawPoint2( QSPt2f(p.x,p.y), point_style );
   	 }
 }

//-------------------------------------------------------------//

void QSNonGriddedContour::search_for_neighbours()
 {
  // create list of all edges in data
  int edges = d->m_ntriangles*3;
  int *edge_buffer = new int[edges];
  for( int edge=0; edge<edges; edge++ ) edge_buffer[edge]=edge;

  // sort this list this list first by p1, second by p2
  // if the two edges have the same points (p1,p2)
  // they will be neighbours on this list after sorting
  sort( edge_buffer, edge_buffer+edges, d->edge_comp );

  // fill the neighbours list
  delete[] m_neighbours[0];
  delete[] m_neighbours[1];
  delete[] m_neighbours[2];
  m_neighbours[0] = new int[d->m_ntriangles];
  m_neighbours[1] = new int[d->m_ntriangles];
  m_neighbours[2] = new int[d->m_ntriangles];

  for( int i=0; i<3; i++ )
	for( int j=0; j<d->m_ntriangles; j++ )
		m_neighbours[i][j] = -1;

  for( int i=0; i<edges-1; i++ ) {
	 int edge1 = edge_buffer[i];
	 int edge2 = edge_buffer[i+1];	
	 if ( d->edges_equal( edge1, edge2 ) ) {		
		 m_neighbours[d->triangle_edge(edge1)][d->triangle(edge1)] = edge2;
                 m_neighbours[d->triangle_edge(edge2)][d->triangle(edge2)] = edge1;
		}
	}

  delete[] edge_buffer;
 }

//-------------------------------------------------------------//

bool QSNonGriddedContour::prepare()
// all triangles are visible
 {
  int curr_step = 0;
  int steps = work_steps * 10;
  QSPt3f pts[3];
  QSPt2f clip_pos( 0.0, 0.0 );
  QSPt2f clip_area( 1.0, 1.0 );
  while( d->curr_triangle < d->m_ntriangles ) {
 	pts[0] = dataToWorldV( point( (int )d->m_triangles->value(d->curr_triangle,0) ) );
  	pts[1] = dataToWorldV( point( (int )d->m_triangles->value(d->curr_triangle,1) ) );
  	pts[2] = dataToWorldV( point( (int )d->m_triangles->value(d->curr_triangle,2) ) );
	QSPt2f p1( pts[0].x, pts[0].y );
	QSPt2f p2( pts[1].x, pts[1].y );
	QSPt2f p3( pts[2].x, pts[2].y );
	if ( QSProjection::clipLine( &p1, &p2, clip_pos, clip_area ) ||
	     QSProjection::clipLine( &p2, &p3, clip_pos, clip_area ) ||
	     QSProjection::clipLine( &p3, &p1, clip_pos, clip_area ) ) {
		d->m_visible_triangles.push_back( d->curr_triangle );
		d->m_index_to_visible_list[d->curr_triangle] = d->m_visible_triangles.size()-1;
		} else {
		d->m_index_to_visible_list[d->curr_triangle] = -1;
		}
	 d->curr_triangle++;
	// end with this little part ...
        if ( curr_step++ > steps && m_bkg_handler ) return false;
	}
  d->m_use_visible_list = true;
  return true;
 }

//-------------------------------------------------------------//

void QSNonGriddedContour::loadStateFromStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSContour::loadStateFromStream( stream, factory );
 }

//-------------------------------------------------------------//

void QSNonGriddedContour::saveStateToStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSContour::saveStateToStream( stream, factory );
 }

//-------------------------------------------------------------//

QString QSNonGriddedContour::channelVariable( int channel ) const
 {
  switch( channel ) {
	case VXCoord:	return "x";
	case VYCoord:	return "y";
	case VZCoord:	return "z";
	case Triangles:	return "i";
	}
  return QString::null;
 }

//-------------------------------------------------------------//

QSNonGriddedContour::ColumnType QSNonGriddedContour::columnType( int channel, int column ) const
 {
  if ( channel == VXCoord && column == 0 ) return ColumnX;
  if ( channel == VYCoord && column == 0 ) return ColumnY;
  if ( channel == VZCoord && column == 0 ) return ColumnZ;
 }



