/***************************************************************************
                                 qsaxis.cpp
                             -------------------
    begin                :

    copyright            : (C) 2001 by Kamil Dobkowski
    email                : kamildobk@poczta.onet.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include"qsaxis.h"
#include<assert.h>
#include<qregexp.h>
#include<algo.h>


//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//


QSAxisTic::QSAxisTic( double value, bool major )
 {
  m_value = value;
  m_major = major;
  m_pos   = 0.0;
  m_angle = 0;
  m_is_fill_defined = false;
 }

//-------------------------------------------------------------//

QSAxisTic::~QSAxisTic()
 {
 }

//-------------------------------------------------------------//

/*
bool QSAxisTic::operator<( const QSAxisTic& t )
 {
  return m_value < t.m_value ;
 }
*/

//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//

const double QSAxis::minScaleBase  =  1.001;
const double QSAxis::minLogValue   =  1e-200;
const double QSAxis::minRange      =  1e-200;
const double QSAxis::minRangeValue = -1e200;
const double QSAxis::maxRangeValue =  1e200;


QSAxis::QSAxis( AxisType type, QSAxes *parentAxes, const char *name )
:QSAxesChild( parentAxes, name )
 {
  m_type     = type;
  m_visible  = true;
  m_opposite = false;
  m_scale    = LinearScale;
  m_round    = false;
  m_reversed = false;
  m_scrollable = true;
  m_default    = true;
  m_pos      = 0.0;
  m_min      = 0.0;
  m_max      = 0.0;
  m_vmin     = 1.0;
  m_vmax     = 10.0;
  m_dmax     = 0.0;
  m_dmin     = 0.0;
  m_majd     = -4.0;
  m_mind     = -20.0;
  m_base     = 10.0;
  m_wscale   = 1.0;
  m_wmin     = 0.0;
  m_tics_angle   = 0;
  m_tics_outer   = false;
  m_tics_visible = true;
  m_tics_format = "(TIC)";
  m_tic_label_pos1 = 0.01;
  m_tic_label_pos2 = 0.01;
  m_title_position = 0.5;
  m_title_distance = 0.05;

  switch( m_type ) {
	case XAxisType: setTitle( "X" ); break;
	case YAxisType: setTitle( "Y" ); break;
	case ZAxisType: setTitle( "Z" ); break;
	case VAxisType: setTitle( "V" ); m_mind = 0.0; break;
	}
  for( int i=0; i<3; i++ ) rememberCurrentView( i );

  static const int CHANNELS_NUM = 4;
  static const int  FONTS_NUM	= 2;
  static const int  FILLS_NUM	= 0;
  static const int  LINES_NUM	= 3;
  static const int  POINTS_NUM	= 0;
  initChannelTable( CHANNELS_NUM );
  initAttributeTables( FONTS_NUM, FILLS_NUM, LINES_NUM, POINTS_NUM );

  m_settings.lines[MajorGridLine].style = QSGLine::Invisible;
  m_settings.lines[MinorGridLine].style = QSGLine::Invisible;
 }

//-------------------------------------------------------------//

QSAxis::~QSAxis()
 {
 }
 //-------------------------------------------------------------//

void QSAxis::setTitlePosition( double value )
 {
  SET_PROPERTY( m_title_position, value );
 }

//-------------------------------------------------------------//

void QSAxis::setTitleDistance( double value )
 {
  SET_PROPERTY( m_title_distance, value );
 }

//-------------------------------------------------------------//

void QSAxis::setTicLabelPos1( double pos )
 {
  SET_PROPERTY( m_tic_label_pos1, pos );
 }

//-------------------------------------------------------------//

void QSAxis::setTicLabelPos2( double pos )
 {
  SET_PROPERTY( m_tic_label_pos2, pos );
 }

//-------------------------------------------------------------//

void QSAxis::setVisible( bool enabled )
 {
  SET_PROPERTY( m_visible, enabled );
 }

//-------------------------------------------------------------//

void QSAxis::setOppositePosition( bool enabled )
 {
  SET_PROPERTY( m_opposite, enabled );
 }

//-------------------------------------------------------------//

void QSAxis::setDefaultPosition( bool enabled )
 {
  SET_PROPERTY( m_default, enabled );
 }

//-------------------------------------------------------------//

void QSAxis::setScrollable( bool enabled )
 {
  SET_PROPERTY( m_scrollable, enabled );
 }

//-------------------------------------------------------------//

void QSAxis::setArrow1( const QSGArrow& arrow )
 {
  SET_PROPERTY( m_arrow1, arrow );
 }

//-------------------------------------------------------------//

void QSAxis::setArrow2( const QSGArrow& arrow )
 {
  SET_PROPERTY( m_arrow2, arrow );
 }

//-------------------------------------------------------------//

void QSAxis::set_arrow1_property( const QString& data )
 {
  setArrow1(toQSGArrow(data));
 }

//-------------------------------------------------------------//

void QSAxis::set_arrow2_property( const QString& data )
 {
  setArrow2(toQSGArrow(data));
 }

//-------------------------------------------------------------//

QString QSAxis::arrow1_property() const
 {
  return toQString(arrow1());
 }

//-------------------------------------------------------------//

QString QSAxis::arrow2_property() const
 {
  return toQString(arrow2());
 }

//-------------------------------------------------------------//

void QSAxis::setRange( double min, double max )
  {
   if ( min > max ) { double temp = min; min = max; max = temp; }
   if ( max == min ) max = min = 0.0;
   if ( m_min != min || m_max != max ) {
	parametersChanging();
        if ( m_min != min ) m_min = min;
        if ( m_max != max ) m_max = max;
        parametersChanged();
       }
  }

//-------------------------------------------------------------//

void QSAxis::setRangeMin( double min )
 {
  if ( min != m_min ) {
	parametersChanging();
	m_min = min;
	parametersChanged();
	}
 }

//-------------------------------------------------------------//

void QSAxis::setRangeMax( double max )
 {
  if ( max != m_max ) {
	parametersChanging();
	m_max = max;
	parametersChanged();
	}
 }

//-------------------------------------------------------------//

void QSAxis::setScale( AxisScale scale, double base )
  {
   if ( scale != LinearScale && scale != LogScale ) return;
   if ( base < minScaleBase ) base = minScaleBase;

   if ( scale != m_scale || base != m_base ) {
         parametersChanging();
         m_scale = scale;
         m_base  = base;
         parametersChanged();
        }
  }

//-------------------------------------------------------------//

void QSAxis::setScaleType( int scale )
 {
  if ( scale != LinearScale && scale != LogScale ) return;
  SET_PROPERTY( m_scale, (AxisScale )scale );
 }

//-------------------------------------------------------------//

void QSAxis::setScaleBase( double base )
 {
  if ( base < minScaleBase ) base = minScaleBase;
  SET_PROPERTY( m_base, base );
 }

//-------------------------------------------------------------//

void QSAxis::setPosition( double pos )
  {
   if ( m_pos != pos ) {
	parametersChanging();
	if ( pos == 0.0 ) m_default = true;
		     else m_default = false;
   	m_pos = pos;
	parametersChanged();
	}
  }

//-------------------------------------------------------------//

void QSAxis::setTicsFormat( const QString& format )
  {
   SET_PROPERTY( m_tics_format, format );
  }

//-------------------------------------------------------------//

void QSAxis::setReversed( bool reversed )
  {
   SET_PROPERTY( m_reversed, reversed );
  }

//-------------------------------------------------------------//

double QSAxis::min( AxisRange type ) const
  {
   switch( type ) {
          case RangeSet:     return m_min ; break;
          case RangeVisible: return m_vmin; break;
          case RangeData:    return m_dmin; break;
         }

   return 1.0;
  }

//-------------------------------------------------------------//

double QSAxis::max( AxisRange type ) const
  {
   switch( type ) {
          case RangeSet:     return m_max ; break;
          case RangeVisible: return m_vmax; break;
          case RangeData:    return m_dmax; break;
         }
   return 10.0;
  }

//-------------------------------------------------------------//

void QSAxis::setRoundRangeToTicStep( bool enabled )
  {
   SET_PROPERTY( m_round, enabled );
  }

//-------------------------------------------------------------//

void QSAxis::setGridStep( double major, double minor )
   {
    if ( m_majd != major ||
         m_mind != minor ) {
           parametersChanging();
           m_majd = major;
           m_mind = minor;
           parametersChanged();
           }
   }

//-------------------------------------------------------------//

void QSAxis::setMajorGridStep( double step )
 {
  SET_PROPERTY( m_majd, step );
 }

//-------------------------------------------------------------//

void QSAxis::setMinorGridStep( double step )
 {
  SET_PROPERTY( m_mind, step );
 }

//-------------------------------------------------------------//

void QSAxis::setTicsOuter( bool enabled )
   {
    SET_PROPERTY( m_tics_outer, enabled );
   }

//-------------------------------------------------------------//

void QSAxis::setTicsVisible( bool visible )
  {
   SET_PROPERTY( m_tics_visible, visible );
  }

//-------------------------------------------------------------//

void QSAxis::setTicsAngle( int angle )
  {
   SET_PROPERTY( m_tics_angle, angle );
  }

//-------------------------------------------------------------//

#define QSAXES_MINUS_INF	-10
//
// w = (value-min)/(max-min)
//	 wscale = 1.0/(max-min)
//	 wmin   = min/(max-min)
//
#define AXIS_TO_WORLD_LINEAR(value) (m_wscale*(value)-m_wmin)
//
// w = (log(value)-log(min))/(log(max)-log(min))
//	 wscale = 1.0/[log10(logbase)*(log(max)-log(min))]
//	 wmin   = log(min)/(log(max)-log(min))	
//
#define AXIS_TO_WORLD_LOG(value) ((value)>0.0?m_wscale*log10(value)-m_wmin:QSAXES_MINUS_INF)
//
// Summary:
//
#define AXIS_TO_WORLD_NORMAL(value) (m_scale==LinearScale?AXIS_TO_WORLD_LINEAR(value):AXIS_TO_WORLD_LOG(value))
#define AXIS_TO_WORLD(w) (m_reversed?1.0-AXIS_TO_WORLD_NORMAL(w):AXIS_TO_WORLD_NORMAL(w))

//-------------------------------------------------------------//

double QSAxis::dataToWorld( double value ) const
  {
   return AXIS_TO_WORLD(value);
  }

//-------------------------------------------------------------//

//
// value = w*(max-min)+min
//	 wscale = 1.0/(max-min)
//	 wmin   = min/(max-min)
//
#define WORLD_TO_AXIS_LINEAR(w) (((w)+m_wmin)/m_wscale)
//
// value = 10^(w*(log(max)-log(min))+log(min))
//	 wscale = 1.0/[log10(logbase)*(log(max)-log(min))]
//	 wmin   = log(min)/(log(max)-log(min))
//
#define WORLD_TO_AXIS_LOG(w) (pow(10,(w+m_wmin)/m_wscale))
//
// Summary:
//
#define WORLD_TO_AXIS_NORMAL(w) (m_scale==LinearScale?WORLD_TO_AXIS_LINEAR(w):WORLD_TO_AXIS_LOG(w))
#define WORLD_TO_AXIS(w) (m_reversed?WORLD_TO_AXIS_NORMAL(1.0-(w)):WORLD_TO_AXIS_NORMAL(w))

//-------------------------------------------------------------//

double QSAxis::worldToData( double value ) const
  {
   return WORLD_TO_AXIS(value);
  }

//-------------------------------------------------------------//

void QSAxis::initAxis( double dataMin, double dataMax, bool is_data )
 {
  m_dmin = dataMin;
  m_dmax = dataMax;
  init_ranges(is_data);
  init_axis_mappings();
  init_tics();
 }

//-------------------------------------------------------------//

void QSAxis::init_ranges( bool is_data )
 {  	
  double visible_min = 1.0;
  double visible_max = 10.0;

  if ( m_min < m_max ) {
	// user set range
	visible_min = m_min;
        visible_max = m_max;
       } else {
	  if ( is_data )
	 	if ( m_dmax > m_dmin ) {
		      // auto-calculated range 1
                 	visible_min = m_dmin;
                        visible_max = m_dmax;
                       } else {
                      // auto-calculated range 2
                        visible_min = floor( m_dmin );
                        visible_max = ceil( m_dmax );
                       }
	}

  // NaN
  if ( visible_min != visible_min ) visible_min = 0.0;	
  if ( visible_max != visible_max ) visible_max = 0.0;

  // visible_min must be ALWAYS > visible_max
  if ( visible_min == visible_max ) {
	// we are here only if floor( m_dmin ) == ceil( m_dmax )
	visible_min -= 1.0;
	visible_max += 1.0;
       	}    			
  if ( visible_min >= visible_max ) {
	// only if some NaN
	visible_max = visible_min + 1.0;
	}
  	
  // adjust log scale
  if ( m_scale == LogScale && visible_min < minLogValue ) {
       	visible_min = minLogValue;
       	if ( visible_max <= visible_min ) visible_max = 1.0;
      	}

  // max values
  visible_min = QMAX( visible_min, minRangeValue );
  visible_max = QMIN( visible_max, maxRangeValue );

  // min range
  if ( visible_max-visible_min < minRange ) {
	if ( visible_min + minRange <= maxRangeValue ) visible_max = visible_min + minRange;
						  else visible_min = visible_max - minRange;
	}
	
  m_vmin = visible_min;
  m_vmax = visible_max; 	
 }

//-------------------------------------------------------------//

void QSAxis::init_axis_mappings()
 {
  double min = m_vmin;
  double max = m_vmax;
  if ( m_scale==LogScale ) {
         min = _log( min, m_base );
         max = _log( max, m_base );
        }

  m_wmin   = min/(max-min);
  m_wscale = 1.0/(max-min);

  if ( m_scale==LogScale ) m_wscale = m_wscale/log10(m_base);	
 }

//-------------------------------------------------------------//

void QSAxis::find_closest( double *valM, int *valE, const double MSD[], int MSD_number )
// Rounds valM*10^valE to closest value that have the most significant digit
// as in table MSD. MSD must contain numbers in range < 1.0, 10.0 ). Numbers
// with more than one digit are allowed.
 {
  double d1, d2;
  double resultM;
  int    resultE;

  int    i;

  if ( MSD_number > 0 ) {
         double minMSD = MSD[0];
         double maxMSD = MSD[0];

         for ( i=0; i< MSD_number; i++ ) {
                if ( MSD[i] < minMSD ) minMSD = MSD[i];
                if ( MSD[i] > maxMSD ) maxMSD = MSD[i];
               }

         // check max(MSD[]) * 10^(valE-1)
         d1 = fabs(*valM - 0.1*maxMSD);
         resultM =  maxMSD;
         resultE = *valE-1;

         // check all values: MSD[]*10^valE
         for( i=0; i < MSD_number; i++ ) {
                d2 = fabs(*valM - MSD[i]);
                if ( d2 < d1 ) {
                        d1 = d2;
                        resultM = MSD[i];
                        resultE = *valE;
                       }
              }

         // check min(MSD[]) * 10^(valE+1)
         d2 = fabs(*valM - 10.0*minMSD);
         if ( d2 < d1 ) {
                 resultM =  minMSD;
                 resultE = *valE+1;
                }

         *valM = resultM;
         *valE = resultE;
        }
 }

//-------------------------------------------------------------//

void QSAxis::to_float_point( double value, double *mantissa, int *exponent )
// Decompose 'value' into 'mantissa'*10^'exponent'
// Mantissa is in the range <1.0, 10.0)
// value must be greater than 0.
 {
  if ( value > 0.0 ) {
	int e = (int )floor( log10(value) );
  	if ( mantissa ) *mantissa = value/pow(10.0,e);
  	if ( exponent ) *exponent = e;
	} else {
	if ( mantissa ) *mantissa = value;
  	if ( exponent ) *exponent = 1;
	}

 }

  /*
  int e = 0;
  while( value >= 10.0 )            { e+=1; value/=10.0; }
  while( value < 1.0 && value > 0 ) { e-=1; value*=10.0; }
  */

//-------------------------------------------------------------//

void QSAxis::init_tics()
 {
  m_tics.erase( m_tics.begin(), m_tics.end() );

  //  Default Style for auto-generated tics
  init_auto_tics_values();
  list<QSAxisTic>::iterator curr_tic = m_tics.begin();
  list<QSAxisTic>::iterator end_tic  = m_tics.end();
  while ( curr_tic != end_tic ) {
         curr_tic->m_pos   = dataToWorld(curr_tic->m_value);
	 curr_tic->m_font  = font( TicsFont );
	 curr_tic->m_angle = m_tics_angle;
	 if ( curr_tic->m_major ) {
                curr_tic->m_line = line( MajorGridLine );
		sprintfTic(curr_tic->m_label,curr_tic->m_value);
		} else {
                curr_tic->m_line = line( MinorGridLine );		
		}
	 curr_tic++;
	};

  // Add user tics
  m_last_tic.m_value = m_vmin;
  QSMatrix* tics_values = matrix(TicsChannel);
  QSGFill default_fill; default_fill.color.set( 0, 0, 0 );
  if ( tics_values ) {
	int cols = tics_values->cols();
	if ( cols > 0 )
        for ( int row=0; row<tics_values->rows(); row++ ) {
				// default values
				QSAxisTic new_tic( tics_values->value(row,0) );	
 				new_tic.m_pos   = dataToWorld( new_tic.m_value );
	 			new_tic.m_angle = m_tics_angle;
	 			new_tic.m_major = true;
				sprintfTic( new_tic.m_label, new_tic.m_value);

                                // user provided values ( if any )
				if ( cols > 1 && !tics_values->string(row,1).isEmpty() ) sprintfTic( new_tic.m_label, new_tic.m_value, tics_values->string(row,1) );
				if ( cols > 2 && !tics_values->string(row,2).isEmpty() ) new_tic.m_major = (bool )tics_values->value(row,2);
				if ( cols > 3 && !tics_values->string(row,3).isEmpty() ) new_tic.m_angle = (int )tics_values->value(row,3);				
			
				new_tic.m_line = lineFromData( matrix(LineStyles), row, 0, line( new_tic.m_major ? MajorGridLine : MinorGridLine ) );
				new_tic.m_font = fontFromData( matrix(FontStyles), row, 0, font( TicsFont )     );
				new_tic.m_fill = fillFromData( matrix(FillStyles), row, 0, default_fill ); 			

				// use provided fill or plot's default fill
				if (  matrix(FillStyles) &&
				      matrix(FillStyles)->rows() > row &&
				      matrix(FillStyles)->cols() > 0 &&
				     !matrix(FillStyles)->string(row,0).isEmpty() ) new_tic.m_is_fill_defined = true;
				
				// Only fill is important - how to fill area above the last level
				if ( new_tic.m_value >= m_vmin && new_tic.m_value <= m_vmax ) m_tics.push_back( new_tic );
				else if ( new_tic.m_value > m_vmax && ( new_tic.m_value < m_last_tic.m_value || m_last_tic.m_value <= m_vmax) ) m_last_tic = new_tic;
				}
       }
  // Sort tics
  if ( !is_sorted(m_tics.begin(),m_tics.end()) ) m_tics.sort();

 }

//-------------------------------------------------------------//


void QSAxis::init_auto_tics_values()
// TODO:
// adjust when user tics are given,
 {
  double majnumber = m_majd;
  double minnumber = m_mind;

  // no tics
  if ( majnumber == 0.0 ) {
	return;
	}


  //******************************************
  //*             tics steps                 *
  //******************************************

  if ( majnumber > 0.0 ) {
	// minor tics
	if ( minnumber != 0.0 ) {
 		minnumber = fabs(minnumber);
		for( double i=ceil(m_vmin/minnumber)*minnumber; i<m_vmax; i+=minnumber )
			m_tics.push_back( QSAxisTic(i,false) );
		}
	// major tics
        for( double i=ceil(m_vmin/majnumber)*majnumber; i<m_vmax; i+=majnumber )
		m_tics.push_back( QSAxisTic(i,true) );
	return;
	}

  //******************************************
  //*             tics density               *
  //******************************************

  majnumber = fabs(majnumber);
  minnumber = fabs(minnumber);

  double max;
  double min;

  if ( m_scale == LinearScale ) {
        max = m_vmax/m_base;
        min = m_vmin/m_base;
       } else {
        max = _log( m_vmax, m_base );
        min = _log( m_vmin, m_base );
       }

  double M;
  int E;

  // preferred distance - in the next step it wiil be rounded
  // to find some good-looking value.
  double d = ( max - min ) / majnumber;

  double base = m_base;
  AxisScale s = m_scale;

  //
  //  log scale that will have linear tics
  //
  // if step is lower than 1 and no tic base^integer is visible
  // use an ordinary linear grid
  // 11 (11.2) (11.4) (11.6) (11.8) 12
  if ( s == LogScale && d <= 1.5 && floor(max) < min ) {

         base = 10.0;

         s   = LinearScale;
         d   = (pow( base, max ) - pow( base, max-d )) / base;
         max = pow( base, max ) / base;
         min = pow( base, min ) / base;
        }

  //
  // ------------ Log-Linear (logit ) scale -------------
  //
  // if step is one, or is lower than 1 but at least
  // one tic base^integer is visible.
  // (10^1) 20 30 40 50 60 70 80 90 (10^2) 100 200 300 400 500 ...
  if ( s == LogScale && d <= 1.5 && floor(max) >= min ) {

         // fill the list
         // major step is always base^1
         if ( m_round ) {
                 max =  ceil( max );
                 min = floor( min );

                 m_vmin = pow( base, min );
                 m_vmax = pow( base, max );
                }

         int minor = 0;
         int step  = 1;

         // try to adjust the minor tics number according to
         // the given value of minnumber
         if ( minnumber>0.0 ) {
                 minor = int(ceil(base))-1;
                 step  = int(floor( minor * (max-min) / minnumber + 0.5 )); if ( step < 1 ) step = 1;
                 minor = minor/step;
                }

         for ( double i = floor(min); i<=max; i+=1.0 ) {

                 if ( i >= min ) m_tics.push_back( QSAxisTic(pow(base,i)) );

                 if ( minnumber ) {
                        to_float_point( pow(base,i), &M, &E );      // minor tic step
	
	                double mind = floor( M ) * pow( 10.0, E ) * step;
	
                        for( int j=0; j<minor-1; j++ ) {
                                double minpos = pow(base,i)+(j+1.0)*mind;
                                if ( minpos >= m_vmin &&
                                     minpos <= m_vmax )
                                                m_tics.push_back( QSAxisTic(minpos,false) );
                               }
                        }
                }

          return;
        }

  //
  // ----------- Log scale ----------------------------
  //
  // step is greater than 1.
  // scale is of the form - major (minor) (minor) ...
  // 2^10 (2^12) (2^14) (2^16) (2^18) 2^20 ....
  if ( s == LogScale && d >= 1.5 ) {

         double mind = 1.0;

         // minor step - round to integer value
         if ( minnumber > 0.0 ) {
                mind = (max-min) / minnumber;
                mind = floor( mind+0.5 );
                if ( mind < 1.0 ) mind = 1.0;
               }

         int MTN;

         // rounded major step
         if ( d > mind ) { MTN = (int )floor( d/mind + 0.5 ); d = MTN*mind; }
                    else { MTN = 1; d = floor( d + 0.5 ); }

         if ( m_round ) {
                 max =  ceil( max/d ) * d;
                 min = floor( min/d ) * d;
                 m_vmin = pow( base, min );
                 m_vmax = pow( base, max );
                }

         for ( double i = floor(min); i<=max; i+=d ) {
                 if ( i >= min ) m_tics.push_back( QSAxisTic(pow(base,i)) );
		 if ( minnumber )
                 for( int j=0; j<MTN-1; j++ ) {
                         double minpos = i+(j+1)*mind;
                         if ( minpos >= min && minpos <= max )
                                        m_tics.push_back( QSAxisTic(pow(base,minpos),false) );

                        }
                }
          return;
        }

  //
  // -------- Linear scale ------------
  //
  if ( s == LinearScale ) {
         double r[7];

         to_float_point( d, &M, &E );

         double MSN = M;

         if ( E >=0 ) {
                // round if step > 1.0
                // avaliable, the most significant numbers are: 1, 15, 2, 25, 3, 4, 5
                r[0] = 1.0; r[1] = 2.0; r[2] = 3.0; r[3] = 4.0; r[4] = 5.0;
                find_closest( &MSN, &E, r, 5 );
               }  else {
                // round if step < 1.0
                // avialiable, the most significant numbers are: 1, 2, 5
                r[0] = 1.0; r[1] = 2.0; r[2] = 2.5; r[3] = 5.0;
                find_closest( &MSN, &E, r, 4 );
               }
		 	
         double major = MSN*pow( 10.0, E )*base;

         double MTN;  // minor tics per major tic

         if ( minnumber > 0.0 ) {
                // Calculate preferred minor tick distance in data units
                double mind = (max-min) / minnumber ; if ( mind <= 0.0 ) mind = 1.0; // assert( mind > 0.0 );

                // minor tics number per major tick
                to_float_point( MSN*pow( 10.0, E )/mind, &MTN, &E );

                // round M to values 1, 2, 5, 2*MSN
                r[0] = 1.0; r[1] = 2.0; r[2] = 2.0*MSN; r[3] = 5.0;
                find_closest( &MTN, &E, r, 4 );
               } else {
                // no minor tics
                E = -1;
               }

         // no minor tics
         if ( E < 0 ) { MTN = 1.0; E = 0; }

         int minor = int(floor(MTN) * pow(10.0, E))-1;

         if ( m_round ) {
                 m_vmin = floor( m_vmin/major ) * major;
                 m_vmax =  ceil( m_vmax/major ) * major;
		 init_axis_mappings();
                }

         // fill the list
	 double tic_first = floor(m_vmin/major)*major;
	 double tic_last  = m_vmax;
	 double tic_value = tic_first;
	 int tic_number   = (int )ceil((tic_last-tic_first)/major)+1;
         for ( int i=0; i<tic_number; i++ ) {
		if ( tic_value >= m_vmin && tic_value <= m_vmax ) m_tics.push_back( QSAxisTic(round_tic(tic_value)) );
		for( int j=0; j<minor; j++ ) {
                         double tic_minor_value = tic_value+(j+1.0)*major/(minor+1.0);
                         if ( tic_minor_value >= m_vmin && tic_minor_value <= m_vmax ) m_tics.push_back( QSAxisTic(round_tic(tic_minor_value),false) );
                        }
		tic_value += major;
                }
        }
 }

//cout << "adding tic " << round_tic(i) << " number " << QString::number((tic_to-tic_from)/major,'g',16) << endl;

//-------------------------------------------------------------//

double QSAxis::round_tic( double v )
 {
  double almost_zero = (m_vmax-m_vmin) * 10e-14;
  return fabs(v)<almost_zero?0.0:v;
 }

//-------------------------------------------------------------//

void QSAxis::sprintfTic( QString& buffer, double value, const QString& format  )
 {
  double v[4];

  // 0 - value
  // 1 - factor
  // 2 - base
  // 3 - exponent
  v[0] = value;
  v[2] = m_base;
  v[1] = (m_scale != LinearScale) ? 1.0 : value/v[2];
  v[3] = (m_scale == LinearScale) ? 1.0 : _log(value,v[2]);

  sprintfTic( buffer, format==QString::null?m_tics_format:format, v[0], v[1], v[2], v[3] );
 }

//-------------------------------------------------------------//

void QSAxis::sprintfTic( QString& buffer, const QString& tic_format, double value, double factor, double base, double exponent )
 {

  buffer = tic_format;

  int pos = 0;
  int len = buffer.length();

  while ( pos < len ) {
        int beg = buffer.find( '(', pos ); if ( beg == -1 ) break;
        int end = buffer.find( ')', beg ); if ( end == -1 ) break;

        QString format = buffer.mid( beg, end-beg+1 );

        if ( format.find("TIC",0) != -1 ) {
                 int mpos = 0;

                 //
                 // Default values
                 //
                 QString type("V");
                 QString cformat("%g");

                 //
                 //  Check type:
                 //   V -value,
                 //   B - base,
                 //   F - factor,
                 //   X - exponent
                 //
                 if ( (mpos=format.find(QRegExp("TYPE=\"[VFBX]\""),0)) != -1 )
                         type = format.mid(mpos+6,1);

                 //
                 // C-styled formatting codes.
                 //
                 QRegExp cstyle("FORMAT=\"%[\\-\\+\\s]*[0-9]*\\.?[0-9]*[fFgGeE]\"");
                 if ( (mpos=format.find(cstyle,0)) != -1 )
                         cformat = format.mid( mpos+8, format.find("\"",mpos+8)-mpos-8 );


                 //
                 // Replace a marker vith a C-formatted value
                 //
                 QString result;

                 if ( type == "F" ) result.sprintf( cformat, factor );
                 else if ( type == "B" ) result.sprintf( cformat, base );
                 else if ( type == "X" ) result.sprintf( cformat, exponent );
                 else result.sprintf( cformat, value );

                 buffer.replace( beg, end-beg+1, result );
                 end = beg + result.length() - 1;
                }

        pos = end+1;
       }

  }

//-------------------------------------------------------------//

void QSAxis::rememberCurrentView( int index )
 {
  if ( index >=0 && index < 3 ) {
   	rememberedViews[index].min = m_min;
	rememberedViews[index].max = m_max;
	rememberedViews[index].base = m_base;
	rememberedViews[index].scale = m_scale;
	rememberedViews[index].round = m_round;
	rememberedViews[index].reversed = m_reversed;
	}
 }

//-------------------------------------------------------------//

void QSAxis::setRememberedView( int index )
 {
  if ( index >=0 && index < 3 ) {
   	setRange( rememberedViews[index].min, rememberedViews[index].max );
	setScaleType( rememberedViews[index].scale );
	setScaleBase( rememberedViews[index].base );
	setRoundRangeToTicStep( rememberedViews[index].round );
	setReversed( rememberedViews[index].reversed );
	}
 }

//-------------------------------------------------------------//

void QSAxis::loadStateFromStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSAxesChild::loadStateFromStream( stream, factory );
 }

//-------------------------------------------------------------//

void QSAxis::saveStateToStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSAxesChild::saveStateToStream( stream, factory );
 }

//-------------------------------------------------------------//

QSAxis::ColumnType QSAxis::columnType( int channel, int column ) const
 {
 }





