"""Low-level interface to KMatplot.

Does roughly the same as interface/common.c.
All functions are members of class kmpProcess. The constructor starts and
connects to a KMatplot session.
"""

__version__ = 0.3
__author__ = 'Martin Wiechert <martin.wiechert@gmx.de>'
__date__ = 'December 19, 2001'

import socket
import os
import struct
import tempfile
import commands
import paths

class Error (Exception):
    pass

# Build environment for kmatplot call.
env = os.environ.copy ()
for name in ('KDEDIR', 'KDEDIRS'):
    if env.has_key (name):
        if not paths.KMPDIR in env [name].split (':'):
            env [name] = env [name] + ':' + paths.KMPDIR
    else:
        env [name] = paths.KMPDIR

SizeOfInt = len (struct.pack ('i', 1))  # FIXME: There must be a better way...

# Enums etc. from interface/msg.h and interface/etype.h.
MsgChannel, MsgProperty, MsgAddAxes, MsgRemoveAxes, MsgAddDataset, \
MsgRemoveDataset, MsgRemoveAllDatasets = range (7)

PlotCurve, PlotImage, PlotContour, PlotSurface, PlotFigure = range (5)

EDouble, EFloat, ELong, EUShort, EShort, EUChar = range (6)

esizeof = 6 * [0]
esizeof [EUChar] = len (struct.pack ('B', 1))
esizeof [EShort] = len (struct.pack ('h', 1))
esizeof [EUShort] = len (struct.pack ('H', 1))
esizeof [ELong] = len (struct.pack ('L', 1))
esizeof [EFloat] = len (struct.pack ('f', 1))
esizeof [EDouble] = len (struct.pack ('d', 1))

# A dictionary Numeric typecodes to 'ETypes'.
etype = {}
etype ['l'] = etype ['i'] = ELong
etype ['s'] = EShort
etype ['1'] = EUChar
etype ['f'] = EFloat
etype ['d'] = EDouble

class header:  # Mimics union hdr_t from interface/msg.h.
    def __init__ (self):
        self.keys = ['type', 'dlen', 'plot', 'axes', 'dnum', 'plen', 'ptype',
                     'chan', 'vlen', 'rows', 'cols', 'etype', 'lineo',
                     'pixelo']
        self.maxkeys = 10 
    def __str__ (self):
        data = [getattr (self, a) for a in self.keys if hasattr (self, a)]
        data += (self.maxkeys - len (data)) * [0]
        return apply (struct.pack, [self.maxkeys * 'i'] + data)

class kmpProcess:
    def __init__ (self):
        s = socket.socket (socket.AF_UNIX, socket.SOCK_STREAM)
        sname = tempfile.mktemp (os.environ ['USER'] + '.kmatplot.py')
        s.bind (sname)
        s.listen (1)
        os.spawnve (os.P_NOWAIT, paths.KMPEXEC,
                    [paths.KMPEXEC, '--fd', str (s.fileno ())], env)
        s.close ()
        self.socket = socket.socket (socket.AF_UNIX, socket.SOCK_STREAM)
        self.socket.connect (sname)

    def get_reply_code (self):
        return struct.unpack ('i', self.socket.recv (SizeOfInt)) [0]

    def add_axes (self, axes_type):
        hdr = header ()
        hdr.type = MsgAddAxes
        hdr.dlen = 0
        hdr.plot = -1
        hdr.axes = axes_type
        self.socket.send (str (hdr))
        return self.get_reply_code ()

    def remove_axes (self, axes_id):
        hdr = header ()
        hdr.type = MsgRemoveAxes
        hdr.dlen = 0
        hdr.plot = axes_id
        hdr.axes = -1  # Dummy.
        self.socket.send (str (hdr))
        return self.get_reply_code ()

    def add_dataset (self, plot, type):
        hdr = header ()
        hdr.type = MsgAddDataset
        hdr.dlen = 0
        hdr.dnum = -1
        hdr.plot = plot
        hdr.ptype = type
        self.socket.send (str (hdr))
        return self.get_reply_code ()
        
    def remove_dataset (self, plot, number):
        hdr = header ()
        hdr.type = MsgRemoveDataset
        hdr.dlen = 0
        hdr.dnum = number
        hdr.plot = plot
        hdr.ptype = -1  # Dummy.
        self.socket.send (str (hdr))
        return self.get_reply_code ()
        
    def remove_all_datasets (self, plot):
        hdr = header ()
        hdr.type = MsgRemoveAllDatasets
        hdr.dlen = 0
        hdr.dnum = -1  # Dummy.
        hdr.plot = plot
        hdr.ptype = -1  # Dummy.
        self.socket.send (str (hdr))
        return self.get_reply_code ()

    def set_channel (self, plot, dataset, channel, data):
        hdr = header ()
        hdr.type = MsgChannel
        s = data.shape
        if len (s) == 1:
            s = (s[0], 1)
        hdr.rows, hdr.cols = s
        hdr.etype = etype [data.typecode ()]
        hdr.pixelo = esizeof [hdr.etype]
        hdr.lineo = hdr.cols * hdr.pixelo
        hdr.dlen = hdr.rows * hdr.cols * hdr.pixelo
        hdr.dnum = dataset
        hdr.plot = plot
        hdr.chan = channel
        self.socket.send (str (hdr))
        self.socket.send (data.tostring ())
        return self.get_reply_code ()
    
    def set_property (self, plot, property, value):
        # Not functional yet.
        hdr = header ()
        hdr.type = MsgProperty
        hdr.plen = len (property)
        hdr.vlen = len (value)
        hdr.dlen = hdr.plen + hdr.vlen
        hdr.plot = plot
        self.socket.send (str (hdr))
        self.socket.send (property + value)  # FIXME: verify syntax
        return self.get_reply_code ()
        
    
