/***************************************************************************
                          	mpsymbol.h
                             -------------------
    begin                : Tue Nov 20 2001
    copyright            : (C) 2001 by Kamil
    email                : kamil@localhost.localdomain
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef MPSYMBOL_H
#define MPSYMBOL_H

#include<deque.h>

//---------------------------------------------------------------------------------------------//

class MPError;
class MPSymbol;
class MPSymbolList;

/**
  * General symbol class, it can be function or variable and generally is implemented as
  * a matrix, but can be a scalar if isScalar() returns true. You get matrix arguments and have to reimplement value()
  * to return function value and you can reimplement rows() and cols() to return a resulting matrix size.
  * Current implementation of rows() and cols() returns a size equal to a size of arguments
  * You can also reimplement checkArgs() to check a number and size of all arguments and do other init stuff.
  * Default implementation returns error if the size of all arguments is not the same.
  * Remember that the scalar values should be treaten in a different way. If you
  * are writting, for example, ADD function which looks like this: ADD( arg1, arg2 ), you
  * have to check ( in checkArgs() ) whether there are exactly two arguments and if they have the
  * same size, but scalars have the size equal to any other size, like in following expression: [ 2 3 0 ] + 2 = [ 4 5 2 ]
  * If two following matrices are added [ 2 3 ] + [3 4 5 ] = 'error: nonconformant arguments' should be raised
  * @author Kamil
  */
class MPSymbol {
  public:
	/**
	  * Constructor. It takes a list of arguments ( can contain 0 elements ), and
	  * position of this expression in the formula string ( used to build error messages ).
	  * Identifier is copied via strdup()
	  */
	MPSymbol( MPSymbolList *args, int columnFrom, int columnTo, const char *identifier );
	/**
	  * Destructor
	  */
	virtual ~MPSymbol();
	/**
          * Sets arguments. Usually you won't need it as you will be passing arguments to constructor
          */
	virtual void setArgs( MPSymbolList *args );
	/**
	  * Checks if there are no syntax errors: if there are correct number of arguments,
	  * arguments have desired size, etc. Default implementation check if all args have
	  * the same size and sets the size of this expression to it. You can omit checking if
	  * MPError::hasError() is true, because only a first error is recordered. Rememeber
	  * to call firts an original implementation of this method which calls checkArgs() for all arguments.
	  */
	virtual void checkArgs( MPError& info );
	/**
	  * Returns a value of this expression for 'row' and 'col'. If this expression is a
	  * scalar value it has to return the same value for all 'row' and 'col' arguments.
	  * See General usage section for details.
	  * Return NaN ( =sqrt(-1.0) ) in the case of error.
	  *  Examples:
	  *    value( int row, int col ) { return args[0]->value(row,col)+args[1]->value(row,col); }.
	  *    value( int row, int col ) { return sin( args[0]->value(row,col) ); }.
	  */
	virtual double value( int row, int col ) = 0;
	/**
	  * Rows. Scalar functions must have only one row.
	  */
	int rows() const { return m_rows; }
	/**
	  * Columns. Scalar function must have only one columns.
	  */
	int cols() const { return m_cols; }
	/**
	  * Returns true if rows == 1 and cols == 1. A scalar function
	  * must return the same value for all arguments 'row' and 'col'.
	  * See value()
	  */
	bool isScalar() const { return rows() == 1 && cols() == 1; }
	/**
          * Returns true if symbol is constant - it means that no arguments will
	  * be passed to it. If this method returns false the symbol is a function
	  * and it must be called with parenthesis, even if it accepts no arguments.
	  * For example: 'pi' - is a constant, 'rand()' is a function.
	  * Default implementation returns false.
          */
	virtual bool isVariable() const { return false; }
	/**
	  * Returns an identifier of this expression
	  * ( it is a name of funtion or variable: 'sin, 'cos', 'pi' ).
          * Example implementation:
	  * virtual const char *identifier() const { return "sin"; }
	  */
	const char *identifier() const { return m_id; }
	/**
	  * Position of the expression in the formula string. Used to
	  * to build error messages.
	  */
	int colFrom() const { return m_col_from; }
	/**
	  * Position of the expression in the formula string. Used to
	  * to build error messages.
	  */
	int colTo() const { return m_col_to; }
	/**
	  * Returns true if the size of two expressions is the same ( checking isScalar() also ).
	  */
	static bool isSizeEqual( const MPSymbol *expression1, const MPSymbol *expression2 );

  protected:
	int m_col_from;
	int m_col_to;
	int m_rows;
	int m_cols;
	MPSymbolList *m_args;
	char *m_id;
 };

//---------------------------------------------------------------------------------------------//

/**
  * Expression list. It is usualy used as an argument list for expressions.
  * For example in: sheet( 1:10, 2:3 ) - expression list passed as argument list
  * for 'sheet' contains two  expressions '1:10' and '2:3'. In this example : sin(3.14)
  * argument list for 'sin' contains only one expression single value = '3.14'.
  */
class MPSymbolList {
    public:
	/**
	  * Constructor
	  */
	MPSymbolList();
	/**
	  * Constructor of the single element list.
	  */
	MPSymbolList( MPSymbol *expression );
	/**
	  * Constructor of the two element list.
	  */
	MPSymbolList( MPSymbol *expression1, MPSymbol *expression2 );
	/**
	  * Destructor of the two element list.
	  */
       virtual ~MPSymbolList();
	/**
	  * Add a new expression to the list. All expression are deleted in destructor,.
	  */
	void add( MPSymbol *expression );
	/**
	  * Returns a number of expressions.
	  */
	int count() const { return (int )m_list.size(); }
	/**
	  * Returns expression at position 'number' on this list.
	  */
	MPSymbol *at( int number ) const { return m_list[number]; }

   protected:
	deque<MPSymbol*> m_list;
 };

//--------------------------------------------------------------------------------------------//

/**
  * Error info object
  */
class MPError {
      public:
	enum ErrorType { None = 0, WrongNumberOfArgs, NonconformantArgs, UndefinedSymbol, ParseError, Custom };
	/**
	  * Constructor
	  */
	MPError();
	/**
	  * Destructor
	  */
	virtual ~MPError();
	/**
	  * Wrong number of arguments.
	  */
	virtual void setWrongNumberOfArguments( int currArgs, int correctArgs, MPSymbol *sym );
	/**
	  * Nonconformant argument
	  */
	virtual void setNonconformantArgument( int arg_nr, MPSymbol *arg, MPSymbol *sym );
	/**
	  * Undefined symbol
	  */
	virtual void setUndefinedSymbol( const char *symbol, int colFrom, int colTo );
	/**
	  * Parse error.
	  */
	virtual void setParseError( int colFrom, int colTo );
	/**
	  * Sets a custom error message.
	  */
	virtual void setError( const char *message, MPSymbol *sym );
	/**
	  * Returns 'true' if there is an error already set in this object.
	  * Only the first error is recordered and and further setError calls
          * have no effect.
	  */
	bool hasError() const { return m_type != None; }
	/**
	  * Column from
	  */
	int colFrom() const { return m_col_from; }
	/**
	  * Column to
	  */
	int colTo() const { return m_col_to; }

   protected:
	ErrorType m_type;
	int m_col_from;
	int m_col_to;
   };

//---------------------------------------------------------------------------------------------//

/**
  * Expression factory is used by the parser to create expression objects based on their identifiers.
  */
class MPSymbolFactory {
  public:
	/**
	  * Constructor
	  */
	MPSymbolFactory();
	/**
	  * Destructor
	  */
	virtual ~MPSymbolFactory();
	/**
	  * Create expression with given identifier ( for exaple 'sin', 'cos', 'pi' etc )
	  * and arguments. Return NULL if this factory does not provide requested object.
	  * UNICODE symbols not supported yet
	  */
	virtual MPSymbol *create( const char *identifier, MPSymbolList *args, int colFrom, int colTo ) = 0;
	/**
	  * Return a total number of objects providen by this factory.
	  */
	virtual	int symbolCount() const = 0;
	/**
	  * Return an identifier of an object number 'expressionNumber'.
	  */
	virtual const char *symbolIdentifier( int symbolNumber ) = 0;
	/**
	  * Return a short ( few sentences ) description of an object number 'expressionNumber'.
	  */
	virtual const char *symbolDescription( int symbolNumber ) = 0;
	/**
	  * Name of this factory
	  */
	virtual const char *name() const { return "Unnamed"; }


 };




//----------------------------------------------------------------------------------------------//

#endif
