/*************************************************/
/* methods for class XDeviceNet                  */
/*                                               */
/* widget containing a circuit                   */
/* actions with/at objects in the net-widget     */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <qfile.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <qstring.h>
#include <qpoint.h>
#include <qpaintdevice.h>

#include "klogic.h"
#include "klogicIO.h"
#include "mainw.h"
#include "netw.h"
#include "xnet.h"
#include "xwire.h"
#include "devLib.h"
#include "selection.h"
#include "xmlExport.h"
#include "deviceTypes.h"
#include "deviceMuxDemux.h"
#include "deviceOsc.h"
#include "deviceRAM.h"
#include "deviceMatrix.h"
#include "deviceJKFF.h"

/***************************************************/
/* methods of XDeviceNet                           */
/***************************************************/
int XDeviceNet::instance = 0;
int XDeviceNet::magic_version = 0;	// to be removed soon!

XDeviceNet::XDeviceNet()
	:XDevice(DeviceType::fNET, 0, 0)
{

	instance++;

	_parent = (XDeviceNet *)NULL;
}

XDeviceNet::XDeviceNet(XDeviceNet *parent, int x, int y)
	: XDevice(DeviceType::fNET, x, y)
{

	instance++;

	_parent = parent;
}

XDeviceNet::XDeviceNet(XDeviceNet *parent, int x, int y, int size)
	: XDevice(DeviceType::fNET, x, y, size)
{

	instance++;

	_parent = parent;
}

XDeviceNet::~XDeviceNet()
{
	instance--;

	KlogicList<XDevice> *d = netdev.First();
	KlogicList<XWire> *w = netwire.First();
	KlogicList<XDeviceNet> *n = netnet.First();

	// destroy all subcircuits
	while(n) {
		delete n->Get();
		netdev.Destroy((XDevice *)(n->Get()));
		n = n->Next();
	}
	// destroy all devices
	while(d) {
		delete d->Get();
		d = d->Next();
	}
	// destroy all wires
	while(w) {
		delete w->Get();
		w = w->Next();
	}
	// destroy lists
	netnet.Destroy();
	netdev.Destroy();
	netwire.Destroy();
	netin.Destroy();
	netout.Destroy();

	initImport();
}

bool XDeviceNet::hasImpossibleEquations()
{
	KlogicList<XDevice> *ld = netdev.First();
	while(ld) {
		if (ld->Get()->type() == DeviceType::fRAM || ld->Get()->isTristate()) return true;
		ld = ld->Next();
	}
	return false;
}

// returns 1 if all names are unique, otherwise 0
int XDeviceNet::checkCircuitNames()
{
	KlogicList<XDevice> *ld1 = netdev.First();
	KlogicList<XDevice> *ld2;

	while(ld1) {
		// sub circuits ...
		XDeviceNet *net = ld1->Get()->devIsNet();
		if (net && !net->checkCircuitNames()) return 0;

		// devices ...
		ld2 = netdev.First();
		while(ld2) {
			if (ld2->Get() != ld1->Get()) {
				if (!strcmp(ld1->Get()->getText(), ld2->Get()->getText())) return 0;
			}
			ld2 = ld2->Next();
		}

		ld1 = ld1->Next();
	}
	return 1;
}

// change all duplicate names to be unique
int XDeviceNet::unifyCircuitNames()
{
	KlogicList<XDevice> *ld1 = netdev.First();
	KlogicList<XDevice> *ld2;

	while(ld1) {
		// sub circuits ...
		XDeviceNet *net = ld1->Get()->devIsNet();
		if (net && !net->checkCircuitNames()) return 0;

		// devices ...
		ld2 = netdev.First();
		while(ld2) {
			if (ld2->Get() != ld1->Get()) {
				if (!strcmp(ld1->Get()->getText(), ld2->Get()->getText())) {
					QString buffer;
					buffer.sprintf("uni%d", ld2->Get()->getID());
					setText(ld2->Get()->getID(), buffer);
				}
			}
			ld2 = ld2->Next();
		}

		ld1 = ld1->Next();
	}
	return 1;
}

KlogicList<OutputInfo> * XDeviceNet::parseCircuit(const char *qualifier, KlogicList<OutputInfo> *ls)
{
	KlogicList<XDevice> *ld = netdev.First();
	char buf[10000];
	XDevice *dev;

	// init level
	if (!ls) {
		ls = new KlogicList<OutputInfo>;
	}

	while(ld) {
		dev = ld->Get();
		XDeviceNet *net = dev->devIsNet();
		if (net) {
			if (qualifier) sprintf(buf, "%s%s.", qualifier, dev->getText());
			else sprintf(buf, "%s.", dev->getText());

			// remove all spaces from the used names!
			char *s;
			while (NULL != (s = strchr(buf, ' '))) *s = '_';

			net->parseCircuit(buf, ls);		// recursive call
		}
		dev->getAllEquations(ls, qualifier);		// device equations
		ld = ld->Next();
	}

	return ls;
}

void XDeviceNet::deleteNet(QPainter *p)
{
	// destroy all subcircuits
	KlogicList<XDeviceNet> *n = netnet.First();
	while(n) {
		//n->Get()->deleteNet(p);
		delete n->Get();
		netdev.Destroy((XDevice *)(n->Get()));
		n = n->Next();
	}
	// destroy all devices
	KlogicList<XDevice> *ld = netdev.First();
	while(ld) {
		deleteDevice(p, ld->Get());
		ld = netdev.First();
	}
	// destroy all wires
	KlogicList<XWire> *lw = netwire.First();
	while(lw) {
		deleteWire(lw->Get());
		lw = netwire.First();
	}
	// destroy lists
	netnet.Destroy();
	netdev.Destroy();
	netwire.Destroy();
	netin.Destroy();
	netout.Destroy();
	Device::resetCounter();
}

XDeviceNet * XDeviceNet::rootParent()
{
	XDeviceNet *top;

	top = this;
	while(top->parent()) top = top->parent();
	return top;
}

XDeviceNet * XDeviceNet::parent()
{
	return _parent;
}

KlogicList<XDevice> *XDeviceNet::devPointer()
{
	return &netdev;
}

int XDeviceNet::empty()
{
	if (!netnet.First() && !netdev.First() &&
	    !netwire.First() && !netin.First() && !netout.First())
		return 1;
	return 0;
}

void XDeviceNet::applyDefaults()
{
	KlogicList<XDevice> *ld = netdev.First();
	XDeviceNet *net;

	setDelay(Device::defDelay());
	setInverted(XDevice::isDefInverted());
	setWidth(XDevice::defWidth());
	setUndef(Device::defUndefined());
	invertTrigger(Device::triggerInverted());
	displayText(XDevice::textDisplayedGlobal());
	setImage();

	while(ld) {
		net = ld->Get()->devIsNet();
		if (net) net->applyDefaults();
		else {
			ld->Get()->setDelay(Device::defDelay());
			ld->Get()->setInverted(XDevice::isDefInverted());
			ld->Get()->setWidth(XDevice::defWidth());
			ld->Get()->setUndef(Device::defUndefined());
			ld->Get()->invertTrigger(Device::triggerInverted());
			ld->Get()->displayText(XDevice::textDisplayedGlobal());
			ld->Get()->setEquation();
			ld->Get()->parseEquation();
			setText(ld->Get()->getID(), ld->Get()->getText());
			ld->Get()->setImage();
		}
		ld = ld->Next();
	}
}

void XDeviceNet::correctIO(int dev_id, int old_id)
{
	XDevice *devi = netin.Get(dev_id);
	XDevice *devo = netout.Get(dev_id);

	if (devi) {
		KlogicList<Value> *lv = named_input.With(old_id);
		if (lv) {
			lv->setText(devi->getText());
			lv->setID1(dev_id);
		}
	}

	if (devo) {
		KlogicList<opStack> *lop = named_output.With(old_id);
		if (lop) {
			lop->setText(devo->getText());
			lop->setID1(dev_id);
		}
	}
}

void XDeviceNet::setText(int dev_id, const QString& new_text)
{
	XDevice *d = netdev.Get(dev_id);
 
	if (d) {
		d->setText(new_text);
		if ((d->type() == DeviceType::fIN || d->type() == DeviceType::fOUT) && !new_text.isEmpty() && new_text.length()) {
			if (d->type() == DeviceType::fIN) {
				KlogicList<Value> *lv = named_input.With(dev_id);
				if (lv) lv->setText(new_text);
				if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
			}
			if (d->type() == DeviceType::fOUT) {
				KlogicList<opStack> *lop = named_output.With(dev_id);
				if (lop) lop->setText(new_text);
				if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
			}
		}
	}
}

void XDeviceNet::garbageCollection(QPainter *p)
{
	KlogicList<XWire> *lw = netwire.First();
 
	while(lw) {
		lw->Get()->Wire::garbageCollection();
		if (!wireOK(p, lw->Get())) {
			lw = netwire.First();
		} else {
			lw = lw->Next();
		}
	}
}

//*************************************************************************
//* drawing methods
//*************************************************************************

// erase device, then draw all
void XDeviceNet::drawDev(QPaintDevice *paintd, QPainter *p, XDevice *dev)
{
	// erase device and its connected wires
	dev->erase(p);
	// redraw all
	drawAll(paintd, p);
}

// draw all devices and wires in the net
// if paintd is NULL then drawing device is a printer
void XDeviceNet::drawAll(QPaintDevice *paintd, QPainter *p)
{
	KlogicList<XDevice> *ld = netdev.First();
	KlogicList<XWire> *lw = netwire.First();

	while(ld) {
		if (paintd) ld->Get()->drawImage(paintd, p);
		else ld->Get()->drawImage(p);
		ld = ld->Next();
	}
	while(lw) {
		if (paintd) {
			lw->Get()->drawImage(p);
		}
		else {
			// printer output
			lw->Get()->Wire::setColor(Qt::black);
			lw->Get()->drawImage(p);
			lw->Get()->setColor(true);
		}
		lw = lw->Next();
	}
}

// force setting status dependant images
void XDeviceNet::forceOutputChange()
{
	Device::forceOutputChange();
	KlogicList<XDevice> *ld = netdev.First();
	while(ld) {
		ld->Get()->forceOutputChange();
		ld = ld->Next();
	}
}

// draw all wires with its actual potential
// draw all devices with a changed image
// returns wether a runtime shortcut occured (WSHORT) or not (0)
int XDeviceNet::drawStatus(QPaintDevice *paintd, QPainter *p)
{
	int ret_code = 0;
	int ret_tmp;

	// some devices like 7S or LED
	KlogicList<XDevice> *ld = netdev.First();
	while(ld) {
		// if image changes..
		if (ld->Get()->setColor()) {
			// ..draw device
			ld->Get()->drawImage(paintd, p);
		}
		ld = ld->Next();
	}

	// draw all wires in color depending of device-output
	KlogicList<XWire> *lw = netwire.First();
	while(lw) {
		// if color changes..
		if (0 != (ret_tmp = lw->Get()->setColor())) {
			// ..draw wire
			lw->Get()->drawImage(p);
			// .. runtime shortcut detection
			if (ret_tmp == WSHORT) ret_code = WSHORT;
		}
		lw = lw->Next();
	}

	// .. runtime shortcut detection
	return ret_code;
}

void XDeviceNet::setUndef(int undef)
{
	KlogicList<XDevice> *ld = netout.First();
	while(ld) {
		ld->Get()->setUndef(undef);
		ld = ld->Next();
	}
	Device::setUndef(undef);
}

//*************************************************************************
//* device methods
//*************************************************************************

// create new device
// if import_size is != -1, its an import!
XDevice * XDeviceNet::newDevice(int function, int x, int y, int import_size)
{
	// instanciate lib device
	if (function >= DeviceType::fFIRST_LIBDEV && function <= DeviceType::fLAST_LIBDEV) {
		if (MainWidget::activeLib) {
			XDevice *dev = MainWidget::activeLib->createInstance(this, function, x, y);
			if (dev && dev->devIsNet()) return dev->devIsNet();
			return dev;
		}
		return (XDevice *)NULL;
	}

	// new device
	XDevice *dev;
	XDeviceNet *net;

	switch(function) {
		case DeviceType::fNET:
			// new net device
			if (import_size == -1) net = new XDeviceNet(this, x, y);
			else net = new XDeviceNet(this, x, y, import_size);
			netnet.Append(net, net->getID());
			dev = (XDevice *)net;
			break;
		case DeviceType::fRAM:
			dev = new RAMDevice(x, y, import_size);
			break;
		case DeviceType::fOSC:
			dev = new OscillatorDevice(x, y, import_size);
			break;
		case DeviceType::fIN:
			// append named input/output to this device
			dev = new XDevice(function, x, y, import_size);
			if (import_size == -1) {
				QString text = dev->getText();
				if (ADDNAMED_OK > addInputName(text, -1, dev->getID())) {
					delete dev;
					return (XDevice *)NULL;
				}
			}
			netin.Append(dev, dev->getID());
			break;
		case DeviceType::fOUT:
			dev = new XDevice(function, x, y, import_size);
			if (import_size == -1) {
				QString text = dev->getText();
				if (ADDNAMED_OK > addOutputName(text, -1, dev->getID())) {
					delete dev;
					return (XDevice *)NULL;
				}
			}
			netout.Append(dev, dev->getID());
			dev->setUndef(undef());
			break;
		case DeviceType::fBUS:
			dev = new BusDevice(x, y, import_size);
			break;
		case DeviceType::fMATRIX:
			dev = new MatrixDevice(x, y, import_size);
			break;
		case DeviceType::fJK:
			dev = new JKFFDevice(x, y, import_size);
			break;
		default:
			dev = new XDevice(function, x, y, import_size);
			break;
	}
	netdev.Append(dev, dev->getID());
	return dev;

}

// virtual
// returns pointer to the net-device if device is a net-device
XDeviceNet * XDeviceNet::devIsNet()
{
	return this;
}

// get subnet-pointer by name (used by import methods)
XDeviceNet * XDeviceNet::subNet(QString& s)
{
	KlogicList<XDeviceNet> *ln;
	int epos;
	QString sid;

	epos = s.find(DLM, 0);
	if (epos == -1) epos = s.length();
	sid = s.mid(1, epos - 1);

	ln = netnet.With(devmap.mapID(sid));
	if (ln) return ln->Get();
	return (XDeviceNet *)NULL;
}

// look for a device containing point
XDevice * XDeviceNet::containsDev(QPoint p)
{
	KlogicList<XDevice> *l = netdev.First();

	while(l) {
		if (l->Get()->contains(p))
			return l->Get();
		l  = l->Next();
	}
	return (XDevice *)NULL;
}

// look for a device containing device
XDevice * XDeviceNet::containsDev(XDevice *dev)
{
	KlogicList<XDevice> *l = netdev.First();

	while(l) {
		if (l->Get() != dev) {
			if (l->Get()->contains(dev->getPos())) {
				return l->Get();
			}
		}
		l  = l->Next();
	}
	return (XDevice *)NULL;
}

// mark all devices in rect as selected
void XDeviceNet::selectDev(QRect r, KlogicList<XDevice> *d)
{
	KlogicList<XDevice> *l = netdev.First();

	while(l) {
		if (l->Get()->contains(r)) {
			if (!d->With(l->Get())) d->Append(l->Get());
			l->Get()->select(true);
		}
		else {
			d->Destroy(l->Get());
			l->Get()->select(false);
		}
		l = l->Next();
	}
}

// called from class Selection
// add all selected devices to the selection (paste operation)
void XDeviceNet::grabSelection(Selection *sel)
{
	KlogicList<XDevice> *ld = netdev.First();
	KlogicList<XWire> *lw = netwire.First();

	while(ld) {
		if (ld->Get()->isSelected()) sel->add(ld->Get());
		ld = ld->Next();
	}
	while(lw) {
		if (lw->Get()->isSelected()) sel->add(lw->Get());
		lw = lw->Next();
	}
}

// delete a device
void XDeviceNet::deleteDevice(QPainter *p, XDevice *dev)
{
	if (dev->type() == DeviceType::fIN) {
		netin.Destroy(dev);
		removeInputName(dev->getID());
	}
	if (dev->type() == DeviceType::fOUT) {
		netout.Destroy(dev);
		removeOutputName(dev->getID());
	}

	// erase bitmap from screen
	dev->erase(p);

	// delete device reference in list
	XDeviceNet *net = dev->devIsNet();
	if (net) {
		net->deleteNet(p);
		netnet.Destroy(net);
	}
	netdev.Destroy(dev);

	// delete device, in destructor: delete connections
	if (net) delete net;
	else delete dev;
}

//*************************************************************************
//* wire methods
//*************************************************************************

// look for a wire containing point
XWire * XDeviceNet::containsWire(QPoint p)
{
	KlogicList<XWire> *l = netwire.First();

	while(l) {
		// if wire contains point, the node will be locked automaticly
		// if not, a new node is created
		if (l->Get()->createNode(p)) return l->Get();
		l = l->Next();
	}
	return (XWire *)NULL;
}

// look for a wire containing (parts of) rect and select them
void XDeviceNet::selectWire(QRect r, KlogicList<XWire> *w)
{
	KlogicList<XWire> *l = netwire.First();

	while(l) {
		// catch nodes, mark them as selected
		if (l->Get()->select(r)) {
			if (!w->With(l->Get())) w->Append(l->Get());
		}
		else {
			w->Destroy(l->Get());
		}
		l = l->Next();
	}
}

int XDeviceNet::removeNode(QPainter *p, QPoint pt)
{
	KlogicList<XWire> *l = netwire.First();
	XWire *w1, *w2;
	int ret = 0;
	int nextl = 1;

	// can remove more than one node if required
	while(l) {
		w1 = l->Get();
		if (w1->contains(pt)) {
			w1->erase(p);
			w2 = w1->removeNode();
			if (!wireOK(p, w1)) {
				nextl = 0;
				l = netwire.First();
			}
			if (w2 && !wireOK(p, w2)) {
				nextl = 0;
				l = netwire.First();
			}
			ret++;
		}
		if (nextl) l = l->Next();
		else nextl = 1;
	}
	return ret;
}

// look for a device or wire, connected to given wire
// connections will be made automaticly on active node of wire
int XDeviceNet::checkConnection(XWire *w)
{
	KlogicList<XDevice> *ld = netdev.First();
	KlogicList<XWire> *lw = netwire.First();
	int ret;

	// look for device connection/disconnection
	while(ld) {
		if (DOK != (ret = ld->Get()->checkConnection(w))) {
			w->setColor(true);
			return ret;
		}
		ld = ld->Next();
	}
	// look for wire connection/disconnection
	while(lw) {
		if (w != lw->Get()) {
			if (WOK != (ret = lw->Get()->checkConnection(w))) {
				w->setColor(true);
				return ret;
			}
	        }
		lw = lw->Next();
	}
	return NOK;
}

// look for a device, connected to given wire
// (used as import-method only)
int XDeviceNet::checkConnection(int devid, int inverted, XWire *w)
{
	XDevice *d = netdev.Get(devid);
	int ret;

	// look for new device connection
	if (d) {
		if (DOK != (ret = d->checkConnection(w, inverted))) return ret;
	}
	return NFAIL;
}

// create a new wire (import)
XWire *XDeviceNet::newWire()
{
	XWire *w = new XWire();
	netwire.Append(w, w->getID());
	return w;
}

// get a wire (import)
XWire *XDeviceNet::getWire(int id)
{
	return netwire.Get(id);
}

// create a new wire
XWire *XDeviceNet::newWire(QPainter *p, QPoint pt)
{
	XWire *w;
	int ret;

	// create new wire and its first node
	w = new XWire(pt);

	// look for connection in the first point
	ret = checkConnection(w);
	switch (ret) {
		case WCONN:
		case DCONN:
			pt = w->getActive();	// same point, but aligned to the grid
			break;
	}

	// wire possibly has changed
	w->erase(p);
	w->draw(p);

	// lock new wire, create second node
	if (!w->createNode(pt)) {
		delete w;
		warning(NEWWIRE);
		return (XWire *)NULL;
	}
	// append new wire to wirelist
	netwire.Append(w, w->getID());
	return w;
}

// no wires with single nodes allowed
int XDeviceNet::wireOK(QPainter *p, XWire *w)
{
	XWire *w2;

	if (w->countNodes() < 2) {
		w->lockLastNode();
		// erase last locked node
		w->erase(p);
		w2 = w->removeNode();
		if (w2) wireOK(p, w2);	// recursive call
		// remove nodeless wire
		netwire.Destroy(w);
		delete w;
		return 0;
	}
	return 1;
}

// wire released after clicked with the mouse
int XDeviceNet::releaseWire(QPainter *p, XWire *w)
{
	w->Wire::garbageCollection();

	// check, that wire has at leat two nodes
	if (!wireOK(p, w)) return 0;
	w->releaseNode();
	return 1;
}

// delete wire
void XDeviceNet::deleteWire(XWire *w)
{
	netwire.Destroy(w);
	delete w;
}

//*************************************************************************
//* calculation methods
//*************************************************************************

// set net to init-state (flush all calculation-fifos)
void XDeviceNet::flush()
{
	KlogicList<XDevice> *ld = netdev.First();
	while(ld) {
		ld->Get()->flush(0);
		ld = ld->Next();
	}
}

// calculate burst (normally bust is 1)
void XDeviceNet::Burst(int burst)
{	int burst_step;

	switchInterface(IF_INPUT);

	// calculate new values
	for (burst_step = 0; burst_step < burst; burst_step++) {
		Calculate(burst_step);
		Propagate(burst_step);
	}

	switchInterface(IF_OUTPUT);
}

// switch input- or output devices of this
void XDeviceNet::switchInterface(int if_type)
{
	KlogicList<XDevice> *ld;
	KlogicList<XDeviceNet> *ln;
	Device *d;

	// set output-values of input-devices by requesting named inputs of this sub circuit
	// input devices are part of this, and connected to named inputs of this
	if (if_type == IF_INPUT) {
		ld = netin.First();
		while(ld) {
			d = ld->Get();
			// set input device from the named input value
			d->setStaticInput(input(d->getID()));
			ld = ld->Next();
		}
	}

	// set output values of named outputs by requesting output devices
	// output devices are part of this, and named outputs are connected to them
	if (if_type == IF_OUTPUT) {
		ld = netout.First();
		while (ld) {
			d = ld->Get();
			// set the named output value
			setStaticOutput(d->getID(), d->output());
			ld = ld->Next();
		}
	}

	ln = netnet.First();
	while(ln) {
		ln->Get()->switchInterface(if_type);
		ln = ln->Next();
	}
}

// virtual
// calculate all devices of this circuit and all sub circuits
void XDeviceNet::Calculate(int burst_step)
{
	KlogicList<XDevice> *ld = netdev.First();
	while(ld) {
		ld->Get()->Calculate(burst_step);       // recursive by virtual
		ld = ld->Next();
	}
}

// virtual
// propagate values to the outputs of this circuit and all sub circuits
void XDeviceNet::Propagate(int burst_step)
{
	KlogicList<XDevice> *ld;
	ld = netdev.First();
	while(ld) {
		ld->Get()->Propagate(burst_step);	// recursive by virtual
		ld = ld->Next();
	}
}

/***************************************************/
/* import/export operations on a net-widget        */
/***************************************************/
// XML export method for library devices: export to XmlDevice
bool XDeviceNet::exportXML(XmlNet& net)
{
	return exportXML((QTextStream *)NULL, &net, false, 0, 0, 0);
}

// XML export method for files: write directly to the file
bool XDeviceNet::exportXML(QTextStream &ts, bool selected, int level, int dx, int dy)
{
	// top level circuit with selected flag ignored!
	XmlNet net(this, false, level, dx, dy);
	return exportXML(&ts, &net, selected, level, dx, dy);
}

// XML export method
bool XDeviceNet::exportXML(QTextStream *ts, XmlNet *net, bool selected, int level, int dx, int dy)
{
	bool ownNet = (net == (XmlNet *)NULL);

	// circuit parameter and header
	if (ownNet) {
		net = new XmlNet(this, selected, level, dx, dy);
		if (!net->isValid()) {
			delete net;
			return false;
		}
	}

	if (ts) {
		*ts << net->getFieldStart();
		*ts << net->getFieldContent();
		level++;
	}

	// circuit wires
	KlogicList<XWire> *lw = netwire.First();
	while(lw) {
		XmlWire wire(lw->Get(), selected, level, dx, dy);
		if (wire.isValid()) {
			if (ts) *ts << wire.getField();
			else net->addContent(wire.getField());
		}

		lw = lw->Next();
	}

	// circuit devices
	KlogicList<XDevice> *ld = netdev.First();
	XDeviceNet *pNet;
	while(ld) {
		pNet = ld->Get()->devIsNet();
		if (pNet) {
			XmlNet subCircuit(pNet, selected, level, dx, dy);
			if (subCircuit.isValid()) {
				// for sub-circuits always set selected=false!
				pNet->exportXML(ts, &subCircuit, false, level, 0, 0);	// recursion!

				// add net content (if ts avail: content already added)
				if (!ts) net->addContent(subCircuit.getField());
			}
		} else {
			XmlDevice dev(ld->Get(), selected, level, dx, dy);
			if (dev.isValid()) {
				if (ts) *ts << dev.getField();
				else net->addContent(dev.getField());
			}
		}
		ld = ld->Next();
	}

	// circuit wire <-> wire connections
	lw = netwire.First();
	while(lw) {
		XmlWireConnection wire_conn(lw->Get(), selected, level, dx, dy);
		if (wire_conn.isValid()) {
			if (ts) *ts << wire_conn.getField();
			else net->addContent(wire_conn.getField());
		}

		lw = lw->Next();
	}

	// circuit device <-> wire connections
	lw = netwire.First();
	while(lw) {
		XmlDeviceConnection dev_conn(lw->Get(), selected, level, dx, dy);
		if (dev_conn.isValid()) {
			if (ts) *ts << dev_conn.getField();
			else net->addContent(dev_conn.getField());
		}

		lw = lw->Next();
	}

	if (ts) {
		level--;
		*ts << net->getFieldEnd();
	}

	if (ownNet) {
		delete net;
	}
	return true;
}

// proprietary export format, recursively called!
// subnet is true for the very first circuit, if a sub circuit should be stored
// for all lower circuits, subnet is always true!
bool XDeviceNet::exportNet(QTextStream &ts, bool selected, bool subnet, int dx, int dy)
{
	QString s;

	if (subnet) ts << IO_B_SUBNET << "\n";
	else ts << IO_B_NET << "\n";
 
	ts << exportInstance() << "\n";
 
	// write wire position information
	ts << IO_B_WIRE << "\n";
	s = exportWires(selected, dx, dy);
	if (!s.isEmpty()) ts << s << "\n";
	ts << IO_E_WIRE << "\n";
 
	// write device type/position information
	ts << IO_B_DEVICE << "\n";
	s = exportDevices(selected, dx, dy);
	if (!s.isEmpty()) ts << s << "\n";
	ts << IO_E_DEVICE << "\n";
 
	// write subnet information
	KlogicList<XDeviceNet> *ln = netnet.First();
	while(ln) {
		if ((selected && ln->Get()->isSelected()) || !selected) {
			if (!ln->Get()->exportNet(ts, false, true, 0, 0)) {
				return false;
			}
		}
		ln = ln->Next();
	}
 
	// write wire-wire connections
	ts << IO_B_CONNWIRE << "\n";
	s = exportWireConn(selected, dx, dy);
	if (!s.isEmpty()) ts << s << "\n";
	ts << IO_E_CONNWIRE << "\n";
 
	// write wire-device connections
	ts << IO_B_CONNDEVICE << "\n";
	s = exportDevConn(selected, dx, dy);
	if (!s.isEmpty()) ts << s << "\n";
	ts << IO_E_CONNDEVICE << "\n";

        if (subnet) ts << IO_E_SUBNET << "\n";
        else ts << IO_E_NET << "\n";
 
	return true;
}

