# Copyright (c) 2008 Hideki Ikemoto
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

class BoardListView < Qt::TreeWidget
  slots 'load_board_list()'
  slots 'show_board(QTreeWidgetItem*)'
  slots 'add_to_favorite()', 'delete_from_favorite()'
  slots 'copy_url()', 'open_with_web_browser()', 'copy_title_and_url()'

  COLUMN_TITLE = 0
  COLUMN_URL = 1

  def initialize(parent)
    super(parent)

    setRootIsDecorated(true)
    setHeaderLabels(["board name"])

    connect(self, SIGNAL("itemClicked(QTreeWidgetItem*,int)"),
            self, SLOT("show_board(QTreeWidgetItem*)"))

    str = nil
    begin
      str = Downloader.load_menu_list
    rescue
      str = ""
    end
    parse_board_list(str)
  end

  BOARD_REGEXP = /<A HREF=([^ ]*).*>(.*)<\/A>/
  GROUP_REGEXP = /<BR><BR><B>(.*)<\/B><BR>/

  def load_board_list
    str = Downloader.download_menu_list
    parse_board_list(str)
  end

  def parse_board_list(str)
    group_list = []
    current_group = nil
    str.each {|line|
      if GROUP_REGEXP.match(line)
        group_name = Regexp.last_match[1]

        current_group = BoardGroup.new(group_name)
        group_list.push(current_group)
      elsif BOARD_REGEXP.match(line)
        board_url = Regexp.last_match[1]
        board_name = Regexp.last_match[2]

        next unless Util.board_url?(board_url)
        board_id = Util.board_url_to_board_id(board_url)
        board_type = Util.board_url_to_board_type_name(board_url)

        board = Board.new(board_url, board_name)
        BoardManager.instance.register_name(board_url, board_name)
        BoardManager.instance.register_url(board_type, board_id, board_url)
        current_group.push(board) if current_group
      end
    }
    group_list.delete_if {|group|
      group.list.empty?
    }

    clear

    @favorite_item = Qt::TreeWidgetItem.new(self)
    @favorite_item.setText(0, "お気に入り")
    load_favorites

    kitabbs_url = "http://jbbs.livedoor.jp/computer/18420/"
    BoardManager.instance.register_name(kitabbs_url, "Kita板")
    BoardManager.instance.register_url("jbbs.livedoor.jp", "computer_18420", kitabbs_url)
    item = Qt::TreeWidgetItem.new(self)
    item.setText(0, "Kita板")
    item.setText(1, "http://jbbs.livedoor.jp/computer/18420/")

    group_list.each {|group|
      parent_item = Qt::TreeWidgetItem.new(self)
      parent_item.setText(0, group.name)
      group.list.each {|board|
        item = Qt::TreeWidgetItem.new(parent_item)
        item.setText(0, board.title)
        item.setText(1, board.url)
      }
    }
  end

  def show_board(item)
    return if item == nil

    board_url = item.text(COLUMN_URL)
    return if board_url == nil

    ViewMediator.show_board(board_url)
  end

  def contextMenuEvent(event)
    popup = KDE::Menu.new
    popup.addAction("ウェブブラウザで開く", self, SLOT('open_with_web_browser()'))
    popup.addAction("Copy URL", self, SLOT('copy_url()'))
    popup.addAction("Copy title and URL", self, SLOT('copy_title_and_url()'))
    if currentItem.parent == @favorite_item then
      popup.addAction("お気に入りから削除", self, SLOT('delete_from_favorite()'))
    else
      popup.addAction("お気に入りに追加", self, SLOT('add_to_favorite()'))
    end
    popup.exec(event.globalPos())
  end

  def copy_url
    board_url = currentItem.text(COLUMN_URL)
    clipboard = Qt::Application.clipboard
    clipboard.setText(board_url, Qt::Clipboard::Clipboard)
    clipboard.setText(board_url, Qt::Clipboard::Selection)
  end

  def copy_title_and_url
    board_url = currentItem.text(COLUMN_URL)
    board_title = currentItem.text(COLUMN_TITLE)
    clipboard = Qt::Application.clipboard
    cliptext = board_title + "\n" + board_url
    clipboard.setText(cliptext, Qt::Clipboard::Clipboard)
    clipboard.setText(cliptext, Qt::Clipboard::Selection)
  end

  def open_with_web_browser
    board_url = currentItem.text(COLUMN_URL)
    KDE::ToolInvocation.invokeBrowser(board_url)
  end

  def add_to_favorite
    board_url = currentItem.text(COLUMN_URL)
    board_type = Util.board_url_to_board_type_name(board_url)
    board_id = Util.board_url_to_board_id(board_url)
    registered = FavoriteBoards.instance.register(board_type, board_id)

    if registered then
      board_name = BoardManager.instance.get_name(board_url)
      board_item = Qt::TreeWidgetItem.new(@favorite_item)
      board_item.setText(0, board_name)
      board_item.setText(1, board_url)
    end
  end

  def delete_from_favorite()
    board_url = currentItem.text(COLUMN_URL)
    board_type = Util.board_url_to_board_type_name(board_url)
    board_id = Util.board_url_to_board_id(board_url)
    FavoriteBoards.instance.delete(board_type, board_id)

    @favorite_item.removeChild(currentItem)
  end

  def load_favorites
    FavoriteBoards.instance.each {|type, id|
      board_url = BoardManager.instance.get_url(type, id)
      board_name = BoardManager.instance.get_name(board_url)
      board_item = Qt::TreeWidgetItem.new(@favorite_item)
      board_item.setText(0, board_name)
      board_item.setText(1, board_url)
    }
  end
end
