#include "config.h"
/*
 * store.c
 *
 * Very simple store. PIDs are the (negated) offset of the object in the
 * data file. The size of an object is stored preceeding the actual
 * object data in the store. Transfer to and from disk is done with fread
 * and fwrite, so the store can only be used on one platform after it's
 * created.
 *
 */

#ifdef DEBUG
#endif

#include <stdio.h>
#include <stdlib.h>

#include "vm/jni.h"
#include "vm/store.h"
#include "vm/interp_methods.h"

#ifndef TEASEME
static FILE* datafile;
static int firstavail;
#endif

/*
 * Opens an existing persistent store.
 *
 */

int STORE_OpenStore
(
  const char* storename
)
{
#ifdef TEASEME
  return 1;
#else
  datafile = fopen(storename, "rb+");
  if (datafile)
  {
    fread(&firstavail, sizeof(firstavail), 1, datafile);
    return 1;
  }
  else
  {
    return 0;
  }
#endif
}

/*
 * Opens a new persistent store.
 *
 */

int STORE_NewStore
(
  const char* storename
)
{
  
#ifdef TEASEME
  return 0;
#else
  tracePersist("NewStore with %s", storename);
  /* try opening file with that name */
  datafile = fopen(storename, "rb");
  if (datafile)
  {
    /* store already exists, so close it and return 0 */
    fclose(datafile);
    return 0;
  }
  else
  {
    /* store does not exist, so create it */
    tracePersist("Creating store %s...", storename);
    datafile = fopen(storename, "wb+");
    /* firstavail must point to after itself and the root PID */
    firstavail = sizeof(firstavail) + sizeof(PID);
    fwrite(&firstavail, sizeof(firstavail), 1, datafile);
    STORE_SetRootPID(0);
    return 1;
  }
#endif
}

/*
 * Closes an open store.
 *
 */

void STORE_CloseStore
(
  void
)
{
#ifdef TEASEME
  return;
#else
  tracePersist0("STORE_CloseStore: closing store");
  fseek(datafile, 0, SEEK_SET);
  fwrite(&firstavail, sizeof(firstavail), 1, datafile);
  fclose(datafile);
  datafile = NULL;
  firstavail = 0;
  tracePersist0("STORE_CloseStore: done");
#endif
}

/*
 * Gets the root object's PID
 *
 */

PID STORE_GetRootPID
(

  void
)
{
#ifdef TEASEME
  return 0;
#else

  PID pid;
  tracePersist("STORE_GetRootPID: seeking to %i", sizeof(firstavail));
  fseek(datafile, sizeof(firstavail), SEEK_SET);
  tracePersist("STORE_GetRootPID: reading %i bytes", sizeof(pid));
  fread(&pid, sizeof(pid), 1, datafile);
  tracePersist("STORE_GetRootPID: read pid %i", pid);
  return pid;
#endif
}

/*
 * Set the root object's PID: NEVER USE! (except where needed)
 *
 */

void STORE_SetRootPID
(
  PID pid
)
{
#ifdef TEASEME
  return;
#else
  tracePersist("STORE_SetRootPID: seeking to %i", sizeof(firstavail));
  fseek(datafile, sizeof(firstavail), SEEK_SET);
  fwrite(&pid, sizeof(pid), 1, datafile);
  tracePersist("STORE_SetRootPID: wrote pid %i", pid);
#endif
}

/*
 * Gets the size of an object in the store.
 *
 */

int STORE_GetObjectSize
(
  PID pid
)
{
#ifdef TEASEME
  return 0;
#else
  int objsize;
  
  assert((int32*) pid != NULL);
  fseek(datafile, -pid, SEEK_SET);
  fread(&objsize, sizeof(int), 1, datafile);
  return objsize;
#endif
}

/*
 * Loads an object from the store.
 *
 */
void STORE_LoadObject
(
  PID pid,
  char* obj
)
{
#ifdef TEASEME
  return;
#else
  int objsize;
  int result;

  if (-pid >= firstavail) {
    panic("Loading past end of store");
  }
  assert((int32*) pid != NULL);

#ifdef STORE_VERBOSE
  tracePersist("STORE_LoadObject: pid = %x, obj = %x", pid, obj);
#endif 
  fseek(datafile, -pid, SEEK_SET);
  fread(&objsize, sizeof(int), 1, datafile);
  result = fread(obj, objsize, 1, datafile);
  if (result != 1) {
    panic("didn't read whole object");
  }

#ifdef PERSIST_VERBOSE
  tracePersist("STORE_LoadObject: read %i, objsize %i, pid %x obj %x", 
	       result, objsize, pid, obj);
#endif
#endif
}

/*
 * Updates an existing object in the store with new data given.
 *
 */


#include "vm/classfil.h"

void STORE_UpdateObject
(
  PID pid,
  void* obj
)
{
#ifdef TEASEME

#else
  int objsize;
  jobject dummy;

  dummy = obj;
#ifdef PERSIST_VERBOSE
  tracePersist("STORE_UpdateObject: pid %x, obj %p", (int) pid, obj);
#endif
  assert((int32*) pid != NULL);
  fseek(datafile, -pid, SEEK_SET);
  fread(&objsize, sizeof(int), 1, datafile);
  fseek(datafile, 0, SEEK_CUR);
  fwrite(obj, objsize, 1, datafile);
  tracePersist("STORE_UpdateObject: wrote %i bytes", objsize);
#endif
}

/*
 * Allocates a new object in the store - doesn't write it though... just
 * gives it a pid and writes its size... use STORE_UpdateObject to write
 * it once you have its pid.
 *
 */

PID STORE_NewObject
(
  int objsize
)
{
#ifdef TEASEME
  return -1;
#else

  PID pid;

#ifdef PERSIST_VERBOSE
  tracePersist("STORE_NewObject: firstAvail %i, objsize %i", 
	       firstavail, objsize);
#endif
  /* seek to first available space */
  fseek(datafile, firstavail, SEEK_SET);
  /* write new object's size */
  fwrite(&objsize, sizeof(objsize), 1, datafile);

  /* write new object */ /* don't write... we just want to allocate */
/*  fwrite(obj, objsize, 1, datafile);*/
  /* new object's PID is its location negated */
  pid = -firstavail;
  /* first available space is now after this object (and its size) */
  firstavail += (objsize + sizeof(objsize));
#ifdef PERSIST_VERBOSE
  tracePersist("STORE_NewObject: firstavail is now %i", firstavail);
#endif
  /* seek to beginning and update record of first available space */
  /* don't write it yet - much more efficient just to write when closing */
/*  fseek(datafile, 0, SEEK_SET);
  fwrite(&firstavail, sizeof(firstavail), 1, datafile);*/
  /* return new object's PID */
  return pid;
#endif
}



