/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#ifdef KISSME_LINUX_KERNEL

#include <linux/module.h>
#include <linux/types.h>
#include <linux/file.h>
#include <linux/stat.h>
#include <linux/fcntl.h>
#include <asm/uaccess.h>
#include <linux/fs.h>
#include "sys_linux_kernel/file.h"
#endif

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <pthread.h>
#endif

#include "vm/jni.h"

#include "vm/classfil.h"
#include "sys_linux_host/kissme_classload.h"

#include "lib/indigenous/java.lang/VMClass.h"  
#include "lib/indigenous/java.lang/VMSystem.h"
#include "lib/indigenous/java.lang/VMClassLoader.h"
#include "lib/indigenous/java.lang/Class_Reflection.h"
#include "vm/threads_native.h"
#include "vm/interp.h"
#include "vm/interp_methods.h"
#include "vm/classfile_methods.h"
#include "vm/garbage.h"
#include "vm/garbage2.h"
#include "vm/newobject.h"
#include "vm/interp/methodstacks.h"
#include "vm/threadinfo.h"
#include "vm/cptuplelist.h"

#include "vm/initialize.h"
#include "vm/startup.h"
#include "vm/oom_exception_object.h"

#ifdef TEASEME
extern volatile int iSystemHeapDestroyed; //flag indicating that the system heap has been free'ed
#endif
extern int use_persistent_store;
extern int canStartOptimising; //for the JIT
extern volatile int totalThreadsInSystem; //we count all threads that are created and destroyed
extern tClassLoaderTuple* pstObjectType;  // the type for java.lang.Object
extern tClassLoaderTuple* pstStringType;  // the type for java.lang.String
extern tClassLoaderTuple* pstClassType;   // the type for java.lang.Class
extern tClassLoaderTuple* pstVMClassType; // the type for java.lang.VMClass

#ifdef TEASEME
#ifdef KISSME_LINUX_KERNEL
int run_thread(void* arg);
#else
void* run_thread(void* arg);
#endif
#include "lib/indigenous/jos.system/processes.h"
#endif

jobject OOMExceptionObject = NULL;
int INTERP_FirstThreadRunning;
tAllocatorHeap* system_heap;


static tClassLoaderTuple* LoadMainClass(JNIEnv* env, char *pszMainClass);


int STARTUP_startup(char* pszMainClass, char* pszStoreName, int argc, 
		    char** argv, int argsused) {
  
  int32 i32MainStackPlace;
  tStackFrame* pstFrame;
  tClassLoaderTuple* pstMainClass;
  tMethod* pstMainMethod;
  tOBREF hMainThread;
  tOBREF pstExOb;
  
  tJNIData* pstJNIData;
  tARREF pstArgArray;
  JNIEnv* jni_env;
  
  int i;
  int failure;
  tThreadNode* node;
  
#ifdef TEASEME
  tClassLoaderTuple* tclass;
  tAllocatorHeap* pstHeap; //the heap for the 'init' process
  struct run_thread_args* targs;
#endif
  
#ifndef TEASEME
#ifdef USE_DISTRIBUTED_STORE
  /* We'll open the store on demand*/
#else

#ifdef PERSIST
  if (use_persistent_store) {
    if (STORE_OpenStore(pszStoreName) == 0) {
      // XXX - what if this fails?  how do the native methods know?
      STORE_NewStore(pszStoreName);
    }
  }
#endif
#endif
#endif

#ifdef TEASEME
  CLASSFILE_ppszClassPaths = sys_malloc(sizeof(char*) * 3);
  CLASSFILE_u16NumClassPaths = 3; 
  CLASSFILE_ppszClassPaths[0] = "/home/jewel/teaseme/cvs/classes";
  CLASSFILE_ppszClassPaths[1] = "/home/jewel/kissme/cvs/classes";
  CLASSFILE_ppszClassPaths[2] = "/home/jewel/java/cvsclasspath/classpath";
//  CLASSFILE_ppszClassPaths[1] = "/usr/share/teaseme/classes";

  startupTrace("teaseme JVM loaded (tid %x %i)", 
	       (int) sys_current_thread(), (int) sys_current_thread());
  startupTrace("Using class path: %s",   CLASSFILE_ppszClassPaths[0]);
  startupTrace("  and class path: %s",   CLASSFILE_ppszClassPaths[1]);
  startupTrace("  and class path: %s",   CLASSFILE_ppszClassPaths[2]);

  iSystemHeapDestroyed = 0;
#endif

  OOMExceptionObject = NULL;

  CLASSFILE_Init();

  totalThreadsInSystem = 0;

  sys_mutex_provider_init();
  sys_condition_provider_init();

  JNI_zeroJNITable();

  THREAD_Init();
#ifdef TEASEME
  JOSPROCESS_Init();
#else
  system_heap = GARBAGE_Init();  
#endif

  /* create this thread's JNIData structure */
  pstJNIData = THREAD_createJNIData(NULL, NULL);
#ifdef TEASEME
  system_heap = pstHeap = GARBAGE_Init();
  pstJNIData->pstHeap = pstHeap;
#else
  pstJNIData->pstHeap = system_heap;
#endif

  jni_env = &pstJNIData->functions;

  node = THREADINFO_CreateNewNodeForThread(sys_current_thread(), 
					   &i32MainStackPlace);
  THREADINFO_addThreadNodeToList(node);

  THREADINFO_IncrementThreadCount();

  traceStartup0("preloading java.lang.VMClass");
  pstVMClassType = CLASSFILE_FindOrLoad(jni_env, "java/lang/VMClass", NULL);
  if (pstVMClassType == NULL) {
    //XXX Check the loading tuple
    return fatalError0(-2, "Could not find java/lang/VMClass");
  }

  traceStartup0("preloading java.lang.Class");
  pstClassType = CLASSFILE_FindOrLoad(jni_env, "java/lang/Class", NULL);
  if (pstClassType == NULL) {
    //XXX Check the loading tuple
    return fatalError0(-2, "Could not find java/lang/Class");
  }

  /* initialise JNI */
  traceStartup0("calling JNI_Init");
  JNI_Init(jni_env);   // Note that this completes the java/lang/Class objec

  /* register the natives for VMClass,  These are needed for performing
     class initialisation. */
  NATIVES_InitNatives(pstVMClassType, 0);

  if (CLASSLOADER_TUPLE_FixupClassObjects(jni_env)) {
    return fatalError0(-2, "Could not make class objects during startup");
  }

  traceStartup0("preloading java.lang.Object");
  pstObjectType = CLASSFILE_FindOrLoad(jni_env, "java/lang/Object", NULL);
  if (pstObjectType == NULL) {
    return fatalError0(-2, "Could not find java/lang/Object");
  }

  traceStartup0("preloading java.lang.String");
  pstStringType = CLASSFILE_FindOrLoad(jni_env,"java/lang/String", NULL);
  if (pstStringType == NULL) {
    return fatalError0(-2, "Could not find java/lang/String");
  }

  traceStartup0("initializing java.lang.String");
  if (INITIALIZE_InitializeClass(jni_env, 
				 pstStringType->classObject) != NULL) {
    return fatalError0(-2, "Could not initialize java/lang/String");
  }

  traceStartup0("making primitive types");
  if (java_lang_VMClassLoader_makePrimitiveTypes(jni_env) != 0) {
    return fatalError0(-7, "Could not create primitive types and classes");
  }
  traceStartup0("preloading java.lang.Thread");
  assert(CLASSFILE_FindOrLoad(jni_env, "java/lang/Thread", pstClassType));
  traceStartup0("preloading java.lang.NullPointerException");
  assert(CLASSFILE_FindOrLoad(jni_env, "java/lang/NullPointerException", 
			      pstClassType));

  INTERP_FirstThreadRunning = 1;

  traceStartup0("preloading java.lang.System");
  if ((CLASSFILE_FindOrLoad(jni_env,"java/lang/System", NULL)) == NULL) {
    return fatalError0(-2, "Could not find java/lang/System");
  }
  
  {
    jclass tClazz;
    jmethodID methodID;
    tClazz = (*jni_env)->FindClass(jni_env, "java/lang/Double");
    assert(tClazz);
    methodID = (*jni_env)->GetStaticMethodID(jni_env, tClazz, 
					     "initIDs", "()V");
    assert(methodID);
    traceStartup0("calling java.lang.Double.initIds()");
    (*jni_env)->CallStaticVoidMethod(jni_env, tClazz, methodID);
    if((*jni_env)->ExceptionOccurred(jni_env))
      {
	INTERP_printStackTrace(jni_env, (*jni_env)->ExceptionOccurred(jni_env));
	return fatalError0(-2, "java/lang/Double.initIDs failed in startup, check that the native shared libraries can be loaded");
      }
  }

#ifdef TEASEME
  targs = sys_malloc(sizeof(struct run_thread_args));

  targs->pstOrigHeap = pstHeap;
  targs->pstNewHeap = pstHeap;
  targs->pstJNIData = pstJNIData;
  targs->className = INTERP_NewStringFromAsciz(jni_env, "teaseme/system/init");
  if (targs->className == NULL) {
    return -1;
  }
  targs->args = INTERP_NewArray(jni_env, T_OBJECT, 0, "java/lang/String",
				pstClassType);
  if (targs->args == NULL) {
    return -2;
  }

  {
    jclass tgClazz;
    jmethodID methodID;
    jstring cstring = INTERP_NewStringFromAsciz(jni_env, "init");
    assert(cstring);

    tgClazz = (*jni_env)->FindClass(jni_env, "java/lang/ThreadGroup");
    assert(tgClazz);
    methodID = (*jni_env)->GetMethodID(jni_env, tgClazz, "<init>", 
				       "(Ljava/lang/String;)V");
    assert(methodID);
    targs->threadGroup = (*jni_env)->NewObject(jni_env, tgClazz, methodID, 
					       cstring);
    
    if (!targs->threadGroup) {
      return -3;
    }
  }

  {
    jclass tClazz;
    jmethodID methodID;
    jstring cstring = INTERP_NewStringFromAsciz(jni_env, "initThread");
    assert(cstring);
    tClazz = (*jni_env)->FindClass(jni_env, "java/lang/Thread");
    assert(tClazz);
    methodID = 
      (*jni_env)->GetMethodID(jni_env, tClazz, "<init>",
			      "(Ljava/lang/ThreadGroup;Ljava/lang/String;)V");
    assert(methodID);
    targs->threadObject = (*jni_env)->NewObject(jni_env, tClazz, methodID,
						targs->threadGroup, cstring);
    
    if (!targs->threadObject) {
      return -4;
    }
  }

  {
    jclass jClazz;
    jmethodID methodID;

    jClazz = 
      (*jni_env)->FindClass(jni_env, 
			    "teaseme/system/servers/process/JOSProcess");
    assert(jClazz);
    methodID = (*jni_env)->GetMethodID(jni_env, jClazz, "<init>", "(I)V");
    assert(methodID);
    targs->josProcessObject = (*jni_env)->NewObject(jni_env, jClazz,
						    methodID, (jint) pstHeap);
    
    if (!targs->josProcessObject) {
      return -5;
    }
  }
#endif

#ifndef GARBAGE_GENERATIONAL
#ifdef PERSIST
  if (use_persistent_store) {
    traceStartup0("initialising RPOT and preloading plava.PStore");
    RPOT_Init();
    CLASSFILE_FindOrLoad(jni_env, "plava/PStore", pstClassType);
  }
#endif
#endif
  

#ifdef TEASEME
  traceStartup0("Starting main thread");
#ifdef KISSME_LINUX_KERNEL
  kernel_thread(&run_thread, (void*) targs, 0);
#else
#ifdef KISSME_LINUX_USER
  {
    pthread_t id;
    traceStartup0("Starting thread now");
    assert(pthread_create(&id, NULL, &run_thread, (void*) targs) == 0);
  }
#else
  kinterface->kthread_create(&run_thread, (void*) targs, 16, 0);
#endif
#endif
  
  totalThreadsInSystem++;
#ifdef KISSME_LINUX_USER
  while(iSystemHeapDestroyed == 0) {
#ifdef HAVE_PTHREAD_YIELD
    pthread_yield();
#endif
  }
#endif
#endif
  if (STARTUP_createOOMExceptionObject(jni_env) != 0) {
    return -6;
  }
  
  setBootstrapped();
  
#ifndef TEASEME
  /* Construct a thread object for the main thread */
  hMainThread = INTERP_ConstructMainThread(jni_env);
  assert(hMainThread);

  pstMainClass = LoadMainClass(jni_env, pszMainClass);
  if (pstMainClass == NULL) {
    return fatalError(1, "Cannot load the main class %s.", pszMainClass);
  }
  
  pstMainMethod = CLASSFILE_FindMethod(jni_env, pstMainClass, 
				       "main", "([Ljava/lang/String;)V");
  if (!pstMainMethod) {
    //XXX We have to throw a "NoSuchMethodError" exception
    //    perhaps we should use reflection to find the method
    //    Or ..
    //    Create a kissme specific Runner class which runs the "main" thread 
    //    and looks for the user's class
    return fatalError(2, "No static void main(String[]) method found in "
		      "class %s.", pszMainClass);
  }

  /* XXX - should check that the 'main' method is public static */

  /* handle command-line arguments */
  traceStartup("creating Java objects for args: first is argv[%d]", argsused);
  pstArgArray = INTERP_NewArray(jni_env, T_OBJECT, argc - argsused,
				"java/lang/String", pstClassType, &failure);
  for (i = argsused; i < argc; i++) {
    ((tOBREF*) ADEREF(pstArgArray)->pvElements)[i - argsused] = 
      INTERP_NewStringFromAsciz(jni_env, argv[i]);
  }
  
  totalThreadsInSystem++;
  
  /* We create a frame for the static main method, it has no
     pstNewObject because it is static */
  traceStartup0("creating initial stack frame");
  pstFrame = InitialStackFrame(pstMainMethod, NULL, (int32*) &pstArgArray,
			       system_heap);
  THREADINFO_SetNodeInfo(node, hMainThread, &i32MainStackPlace);
  THREADINFO_addInitialStackFrameToNode(node, pstFrame);

  traceStartup0("calling interpreter to run main(args)");
  pstExOb = Interpret(jni_env, pstFrame, 1);
  traceStartup0("interpreter has returned");
  THREADINFO_removeInitialStackFrameFromNode(node, pstFrame);

  traceStartup0("interpreter has exitted normally");
  
  node->state = THREAD_STATE_TERMINATED;
  THREADINFO_DecrementThreadCount();

  i32MainStackPlace = 22; //stop this being optimised away

  return shutdown_machine(pstExOb, 0, 0);
#endif
  
#ifdef TEASEME
  i32MainStackPlace = 0x3839272; //stop optimisation
  traceStartup("STARTUP_startup returning 0 (threads %i)\n", 
	       totalThreadsInSystem);
  return 0;
#endif
}

int shutdown_machine(tOBREF pstExOb, int forced_shutdown, int retval)
{
  // The main thread is handled differently to the others, which is why
  // this is called here.  But shutdown_machine can be called from
  // another thread (maybe through System.exit()), in which case we
  // must be careful about shutting down (the forced_shutdown flag
  // could help us here)
  
  traceStartup("Called shutdown_machine: pstExOb = %p, forced = %d, rc = %d",
	       pstExOb, forced_shutdown, retval);
  THREADINFO_DecrementThreadCount(); //decrement the calling thread's count
  
  if (!forced_shutdown) {
    traceStartup0("kissme: main thread has finished, waiting for others");
    {  
      tJNIData* pstJNIData = JNI_getJNIData(sys_current_thread());
      JNIEnv* jni_env = &pstJNIData->functions;

      THREAD_WaitForNativeThreads(jni_env);
    }
    traceStartup0("All threads finished");
  }

  CPTUPLELIST_Empty();
  JNI_Finish();

#ifdef PERSIST
  if (use_persistent_store && pstExOb == NULL && 
      (retval == 0 || !forced_shutdown)) {
    traceStartup0("Performing final stabilise");
    RPOT_StabiliseAll();
  }
  else {
    traceStartup0("Skipping final stabilise");
  }
#endif

#ifdef GARBAGE
  GARBAGE_Finish(system_heap);
#endif
  
#ifdef PERSIST
  if (use_persistent_store) {
    traceStartup0("Closing store");
    STORE_CloseStore();
  }
#endif
  
  // XXX -- this could be a lie.
  traceStartup0("kissme exiting");
  
  // XXX -- what happens when user calls (say) "System.exit(42)"?
  if (forced_shutdown) {
    return retval;
  }
  else if (pstExOb) {
    return -1;
  }
  else {
    return 0;
  }
}


static tClassLoaderTuple* LoadMainClass(JNIEnv* env, char* pszMainClass)
{
  jmethodID mid;
  jobject hSystemLoader, hClassLoader, hClassName;
  jclass hMainClass;
  jthrowable pstExObject;
  
  traceStartup0("Finding java.lang.ClassLoader");
  hClassLoader = (*env)->FindClass(env, "java/lang/ClassLoader");
  if (hClassLoader == NULL) {
    pstExObject = (*env)->ExceptionOccurred(env);
    if (pstExObject) {
      INTERP_printStackTrace(env, pstExObject);
    }
    return NULL;
  }

  traceStartup0("Finding java.lang.ClassLoader.getSystemClassLoader");
  mid = (*env)->GetStaticMethodID(env, hClassLoader, 
				  "getSystemClassLoader",
				  "()Ljava/lang/ClassLoader;");
  if (mid == NULL) {    
    pstExObject = (*env)->ExceptionOccurred(env);
    if (pstExObject) {
      INTERP_printStackTrace(env, pstExObject);
    }
    return NULL;
  }

  traceStartup("Calling java.lang.ClassLoader.getSystemClassLoader()");
  hSystemLoader = (*env)->CallStaticObjectMethod(env, hClassLoader, mid);
  pstExObject = (*env)->ExceptionOccurred(env);
  if (pstExObject) {
    INTERP_printStackTrace(env, pstExObject);
    return NULL;
  }

  traceStartup("Finding SystemClassLoader.findClass()");
  mid = (*env)->GetMethodID(env,
			    (*env)->GetObjectClass(env, hSystemLoader), 
			    "findClass",
			    "(Ljava/lang/String;)Ljava/lang/Class;");
  if (mid == NULL) {
    pstExObject = (*env)->ExceptionOccurred(env);
    if (pstExObject) {
      INTERP_printStackTrace(env, pstExObject);
    }
    return NULL;
  }

  traceStartup("Calling SystemClassLoader.findClass(%s)", pszMainClass);
  hClassName = INTERP_NewStringFromAsciz(env, pszMainClass);
  hMainClass = (*env)->CallObjectMethod(env, hSystemLoader, mid, hClassName);
				   
  pstExObject = (*env)->ExceptionOccurred(env);
  if (pstExObject) {
    INTERP_printStackTrace(env, pstExObject);
    return NULL;
  }

  assert(DEREF(hMainClass)->pstType == pstClassType);

  traceStartup0("LoadMainClass succeeded");
  return CLASS_GetClassStruct(env, hMainClass);
}



