#include "config.h"

/*
 * RPOT.c
 *
 * Contains the Resident Persistent Object Table.
 *
 */

#ifdef PERSIST

#ifndef GARBAGE
#error GARBAGE must be defined for RPOT.c
#endif

#ifndef GARBAGE_GENERATIONAL
#ifndef GARBAGE_COPYING


#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#endif

#include "vm/global.h"
#include "vm/garbage.h"
#include "vm/store.h"
#include "vm/rpot.h"
#include "vm/jni.h"
#include "vm/jni_data.h"
#include "vm/classfile_methods.h"

#ifdef REVRPOT
tRPOTElement** RPOT_table = NULL;
tRPOTElement** RPOT_revtable = NULL;
#else
tRPOTElement* RPOT_table = NULL;
#endif

/* The number of objects in the RPOT */
static int iNumItems;


/*
   We allocate a large bucket to accomodate a number of
   overflow entries. We keep track of all these bucket so that we can
   find easily if an arbitrary address is within the RPOT.

   The buckets we create a PHYSICAL buckets, not LOGICAL buckets in the
   way that they are usually associated with hash tables. The logical
   buckets are created by the linked lists that are kept, but these lists
   can include elements from any of the physical buckets.

   The entries have the same format as before - with a "next" pointer
   pointing to overflows.

*/

typedef struct rpotoverflowbucket
{
  tRPOTElement* pstFirstFreeElement;
  struct rpotoverflowbucket* pstNext;
  struct rpotoverflowbucket* pstNextFree;
  tRPOTElement  astElements[1]; /* must be last member */
} tRPOTOverflowBucket;

tRPOTOverflowBucket* pstBucketList = NULL;
tRPOTOverflowBucket* pstFreeBucketList = NULL;


static tRPOTOverflowBucket* WhichBucket(tRPOTElement* elem)
{
  tRPOTOverflowBucket* temp;
  
  temp = pstBucketList;
  while (temp) {
    /* if the element is within the memory range of this bucket, we've
       found it */
    if ((void*) elem > (void*) temp &&
        (void*) elem <= (void*) &(temp->astElements[RPOT_OVERFLOWSIZE - 1])) {
      return temp;
    }
    temp = temp->pstNext;
  }
  return NULL;
}


int RPOT_InRPOT(void* ref)
{
  if (RPOT_table == NULL) {
    /* No RPOT_table */
    return 0;
  }
  /* Test if the ref is within the range of the main table. */
  else if (ref > (void *) RPOT_table &&
      ref < ((void *) RPOT_table + sizeof(tRPOTElement) * RPOT_TABLESIZE)) {
    return 1;
  }
  else {
    /* Otherwise, just pass ref to WhichBucket - it doesn't have to be a
       real tRPOTElement pointer - will work if it points to the la
       field of the tRPOTElement and not to the front */
    return (WhichBucket(ref) != NULL);
  }
}


void RPOT_Init(void)
{
  int i;

  /* only need main table */

  RPOT_table = (tRPOTElement*) sys_malloc(sizeof(tRPOTElement) * 
					  RPOT_TABLESIZE);
  for (i = 0; i < RPOT_TABLESIZE; i++) {
    RPOT_table[i].pid = (PID) NULL;
    RPOT_table[i].la = (PID) NULL;
    RPOT_table[i].next = NULL;
  }
}

void RPOT_Kill(void)
{
  tRPOTOverflowBucket* temp;
  tRPOTOverflowBucket* next;

  sys_free(RPOT_table);
  temp = pstBucketList;
  while (temp) {
    next = temp->pstNext;
    sys_free(temp);
    temp = next;
  }
}

tObject* RPOT_GetLA(PID pid)
{
  tRPOTElement* temp;

  if ((temp = &(RPOT_table[HASHFUNC(pid)]))->pid == pid) {
    return temp->la;
  }
  temp = temp->next;
  while (temp) {
    if (temp->pid == pid) {
      return temp->la;
    }
    temp = temp->next;
  }
  return NULL;
}


/* Adding is a simple process if there is nothing in the required slot
   in the main table.  If this slot is occupied, however, we must find
   a free slot in an overflow bucket.  If none is available, we must
   create a bucket into which to put it. */
tRPOTElement* RPOT_Add(PID pid, void* la)
{
  tRPOTElement* temp;
  tRPOTElement* newtemp;
  tRPOTElement* retval;
  tRPOTOverflowBucket* pstNewBucket;
  int i;

  temp = &(RPOT_table[HASHFUNC(pid)]);
  if (temp->pid) {
    /* this space is used, so go to the end of the overflow chain
       (will be 0 steps if there is no overflow */
    while (temp->next) {
      temp = temp->next;
    }
    /* now we need to add a new entry, so look to see if there is space
       in an overflow bucket (if there is an overflow bucket) */
    if (pstFreeBucketList) {
      /* use the first available space in first available bucket */
      newtemp = pstFreeBucketList->pstFirstFreeElement;
      pstFreeBucketList->pstFirstFreeElement = newtemp->next;
      /* if that was the last free space in that bucket then we need to remove
         the bucket from the list of free buckets */
      if (pstFreeBucketList->pstFirstFreeElement == NULL) {
        pstFreeBucketList = pstFreeBucketList->pstNextFree;
      }
      newtemp->pid = pid;
      newtemp->la = la;
      newtemp->oldHandle = NULL;
      newtemp->next = NULL;
      temp->next = newtemp;
      retval = newtemp;
    }
    else {
      /* allocate new bucket */
      /* put in -1 because the bucket definition contains 1 */
      pstNewBucket = (tRPOTOverflowBucket*) 
	sys_malloc(sizeof(tRPOTOverflowBucket) + 
		   sizeof(tRPOTElement) * (RPOT_OVERFLOWSIZE - 1));
#ifdef DEBUG
      assert(pstNewBucket);
#endif

      /* add bucket to bucket list */
      pstNewBucket->pstNext = pstBucketList;
      pstBucketList = pstNewBucket;
      /* add bucket to free bucket list */
      pstNewBucket->pstNextFree = pstFreeBucketList;
      pstFreeBucketList = pstNewBucket;

      /* go through all elements and initialise them... use their next
         pointers to point to the next free element - set them all up
         beforehand */
      for (i = 0; i < (RPOT_OVERFLOWSIZE - 2); i++) {
        pstNewBucket->astElements[i].pid = (PID) NULL;
        pstNewBucket->astElements[i].la = (PID) NULL;
        pstNewBucket->astElements[i].next = 
	  &(pstNewBucket->astElements[i + 1]);
      }
      pstNewBucket->astElements[i].pid = (PID) NULL;
      pstNewBucket->astElements[i].la = (PID) NULL;
      pstNewBucket->astElements[i].next = NULL;

      /* set first free to second slot coz we're about to use the first
         slot right now */
      pstNewBucket->pstFirstFreeElement = &(pstNewBucket->astElements[1]);

      /* put info into first slot */
      pstNewBucket->astElements[0].pid = pid;
      pstNewBucket->astElements[0].la = la;
      pstNewBucket->astElements[0].oldHandle = NULL;
      pstNewBucket->astElements[0].next = NULL;
      temp->next = &(pstNewBucket->astElements[0]);
      retval = &(pstNewBucket->astElements[0]);
    }
  }
  else {
    /* this space is empty, so just set the values */
    temp->pid = pid;
    temp->la = la;
    temp->oldHandle = NULL;
    retval = temp;
  }

  iNumItems += 1;

  return retval;
}


/* Removal can be complicated. If an overflow element is freed, it must
   be added to its bucket's free list. If the bucket previously had no
   free elements, it must be added to the free bucket list. */
void RPOT_Remove (PID pid)
{
  tRPOTElement* temp;
  tRPOTElement* next;
  tRPOTElement* prev;
  tRPOTOverflowBucket* pstTempBucket;

  temp = &(RPOT_table[HASHFUNC(pid)]);
  if (temp->pid == pid) {
    /* it's in the main table... but we might have to move the info from
       the next overflow element into here if there is an overflow */
    if (temp->next) {
      /* put next item's info into main table */
      next = temp->next;
      temp->pid = next->pid;
      temp->la = next->la;
      temp->next = next->next;

      /* free up the overflow item */
      pstTempBucket = WhichBucket(next);
      next->next = pstTempBucket->pstFirstFreeElement;
      pstTempBucket->pstFirstFreeElement = next;
      /* check if this bucket has to be put onto the free bucket list -
         will be true if it previously had no free elements (ie if the
         newly freed element's next pointer is NULL */
      if (next->next == NULL) {
        pstTempBucket->pstNextFree = pstFreeBucketList;
        pstFreeBucketList = pstTempBucket;
      }
    }
    else {
      temp->pid = (PID) NULL;
      temp->la = (PID) NULL;
    }
  }
  else {
    prev = temp;
    temp = temp->next;
    while (temp) {
      if (temp->pid == pid) {
        /* link previous one to next one, skipping this one */
        prev->next = temp->next;
        /* deal with the free list in the bucket */
        pstTempBucket = WhichBucket(temp);
        temp->next = pstTempBucket->pstFirstFreeElement;
        pstTempBucket->pstFirstFreeElement = temp;
        if (temp->next == NULL) {
          pstTempBucket->pstNextFree = pstFreeBucketList;
          pstFreeBucketList = pstTempBucket;
        }
        temp = NULL; /* to make the loop end without using "break" */
      }
      else {
        prev = temp;
        temp = temp->next;
      }
    }
  }
  iNumItems -= 1;
}


tOBREF RPOT_GetObref(PID pid)
{
  tRPOTElement* temp;

  if ((temp = &(RPOT_table[HASHFUNC(pid)]))->pid == pid) {
    return &(temp->la);
  }
  temp = temp->next;
  while (temp) {
    if (temp->pid == pid) {
      return &(temp->la);
    }
    temp = temp->next;
  }

  return NULL;
}


/*
 * Faults in an object, returning its handle.
 *
 */
tOBREF RPOT_FaultObject(PID pid)
{
  tObject* la;
  tObject* obj;
  tArray* arr;
  char* className;
  PID classPID;
  int objSize;
  
#ifdef RPOTREFS
  tRPOTElement* rpotelem;
#else
  tOBREF objref;
#endif
#ifdef EAGERINDIRECT
  PID refPID;
  tClass* pstCurrType;
  int iFinished = 0;
  int i;
  tField* field;
  tObject* ref;
  tOBREF refH;
  tRPOTElement* refrpotelem;
#endif
  
#ifdef DEBUG
  assert(ISPID(pid));
#endif

  /* residency check first */
  la = RPOT_GetLA(pid);
  if (la) {
#ifdef EVICTIONS
    if (ISPID(la)) {
      /* the object has been faulted in but subsequently swopped out
         so....... we swop it back in here to make the interpreter
         happy */
      return RPOT_SwapIn(pid);
    }
#endif
#ifdef RPOTREFS
    /* it's already resident (false object fault) so just return the handle */
    if (la->hHandle) {
      return la->hHandle;
    }
    else {
      return RPOT_GetObref(pid);
    }
#else
    return la->hHandle;
#endif
  }

  //  printf("PID not resident %x\n", pid);
  
#ifdef DEBUG
#ifndef USEOSKIT
  assert(pid < 0);
#endif
#endif
#ifdef EVICTIONS
  la = TPOT_GetLA(pid);
  if (la) {
    if (ISPID(la)) {
      /* the object has been faulted in but subsequently swopped out
         so....... we swop it back in here to make the interpreter
         happy */
      return RPOT_SwapIn(pid);
    }
    else {
      return la->hHandle;
    }
  }
#endif
  
  /* get object from store */
  objSize = STORE_GetObjectSize(pid);
#ifdef DEBUG
#ifndef USEOSKIT
  assert(objSize >= 0);
#endif
#endif

#ifdef RPOTREFS
  obj = GARBAGE_MallocObj(objSize, JNI_getCurrentHeap()); 
#else
  objref = GARBAGE_Malloc(objSize, JNI_getCurrentHeap());
  obj = PDEREF(objref);
#endif
  STORE_LoadObject(pid, (void*) obj);

#ifdef RPOTREFS
  rpotelem = RPOT_Add(pid, obj);
  obj->hHandle = (tOBREF) rpotelem;
#else
  RPOT_Add(pid, obj);
  obj->hHandle = objref;
#endif
  
#ifndef REVRPOT
  obj->pid = pid;
#endif
  
  /* point field pointer to end of object struct data */
  if ((obj->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_ARRAY) {
    arr = (tArray*) obj;
    arr->pvElements = (((byte*) obj) + ROUND32BIT(sizeof(tArray)));
  }
  else {
    obj->pi32Vars = (int32*) (((byte*) obj) + ROUND32BIT(sizeof(tObject)));
  }
  
  /* ### deal with locks, etc */
  
  /* get the object's class */
  objSize = STORE_GetObjectSize((PID) obj->pstType);
#ifdef DEBUG
#ifndef USEOSKIT
  assert(objSize >= 0);
#endif
#endif
  
  if (objSize < 0) {
    eprintf("RPOT_FaultObject, obj size is %i, class pid is %p\n", 
	    objSize, obj->pstType);
    assert(objSize >= 0);
  }

  tracePersist("RPOT_FaultObject, obj size is %i, class pid is %p", 
	       objSize, obj->pstType);

  className = (char*) sys_malloc(sizeof(char) * (objSize + 1));
  classPID = (PID) obj->pstType;
  STORE_LoadObject(classPID, className);
  className[objSize] = 0;

  // XXXX The following will use the default class loader.  How
  // XXXX do we work out which is the real loading class?  [It could be
  // XXXX important ... ]
  obj->pstType = CLASSFILE_FindOrLoad(JNI_getJNIEnvPtr(), className, NULL);
  assert(obj->pstType);
  if (obj->pstType->pstClass->persIndex == (PID) NULL) {
    obj->pstType->pstClass->persIndex = classPID;
  }
  sys_free(className);
  
#ifdef EVICTIONS
  SETFLAG(obj,GARBAGE_PERSISTENT);
#endif
  
#ifdef EAGERINDIRECT
  /* make handles for all the object's references */
  /* for each reference field, if the object referred to by the PID is
     already loaded then we must use its handle, otherwise allocate a
     new handle and put the PID in it */
  if ((obj->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT) {
    if (obj->pi32Vars) {
      /* Check the fields (looking at parent classes too) */
      for (pstCurrType = obj->pstType; 
	   iFinished == 0; 
	   pstCurrType = pstCurrType->pstSuperClass) {
        for (i = 0; i < pstCurrType->u16InstCount; i++) {
          field = &(pstCurrType->pstInstOffsets[i]);
          switch (field->uidFieldSig[0]) {
	  case '[': 
	  case 'L': 
	    {
	      refPID = obj->pi32Vars[field->u16Offset];
	      if (refPID) {
		/* look in RPOT */
		ref = RPOT_GetLA(refPID);
		if (ref) {
#ifdef RPOTREFS
		  /* ### maybe check in case this object somehow has a handle
		     as well as being in the RPOT */
		  refH = RPOT_GetObref(refPID);
#else
		  refH = ref->hHandle;
#endif
		}
		else {
		  ref = TPOT_GetLA(refPID);
		  if (ref) {
		    /* should have a handle */
		    refH = ref->hHandle;
		  }
		  else {
		    /* object not resident so allocate a handle / RPOT entry
		       for it in its absence */
#ifdef RPOTREFS
		    refrpotelem = RPOT_Add(refPID, (void*) refPID);
		    refH = &(refrpotelem->la);
#else
		    refH = GARBAGE_AllocHandle();
		    PDEREF(refH) = (tObject*) refPID;
#endif
		  }
		}
		
		/* overwrite reference with the handle */
		obj->pi32Vars[field->u16Offset] = (int32) refH;
	      }
	      
	      break;
	    }
	  }
        }
        if (pstCurrType->pstSuperClass == NULL ||
            ((pstCurrType->pstInstOffsets) && 
	     (pstCurrType->pstInstOffsets[0].u16Offset == 0))) {
          iFinished = 1;
        }
      }
      
    }
  }
  else {
    arr = (tArray*) obj;

    if ((arr->enType == T_ARRAY) || (arr->enType == T_OBJECT)) {
      for (i = 0; i < arr->i32Number; i++) {
        refPID = ((PID*) arr->pvElements)[i];
	
	if (refPID) {
	  /* look in RPOT */
	  ref = RPOT_GetLA(refPID);
	  if (ref) {
#ifdef RPOTREFS
	    /* ### maybe check in case this object somehow has a handle
	       as well as being in the RPOT */
	    refH = RPOT_GetObref(refPID);
#else
	    refH = ref->hHandle;
#endif
	  }
	  else {
	    ref = TPOT_GetLA(refPID);
	    if (ref) {
	      /* should have a handle */
	      refH = ref->hHandle;
	    }
	    else {
#ifdef RPOTREFS
	      refrpotelem = RPOT_Add(refPID, (void*) refPID);
	      refH = &(refrpotelem->la);
#else
	      refH = GARBAGE_AllocHandle();
	      PDEREF(refH) = (tObject*) refPID;
#endif
	    }
	  }
	  
	  /* overwrite reference with the handle */
	  ((tOBREF*) arr->pvElements)[i] = refH;
	}
      }
    }
  }
#endif
  
  
#ifdef RPOTREFS
  /* return handle */
  return &(rpotelem->la);
#else
  return obj->hHandle;
#endif
}


/*
 * Handles everything concerned with promoting an object to persistence:
 *
 * 1) Stores its class's name in the store if it is not already there.
 * 2) Stores the object in the store.
 * 3) Puts the PID->LA mapping into the RPOT.
 *
 */
#ifndef EVICTIONS

int calcArrayElementSize(tAType enType, /* @parm Type of elements in array */ 
			 char* pszSig)
{
  int32 i32ElSize;
  char* pszThisSig;
  int iSigLen;

  switch (enType) {
    /* pointer */
  case T_ARRAY:
    {
      i32ElSize = sizeof(tARREF);
      pszThisSig = (char*) sys_malloc(strlen(pszSig) + 2);
      pszThisSig[0] = '[';
      strcpy(pszThisSig + 1, pszSig);
      break;
    }
    /* pointer... same as T_ARRAY but do it separately for fun */
  case T_OBJECT:
    {
      i32ElSize = sizeof(tOBREF);
      iSigLen = strlen(pszSig) + 4;
      pszThisSig = (char*) sys_malloc(iSigLen);
      pszThisSig[0] = '[';
      pszThisSig[1] = 'L';
      strcpy(pszThisSig + 2, pszSig);
      pszThisSig[iSigLen - 2] = ';';
      pszThisSig[iSigLen - 1] = 0;
      break;
    }
    /* 8 bits */
  case T_BOOLEAN:
    {
      i32ElSize = 1;
      pszThisSig = "[Z";
      break;
    }
  case T_BYTE:
    {
      i32ElSize = 1;
      pszThisSig = "[B";
      break;
    }
    /* 16 bits */
  case T_CHAR:
    {
      i32ElSize = 2;
      pszThisSig = "[C";
      break;
    }
  case T_SHORT:
    {
      i32ElSize = 2;
      pszThisSig = "[S";
      break;
    }
    /* 32 bits */
  case T_INT:
    {
      i32ElSize = 4;
      pszThisSig = "[I";
      break;
    }
  case T_FLOAT:
    {
      i32ElSize = 4;
      pszThisSig = "[F";
      break;
    }
    /* 64 bits */
  case T_LONG:
    {
      i32ElSize = 8;
      pszThisSig = "[J";
      break;
    }
  case T_DOUBLE:
    {
      i32ElSize = 8;
      pszThisSig = "[D";
      break;
    }
  default:
    {
      panic("calcArrayElementSize: unexpected enType %d", enType);
    }
  }
  return i32ElSize;
}
#endif


/*
 * This is the first part of creating a persistent object.  The
 * object's class is made persistent if it isn't already persistent,
 * and the object is allocated space in the store and a PID.  
 * 
 * Note: the object's state is not written out at this stage.  
 */
PID RPOT_PromoteObject(tObject* ref)
{
  PID pid;
  tClassLoaderTuple* type;
  tClassLoaderTuple* origType;
  tRPOTElement* tempRPOTElement;

#ifdef DEBUG
  /* XXXX - is this correct?  Ref is a tObject not a tOBREF */
  assert(ISPID((PID) ref) == 0);

  assert(ref->hHandle && ref == *(ref->hHandle));
#endif

  /* sort out this object's class first */
  origType = type = ref->pstType;
  if (type->pstClass->persIndex == (PID) NULL) {
    /* not in store so we must add it */
    type->pstClass->persIndex = STORE_NewObject(strlen(type->uidName));
    STORE_UpdateObject(type->pstClass->persIndex, type->uidName);
  }
  
  /* ### deal with locks, etc */
#ifdef EVICTIONS
  ref->i32Flags |= GARBAGE_PERSISTENT;
#endif

  /* put object into store */
  if ((ref->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT) {
    pid = STORE_NewObject(ROUND32BIT(sizeof(tObject)) +
			  sizeof(int32) * origType->pstClass->u16InstSize);
  }
  else if ((ref->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_ARRAY) {
    int i32Size = calcArrayElementSize(((tArray*) ref)->enType,
				       origType->uidName);
    pid = STORE_NewObject(ROUND32BIT(sizeof(tArray)) + 
			  i32Size * ((tArray*) ref)->i32Number);
  }
  else {
    assert(JNI_FALSE);
  }

  /* add the PID->LA mapping to RPOT_table */
  tempRPOTElement = RPOT_Add(pid, ref);

  /* record the old handle in the RPOT table, then set the object's handle
     to point at the RPOT */
  tempRPOTElement->oldHandle = ref->hHandle;
  ref->hHandle = &(tempRPOTElement->la);

  /* record PID in the object */
#ifndef REVRPOT
  ref->pid = pid;
#endif

  return pid;
}


/*
 * Write an object's state to the store.  The object's type pointer is
 * temporarily replaced with its classes persistent id while this is
 * done.
 *
 * Note: it is assumed that the object has a PID and that its refs
 * have been swizzled before this routine is called.  
 */
void RPOT_UpdateObject(PID pid, tObject* ref)
{
  tClassLoaderTuple* type;

#ifdef DEBUG
  assert(ISPID((PID) ref) == 0);
#endif
  
  type = ref->pstType;
  ref->pstType = (tClassLoaderTuple*) type->pstClass->persIndex;
  STORE_UpdateObject(pid, ref);
  ref->pstType = type;
}


#ifdef EVICTIONS
tOBREF RPOT_SwapIn(PID pid)
{
  tArray* a;
  tObject* o;
  int32 iSize;
  char* className;
  PID classPID;
#ifdef MEMSWAP
  tClass* pstCurrType;
  int i;
  int iFinished;
  tField* field;
  tOBREF  refH;
  tObject* ref;
#endif
#ifdef EAGERINDIRECT
  PID refPID;
  tClass* pstCurrType;
  int iFinished = 0;
  int i;
  tField* field;
  tObject* ref;
  tOBREF refH;
  tRPOTElement* refrpotelem;
#endif
  
  o = RPOT_GetLA(pid);
  if (o && !ISPID(o)) {
#ifdef RPOTREFS
    if (o->hHandle) {
      return o->hHandle;
    }
    else {
      return RPOT_GetObref(pid);
    }
#else
    return o->hHandle;
#endif
  }
  o = TPOT_GetLA(pid);
  if (o && !ISPID(o)) {
    return o->hHandle;
  }
  
  iSize = STORE_GetObjectSize(pid);
  
  // XXXX - This is wrong.  if RPOTREFS is defined, (I think) we need
  // to use an RPOT element as a handle.
  o = GARBAGE_MallocObj(iSize, JNI_getCurrentHeap());
  STORE_LoadObject(pid, o);

  /* do this coz they don't all have their PIDs done properly in the store */
  if (o->pid == NULL) {
    o->pid = pid;
  }
  
  /* get the object's class */
  classPID = (PID) o->pstType;
  if (ISPID(classPID)) {
    iSize = STORE_GetObjectSize(classPID);
    className = (char*) sys_malloc(sizeof(char) * (iSize + 1));
    STORE_LoadObject(classPID, className);
    className[iSize] = 0;
    o->pstType = CLASSFILE_FindOrLoad(className);
    if (o->pstType->persIndex == NULL) {
      o->pstType->persIndex = classPID;
    }
    sys_free(className);
  }
  
  if ((o->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT) {
    if (o->pstType->u16InstSize) {
      o->pi32Vars = (int32*) (((byte*) o) + ROUND32BIT(sizeof(tObject)));
#ifdef MEMSWAP
      if (!HASFLAG(o, GARBAGE_MEMSWOPPED)) {
        iFinished = 0;
        for (pstCurrType = o->pstType; 
	     iFinished == 0; 
	     pstCurrType = pstCurrType->pstSuperClass) {
          for (i = 0; i < pstCurrType->u16InstCount; i++) {
            field = &(pstCurrType->pstInstOffsets[i]);
            switch (field->uidFieldSig[0]) {
	    case '[':
	    case 'L':
              {
                refH = (tOBREF) o->pi32Vars[field->u16Offset];
                /* only bother if it's not null or a PID */
                if (refH && (!ISPID(refH))) {
                  ref = PDEREF(refH);
                  if (!ISPID(ref)) {
                    ref->i32ExternalRefCount -= 1;
                  }
                }
                break;
              }
            }
          }
          /* if we've got up to the Object class or we've got to a class that
             does not inherit anything then we have finished */
	  if ((pstCurrType->pstSuperClass == NULL) ||
              ((pstCurrType->pstInstOffsets) && 
	       (pstCurrType->pstInstOffsets[0].u16Offset == 0))) {
            iFinished = 1;
          }
        }
      }
#endif
    }
    else {
      o->pi32Vars = NULL;
    }
  }
  /* if object is an array */
  else {
    a = (tArray*) o;
    if (a->i32Number) {
      a->pvElements = (((byte*) a) + ROUND32BIT(sizeof(tArray)));
#ifdef MEMSWAP
      /* if it's an array of references, fix the refs */
      if (a->enType == T_OBJECT) {
        for (i = 0; i < a->i32Number; i++) {
          refH = ((tOBREF*) a->pvElements)[i];
          /* if not a null reference */
          if (refH && (!ISPID(refH))) {
            ref = PDEREF(refH);
            if (!ISPID(ref)) {
	      /* only worry about ref count if it's not a PID */
	      ref->i32ExternalRefCount -= 1;
            }
          }
        }
      }
#endif
    }
    else {
      a->pvElements = NULL;
    }
  }
  
#ifdef EAGERINDIRECT
  /* make handles for all the object's references */
  /* for each reference field, if the object referred to by the PID is
     already loaded then we must use its handle, otherwise allocate a
     new handle and put the PID in it */
  if ((o->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT) {
    if (o->pi32Vars) {
      /* Check the fields (looking at parent classes too) */
      for (pstCurrType = o->pstType; 
	   iFinished == 0; 
	   pstCurrType = pstCurrType->pstSuperClass) {
        for (i = 0; i < pstCurrType->u16InstCount; i++) {
	  field = &(pstCurrType->pstInstOffsets[i]);
          switch (field->uidFieldSig[0]) {
	  case '[':
	  case 'L':
            {
              refPID = o->pi32Vars[field->u16Offset];
              if (refPID) {
                /* look in RPOT */
                ref = RPOT_GetLA(refPID);
                if (ref) {
#ifdef RPOTREFS
                  /* ### maybe check in case this object somehow has a handle
		     as well as being in the RPOT */
                  refH = RPOT_GetObref(refPID);
#else
                  refH = ref->hHandle;
#endif
                }
                else {
		  ref = TPOT_GetLA(refPID);
                  if (ref) {
                    /* should have a handle */
                    refH = ref->hHandle;
                  }
                  else {
                    /* object not resident so allocate a handle / RPOT entry
                       for it in its absence */
#ifdef RPOTREFS
                    refrpotelem = RPOT_Add(refPID, (void*) refPID);
                    refH = &(refrpotelem->la);
#else
                    refH = GARBAGE_AllocHandle();
                    PDEREF(refH) = (tObject*) refPID;
#endif
                  }
                }
		
                /* overwrite reference with the handle */
                o->pi32Vars[field->u16Offset] = (int32) refH;
              }
	      
              break;
            }
          }
        }
        if ((pstCurrType->pstSuperClass == NULL) ||
            ((pstCurrType->pstInstOffsets) && 
	     (pstCurrType->pstInstOffsets[0].u16Offset == 0))) {
          iFinished = 1;
        }
      }
      
    }
  }
  else {
    a = (tArray*) o;

    if ((a->enType == T_ARRAY) || (a->enType == T_OBJECT)) {
      for (i = 0; i < a->i32Number; i++) {
        refPID = ((PID*) a->pvElements)[i];
	
	if (refPID) {
	  /* look in RPOT */
	  ref = RPOT_GetLA(refPID);
	  if (ref) {
#ifdef RPOTREFS
	    /* ### maybe check in case this object somehow has a handle
	       as well as being in the RPOT */
	    refH = RPOT_GetObref(refPID);
#else
	    refH = ref->hHandle;
#endif
	  }
	  else {
	    ref = TPOT_GetLA(refPID);
	    if (ref) {
	      /* should have a handle */
	      refH = ref->hHandle;
	    }
	    else {
#ifdef RPOTREFS
	      refrpotelem = RPOT_Add(refPID, (void*) refPID);
	      refH = &(refrpotelem->la);
#else
	      refH = GARBAGE_AllocHandle();
	      PDEREF(refH) = (tObject*) refPID;
#endif
	    }
	  }
	  
	  /* overwrite reference with the handle */
	  ((tOBREF*) a->pvElements)[i] = refH;
	}
      }
    }
  }
#endif
  
  
  /* check if this is a swopped-out object or a persistent one */
  if (HASFLAG(o, GARBAGE_PERSISTENT)) {
    tOBREF refThing;
    
#ifndef RPOTREFS
    /* the la field is set to the handle when it's swopped out, so
       write the correct handle pointer back now */
    o->hHandle = (tOBREF) TPOT_GetLA(o->pid);
#endif
    
    /* persistent objects use RPOT entries as refs */
    refThing = RPOT_ChangeLA(o->pid, o);
    
    /* persistent objects that have been loaded from the store and were
       not actually created during this run won't have handles, so only
       change its handle if it actually has one */
    
#ifdef RPOTREFS
    if (o->hHandle) {
      *(o->hHandle) = o;
      return o->hHandle;
    }
    else {
      return refThing;
    }
#else
    *(o->hHandle) = o;
    return o->hHandle;
#endif
  }
  else {
#ifndef RPOTREFS
    o->hHandle = (tOBREF) TPOT_GetLA(o->pid);
#endif
    
    /* swopped-out objects use handles as refs */
    *(o->hHandle) = o;
    /* change TPOT entry */
    TPOT_ChangeLA(o->pid, o);
    return o->hHandle;
  }
}
#endif

/*
 * Change the local address for an RPOT entry.  This version
 * is used by the eviction / swap-in code, and updates the
 * oldHandle in the RPOT entry.
 */
tOBREF RPOT_ChangeLA (PID pid, void* newLA)
{
  tRPOTElement* temp;
  
#ifdef REVRPOT
#error how do we update the old handle?
  RPOT_Remove(pid);
  temp = RPOT_Add(pid, newLA);
#else
  /* find the RPOT entry to update */
  if ((temp = &(RPOT_table[HASHFUNC(pid)]))->pid != pid) {
    temp = temp->next;
    while (temp && temp->pid != pid) {
      temp = temp->next;
    }
  }
  assert(temp != NULL && temp->pid == pid);
  temp->la = newLA;
  if (temp->oldHandle) {
    /* update the old handle */
    *(temp->oldHandle) = (tObject *) newLA;
  }
#endif
  return &(temp->la);
}

/*
 * Change the local address for the RPOT entry corresponding to a
 * handle.  This version is used by the garbage collector and frees
 * the oldHandle in the RPOT entry.  
 */
tOBREF RPOT_ChangeLA2 (tAllocatorHeap* heap, tOBREF h, void* newLA)
{
  tRPOTElement* elem = (tRPOTElement *) h;
#ifdef REVRPOT
  PID pid = elem->pid;
#endif

#ifdef DEBUG
  assert(h);
  assert(RPOT_InRPOT(h));
#endif
  
  if (elem->oldHandle) {
#ifdef DEBUG
    assert(elem->oldHandle != h);
#endif
    HANDLES_ReleaseHandle(heap, elem->oldHandle);
    elem->oldHandle = NULL;
  }

#ifdef REVRPOT
  RPOT_Remove(pid);
  elem = RPOT_Add(pid, newLA);
#else
  elem->la = newLA;
#endif

#ifdef DEBUG
  assert(((tObject *) newLA)->hHandle == h);
#endif
  return &(elem->la);
}


/*
 * Handles updating an object, including all objects reachable from it
 * that also need updating (or promoting).
 */

#ifndef EVICTIONS
void RPOT_Stabilise (tObject* o)
{
  int i;
  tField* field;
  tOBREF refH;
  tObject* ref;
  tArray* a;
#ifdef RESWIZZLE
  void* pvStore = NULL;
#endif
  int iFinished = 0;
  tClassLoaderTuple* pstCurrType;
  
  /* if this object has neither its PERSIST_Updated or PERSIST_RefsUpdated
     flags set and is already persistent then we can forget about it */
#ifdef REVRPOT
  if (((o->i32Flags & PERSIST_DirtyMask) == 0) && RPOT_GetPID(o)) {
    return;
  }
#else
  
#ifdef DEBUG
  assert(o->pstType);
#endif
  if (((o->i32Flags & PERSIST_DirtyMask) == 0) && o->pid) {
    return;
  }
#endif
  
  /* set persistent flag */
  o->i32Flags |= GARBAGE_PERSISTENT;

  /* clear update flags we do this here instead of at the end so that
     this object will be skipped if it is reachable via one of its own
     references */
  o->i32Flags &= ~PERSIST_DirtyMask;

  /* if the object is non-persistent, promote it now ... for the same
     reason as above */
  if (o->pid == 0) {
    RPOT_PromoteObject(o);
  }
  
  /* if object is a class instance */
  if ((o->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT) {
    /* if this object doesn't have any variables then we don't need to
       do anything here */
    if (o->pi32Vars) {
#ifdef RESWIZZLE
      /* store fields so that we can write them back afterwards */
      pvStore = sys_malloc(o->pstType->u16InstSize * sizeof(int32));
#ifdef DEBUG
      assert(pvStore);
#endif
      memcpy(pvStore, o->pi32Vars, o->pstType->u16InstSize * sizeof(int32));
#endif
      
      /* Checking the fields is complicated because information about
         fields that are inherited from a parent class are not stored
         in the child class's class definition. We have to check the
         parent classes until we find one whose first field is stored
         in slot 0, i.e. one of its own fields is the first one which
         means it does not inherit any. */

      for (pstCurrType = o->pstType; 
	   iFinished == 0; 
	   pstCurrType = pstCurrType->pstClass->pstSuperClass) {
        for (i = 0; i < pstCurrType->pstClass->u16InstCount; i++) {
          field = &(pstCurrType->pstClass->pstInstOffsets[i]);
          switch (field->uidFieldSig[0]) {
	  case '[':
	  case 'L':
            {
              refH = (tOBREF) o->pi32Vars[field->u16Offset];
              /* only bother if it's not null */
              if (refH) {
                if (ISPID(refH)) {
                  continue;
                }
		/* keep this like this... hope for no swopped out objects */
                ref = DEREF(refH);  
                RPOT_Stabilise(ref);
#ifdef DEBUG
                assert(ref->pid);
#endif
                /* unswizzle the reference field */
#ifdef REVRPOT
                o->pi32Vars[field->u16Offset] = RPOT_GetPID(ref);
#else
                o->pi32Vars[field->u16Offset] = ref->pid;
#endif
              }
              break;
            }
          }
        }

        /* if we've got up to the Object class or we've got to a class that
           does not inherit anything then we have finished */
        if ((pstCurrType->pstClass->pstSuperClass == NULL) ||
            ((pstCurrType->pstClass->pstInstOffsets) &&
	     (pstCurrType->pstClass->pstInstOffsets[0].u16Offset == 0))) {
          iFinished = 1;
        }
      }
    }
  }
  /* if object is an array */
  else {
    a = (tArray*) o;
    
    /* if the array is not an array of arrays or of objects then we don't
       have to do anything */
    if ((a->enType == T_ARRAY) || (a->enType == T_OBJECT)) {
#ifdef RESWIZZLE
      /* store fields so that we can write them back afterwards */
      pvStore = sys_malloc(a->i32Number * sizeof(int32));
#ifdef DEBUG
      assert(pvStore);
#endif
      memcpy(pvStore, a->pvElements, a->i32Number * sizeof(int32));
#endif

      for (i = 0; i < a->i32Number; i++) {
        refH = ((tOBREF*) a->pvElements)[i];
        /* if not a null reference */
        if (refH) {
          /* stabilise object */
          /* if object is not already persistent then it'll be promoted during
             the call */
          if (ISPID(refH)) {
            continue;
          }
          ref = DEREF(refH);
          RPOT_Stabilise(ref);
	  
#ifdef DEBUG
#ifndef USEOSKIT
          assert(ref->pid);
#endif
#endif
          /* unswizzle the slot so that the current object will
             be saved properly */
#ifdef REVRPOT
          ((tObject**) a->pvElements)[i] = (tObject*) RPOT_GetPID(ref);
#else
          ((tObject**) a->pvElements)[i] = (tObject*) ref->pid;
#endif
        }
      }
    }
  } //end else

  /* Save the object's state.  If this is a newly promoted object, this
     is where it gets written for the first time. */
#ifdef REVRPOT
  RPOT_UpdateObject(RPOT_GetPID(o), o);
#else
  RPOT_UpdateObject(o->pid, o);
#endif
  
#ifdef RESWIZZLE
  /* don't attempt to do this if we didn't actually copy the fields - check
     by checking if pvStore was allocated */
  if (pvStore) {
    if ((o->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT) {
      /* write fields back */
      memcpy(o->pi32Vars, pvStore, o->pstType->u16InstSize * sizeof(int32));
    }
    else {
      memcpy(a->pvElements, pvStore, a->i32Number * sizeof(int32));
    }
    sys_free(pvStore);
  }
#endif
}
#endif


/*
 * Updates all objects in the RPOT that have been changed.
 */

void RPOT_StabiliseAll(void)
{
#if defined(EVICTIONS) && !defined(OLDSTABILISE)
  /*
    What we want to do in the stabilise is record the state of all objects
    onto the store.
    
    We'll go through all objects in memory and write them to the store...
    that should sort it all out. Since we're only swopping in store format,
    any objects that have been swopped out and not swopped back in can be
    left as they are on the store.
  */ 
 
#ifdef TIMING
  GLOBAL_PrintTime("Before stabilise");
#else
  tracePersist0("RPOT_StabiliseAll: before stabilise");
#endif
  panic0("RPOT_StabiliseAll: this case not supported yet");
  GARBAGE_WriteEverything();
#ifdef TIMING
  GLOBAL_PrintTime("After stabilise");
#else
  tracePersist0("RPOT_StabiliseAll: after stabilise");
#endif
#else
  int i;
  tRPOTElement* temp;
  
#ifdef TIMING
  GLOBAL_PrintTime("Before stabilise");
#else
  tracePersist0("RPOT_StabiliseAll: before stabilise");
#endif
  for (i = 0; i < RPOT_TABLESIZE; i++) {
#ifdef REVRPOT
    for (temp = RPOT_table[i]; temp; temp = temp->next) {
      RPOT_Stabilise(temp->la);
    }
#else
    
    if (RPOT_table[i].pid) {
      RPOT_Stabilise(RPOT_table[i].la);
      temp = RPOT_table[i].next;
      while (temp) {
        RPOT_Stabilise(temp->la);
        temp = temp->next;
      }
    }
#endif
  }
#ifdef TIMING
  GLOBAL_PrintTime("After stabilise");
#else
  tracePersist0("RPOT_StabiliseAll: after stabilise");
#endif
  /*  GARBAGE_gc();*/
#endif
}


/*
 * Prints out the RPOT_table for debugging.
 */
void RPOT_PrintTable(void)
{
  int i;
  tRPOTElement* temp;
  for (i = 0; i < RPOT_TABLESIZE; i++) {
    if (RPOT_table[i].pid) {
      eprintf("pid %lx ", (uint32)(RPOT_table[i].pid));
      temp = RPOT_table[i].next;
      while (temp) {
        temp = temp->next;
      }
    }
  }
}

#endif //GARBAGE_COPYING
#endif //GARBAGE_GENERATIONAL
#endif //PERSIST
