/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>


#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <pthread.h>
#endif

#include "lib/indigenous/java.lang/VMSystem.h"
#include "lib/indigenous/java.lang/VMClassLoader.h"
#include "vm/threads_native.h"
#include "vm/interp.h"
#include "vm/jni.h"
#include "vm/interp_methods.h"
#include "vm/classfile_methods.h"
#include "vm/garbage.h"
#include "vm/garbage2.h"
#include "vm/newobject.h"
#include "vm/interp/methodstacks.h"
#include "vm/threadinfo.h"
#include "vm/startup.h"


/* global variables ********************************************************/


extern int INTERP_InitialStackSize; //This can be set by the ProcessArgs routine 
extern int canStartOptimising; //For the run-time-optimiser
int canStartStackMaps = 0; //For the stack maps
int showLoading = 0; //Show the classes loaded by the interpreter
int showClasspath = 0; //Show our classpaths
int respectGCCalls = 0; //Don't ignore calls to java.lang.Runtime.gc()

// Decides whether we are allowed to put objects into the persistent store
int use_persistent_store = 0;


static FILE* findConfigFile(char **filename);
static int scanConfigFile(FILE* fp);


static void initClassPath(tClassPath* classpath) 
{
  classpath->ppszClassPaths = NULL;
  classpath->u16NumClassPaths = 0;
}


/* Add 'pathItem' to 'pathList', reallocating as required */
static void addClasspathEntry(char* pathItem, tClassPath* classpath)
{
  if (classpath->ppszClassPaths == NULL) {
    classpath->ppszClassPaths = (char **) malloc(sizeof(char *) * 2);
    classpath->ppszClassPaths[0] = pathItem;
    classpath->ppszClassPaths[1] = NULL;
    classpath->u16NumClassPaths = 1;
  }
  else {
    int len = sizeof(char *) * (classpath->u16NumClassPaths + 2);
    classpath->ppszClassPaths =
      (char **) realloc(classpath->ppszClassPaths, len);
    classpath->ppszClassPaths[classpath->u16NumClassPaths++] = pathItem;
    classpath->ppszClassPaths[classpath->u16NumClassPaths] = 0;
  }
}  

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

/* Checks an entry and adds it if valid */
static int checkClasspathEntry(char* item)
{
  struct stat buf;
  int statval;

  int isZip = 0;
  int itemLen = strlen(item);
  
  if (itemLen == 0) {
    /* silently drop zero-length classpath entries */
    return 0;
  }
  if (itemLen > 4 && (strstr(item, ".zip") == (item + itemLen - 4))) {
    isZip = 1;
  }
  else if (itemLen > 4 && (strstr(item, ".jar") == (item + itemLen - 4))) {
    isZip = 1;
  }
  statval = stat(item, &buf);
  if (!isZip) {
    if (statval == 0 && S_ISDIR(buf.st_mode)) {
      return 1;
    }
    else {
      eprintf("WARNING, %s is not a valid directory\n", item);
      return 0;
    }
  }
  else {
    if ((statval == 0) && (S_ISREG(buf.st_mode))) {
      return 1;
    }
    else {
      eprintf("WARNING, %s is not a valid zip/jar file\n", item);
      return 0;
    }
  }
}

/* Parse a ':' separated string, adding names to the path structure*/
static void parseClasspath(char *pathString, tClassPath* classpath) 
{
  char* front = pathString;
  char* back = front;

  while ((*back != '\0') && (*back != '\n')) {
    if (*back == ':') {
      char* item = malloc(back - front + 1);
      strncpy(item, front, back - front);
      item[back - front] = '\0';
      
      if (checkClasspathEntry(item)) {
	addClasspathEntry(item, classpath);
      }
      
      front = back + 1;
    }
    else if (*(back + 1) == '\0') {
      char* item = malloc(back - front + 2);
      strncpy(item, front, back - front + 1);
      item[back - front + 1] = '\0';
      
      if (checkClasspathEntry(item)) {
	addClasspathEntry(item, classpath);
      }
      
      return;
    }
    back++;
  }
}

/* Find a .kissmerc file */
static FILE* findConfigFile(char **filename)
{
  char* home;

  FILE* fp = fopen("./.kissmerc", "r");
  if (fp != NULL) {
    *filename = "./.kissmerc";
    return fp;
  }
  home = getenv("HOME");
  if (home != NULL) {
    char *tmp = malloc(strlen(home) + 20);
    strcpy(tmp, home);
    strcat(tmp, "/.kissme/kissmerc");
    fp = fopen(tmp, "r");
    if (fp != NULL) {
      *filename = tmp;
      return fp;
    }
    free(tmp);
  }
  fp = fopen("/etc/kissme/kissmerc", "r");
  if (fp != NULL) {
    *filename = "/etc/kissme/kissmerc";
  }
  return fp;
}

/* Read the contents of 'fp' looking for a classpath line */
static int scanConfigFile(FILE* fp)
{
#define MAX_CONF_PROP_LEN 20
#define CONF_VALUE_LEN 64
  int line = 1;
  int seenClasspath = 0;
  int seenBootClasspath = 0;
  int ch;
  
  do {
    char prop[MAX_CONF_PROP_LEN + 1];
    char *value;
    int i, valueLen;
    
    /* skip whitespace before the property name */
    ch = fgetc(fp);
    while (ch == ' ' || ch == '\t' || ch == '\n') {
      ch = fgetc(fp);
    }

    if ((int) ch == -1) {
      return -1;
    }
    if (ch == '\n') {
      continue;
    }

    /* gather the property name */
    for (i = 0; i < MAX_CONF_PROP_LEN; i++) {
      if (!((ch >= 'A' && ch <= 'Z') || (ch >= 'a' && ch <= 'z') ||
	    (ch >= '0' && ch <= '9') || ch == '.')) {
	break;
      }
      prop[i] = (char) ch;
      ch = fgetc(fp);
    }
    prop[i] = '\0';
    
    /* skip whitespace after the property name */
    while (ch == ' ' || ch == '\t') {
      ch = fgetc(fp);
    }

    /* look for the property/value separator */
    if (ch != '=') {
      eprintf("Error in kissmerc at line %i: missing '='\n", line);
      return 0;
    }

    /* skip whitespace after the '=' */
    ch = fgetc(fp);
    while (ch == ' ' || ch == '\t') {
      ch = fgetc(fp);
    }

    /* gather the value */
    valueLen = CONF_VALUE_LEN;
    value = (char *) malloc(valueLen + 1);
    for (i = 0; ch != '\n' && ch != ' ' && ch != '\t' && ch != EOF; i++) {
      if (i >= valueLen) {
	valueLen += CONF_VALUE_LEN;
	value = (char *) realloc(value, valueLen);
      }
      value[i] = (char) ch;
      ch = fgetc(fp);
    }
    value[i] = '\0';

    while(ch == ' ' || ch == '\t')
      ch = fgetc(fp);

    /* process the property / value pair */
    if (strcmp(prop, "classpath") == 0) {
      if (seenClasspath++) {
	eprintf("Error in kissmerc at line %i: multiple classpaths",
		line);
	return 0;
      }
      parseClasspath(value, &CLASSFILE_UserClassPath);
    }
    else if (strcmp(prop, "bootclasspath") == 0) {
      if (seenBootClasspath++) {
	eprintf("Error in kissmerc at line %i: multiple bootclasspaths",
		line);
	return 0;
      }
      parseClasspath(value, &CLASSFILE_BootClassPath);
    }
    else {
      eprintf("Error in kissmerc at line %i: unknown property %s \n",
	      line, prop);
    }

    /* tidy up */
    free(value);
    line++;

  } while (ch != EOF);
  return -1;
}


static long scaledLong(char *string) 
{
  char *end;
  long res = strtol(string, &end, 10);
  if (end[0] == '\0') {
    return res;
  }
  else if ((end[0] == 'k' || end[0] == 'K') && end[1] == '\0') {
    return res * 1024;
  }
  else if ((end[0] == 'm' || end[0] == 'M') && end[1] == '\0') {
    return res * 1024 * 1024;
  }
  else if ((end[0] == 'g' || end[0] == 'G') && end[1] == '\0') {
    return res * 1024 * 1024 * 1024;
  }
  else if ((end[0] == 't' || end[0] == 'T') && end[1] == '\0') {
    return res * 1024 * 1024 * 1024 * 1024;
  }
  else {
    return -1;
  }
}

static void printVersion()
{
  eprintf("kissme %s (%s) - This program is free software\n\n",
          KISSME_VERSION, KISSME_BUILD_DATE);
}

static void printHelp(char** argv)
{
  printVersion();
  eprintf("Usage: %s [options] <main class> [program arguments]\n", argv[0]);
  eprint("Where [options] can be:\n");
  eprint("  -cp or -classpath <path>  specifies the user class path\n");
  eprint("  -bootclasspath <path>     specifies the classpath for system libraries\n");
  eprint("  -store <name>             enables persistence using store <name>\n");
#ifdef ALLOCATOR
  eprint("  -heap <size>              sets the maximum heap size\n");
#else
  eprint("  -heap <size>              currently ignored\n");
#endif
  eprint("  -oss <size>               sets the pre-thread maximum stack size\n");
#ifdef EVICTIONS
  eprint("  -ef <num>/<num>           currently ignored\n");
  eprint("  -sm <num>,<num>           currently ignored\n");
#endif
#ifdef TRACING
  eprint("  -t (<name>|<num>)[,...][=<path>]\n");
  eprint("  -t all[=<path>]\n");
  eprint("                            enables logging channels\n");
  eprint("                            '=<path>' sends tracing to <path>\n");
  eprint("  -t list                   list all channels before starting\n");
#endif
  eprint("  -v                        turns on verbose mode\n");
  eprint("  -vv                       turns on very verbose mode\n");
  eprint("  -nojit                    currently ignored\n");
  eprint("  -D<propname>=<value>      specifies a System property value\n");
  eprint("\n");
  eprint("  -h or -help               prints this information\n");
  eprint("  -version                  prints kissme version information\n");
  eprint("\n");
  eprintf("Typical usage: %s org.myorg.MyClassFile\n", argv[0]);
}


static int processTraceOption(char *arg) 
{
  char *buffer = malloc(strlen(arg) + 1);
  char *cp, *cp2;
  FILE *file = NULL;

  strcpy(buffer, arg);
  cp = buffer;
  cp2 = strchr(cp, '=');

  // Separate the fileName (if any)
  if (cp2) {
    *cp2 = 0;
    cp2++;
    if (*cp2) {
      file = fopen(cp2, "w");
      if (file == NULL) {
#ifdef KISSME_LINUX_USER
	eprintf("Cannot open log file %s: %s\n", cp2, sys_errlist[errno]);
#else
	eprintf("Cannot open log file %s\n", cp2);
#endif
	return 0;
      }
    }
    else {
      eprint("Missing <file> following '=' in trace argument\n");
      return 0;
    }
  }

  // Interpret the channel name or number
  if (strcmp(cp, "all") == 0) {
    int i;
    for (i = 0; i <= LAST_DIAG_CHANNEL; i++) {
      DIAG_SetChannelState(i, 1);
      if (file) {
	DIAG_SetChannelFile(i, file);
      }
    }
  }
  else {
    long channel = DIAG_LookupChannel(cp);
    if (channel == -1) {
      char *end;
      channel = strtoul(cp, &end, 10);
      if (*end != 0) {
	eprintf("Unrecognizable channel name or number (%s)\n", cp);
	return 0; 
      }
      if (channel < 0 || channel > LAST_DIAG_CHANNEL) {
	eprintf("Channel number out of range(%s)\n", cp);
	return 0;
      }
    }
    DIAG_SetChannelState(channel, 1);
    if (file) {
      DIAG_SetChannelFile(channel, file);
    }
  }
  return 1;
}


/*
 * @doc FUNC
 * @func
 * This function scans the command line args for kissme options
 * and processes them.  Also loads the kissmerc file if required.
 * @rdesc Returns the index of the first non-option argument or
 *        negative value if there was an error.
 */

static int processOptions
(
  int argc,               /* @parm Number of arguments */
  char** argv,            /* @parm Argument vector */
  char** ppszStoreName    /* @parm Name of persistent store */
)
{
  int i;
#ifdef TRACING
  int listChannels = 0;
#endif
  char *classpathArg = NULL;
  char *bootClasspathArg = NULL;

#ifdef TRACING
  DIAG_InitChannels();
#endif
  
  for (i = 1; i < argc && argv[i][0] == '-'; i++) {
    if (strcmp(argv[i], "--help") == 0 ||
        strcmp(argv[i], "-help") == 0 || 
	strcmp(argv[i], "-h") == 0) {
      printHelp(argv);
      return -2;
    }
    else if (strcmp(argv[i], "--version") == 0 ||
	     strcmp(argv[i], "-version") == 0) {
      printVersion();
      return -2;
    }
    else if (strcmp(argv[i], "-cp") == 0 || 
	     strcmp(argv[i], "-classpath") == 0 ||
	     strcmp(argv[i], "--cp") == 0 || 
	     strcmp(argv[i], "--classpath") == 0) {
      if (i + 1 >= argc) {
	eprintf("Missing classpath after '%s'\n", argv[i]);
	return -1;
      }
      classpathArg = argv[++i];
    }
    else if (strcmp(argv[i], "-oss") == 0 ||
	     strcmp(argv[i], "--oss") == 0) {
      long oss;
      if (i + 1 >= argc) {
	eprintf("Missing stack size after '%s'\n", argv[i]);
	return -1;
      }
      oss = scaledLong(argv[++i]);
      if (oss < 0) {
	eprintf("Invalid stack size\n");
	return -1;
      }
      INTERP_InitialStackSize = (int) oss;
    }
    else if (strcmp(argv[i], "-bootclasspath") == 0 ||
	     strcmp(argv[i], "--bootclasspath") == 0) {
      if (i + 1 >= argc) {
	eprintf("Missing classpath after '%s'\n", argv[i]);
	return -1;
      }
      bootClasspathArg = argv[++i];
    }
    else if (strcmp(argv[i], "-store") == 0 ||
	     strcmp(argv[i], "--store") == 0) {
      if (i + 1 >= argc) {
	eprintf("Missing store name after '%s'\n", argv[i]);
	return -1;
      }
      use_persistent_store = 1;
      *ppszStoreName = argv[++i];
    }
    else if (strcmp(argv[i], "-heap") == 0 ||
	     strcmp(argv[i], "--heap") == 0) {
      long hsize;
      if (i + 1 >= argc) {
	eprintf("Missing heap size after '%s'\n", argv[i]);
	return -1;
      }
#ifdef ALLOCATOR
      hsize = scaledLong(argv[++i]);
      if (hsize < 0) {
	eprintf("Invalid heap size\n");
	return -1;
      }
      ALLOCATOR_SetHeapSize(hsize);
#else
      eprintf("%s option ignored\n", argv[i++]);
#endif
    }
#ifdef EVICTIONS
    // The -ef and -sm options appear to set GC parameters that
    // are not used by anything.
    else if (strcmp(argv[i], "-ef") == 0) {
      if (i + 1 >= argc) {
	eprintf("Missing parameter after '%s'\n", argv[i]);
	return -1;
      }
      sscanf(argv[++i], "%ld/%ld", &GARBAGE_FREEMEM_NUMER, 
	     &GARBAGE_FREEMEM_DENOM);
    }
    else if (strcmp(argv[i], "-sm") == 0) {
      if (i + 1 >= argc) {
	eprintf("Missing parameter after '%s'\n", argv[i]);
	return -1;
      }
      sscanf(argv[++i], "%ld,%ld", &GARBAGE_NUMDEFFRAMES, 
	     &GARBAGE_NUMDEFREACH);
    }
#endif
    else if (strcmp(argv[i], "-nojit") == 0 || 
	     strcmp(argv[i], "--nojit") == 0) {
      eprintf("%s option ignored\n", argv[i]);
    }
    else if (strcmp(argv[i], "-t") == 0) {
#ifdef TRACING
      if (i + 1 >= argc) {
	eprintf("Missing trace parameter after '%s'\n", argv[i]);
	return -1;
      }
      i++;
      if (strcmp(argv[i], "list") == 0) {
	listChannels = 1;
      }
      else if (!processTraceOption(argv[i])) {
	return -1;
      }
#else
      eprintf("Option '-t' is not available.\n");
      return -1;
#endif
    }
    else if (strcmp(argv[i], "-gc") == 0) {
      respectGCCalls = 1; //Don't ignore calls to Runtime.gc()
    }
    else if (strcmp(argv[i], "-v") == 0) {
      showLoading = 1; //Show which and where classes are loaded from 
    }
    else if (strcmp(argv[i], "-vv") == 0) {
      showLoading = 1; //Show which and where classes are loaded from 
      showClasspath = 1; //show what we're using for a classpath
    }
    else if (strncmp(argv[i], "-D", 2) == 0) {
      // A property argument ... just check that it is 'well-formed'
      int eqPos = index(argv[i], '=') - argv[i];
      if (eqPos <= 2 || eqPos == strlen(argv[i]) - 1) {
	eprintf("Malformed property option '%s'\n", argv[i]);
	return -1;
      }
    }
    else {
      eprintf("Unrecognised option '%s'\n", argv[i]);
      return -1;
    }
  }

#ifdef TRACING
  if (listChannels) {
    eprint("The following trace channels are defined\n");
    DIAG_ListChannels();
  }
#endif
  
  initClassPath(&CLASSFILE_BootClassPath);
  initClassPath(&CLASSFILE_UserClassPath);

  if (bootClasspathArg) {
    parseClasspath(bootClasspathArg, &CLASSFILE_BootClassPath);
  }
  else
    {
      char *filename;
      FILE *fp = findConfigFile(&filename);
      if (fp != NULL) {
	if (!scanConfigFile(fp)) {
	  eprintf("Error occurred loading config file '%s'\n", filename);
	  return -1;
	}
      }
    }

  if (classpathArg) {
    parseClasspath(classpathArg, &CLASSFILE_UserClassPath);
  }
  else {
    char *envString = getenv("CLASSPATH");
    if (envString != NULL) {
      parseClasspath(envString, &CLASSFILE_UserClassPath);
    }
    else {
      /* When no user path was specified, use the current directory */
      addClasspathEntry(".", &CLASSFILE_UserClassPath);   
    }
  }

  return i;
}

//we count all threads that are created and destroyed
volatile int totalThreadsInSystem; 


static void usage(char *progName) {
  eprintf("Usage: %s [options] <main class> [program arguments]\n", 
	  progName);
  eprint("Where common [options] are:\n");
  eprint("  -cp | --classpath <path>  to specify a class path\n");
  eprint("  -h | --help               to list the possible options\n");
}


/**
 * The main function processes command line options and then calls
 * the startup routine 
 **/
int main(int argc, char* argv[], char* c_env[])
{
  char* pszMainClass = NULL;  
  char* pszStoreName = "persistent_store";
  int argsUsed = -1;

  if (argc < 2) {
    usage(argv[0]);
    return -1;
  }

  // process the kissme options
  argsUsed = processOptions(argc, argv, &pszStoreName);
  if (argsUsed < 0) {
    if (argsUsed != -2) {
      usage(argv[0]);
    }
    return -2;
  }

  // Check that there is a class
  if (argsUsed >= argc) {
    eprint("You didn't specify the class to be run.\n");
    usage(argv[0]);
    return -3;
  }
  pszMainClass = argv[argsUsed++];

  if (showClasspath == 1) {
    int m = 0; 
    eprint("User classpath:\n");
    for (m = 0; m < CLASSFILE_UserClassPath.u16NumClassPaths; m++) {
      eprintf("%i: %s\n", m + 1, CLASSFILE_UserClassPath.ppszClassPaths[m]);
    }
    eprint("Bootstrap classpath:\n");
    for (m = 0; m < CLASSFILE_BootClassPath.u16NumClassPaths; m++) {
      eprintf("%i: %s\n", m + 1, CLASSFILE_BootClassPath.ppszClassPaths[m]);
    }
  }
  
  // These will be used to extract the properties
  java_lang_VMSystem_setArgsPointer(argv, argc);

  // This little piggy went to market ...
  return STARTUP_startup(pszMainClass, pszStoreName, argc, argv, argsUsed);
}


