/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the teaseme project, which in turn is part of 
 * the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* 
 * @doc MODULE 
 * @module jutils.c | 
 * 
 * This file contains general utility functions. Memory allocation functions 
 * that check allocation and record allocation are provided. Stack and queue 
 * manipulation functions are provided as well. 
 * 
 */ 

#include "config.h"

#include "vm/wrappers.h"

#ifdef KISSME_LINUX_USER
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#else
#ifdef KISSME_LINUX_KERNEL
#include <linux/time.h> 
#include <linux/types.h> 
#include <linux/sched.h> 

#include <linux/kernel.h> 
#include <linux/malloc.h> 
 #else
//RJK
 #endif
#endif

#include "vm/jutils.h" 
 
#include "vm/global.h" 
 
 
#ifdef OLDMEMTRACKING 
/* for counting the total amount of memory that has been allocated with 
   calls to UTILS_mmalloc */ 
 
static unsigned long int iTotalMem = 0; 
static unsigned long int iTotalMem1 = 0; 
 
/* 
 * @doc FUNC 
 * @func 
 * This function allocates the requested amount of memory and checks for 
 * the success of the allocation.  If the allocation was unsuccessful, it 
 * exits with an error message. 
 * 
 * @rdesc Returns a pointer to the allocated memory, if successful. 
 * 
 */ 
 
void* UTILS_mmalloc 
( 
  size_t size   /* @parm Number of bytes to allocate */ 
) 
{ 
  void* ptr; 
 
  ptr = (void*) sys_malloc(size); 
  if (ptr == NULL) 
  { 
    fatalError("UTILS_mmalloc: out of memory allocating %u bytes", size); 
  } 
/*  iTotalMem += size;*/ 
  return ptr; 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * This function allocates the requested amount of memory and checks for 
 * the success of the allocation.  If the allocation was unsuccessful, it 
 * exits with an error message.  The amount that was requested is recorded 
 * so that memory allocation can be checked. 
 * 
 * @rdesc Returns a pointer to the allocated memory, if successful. 
 * 
 */ 
 
void* UTILS_mmalloc1 
( 
  size_t size   /* @parm Number of bytes to allocate */ 
) 
{ 
  void* ptr; 
 
  ptr = (void*)kmalloc(size, GFP_KERNEL); 
  if (ptr == NULL) 
  { 
    fatalError("UTILS_mmalloc1: out of memory allocating %u bytes", size); 
  } 
  iTotalMem1 += size; 
  return ptr; 
} 
 
 
/* 
 * @doc FUNC 
 * @func 
 * This function prints out the memory that has been allocated by the 
 * memory allocation functions. 
 * 
 */ 
 
void UTILS_PrintMemUsed 
( 
  void 
) 
{ 
  eprintf("Total memory allocated by UTILS_mmalloc: %lu bytes\n", 
	  iTotalMem); 
  iTotalMem = 0; 
  eprintf("Total memory allocated by UTILS_mmalloc1: %lu bytes\n", 
	  iTotalMem1); 
  iTotalMem1 = 0; 
} 
 
#endif /* OLDMEMTRACKING */ 
 
/* 
 * @doc FUNC 
 * @func 
 * This function creates a new stack of integers of the specified size.  If 
 * the stack becomes full during use, it is increased in size by the specified 
 * amount. 
 * 
 * @rdesc Returns a pointer to the newly created UTILS_stack structure. 
 * 
 */ 
 
UTILS_stack* UTILS_Stack_New 
( 
  int initsize,   /* @parm Initial size of stack */ 
  int deltasize   /* @parm Amount to increase size by if stack is full */ 
) 
{ 
	UTILS_stack *new_stack; 
	new_stack = (UTILS_stack *) sys_malloc(sizeof(UTILS_stack)); 
	if (new_stack == NULL) 
	  { 
	    panic("UTILS_Stack_New: cannot allocate a UTILS_stack struct"); 
	  } 
	new_stack->count = 0; 
	new_stack->max_len = initsize; 
	new_stack->delta_len = deltasize; 
	new_stack->elements = (int *) sys_malloc(sizeof(int)* initsize); 
	if (new_stack->elements == NULL) 
	  { 
	    panic("UTILS_Stack_New: cannot allocate UTILS_stack elements"); 
	  } 
 
	return new_stack; 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * This function destroys a stack previously created with UTILS_Stack_New 
 * 
 */ 
 
void UTILS_Stack_Destroy 
( 
  UTILS_stack* the_stack  /* @parm Stack to destroy */ 
) 
{ 
	sys_free(the_stack->elements); 
	sys_free(the_stack); 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * This function pops the top element off a stack. 
 * 
 * @rdesc The integer that was at the top of the stack. 
 * 
 */ 
 
int UTILS_Stack_Pop 
( 
  UTILS_stack* the_stack  /* @parm The stack to pop from */ 
) 
{ 
  if (the_stack->count == 0) 
    { 
      panic0("UTILS_Stack_Pop: Error: Stack empty"); 
    } 
  --the_stack->count; 
  return the_stack->elements[the_stack->count]; 
} 
 
int UTILS_Stack_PopTop 
( 
 UTILS_stack* the_stack 
 ) 
{ 
  int temp,i; 
  if(the_stack->count == 0){ 
    panic0("UTILS_Stack_PopTop: Error: Stack empty"); 
  } 
  temp = the_stack->elements[0]; 
   
  for(i = 1; i < the_stack->count; i++) 
    the_stack->elements[i-1] = the_stack->elements[i]; 
   
  /* can't use memcpy: man page = " If src and dest overlap, the behavior of memcpy is undefined. " */ 
   
  --the_stack->count; 
  return temp; 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * This function pushes an element onto a stack. 
 * 
 */ 
 
void UTILS_Stack_Push 
( 
 UTILS_stack* the_stack,  /* @parm Stack to push onto */ 
 int value                /* @parm Value to push */ 
 ) 
{ 
  int *new_elem; 
   
  if(the_stack->count == the_stack->max_len){ 
     
    int i;
    new_elem = (int *) sys_malloc(sizeof(int) * (the_stack->max_len + the_stack->delta_len)); 
    if(new_elem == NULL){ 
      panic("UTILS_Stack_Push: cannot reallocate UTILS_stack elements"); 
    } 

    for(i = 0; i < the_stack->count;i++)
      new_elem[i] = the_stack->elements[i];

    sys_free( the_stack->elements); 
    the_stack->elements = new_elem; 
    the_stack->max_len += the_stack->delta_len; 
  } 
   
  the_stack->elements[the_stack->count] = value; 
  ++the_stack->count; 
} 
 
 
