/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme/teaseme project, which in turn is part 
 * of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

/*
 * @doc MODULE
 * @module interp.c |
 *
 * This file contains the core of the virtual machine.  It contains the
 * Interpret function that executes the bytecode.  It also contains the
 * functions for manipulating the environment stack.
 *
 */

/*
 * @doc TOPIC
 * @topic Interpreter |
 *
 * The interpreter is what executes the bytecode. Functions in other modules
 * are called as the interpreter requires them.
 *
 * Before the Interpret() function is called, all the classes that are
 * required should be loaded with calls to LoadClass(). An environment stack
 * frame must be created in order to pass to Interpret(). This environment
 * frame needs to be created by InitialStackFrame(). The method to be executed
 * has to be found before InitialStackFrame() is called.
 *
 * The interpreter has to be told whether to execute in single-threaded or
 * multi-threaded mode. Multi-threaded mode is the usual mode of execution
 * but single-threaded mode is needed for class initialisations and for
 * Java methods that are called from native methods.
 *
 * @ex Example of interpreter invocation. This example shows the use for
 *     multi-threaded mode. A java.lang.Thread object has to be created and
 *     initialised for the main method. |
 *
 * tClass*      pstClass;
 * tMethod*     pstMethod;
 * tStackFrame* pstFrame;
 * tOBREF     pstMainThread;
 * int32*       i32ArgV;
 *
 * / * Load classes * /
 * LoadClass("classes/java/lang/Object");
 * LoadClass("classes/java/lang/Thread");
 * LoadClass("classes/java/lang/String");
 * LoadClass("classes/mine/MyClass");
 *
 * / * Find main class * /
 * pstClass =  CPLIST_Find("MyClass");
 * / * Find main method in main class * /
 * pstMethod = CLASSFILE_FIndMethod(pstClass, "main", "([Ljava/lang/String;)V");
 * / * Create an environment frame for this method * /
 * pstFrame = InitialStackFrame(pstMethod, NULL, i32ArgV);
 * / * Create a thread for the main method * /
 * pstMainThread = INTERP_NewObjectFromName("java/lang/Thread");
 * / * Initialise the thread for use with this frame * /
 * THREAD_InitMainThread(pstMainThread, 5, pstFrame);
 * / * Call the interpreter in multi-threaded mode * /
 * Interpret(pstFrame, 1);
 * / * Dispose of stack frame afterwards * /
 * PopStackFrame(pstFrame);
 *
 */


#ifdef KISSME_LINUX_USER
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#endif

#ifdef TIMING
#include <time.h>
#endif


#include "vm/wrappers.h"
#include "vm/uid.h"

#include "vm/interp.h"
#include "vm/classfil.h"
#include "vm/jni.h"
#include "vm/interp_methods.h"
#include "vm/classfile_methods.h"
#include "vm/opcodes.h"
#include "vm/stdtypes.h"
#include "vm/disass.h"

#include "vm/cplist.h"
#include "vm/jutils.h"

#include "vm/threadex.h"
#include "vm/threads_native.h"
#include "vm/garbage2.h"
#include "vm/newobject.h"
#include "vm/interp/methodstacks.h"

#include "vm/locks.h"
#include "lib/indigenous/java.lang/Class.h"
#include "lib/indigenous/java.lang/Class_Reflection.h"
#include "vm/initialize.h"

#ifdef GARBAGE
#include "vm/garbage.h"
#endif

#include "vm/global.h"

#ifdef PERSIST
#include "vm/rpot.h"
#endif

#ifdef TEASEME
#include "lib/indigenous/jos.system/processes.h"
#endif

/* local prototypes ********************************************************/

void StackInspect(tStackFrame*, int32*);


/* global variables ********************************************************/

int virtualcalls = 0; 
extern int canStartOptimising;
extern int use_persistent_store;
extern tThreadNode* pstThreadList;

extern tOBREF OOMExceptionObject;
extern tAllocatorHeap* allocHeap; //just for debugging

extern tClassLoaderTuple* pstSystemType;

#ifdef DEBUG_CLASS_FOR_GC
//for debugging, records the number of interpreter iterations since last GC
int instructions_since_gc = 0; 
#endif


/* 
 * This function prints a back trace from the given Java frame. Used
 * for debugging.
 */
void INTERP_printBackTrace(tStackFrame* pstCurrFrame)
{
  tStackFrame* iter = pstCurrFrame;

  while (iter->pstPrevFrame != NULL) {
    iter = iter->pstPrevFrame;
  }

  /*  if (strcmp(iter->pstCurrMethod->uidName, "intern") == 0 ||
      strcmp(iter->pstCurrMethod->uidName, "get") == 0 ||
      strcmp(iter->pstCurrMethod->uidName, "hashCode") == 0 ||
      strcmp(iter->pstCurrMethod->uidName, "equals") == 0) {
    return;
    }*/

  eprintf("/////////////////// Stack Trace (%p)\n", pstCurrFrame);
  while (iter != NULL) {
    eprintf(" %s.%s(%s) %i\n", iter->pstCurrMethod->pstClass->uidName, 
	    iter->pstCurrMethod->uidName, iter->pstCurrMethod->uidSignature,
	    INTERP_FigureOutLineNumber(iter));
    iter = iter->pstChildFrame;
  }
  eprintf("\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\ \n");
}


void dumpCharArray(tArray* array)
{
  int i;
  for (i = 0; i < array->i32Number; i++) {
    eprintf("%c", (byte) (((jchar*)array->pvElements)[i] & 0xff));
  }
  eprintf("\n");
}


void dumpString(jobject string)
{
  char* psz = INTERP_AscizFromString(NULL, string);
  eprintf("%s\n", psz);
  sys_free(psz);
}


/*
 * @doc FUNC
 * @func
 * This function is used to display the contents of the operand stack
 * for debugging purposes.  A pointer to the top of the operand stack
 * has to be passed to it since the operand stack pointer in the
 * environment frame that is passed is unlikely to be up to date
 * unless SAVEENV() has been called recently.
 */
void StackInspect(tStackFrame* pstCurrFrame, 
		  /* @parm Pointer to current environment frame */
		  int32 *optop    
		  /* @parm Pointer to top of operand stack */
		  )
{
  JNIEnv* env;
#ifdef DEBUG_FRAMES
  tStackFrame* checkFrame = pstCurrFrame;
  
  while (checkFrame) {
    assert(checkFrame->pstCurrClass);
    assert(checkFrame->pstCurrMethod);
    assert(checkFrame->pstCurrClass->pstClass->pu32ConstPool);
    assert(checkFrame->pstCurrMethod->uidName);
    assert(checkFrame->pstHeap);
    checkFrame = checkFrame->pstPrevFrame;
  }
#endif
  
  env = &JNI_getJNIData(sys_current_thread())->functions;
  assert(pstCurrFrame->pi32OpTop >= pstCurrFrame->pi32OpBot);
  eprintf("\t\t-- Stack: -- ");
  eprintf("\"%s.%s%s\" (line %i)\n", pstCurrFrame->pstCurrClass->uidName, 
	  pstCurrFrame->pstCurrMethod->uidName, 
	  pstCurrFrame->pstCurrMethod->uidSignature, 
	  INTERP_FigureOutLineNumber(pstCurrFrame));
  eprintf("Curr stack size %i words, max stack %i, vars %i \n", 
	  optop - pstCurrFrame->pi32OpBot, 
	  pstCurrFrame->pstCurrMethod->pstCode->u16MaxStack + 1, 
	  pstCurrFrame->pstCurrMethod->pstCode->u16MaxLocals);

  if (DIAG_ChannelState(FRAME_DIAG_CHANNEL) && pstCurrFrame->pi32Vars) {
    int lv;
    for (lv = 0; 
	 lv < (pstCurrFrame->pstCurrMethod->pstCode->u16MaxLocals);
	 lv++) {
      if (lv < pstCurrFrame->pstCurrMethod->u16ArgSize) {
	eprintf("A");
      }
      else {
	eprintf("V");
      }
      
      if (GARBAGE_InHeap(env, (tOBREF) pstCurrFrame->pi32Vars[lv])) {
	eprintf("\t\t%x %i = %x (%s)\n", 
		(int) pstCurrFrame->pi32Vars + lv,  
		(int) pstCurrFrame->pi32Vars[lv], 
		(int) pstCurrFrame->pi32Vars[lv], 
		ODEREF((tOBREF) pstCurrFrame->pi32Vars[lv])->pstType->uidName);
	if(strcmp(ODEREF((tOBREF)pstCurrFrame->pi32Vars[lv])->pstType->uidName,
		  "[C") == 0) {
	  dumpCharArray(ADEREF((tARREF)pstCurrFrame->pi32Vars[lv]));
	}
      }
      else {
	eprintf("\t\t%x %i = %x\n", 
		(int) pstCurrFrame->pi32Vars + lv,  
		(int) pstCurrFrame->pi32Vars[lv], 
		(int) pstCurrFrame->pi32Vars[lv]);
      }
    }
  }
  if (DIAG_ChannelState(OPSTACK_DIAG_CHANNEL)) {
    int32* temp;
    for (temp = (pstCurrFrame->pi32OpBot + 1); 
	 temp != (optop + 1); 
	 temp++) {
      if (GARBAGE_InHeap(env, (tOBREF) temp)) {
	eprintf("Opstack\t\t%p %i = %x (%s)\n", temp, (int) (*temp),
		(int) (*temp), ODEREF((tOBREF) *temp)->pstType->uidName);
	if (strcmp(ODEREF((tOBREF)*temp)->pstType->uidName, "[C") == 0) {
	  dumpCharArray( ADEREF((tARREF) *temp));
	}
	
      }
      else {
	eprintf("Opstack\t\t%p %i = %x\n", temp, (int) (*temp), (int) (*temp));
      }
    }
  }
  eprintf("\t\t\t------------\n");
}


/* the interpreter procedure ***********************************************/

/*
 * @doc FUNC
 * @func
 * This is the procedure that executes the Java bytecode.  It takes the
 * method to execute and all other environment information from the environment
 * stack frame pointer that is passed to it.  It also needs to be told whether
 * to execute in multi-threaded mode or not.  Single-thread mode is used when
 * initialising classes or when the interpreter is called back from a native
 * function.
 *
 * The environment stack frame that is passed to Interpret() must have been
 * created beforehand by a call to InitialStackFrame().
 *
 * @ex This example shows how Interpret() would be called. i32ArgV will need
 *     to be set to point to the parameter (in this case of main, an array
 *     of strings) before passing it |
 *
 * @rdesc Returns NULL for normal exit, or a pointer to a an exception object
 *        if it exits due to an uncaught exception.
 *
 * tClass*      pstClass;
 * tMethod*     pstMethod;
 * tStackFrame* pstFrame;
 * int32*       i32ArgV;
 *
 * pstClass = CPLIST_Find("MyClass");
 * pstMethod = CLASSFILE_FIndMethod(pstClass, "main", "([Ljava/lang/String;)V");
 * pstFrame = InitialStackFrame(pstMethod, NULL, i32ArgV);
 * Interpret(pstFrame, 0);
 * PopStackFrame(pstFrame);
 */

/* 
 * This hack helps us with calling interpret from a thread. Why though?, 
 * I think Interpret was static 
 */
tOBREF INTERP_Interpret(JNIEnv* env, void* arg1, int arg2)
{
  return Interpret(env, arg1, arg2);
}


tOBREF Interpret(JNIEnv* env,
		 tStackFrame* pstInitFrame, 
		 /* @parm Initial environment frame */
		 int iMultiThread   
		 /* @parm 1 for multithread, 0 for single thread */
		 )
{
  /* the registers */
  register byte* pbPC;  //Pointer to the bytecode currently being interpreted
  register int32* optop; //Pointer to the top of the java stack
  register int32* pi32Vars; //Pointer to local variables for this method
  // Pointer to the class structure (whose method we are executing)
  register tClassLoaderTuple* pstCurrClass;  
					      
  tStackFrame* pstCurrFrame = pstInitFrame;
  assert(iMultiThread);
#ifdef PERSIST
  assert(use_persistent_store || 
	 pstCurrFrame->pstCurrClass->pstClass->persIndex == 0);
#endif

  /* initialise the registers */
  LOADENV();

  /* if the first method is native then run it */
  if (pstCurrFrame->pstCurrMethod->u16AccessFlags & ACC_NATIVE) {
    int32 i32NativeRet;
    tMethod* pstMethodTemp = pstCurrFrame->pstCurrMethod;

    if ((pstMethodTemp->u16AccessFlags & ACC_STATIC) == 0) {
      if ((tOBREF) *(optop - pstMethodTemp->u16ArgSize) == NULL) {
	// minus a + 1 for the native case
	return INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "null pointer encountered when invoking native method");
      }
      pstMethodTemp = DEREF((tOBREF) *(optop - pstMethodTemp->u16ArgSize))->
	pstType->pstClass->ppstVT[pstMethodTemp->u16VTIndex];
      
      SAVEENV();

      pstCurrFrame->pi32OpTop -= pstMethodTemp->u16ArgSize;
      
      i32NativeRet = JNI_CallNativeMethod(pstCurrFrame, pstMethodTemp);
      if (i32NativeRet == -1) {
	// The exception has been pushed onto the stack ... pop it.
	tOBREF pstExOb = (tOBREF) pstCurrFrame->pi32OpTop[0];

	pstCurrFrame->pi32OpTop -= 1;
	return pstExOb;
      }
      else {
	return NULL;
      }
    }
    else {
      SAVEENV();

      pstCurrFrame->pi32OpTop -= 1;

      i32NativeRet = JNI_CallNativeMethod(pstCurrFrame, pstMethodTemp);
      if (i32NativeRet == -1) {
	// The exception has been pushed onto the stack ... pop it.
	tOBREF pstExOb = (tOBREF) pstCurrFrame->pi32OpTop[0];

	pstCurrFrame->pi32OpTop -= 1;
	return pstExOb;
      }
      else {
	return NULL;
      }
    }
  }

  LabelLoopTop:
  while (1) {
    
#ifdef DEBUG_FRAMES
    assert(pstCurrFrame);
    assert(pstCurrFrame->pstCurrClass);
    assert(pstCurrFrame->pstCurrMethod);
    assert(pstCurrFrame->pstCurrClass->pstClass->pu32ConstPool);
    assert(pstCurrFrame->pstCurrMethod->uidName);
    assert((((pstCurrFrame->pstCurrClass->uidName[0] >= 'a') && 
	     (pstCurrFrame->pstCurrClass->uidName[0] <= 'z')) || 
	    ((pstCurrFrame->pstCurrClass->uidName[0] >= 'A') && 
	     (pstCurrFrame->pstCurrClass->uidName[0] <= 'Z')) ||
	    (pstCurrFrame->pstCurrClass->uidName[0] == '[')));
#ifdef PERSIST
    assert(use_persistent_store || 
	   pstCurrFrame->pstCurrClass->pstClass->persIndex == 0);
#endif
#endif
    
#ifdef DEBUG_CLASS_FOR_GC
    assert(pstCurrFrame->pstCurrClass->magic1 == CMAGIC1);
    assert(pstCurrFrame->pstCurrClass->magic2 == CMAGIC2);
    instructions_since_gc++;
#endif
    
    if (DIAG_ChannelState(OPCODE_DIAG_CHANNEL)) {
      eprintf("%s.%s %i: ", pstCurrFrame->pstCurrClass->uidName, 
	      pstCurrFrame->pstCurrMethod->uidName, 
	      (int) (pbPC - pstCurrFrame->pbCode));
      DISASS_PrintInstruction(pbPC, pstCurrFrame->pbCode);
    }
    
    //We take action depending on bytecode instruction
    switch (*pbPC++) {
    case nop:
      {
        break;
      }
      
      //I decided to put this in a separate file just to make it 
      //more manageable
#include "vm/interp/constants.h" 

    case bipush:
      {
	/* #### better to do this by casting. */
        if ((*pbPC) & 0x80) {  /* to see if it's negative */
	  *(++optop) = ((int) (*pbPC++)) - 256;
        }
        else {
          *(++optop) = (*pbPC++);
        }
        break;
      }

    case sipush:
      {
        if ((*pbPC) & 0x80) {
          *(++optop) = (*pbPC++) - 256;
        }
        else {
          *(++optop) = (*pbPC++);
        }
        *optop *= 256;
        *optop += (*pbPC++);
        break;
      }

    case ldc:
      {
	int16        i16Temp;
	Uid          uidBytes;
	
	i16Temp = (*pbPC++);
	
	switch (CONSTTAG(pstCurrClass->pstClass, i16Temp)) {
	case CONSTANT_Integer:
	  {
	    *(++optop) = CONSTGET_Int(pstCurrClass->pstClass, i16Temp);
	    break;
	  }
	case CONSTANT_Float:
	  {
#ifdef DEBUG_FLOAT
	    eprintf("Processing ldc, index is %i -> %12G\n", 
		    i16Temp, CONSTGET_Float(pstCurrClass->pstClass, i16Temp));
#endif
	    *((float*) (++optop)) = CONSTGET_Float(pstCurrClass->pstClass, i16Temp);
	    break;
	  }
	case CONSTANT_String:
	  {
	    /* all strings are stored as Asciz strings in UTF8 format */
	    uidBytes = 
	      CONSTGET_UidAsciz(pstCurrClass->pstClass, 
				CONSTGET_String_Index(pstCurrClass->pstClass,
						      i16Temp));
	    *(++optop) = (int32) INTERP_NewStringFromAsciz(env, uidBytes);
	    if (*optop == 0) {
	      pstCurrFrame = ThrowNullPointer(env, pstCurrFrame);
	    }
	    break;
	  }
	default:
	  {
	    panic("Interpreter error in ldc instruction. pbPC = %u",
		  pbPC - 1 - pstCurrFrame->pstCurrMethod->pstCode->pbCode);
	  }
	}
	break;
      }
      
    case ldc_w:
      {
	int16        i16Temp;
	Uid           uidBytes;
	
	i16Temp = (*pbPC++) * 256;
	i16Temp += (*pbPC++);
	switch (CONSTTAG(pstCurrClass->pstClass, i16Temp)) {
	case CONSTANT_Integer:
	  {
	    *(++optop) = CONSTGET_Int(pstCurrClass->pstClass, i16Temp);
	    break;
	  }
	case CONSTANT_Float:
	  {
#ifdef DEBUG_FLOAT
	eprintf("Processing ldc_w, index is %i -> %12G\n", 
		i16Temp, CONSTGET_Float(pstCurrClass->pstClass, i16Temp));
#endif
	    *((float *)(++optop)) = CONSTGET_Float(pstCurrClass->pstClass,
						   i16Temp);
	    break;
	  }
	case CONSTANT_String:
	  {
	    /* all strings are stored as Asciz strings in UTF8 format */
	    uidBytes = 
	      CONSTGET_UidAsciz(pstCurrClass->pstClass, 
				CONSTGET_String_Index(pstCurrClass->pstClass, 
						      i16Temp));
	    *(++optop) = (int32) INTERP_NewStringFromAsciz(env, uidBytes);
	    break;
	  }
	default:
	  {
	    panic("Interpreter error in ldc_w instruction. pbPC = %u",
		  pbPC - 2 - pstCurrFrame->pstCurrMethod->pstCode->pbCode);
	  }
	}
	break;
      } 

    case ldc2_w:
      {
	int16 i16Temp;
	
	i16Temp = (*pbPC++) * 256;
	i16Temp += (*pbPC++);
	
	switch (CONSTTAG(pstCurrClass->pstClass, i16Temp)) {
	case CONSTANT_Long:
	  {
#if(defined __powerpc__)
	    *(++optop) = CONSTGET_Int(pstCurrClass->pstClass, i16Temp + 1);
	    *(++optop) = CONSTGET_Int(pstCurrClass->pstClass, i16Temp);
#else
	    *(++optop) = CONSTGET_Int(pstCurrClass->pstClass, i16Temp);
	    *(++optop) = CONSTGET_Int(pstCurrClass->pstClass, i16Temp + 1);
#endif
	    break;
	  }
	case CONSTANT_Double:
	  {
	    *(++optop) = CONSTGET_Double_High(pstCurrClass->pstClass, i16Temp);
	    *(++optop) = CONSTGET_Double_Low(pstCurrClass->pstClass, i16Temp);
	    break;
	  }
	default:
	  {
	    panic("Interpreter error in ldc2_w instruction. pbPC = %u\n",
		  pbPC - 2 - pstCurrFrame->pstCurrMethod->pstCode->pbCode);
	  }
	}
	break;
      }
      
      //All the load and store instructions are here
      //Also the array load and store instructions
#include "vm/interp/load_and_store.h" 
      
      /* Misc stack manipulation */
    case pop:
      {
	optop--;
	break;
      }
      
    case pop2:
      {
	optop -= 2;
	break;
      }
      
    case dup_:
      {
	optop++;
	*optop = *(optop - 1);
	break;
      }
      
    case dup_x1:
      {
	optop++;
	*optop = *(optop - 1);
	*(optop - 1) = *(optop - 2);
	*(optop - 2) = *optop;
	break;
      }
      
    case dup_x2:
      {
	optop++;
	*optop = *(optop - 1);
	*(optop - 1) = *(optop - 2);
	*(optop - 2) = *(optop - 3);
	*(optop - 3) = *optop;
	break;
      }
      
    case dup2_:
      {
	optop += 2;
	*optop = *(optop - 2);
	*(optop - 1) = *(optop - 3);
	break;
      }
      
    case dup2_x1:
      {
        optop += 2;
        *optop = *(optop - 2);
        *(optop - 1) = *(optop - 3);
        *(optop - 2) = *(optop - 4);
        *(optop - 3) = *optop;
        *(optop - 4) = *(optop - 1);
        break;
      }
      
    case dup2_x2:
      {
        optop += 2;
        *optop = *(optop - 2);
        *(optop - 1) = *(optop - 3);
        *(optop - 2) = *(optop - 4);
        *(optop - 3) = *(optop - 5);
        *(optop - 4) = *optop;
        *(optop - 5) = *(optop - 1);
        break;
      }
      
    case swap:
      {
	int32         i32Temp;
	i32Temp = *optop;
        *optop = *(optop - 1);
        *(optop - 1) = i32Temp;
	break;
      }
      
#include "vm/interp/arithmetic_and_logic.h"
#include "vm/interp/primitive_conversion.h"

#include "vm/interp/comparison_and_branches.h"

    case lookupswitch:
      {
	int32 i32Temp;
	int32 i;
	int32 i32DefaultOff;
	int32 i32NoPairs;
	byte* pbStartPC;
	
	/* remember PC at beginning of instruction */
	pbStartPC = pbPC - 1;
	
	/* skip 0 to 3 bytes of padding */
	while ((pbPC-pstCurrFrame->pbCode) % 4) {
	  pbPC++;
	}
	
	i32DefaultOff = (((int32)pbPC[0]) << 24) | (((int32)pbPC[1]) << 16) |
	  (((int32)pbPC[2]) << 8) | pbPC[3];
	i32NoPairs = (((int32)pbPC[4]) << 24) | (((int32)pbPC[5]) << 16) |
	  (((int32)pbPC[6]) << 8) | pbPC[7];
	pbPC += 8;
	
	for (i = 0; i < i32NoPairs; i++) {
	  /* read in number to compare */
	  i32Temp = (((int32)pbPC[0]) << 24) | (((int32)pbPC[1]) << 16) |
	    (((int32)pbPC[2]) << 8) | pbPC[3];
	  pbPC += 4;
	  
	  /* compare number to key */
	  if (i32Temp == *optop) {
	    /* re-use i32Temp and set it to the offset */
	    i32Temp = (((int32)pbPC[0]) << 24) | (((int32)pbPC[1]) << 16) |
	      (((int32)pbPC[2]) << 8) | pbPC[3];
	    optop--;
	    pbPC = pbStartPC + i32Temp;
	    goto labelLookupEnd;
	  }
	  else {
	    pbPC += 4;
	  }
	}
	
	/* will get here if there was no match */
	optop--;
	pbPC = pbStartPC + i32DefaultOff;
      labelLookupEnd:
	break;
      }

#include "vm/interp/method_return.h"
#include "vm/interp/get_and_put.h"
#include "vm/interp/method_invocation.h"
#include "vm/interp/optimised_method_invocation.h"

    case newfromname:
      {
	tClassLoaderTuple* pstClassTemp;
	tOBREF pstObjectTemp;
	char* pszAsciz;
	
	if (*optop) {
	  pszAsciz = INTERP_AscizFromString(env, (tOBREF) *optop);
	  
	  traceLoading("newfromname %s", pszAsciz);
	  
	  pstClassTemp = CLASSFILE_FindOrLoad(env, pszAsciz, 
					      pstCurrFrame->pstCurrClass);
	  sys_free(pszAsciz);
	  if (pstClassTemp) {
	    int failure;
	    pstObjectTemp = INTERP_NewObject(env, pstClassTemp, &failure);
	    if (failure) {
	      *optop = (int32) pstObjectTemp;
	      SAVEENV();
	      pstCurrFrame = CatchException(env, pstCurrFrame);
	      LOADENV();
	    }
	    else {
	      if (pstObjectTemp) {
		*optop = (int32) pstObjectTemp;
	      }
	      else {
		SAVEENV();
		pstCurrFrame = ThrowOutOfMemory(env, pstCurrFrame);
		LOADENV();
	      }
	    }
	  }
	  else {
	    SAVEENV();
	    pstCurrFrame = ThrowClassNotFound(env, pstCurrFrame);
	    LOADENV();
	  }
	}
	else {
	  SAVEENV();
	  pstCurrFrame = ThrowNullPointer(env, pstCurrFrame);
	  LOADENV();
	}
        break;
      }
      
    case new_:
      {
	int16 i16Temp;
	tClassLoaderTuple* pstClassTemp;
	tOBREF pstObjectTemp;
	
	i16Temp = (*pbPC++) * 256;
	i16Temp += (*pbPC++);
	
#ifdef DEBUG
	assert(CONSTTAG(pstCurrClass->pstClass, i16Temp) == CONSTANT_Class);
#endif
	
        pstClassTemp = CONSTGET_Class(env, pstCurrClass, i16Temp);
	if (pstClassTemp) {
#ifdef RESURRECT_QUICK
	  /* overwrite operation with new_quick */
	  pbPC[-3] = new_quick;
	  /* constant pool item already overwritten with resolved version */
	  /*          CONSTSET(env,pstCurrClass, i16Temp, pstClassTemp);*/
#endif
	  
	  {
	    int failure;
	    pstObjectTemp = INTERP_NewObject(env, pstClassTemp, &failure);
	    if (failure) {
	      *(++optop) = (int32) pstObjectTemp;
	      SAVEENV();
	      pstCurrFrame = CatchException(env, pstCurrFrame);
	      LOADENV();
	    }
	    else {
	      if (pstObjectTemp) {
		*(++optop) = (int32) pstObjectTemp;
	      }
	      else {
		SAVEENV();
		pstCurrFrame = ThrowOutOfMemory(env, pstCurrFrame);
		LOADENV();
	      }
	    }
	  }
	}
	else {
	  SAVEENV();
	  pstCurrFrame = ThrowClassNotFound(env, pstCurrFrame);
	  LOADENV();
	}
	break;
      }
    case new_quick:
      {
	int16 i16Temp;
	tClassLoaderTuple* pstClassTemp;
	tOBREF pstObjectTemp;
	
	i16Temp = (*pbPC++) * 256;
	i16Temp += (*pbPC++);
	
#ifdef DEBUG
	assert(CONSTTAG(pstCurrClass->pstClass, i16Temp) == CONSTANT_Class);
#endif
	
        pstClassTemp = (tClassLoaderTuple*) CONSTGET(pstCurrClass->pstClass,
						     i16Temp);
	{
	  int failure;
	  pstObjectTemp = INTERP_NewObject(env, pstClassTemp, &failure);
	  if (failure) {
	    *(++optop) = (int32) pstObjectTemp;
	    SAVEENV();
	    pstCurrFrame = CatchException(env, pstCurrFrame);
	    LOADENV();
	  }
	  else {
	    if (pstObjectTemp) {
	      *(++optop) = (int32) pstObjectTemp;
	    }
	    else {
	      SAVEENV();
	      pstCurrFrame = ThrowOutOfMemory(env, pstCurrFrame);
	      LOADENV();
	    }
	  }
	}
        break;
      }

    case newarray:
      {
	tARREF pstObjectTemp;

	//Stack holds number of elements to allocate
        if (*optop >= 0) { //extra byte holds type of element
          int failure;
          pstObjectTemp = INTERP_NewArray(env, *pbPC++, *optop, NULL,
					  pstCurrFrame->pstCurrClass,
					  &failure);
	  if (failure) {
	    *optop = (int32) pstObjectTemp;
	    SAVEENV();
	    pstCurrFrame = CatchException(env, pstCurrFrame);
	    LOADENV();
	  }
	  else {
	    if (pstObjectTemp) {
	      *optop = (int32) pstObjectTemp;
	    }
	    else {
	      SAVEENV();
	      pstCurrFrame = ThrowOutOfMemory(env, pstCurrFrame);
	      LOADENV();
	    }
	  }
        }
        else {
	  pbPC++; //just to make sure
          SAVEENV();
          pstCurrFrame = ThrowNegativeArraySize(env, pstCurrFrame);
          LOADENV();
        }
        break;
      }
      
    case anewarray:
      {
	int16 i16Temp;
	tClassLoaderTuple* pstClassTemp;
	
	i16Temp = (*pbPC++) * 256;
	i16Temp += (*pbPC++);
	
#ifdef DEBUG
	assert(CONSTTAG(pstCurrClass->pstClass, i16Temp) == CONSTANT_Class);
#endif
	
	if (*optop >= 0) {
	  pstClassTemp = CONSTGET_Class(env, pstCurrClass, i16Temp);
          if (pstClassTemp) {
	    tARREF pstObjectTemp;
	    int failure;
#ifdef QUICK
            /* overwrite operation with new_quick */
            pbPC[-3] = anewarray_quick;
            /* constant pool item already overwitten with resolved version */
	    /*            CONSTSET(pstCurrClass, i16Temp, pstClassTemp);*/
#endif
            pstObjectTemp = INTERP_NewArray(env, T_OBJECT, *optop, 
					    pstClassTemp->uidName, 
					    pstCurrFrame->pstCurrClass,
					    &failure);
	    if (failure) {
	      *optop = (int32) pstObjectTemp;
	      SAVEENV();
	      pstCurrFrame = CatchException(env, pstCurrFrame);
	      LOADENV();
	    }
	    else {
	      if (pstObjectTemp) {
		*optop = (int32) pstObjectTemp;
	      }
	      else {
		SAVEENV();
		pstCurrFrame = ThrowOutOfMemory(env, pstCurrFrame);
		LOADENV();
	      }
	    }
          }
	  else {
            SAVEENV();
            pstCurrFrame = ThrowClassNotFound(env, pstCurrFrame);
            LOADENV();
          }
        }
        else {
          SAVEENV();
          pstCurrFrame = ThrowNegativeArraySize(env, pstCurrFrame);
          LOADENV();
	}
        break;
      }
      
#ifdef QUICK
    case anewarray_quick: /* what if size = 0? */
      {
	int16 i16Temp;
	tClassLoaderTuple* pstClassTemp;
	
	i16Temp = (*pbPC++) * 256;
	i16Temp += (*pbPC++);
	
#ifdef DEBUG
	assert(CONSTTAG(pstCurrClass->pstClass, i16Temp) == CONSTANT_Class);
#endif
	
	if (*optop >= 0) {
	  pstClassTemp = (tClassLoaderTuple*) CONSTGET(env, pstCurrClass,
						       i16Temp);
          if (pstClassTemp) {
            int failure;
	    tARREF pstObjectTemp;

	    pstObjectTemp = INTERP_NewArray(env, T_OBJECT, *optop,
					    pstClassTemp->uidName, &failure);
	    if (failure) {
	      *optop = (int32) pstObjectTemp;
	      SAVEENV();
	      pstCurrFrame = CatchException(env, pstCurrFrame);
	      LOADENV();
	    }
	    else {
	      if (pstObjectTemp) {
		*optop = (int32) pstObjectTemp;
	      }
	      else {
		SAVEENV();
		pstCurrFrame = ThrowOutOfMemory(env, pstCurrFrame);
		LOADENV();
	      }
	    }
          }
	  else {
            SAVEENV();
            pstCurrFrame = ThrowClassNotFound(env, pstCurrFrame);
            LOADENV();
          }
        }
        else {
          SAVEENV();
          pstCurrFrame = ThrowNegativeArraySize(env, pstCurrFrame);
          LOADENV();
	}
	
        break;
      }
#endif /* QUICK */
      
    case arraylength:
      {
        if (*optop) {
          *optop = ADEREF((tARREF) *optop)->i32Number;
	  assert(*optop >= 0);
        }
        else {
          SAVEENV();
          pstCurrFrame = ThrowNullPointer(env, pstCurrFrame);
          LOADENV();
        }
        break;
      }
      
    case athrow:
      {
        if (pstCurrFrame->pstException || pstCurrFrame->bIsOrphan) {
#ifdef DEBUG
          assert (pstCurrFrame->pstPrevFrame == NULL);
#endif
	  if (pstCurrFrame->bIsOrphan) {
	    return (tOBREF) *optop; //The exception is on the stack?
	  }
	  else {
	    return pstCurrFrame->pstException;
	  }
        }
        if (*optop) {
	  tOBREF thrownObject = (tOBREF) *optop;
	  tClassLoaderTuple* pstThrowable = 
	    CLASSFILE_FindOrLoad(env, "java/lang/Throwable", NULL);
	  //Check that it is Throwable
	  if (INTERP_CheckCast(env, pstThrowable,
			       ODEREF(thrownObject)->pstType) == 1) {
	    SAVEENV();
	    pstCurrFrame = CatchException(env, pstCurrFrame);
	    LOADENV();
	  }
	  else {
	    SAVEENV();
	    pstCurrFrame = ThrowInternalError(env, pstCurrFrame,
					      "operand for 'athrow' instruction"
					      " is not a java.lang.Throwable");
	    LOADENV();
	  }
	}
        else {
	  SAVEENV();
	  INTERP_printBackTrace(pstCurrFrame);
	  pstCurrFrame = ThrowNullPointer(env, pstCurrFrame);
	  LOADENV();
	}
        break;
      }

    case checkcast:
      {
	int16 i16Temp;
	tClassLoaderTuple* pstClassTemp;
	tClassLoaderTuple* pstTempClass1;
	
        /* check if optop is NULL... a NULL object can be cast to anything */
        if (*optop) {
	  i16Temp = (*pbPC++) * 256;
	  i16Temp += (*pbPC++);
	  
#ifdef DEBUG
	  assert(CONSTTAG(pstCurrClass->pstClass, i16Temp) == CONSTANT_Class);
#endif
	  
	  pstTempClass1 = CONSTGET_Class(env, pstCurrClass, i16Temp);
	  
#ifdef QUICK
	  /* overwrite operation with checkcast_quick */
	  pbPC[-3] = checkcast_quick;
	  /* constant pool item is already overwritten with resolved version */
	  /*          CONSTSET(pstCurrClass, i16Temp, pstTempClass1);*/
#endif
	    
	  pstClassTemp = DEREF((tOBREF) *optop)->pstType;
	  
	  if (INTERP_CheckCast(env, pstTempClass1, pstClassTemp) == 0) {
	    optop--;
	    SAVEENV();
	    pstCurrFrame = ThrowClassCast(env, pstCurrFrame, 
					  pstClassTemp->uidName,
					  pstTempClass1->uidName);
	    LOADENV();
	  }
	}
	else {
	  pbPC += 2;  /* skip the two index bytes */
	}
	break;
      }
      
#ifdef QUICK
    case checkcast_quick:
      {
	int16 i16Temp;
	tClassLoaderTuple* pstClassTemp;
	tClassLoaderTuple* pstTempClass1;
	
        if (*optop) {
	  i16Temp = (*pbPC++) * 256;
	  i16Temp += (*pbPC++);
	  
#ifdef DEBUG
	  assert(CONSTTAG(pstCurrClass->pstClass, i16Temp) == CONSTANT_Class);
#endif
	  
	  pstTempClass1 = (tClass*) CONSTGET(env, pstCurrClass, i16Temp);
	  
	  //Jewel sep
	  pstClassTemp = DEREF(((tOBREF) *optop))->pstType;
	  if (INTERP_CheckCast(pstTempClass1, pstClassTemp) == 0) {
	    optop--;
	    SAVEENV();
	    // XXX - add missing(?) 'env' argument
	    pstCurrFrame = ThrowClassCast(env, pstCurrFrame, 
					  pstClassTemp->uidName,
					  pstTempClass1->uidName);
	    LOADENV();
	  }
	}
	else {
	  pbPC += 2;  /* skip the two index bytes */
	}
	break;
      }
#endif /* QUICK */

    case instanceof:
      {
	int16 i16Temp;
	uint16 u16Temp;
	tClassLoaderTuple* pstClassTemp;
        tClassLoaderTuple* pstTempClass1;
	
        /* check if optop is NULL... a NULL object CANNOT be an
           instance of anything */
	if (*optop) {
	  i16Temp = (*pbPC++) * 256;
          i16Temp += (*pbPC++);
	  
          pstTempClass1 = CONSTGET_Class(env,pstCurrClass, i16Temp);
	  
	  //I deleted the quick overwrite
	  
	  pstClassTemp = DEREF((tOBREF) *optop)->pstType;
          /* check if class can be cast */
	  if (INTERP_CheckCast(env, pstTempClass1, pstClassTemp)) {
	    *optop = 1;
	  }
	  else {
	    *optop = 0;
	  }
        }
        else {
	  //A null object is NOT an instance of ANYTHING!
	  *optop = 0;
          pbPC += 2;  /* skip the two index bytes */
        }
        break;
      }
      
      //I deleted the instanceof_quick
      //jewel April 2001
    case monitorenter:
      {
        if (*optop) {
          THREAD_SynchroniseEnter((tOBREF) *(optop--));
        }
        else {
	  // XXX - shouldn't we assign to pstCurrFrame??
	  INTERP_printBackTrace(pstCurrFrame);
          pstCurrFrame = ThrowNullPointer(env, pstCurrFrame);
        }
        break;
      }
      
    case monitorexit:
      {
        if (*optop) {
          THREAD_SynchroniseExit((tOBREF) *(optop--));
        }
        else {
	  // XXX - shouldn't we assign to pstCurrFrame??
          pstCurrFrame = ThrowNullPointer(env, pstCurrFrame);
        }
        break;
      }
      
    case wide:
      {
	switch(pbPC[0]) {
	case iinc:
	  {
	    int16 constant;
	    int offset;
	    
	    //we get a 16-bit unsigned offset
	    offset = (pbPC[1] << 8) | (pbPC[2]);
	    //and a signed 16-bit constant
	    
	    if (pbPC[3] & 0x80) {
	      constant = pbPC[3] - 256;
	    }
	    else {
	      constant = pbPC[3];
	    }
	    constant *= 256;
	    constant += pbPC[4];
	    
	    pi32Vars[offset] += constant;
	    pbPC += 5;
	    break;
	  }
	default:
	  {
	    nonfatalError("Unknown bytecode %i modified by 'wide'. "
			  "The kissme interpreter does not handle this yet.",
			  (int) pbPC[0]);
	    pstCurrFrame = 
	      ThrowInternalError(env, pstCurrFrame,
				 "Unknown bytecode modified by 'wide'"); 
	    }
	  }
	break;
      }
      
    case multianewarray:   
      {
	int16 i16Temp;
	int32 i32Temp;
	Uid uidBytes;
	int popoffstack = 0;
	
	//The type of reference
	i16Temp = (*pbPC++) * 256;
	i16Temp += (*pbPC++);
	
	//How many dimensions
	i32Temp = (*pbPC++);

	if (i32Temp <= 0) {
	  SAVEENV();
	  pstCurrFrame = ThrowInternalError(env, pstCurrFrame,
					    "multianewarray instruction: "
					    "dimensions operand is invalid");
	  LOADENV();	
	}
	else {
	  int failure;
	  tARREF pstObjectTemp;

	  popoffstack = i32Temp  - 1;
	  optop -= popoffstack; 
	  
	  uidBytes = CONSTGET_Class(env,pstCurrClass, i16Temp)->uidName;
	  pstObjectTemp = 
	    INTERP_MultiNewArray(env, optop, 0, i32Temp, uidBytes, 
				 pstCurrFrame->pstCurrClass, &failure);
	  if (failure) {
	    *optop = (int32) pstObjectTemp;
	    SAVEENV();
	    pstCurrFrame = CatchException(env, pstCurrFrame);
	    LOADENV();
	  }
	  else {
	    if (pstObjectTemp) {
	      *optop = (int32) pstObjectTemp;
	    }
	    else {
	      SAVEENV();
	      pstCurrFrame = ThrowOutOfMemory(env, pstCurrFrame);
	      LOADENV();
	    }
	  }
	}
	break;
      }
      
    /* we get here if we haven't found a valid match for the instruction */
    default:
      {
        nonfatalError("Unknown bytecode instruction encountered\n"
		      "Instruction number: %d\n"
		      "Current method: %s.%s%s\n"
		      "Program counter: %u\n", 
		      pbPC[-1],
		      pstCurrClass->uidName, 
		      pstCurrFrame->pstCurrMethod->uidName,
		      pstCurrFrame->pstCurrMethod->uidSignature,
		      pbPC - pstCurrFrame->pstCurrMethod->pstCode->pbCode - 1);

	StackInspect(NULL, NULL);
	pstCurrFrame = ThrowInternalError(env, pstCurrFrame,
					  "Unknown bytecode instruction");
      }
      
    }
    
    if (DIAG_ChannelState(OPSTACK_DIAG_CHANNEL) ||
	DIAG_ChannelState(FRAME_DIAG_CHANNEL))  {
      eprintf("Optop is at %x vars at %x\n", (int) optop, (int) pi32Vars);
      StackInspect(pstCurrFrame, optop);
    }
  }
}

